// PCSXTrace Code Tracer
// By Klarth (Steve Monaco)

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "Win32.h"
#include "resource.h"
#include "Misc.h"
#include "PsxInterpreter.h"
#include "Tracer.h"

int TraceType = TT_FILTERED;
int bIsTracing = FALSE;
int bShowRegValues = TRUE;
int bLogJalOnly = FALSE;
int bLogBiosCode = FALSE;
int bEnableRange = FALSE;
DWORD TraceRangeStart = 0x80000000;
DWORD TraceRangeEnd = 0x80000000;

FILE* TraceFile = NULL;
char* TraceLogBuf = NULL;
int TraceLogBufPos = 0;
const int TraceLogBufSize = 1024 * 1024 * 2; // 2MB buffer
int bTracedData = FALSE;
char TraceFileName[256];
char startbuf[64];
char endbuf[64];
BOOL IsInStartUpdate = FALSE;
BOOL IsInEndUpdate = FALSE;

void UpdateTracerDialog(HWND hDlg);
void SetCheckBoxState(HWND hButton, BOOL bChecked);
void NewTraceFile();
BOOL CALLBACK TracerDlgProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
DWORD WINAPI TracerWorkThread(LPVOID lpParam);
BOOL MakeValidHexString(char* Buffer);

HANDLE hTracerThread;
HWND hTracerDlg;

// GUI Functionality
BOOL CALLBACK TracerDlgProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	DWORD i;
	HWND hTemp;

	switch(uMsg)
	{
	case WM_CLOSE:
		ShowWindow(hDlg, SW_HIDE);
		CheckMenuItem(gApp.hMenu, ID_DEBUG_TRACE, MF_UNCHECKED);
		return TRUE;

	case WM_INITDIALOG:
		TraceLogBuf = (char*)malloc(TraceLogBufSize);
		hTemp = GetDlgItem(hDlg, IDC_TRACETYPE);
		i = ComboBox_AddString(hTemp, "Filtered Trace");
		i = ComboBox_AddString(hTemp, "Unfiltered Trace");
		i = ComboBox_AddString(hTemp, "Mark Code");

		SendDlgItemMessage(hDlg, IDC_TRACERANGESTART, EM_LIMITTEXT, 8, 0);
		SendDlgItemMessage(hDlg, IDC_TRACERANGEEND, EM_LIMITTEXT, 8, 0);
		UpdateTracerDialog(hDlg);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_TRACE:
			if(!bIsTracing) // Program is not currently tracing
			{
				if(TraceFile == NULL) // Open file for log
				{
					NewTraceFile();
					TraceLog("***** New Trace Started *****\n");
				}
				else
					TraceLog("\n\n***** New Trace Started *****");
			}
			bIsTracing ^= TRUE;
			UpdateTracerDialog(hDlg);
			if(!bIsTracing) // User stopped tracing
			{
				fwrite(TraceLogBuf, 1, TraceLogBufPos, TraceFile);
				fflush(TraceFile);
				TraceLogBufPos = 0;
			}
			return TRUE;
		case IDC_NEWTRACEFILE:
			NewTraceFile();
			return TRUE;
		case IDC_TRACETYPE:
			if(HIWORD(wParam) == CBN_SELCHANGE)
			{
				i = ComboBox_GetCurSel(GetDlgItem(hDlg, IDC_TRACETYPE));
				if(i == 0) TraceType = TT_FILTERED;
				else if(i == 1) TraceType = TT_UNFILTERED;
				else if(i == 2) TraceType = TT_MARKCODE;
				return TRUE;
			}
			break;
		case IDC_CLEARFILTER:
			clearFilter();
			return TRUE;
		case IDC_REGVAL:
			if(HIWORD(wParam) == BN_CLICKED)
				bShowRegValues ^= TRUE;
			return TRUE;
		case IDC_JAL:
			if(HIWORD(wParam) == BN_CLICKED)
				bLogJalOnly ^= TRUE;
			return TRUE;
		case IDC_LOGBIOS:
			if(HIWORD(wParam) == BN_CLICKED)
				bLogBiosCode ^= TRUE;
			return TRUE;
		case IDC_ENABLERANGE:
			if(HIWORD(wParam) == BN_CLICKED)
			{
				bEnableRange ^= TRUE;
				UpdateTracerDialog(hDlg);
			}
			return TRUE;
		case IDC_TRACERANGESTART:
			if(HIWORD(wParam) == EN_CHANGE && !IsInStartUpdate) // Make sure they're valid hex strings
			{
				GetDlgItemText(hDlg, IDC_TRACERANGESTART, startbuf, 64);
				if(!MakeValidHexString(startbuf))
				{
					TraceRangeStart = strtoul(startbuf, NULL, 16);
					sprintf(startbuf, "%X", TraceRangeStart);
					IsInStartUpdate = TRUE;
					SetDlgItemText(hDlg, IDC_TRACERANGESTART, startbuf);
					IsInStartUpdate = FALSE;
				}
				else
					TraceRangeStart = strtoul(startbuf, NULL, 16);
			}
			return TRUE;
		case IDC_TRACERANGEEND:
			if(HIWORD(wParam) == EN_CHANGE && !IsInEndUpdate)
			{
				GetDlgItemText(hDlg, IDC_TRACERANGEEND, endbuf, 64);
				if(!MakeValidHexString(endbuf))
				{
					TraceRangeEnd = strtoul(endbuf, NULL, 16);
					sprintf(endbuf, "%X", TraceRangeEnd);
					IsInEndUpdate = TRUE;
					SetDlgItemText(hDlg, IDC_TRACERANGEEND, endbuf);
					IsInEndUpdate = FALSE;
				}
				else
					TraceRangeEnd = strtoul(endbuf, NULL, 16);
			}
			return TRUE;
		}
	}
	return FALSE;
}

void UpdateTracerDialog(HWND hDlg)
{
	if(TraceType == TT_FILTERED)
		ComboBox_SetCurSel(GetDlgItem(hDlg, IDC_TRACETYPE), 0);
	else if(TraceType == TT_UNFILTERED)
		ComboBox_SetCurSel(GetDlgItem(hDlg, IDC_TRACETYPE), 1);
	else if(TraceType == TT_MARKCODE)
		ComboBox_SetCurSel(GetDlgItem(hDlg, IDC_TRACETYPE), 2);

	sprintf(startbuf, "%X", TraceRangeStart);
	IsInStartUpdate = TRUE;
	SetDlgItemText(hDlg, IDC_TRACERANGESTART, startbuf);
	IsInStartUpdate = FALSE;

	sprintf(endbuf, "%X", TraceRangeEnd);
	IsInEndUpdate = TRUE;
	SetDlgItemText(hDlg, IDC_TRACERANGEEND, endbuf);
	IsInEndUpdate = FALSE;

	if(bEnableRange)
	{
		SetCheckBoxState(GetDlgItem(hDlg, IDC_ENABLERANGE), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACERANGESTART), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACERANGEEND), TRUE);
	}
	else
	{
		SetCheckBoxState(GetDlgItem(hDlg, IDC_ENABLERANGE), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACERANGESTART), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACERANGEEND), FALSE);
	}

	if(bIsTracing)
	{
		SetDlgItemText(hDlg, IDC_TRACE, "Stop Trace");
		EnableWindow(GetDlgItem(hDlg, IDC_NEWTRACEFILE), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACETYPE), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_ENABLERANGE), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACERANGESTART), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACERANGEEND), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_REGVAL), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_JAL), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_LOGBIOS), FALSE);
		EnableWindow(GetDlgItem(hDlg, IDC_CLEARFILTER), FALSE);
	}
	else
	{
		SetDlgItemText(hDlg, IDC_TRACE, "Begin Trace");
		EnableWindow(GetDlgItem(hDlg, IDC_NEWTRACEFILE), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_TRACETYPE), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_ENABLERANGE), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_REGVAL), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_JAL), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_LOGBIOS), TRUE);
		EnableWindow(GetDlgItem(hDlg, IDC_CLEARFILTER), TRUE);
	}
	
	if(bShowRegValues)
		SetCheckBoxState(GetDlgItem(hDlg, IDC_REGVAL), TRUE);
	else
		SetCheckBoxState(GetDlgItem(hDlg, IDC_REGVAL), FALSE);

	if(bLogJalOnly)
		SetCheckBoxState(GetDlgItem(hDlg, IDC_JAL), TRUE);
	else
		SetCheckBoxState(GetDlgItem(hDlg, IDC_JAL), FALSE);

	if(bLogBiosCode)
		SetCheckBoxState(GetDlgItem(hDlg, IDC_LOGBIOS), TRUE);
	else
		SetCheckBoxState(GetDlgItem(hDlg, IDC_LOGBIOS), FALSE);
}

void SetCheckBoxState(HWND hButton, BOOL bChecked)
{
	if(bChecked)
		SendMessage(hButton, BM_SETCHECK, 1, 0);
	else
		SendMessage(hButton, BM_SETCHECK, 0, 0);
}

void TraceLog(char* FormatStr, ...)
{
	va_list args;
	int i;

	bTracedData = TRUE;

	va_start(args, FormatStr);
	i = vsprintf(TraceLogBuf+TraceLogBufPos, FormatStr, args);
	TraceLogBufPos += i;
	
	if(TraceLogBufPos + 1024 >= TraceLogBufSize) // 1024 chars left, about time to flush to the log
	{
		fwrite(TraceLogBuf, 1, TraceLogBufPos, TraceFile);
		fflush(TraceFile);
		TraceLogBufPos = 0;
	}
}

void NewTraceFile()
{
	time_t curtime;
	struct tm* timeinfo;
	char buf[80];

	if(bTracedData) // If it traced something...
	{
		if(TraceLogBufPos > 0) // Need to flush the buffer
		{
			fwrite(TraceLogBuf, 1, TraceLogBufPos, TraceFile);
			fflush(TraceFile);
			TraceLogBufPos = 0;
		}
		fclose(TraceFile);
		TraceFile = NULL;
	}

	bTracedData = FALSE;

	// Open file for log
	time(&curtime);
	timeinfo = localtime(&curtime);
	strftime(buf, 80, "%d%b%Y%H%M.%S", timeinfo);
	sprintf(TraceFileName, "%s %s.log", CdromLabel, buf);
	TraceFile = fopen(TraceFileName, "wt");
	if(TraceFile == NULL) // Open generically named log file
	{
		sprintf(TraceFileName, "trace%d.log", clock());
		TraceFile = fopen(TraceFileName, "wt");
	}
}

BOOL CreateTracerWindow(HWND hWnd)
{
	DWORD dwThread;

	hTracerThread = CreateThread(NULL, 0, TracerWorkThread, NULL, CREATE_SUSPENDED, &dwThread);
	Sleep(100);	
	ResumeThread(hTracerThread);

	return TRUE;
}

DWORD WINAPI TracerWorkThread(LPVOID lpParam)
{
	MSG msg;

	hTracerDlg = CreateDialog(gApp.hInstance, MAKEINTRESOURCE(IDD_TRACER), NULL, (DLGPROC)TracerDlgProc);

	if(hTracerDlg == NULL)
		return FALSE;

	ShowWindow(hTracerDlg, SW_SHOW);

	while(GetMessageA(&msg, NULL, 0, 0))
	{
		if (!IsDialogMessage(hTracerDlg, &msg) )
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}
	return msg.wParam;
}

BOOL MakeValidHexString(char* Buffer)
{
	BOOL retcode = TRUE;
	int len = 0, i = 0, j = 0;
	char buf2[64];
	len = strlen(Buffer);
	for(i = 0; i < len; i++)
	{
		if((Buffer[i] <= '9' && Buffer[i] >= '0') || (Buffer[i] <= 'F' && Buffer[i] >= 'A') 
			|| (Buffer[i] <= 'f' && Buffer[i] >= 'a'))
		{
			buf2[j] = Buffer[i];
			j++;
		}
		else
			retcode = FALSE;
	}

	strcpy(Buffer, buf2);
	return retcode;
}