/***************************************************************************

  gfxdecod.c

  Functions to convert the graphic information stored in the roms to a more
  usable format.

  Back in the early '80s, arcade machines didn't have the memory or the
  speed to handle bitmaps like we do today. They used "character maps",
  instead: they had one or more sets of characters (usually 8x8 pixels),
  which would be placed on the screen in order to form a picture. This was
  very fast: updating a character mapped display is, rougly speaking, 64
  times faster than updating an equivalent bitmap display, since you only
  modify groups of 8x8 pixels and not the single pixels. However, it was
  also much less versatile than a bitmap screen, since with only 256
  characters you had to do all of your graphics. To overcome this
  limitation, some special hardware graphics were used: "sprites". A sprite
  is essentially a bitmap, usually larger than a character, which can be
  placed anywhere on the screen (not limited to character boundaries) by
  just telling the hardware the coordinates. Moreover, sprites can be
  flipped along the major axis just by setting the appropriate bit. This
  saves precious memory, since you need only one copy of the image instead
  of four.

  What about colors? Well, the early machines had a limited palette (let's
  say 16 colors) and each character or sprite could not use all of them at
  the same time. Characters and sprites data would use 2 bits per pixel,
  allowing them to address four different colors. You then needed some way
  to tell to the hardware which, among the available colors, were the four
  colors. This was done using a "color attribute", which typically was an
  index for a lookup table.

  OK, after this brief and incomplete introduction, let's come to the
  purpose of this section: how to interpret the data which was stored in
  the graphic roms. Unfortunately, there is no easy answer: it depends on
  the hardware. The easiest way to find how data is encoded, is to start by
  making a bit by bit dump of the rom. You will usually be able to
  immediately recognize some pattern: if you are lucky, you will see
  letters and numbers right away, otherwise you will see something which
  looks like letters and numbers, but with halves switched, dilated, or
  something like that. You'll then have to find a way to put it all
  together to obtain our standard one byte per pixel representation. Two
  things to remember:
  - keep in mind that every pixel has typically two (or more) bits
    associated with it, and they are not necessarily near to each other.
  - characters might be rotated 90 degrees. That's because many games used a
    tube rotated 90 degrees. Think how your monitor would look like if you
	put it on its side ;-)

  After you have successfully decoded the characters, you have to decode
  the sprites. This is usually more difficult, because sprites are larger,
  maybe have more colors, and are more difficult to recognize when they are
  messed up, since they are pure graphics without letters and numbers.
  However, with some work you'll hopefully be able to work them out as
  well. As a rule of thumb, the sprites should be encoded in a way not too
  dissimilar from the characters.

***************************************************************************/

#include <string.h>
#include "gfxdecod.h"


/***************************************************************************

  Convert the character ROM.

  In Phoenix, characters are 8x8 and use two bitplanes. Data is organized
  in a fairly intuitive way: first the bitmap of the first plane, then of
  the second.

***************************************************************************/
void decodechars(unsigned char *src,unsigned char *dest)
{
	unsigned char bits;
        int x,y,offs,plane,chset;
	int c;

        /* 2 (charsets) * 256 (chars) * 8*8 (charsize) */
        memset(dest,0,2*256*8*8);  /* clear destination array */
        for (chset = 0;chset < 2;chset++)
                for (plane = 0;plane < 2;plane++)
                {
                        for (c = 0;c < 256;c++)
                        {
                                for (x = 0;x < 8;x++)
                                {
                                        bits = src[4096 * chset + 2048 * plane + 8 * c + x];

                                        for (y = 0;y < 8;y++)
                                        {
                                                offs = 8*8 * 256 * chset + 8*8 * c + 8 * (7-y) + (7-x);
                                                dest[offs] <<= 1;
                                                if (bits >= 128) dest[offs]++;
                                                bits <<= 1;
                                        }
                                }
                        }
                }
}
