//-------------------------------------------------------------------
//
//              VZEM - VZ200/VZ300 Color Computer Emulator
//               By Guy Thomason (intertek@powerup.com.au)
//                              
//
//                         **  Revision History  **
// 
//      Date           Description
//   ---------         -----------
//   12.07.2001        Rewrite using DJGPP compiler + Allegro
//   16.07.2001        Add sound emulation 
//   17.07.2001        Add debug menu (needs work)
//
//--------------------------------------------------------------------

#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <dir.h>
#include <process.h>
#include <conio.h>
#include "c:\djgpp\allegro\include\allegro.h"
#include "c:\dosvz\z80\z80.c"

#define SAMPLE_SIZE  221                // audio stream length (11025/50)


// globals used in emulation

Z80             Regs;                   // Z80 Registers
byte            vzfont[3072];           // VZ character font
byte            VZMEM[65535];           // VZ system memory
byte            BKMEM[4194304];         // Bank memory (4 meg!)
byte            vzkbrd[8];              // VZ keyboard array
byte            scanline[64];           // redraw flags
byte            vz_latch;               // copy of hardware latch
byte            debug = 0;              // debug screen
int             active_page = 0;        // bank memory page (0 - 255)
int             TRKSIZE_VZ = 0x9a0;     // tracksize  
int             redraw_screen;          // do we need to redraw whole screen
int             running;                // emulation active
int             spkr_cycles;            // cpu cycles per speaker toggle 
int             bpos;                   // current position in audio buffer
int             retrace;                // video retrace in progress
volatile int    timer = 0;              // interrupt timer
AUDIOSTREAM     *stream;                // sound stream
byte            audio_buffer[SAMPLE_SIZE];  // audio buffer


#include "c:\dosvz\pc_keys.h"           // pc keyboard mapping

int VZ_BLACK;     
int VZ_GREEN;     
int VZ_YELLOW;             
int VZ_BLUE;             
int VZ_RED;             
int VZ_BUFF;             
int VZ_CYAN;             
int VZ_MAGENTA;             
int VZ_ORANGE;             
int VZ_DK_GREEN;     
int VZ_BR_GREEN;     
int VZ_DK_ORANGE;     
int VZ_BR_ORANGE;     


#include "c:\dosvz\vzdio.c"             // disk emulation routines
#include "c:\dosvz\vzhw.c"              // hardware handlers
#include "c:\dosvz\debugger.c"          // debug menu



//------------------------------------------------------------------------
//   Function: Synch();
//   Purpose:  Interrupt handler that fires every 20ms. Use to synch video 
//------------------------------------------------------------------------

void Synch()
{
    timer = 1;
}
END_OF_FUNCTION(Synch);


//------------------------------------------------------------------------
//   Function: ScanKbrd();
//   Purpose:  Scan the PC keyboard and map to VZ keyboard array 
//------------------------------------------------------------------------

void ScanKbrd()
{
   byte k;
   int  n;

   for (n=0;n<8;n++) vzkbrd[n] = 0xFF;

   // keyboard line 0
   k = vzkbrd[0];
   if (key[KEY_R]) k = (k & 0xDF); 
   if (key[KEY_Q]) k = (k & 0xEF); 
   if (key[KEY_E]) k = (k & 0xF7); 
   if (key[KEY_W]) k = (k & 0xFD); 
   if (key[KEY_T]) k = (k & 0xFE); 
   vzkbrd[0] = k;

   // keyboard line 1
   k = vzkbrd[1];
   if (key[KEY_F]) k = (k & 0xDF); 
   if (key[KEY_A]) k = (k & 0xEF); 
   if (key[KEY_D]) k = (k & 0xF7); 
   if (key[KEY_LCONTROL]) k = (k & 0xFB); 
   if (key[KEY_S]) k = (k & 0xFD); 
   if (key[KEY_G]) k = (k & 0xFE); 
   vzkbrd[1] = k;

   // keyboard line 2
   k = vzkbrd[2];
   if (key[KEY_V]) k = (k & 0xDF); 
   if (key[KEY_Z]) k = (k & 0xEF); 
   if (key[KEY_C]) k = (k & 0xF7); 
   if (key[KEY_LSHIFT] ) k = (k & 0xFB); 
   if (key[KEY_X]) k = (k & 0xFD); 
   if (key[KEY_B]) k = (k & 0xFE); 
   vzkbrd[2] = k;

   // keyboard line 3
   k = vzkbrd[3];
   if (key[KEY_4]) k = (k & 0xDF); 
   if (key[KEY_1]) k = (k & 0xEF); 
   if (key[KEY_3]) k = (k & 0xF7); 
   if (key[KEY_2]) k = (k & 0xFD); 
   if (key[KEY_5]) k = (k & 0xFE); 
   vzkbrd[3] = k;

   // keyboard line 4
   k = vzkbrd[4];
   if (key[KEY_M]) k = (k & 0xDF); 
   if (key[KEY_SPACE]) k = (k & 0xEF); 
   if (key[KEY_COMMA]) k = (k & 0xF7); 
   if (key[KEY_STOP]) k = (k & 0xFD); 
   if (key[KEY_N]) k = (k & 0xFE); 
   vzkbrd[4] = k;

   // keyboard line 5
   k = vzkbrd[5];
   if (key[KEY_7]) k = (k & 0xDF); 
   if (key[KEY_0]) k = (k & 0xEF); 
   if (key[KEY_8]) k = (k & 0xF7); 
   if (key[KEY_MINUS]) k = (k & 0xFB); 
   if (key[KEY_9]) k = (k & 0xFD); 
   if (key[KEY_6]) k = (k & 0xFE); 
   vzkbrd[5] = k;

   // keyboard line 6
   k = vzkbrd[6];
   if (key[KEY_U]) k = (k & 0xDF); else k = (k | 0x20);
   if (key[KEY_P]) k = (k & 0xEF); else k = (k | 0x10);
   if (key[KEY_I]) k = (k & 0xF7); else k = (k | 0x08);
   if (key[KEY_ENTER]) k = (k & 0xFB); else k = (k | 0x04);
   if (key[KEY_O]) k = (k & 0xFD); else k = (k | 0x02);
   if (key[KEY_Y]) k = (k & 0xFE); else k = (k | 0x01);
   vzkbrd[6] = k;

   // keyboard line 7
   k = vzkbrd[7];
   if (key[KEY_J]) k = (k & 0xDF); 
   if (key[KEY_COLON]) k = (k & 0xEF); 
   if (key[KEY_K]) k = (k & 0xF7); 
   if (key[KEY_QUOTE]) k = (k & 0xFB); 
   if (key[KEY_L]) k = (k & 0xFD); 
   if (key[KEY_H]) k = (k & 0xFE); 
   vzkbrd[7] = k;

   // extra keys
   if ((key[KEY_LEFT]) || (key[KEY_BACKSPACE])) 
   {
	vzkbrd[4] &= 0xDF;              // press M
	vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (key[KEY_RIGHT])     
   {
       vzkbrd[4] &= 0xF7;              // press comma
       vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (key[KEY_UP])        
   {
       vzkbrd[4] &= 0xFD;              // press period
       vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (key[KEY_DOWN])      
   {
       vzkbrd[4] &= 0xEF;              // press space
       vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (key[KEY_INSERT])    
   {
       vzkbrd[7] &= 0xFD;              // press L
       vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (key[KEY_DEL])       
   {
	vzkbrd[7] &= 0xEF;              // press semicolon
	vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (key[KEY_END])       
   {
	vzkbrd[7] &= 0xFB;              // press apostrophe
	vzkbrd[1] &= 0xFB;          // press ctrl
   }

   //
   // Esc to exit emulation
   //
   if (key[KEY_ESC]) running = 0;

   //
   // F10 key for debug menu
   //
   if (key[KEY_F10]) debug = 1;
}

//------------------------------------------------------------------------
//   Function: InstallGfx();
//   Purpose:  Preload the VZ gfx for fast blitting 
//------------------------------------------------------------------------

void InstallGfx()
{
   int             c,offs, x,y,px,py,ax,ay;
   byte            fb, mask, r;
   int             pcol;           

   //
   // Fill the buffer with black. Offscreen memory starts at offset 240        
   //
   rectfill(screen,0,240,319,479,0);
	
   //
   // Preload VZ Text characters
   //

   for (y=0;y<8;y++)
   {
       for (x=0;x<32;x++)
       {
	   c = 32*y+x;                    // total of 256 characters in font
	   if (c > 128) pcol = VZ_GREEN;           // green
	   if (c > 144) pcol = VZ_YELLOW;          // yellow
	   if (c > 160) pcol = VZ_BLUE;            // blue
	   if (c > 176) pcol = VZ_RED;             // red
	   if (c > 192) pcol = VZ_BUFF;            // buff
	   if (c > 208) pcol = VZ_CYAN;            // cyan
	   if (c > 224) pcol = VZ_MAGENTA;         // magenta
	   if (c > 240) pcol = VZ_ORANGE;          // orange

	   offs = c * 12;  // start of character font
	   for (py = 0; py < 12; py++)
	   {
	       fb = vzfont[offs+py];
	       mask = 1;
	       for (px = 0; px < 8; px++)
	       {
		   if (fb & mask)
		   {
		       if (c <= 128) 
		       {
			   putpixel(screen,x*8+px,240+y*12+py,VZ_BR_GREEN);               
			   putpixel(screen,x*8+px,240+96+y*12+py,VZ_BR_ORANGE);
		       }
		       else
		       {
			   putpixel(screen,x*8+px,240+y*12+py,pcol);              
			   putpixel(screen,x*8+px,240+96+y*12+py,pcol);
		       }
		    }
		    else
		    {
			putpixel(screen,x*8+px,240+y*12+py,VZ_DK_GREEN);
			putpixel(screen,x*8+px,240+96+y*12+py,VZ_DK_GREEN);
		    }
		    mask = mask * 2;
	       }
	   }
	}
    }


    // Preload VZ Graphics characters
    // 128x64 to 256x192 = 2*3 pixel ratio

    fb = 0;
    for (y=0;y<8;y++)
    {
	for (x=0;x<32;x++)
	{
	    r = 6;
	    mask = 0xC0;                    // 11000000 binary
	    for (px = 0; px < 4; px++)
	    {
		if (((fb & mask) >> r) == 0x00) pcol = VZ_GREEN; 
		if (((fb & mask) >> r) == 0x01) pcol = VZ_YELLOW;
		if (((fb & mask) >> r) == 0x02) pcol = VZ_BLUE;
		if (((fb & mask) >> r) == 0x03) pcol = VZ_RED;  

		for (ax=0;ax<2;ax++)
		    for (ay=0;ay<3;ay++)
		    {
			putpixel(screen,x*8+px*2+ax,240+192+y*3+ay,pcol);
		    }
		r -= 2;
		mask = (mask >> 2);
	    }
	    fb++;
	}
    }

    fb = 0;
    for (y=0;y<8;y++)
    {
	for (x=0;x<32;x++)
	{
	    r = 6;
	    mask = 0xC0;                    // 11000000 binary
	    for (px = 0; px < 4; px++)
	    {
		if (((fb & mask) >> r) == 0x00) pcol = VZ_BUFF; 
		if (((fb & mask) >> r) == 0x01) pcol = VZ_CYAN;
		if (((fb & mask) >> r) == 0x02) pcol = VZ_ORANGE;
		if (((fb & mask) >> r) == 0x03) pcol = VZ_MAGENTA;  

		for (ax=0;ax<2;ax++)
		    for (ay=0;ay<3;ay++)
		    {
			putpixel(screen,x*8+px*2+ax,240+216+y*3+ay,pcol);
		    }
		r -= 2;
		mask = (mask >> 2);
	    }
	    fb++;
	}
     }
}

void LoadVZFile(char *s)
{
    FILE *vzfile;
    unsigned short  n,start_addr;
    byte b;

    typedef struct vzFile
    {
	byte    vzmagic[4];
	byte    filename[17];
	byte    ftype;
	byte    start_addrl;
	byte    start_addrh;
    } VZFILE;


    VZFILE vzf;

    vzfile = fopen(s,"rb");
    if (!vzfile)
    {
	exit(1);
    }
    
    
    // read the vzheader
    
    fread(&vzf,sizeof(vzf),1,vzfile);
	start_addr = vzf.start_addrh; 
	start_addr = start_addr * 256 + vzf.start_addrl;
	

    n=0;
    fread(&b,1,1,vzfile);
    while (!feof(vzfile))
    {
	   VZMEM[start_addr+n] = b;
	   n++;
	   fread(&b,1,1,vzfile);
    }
    fclose(vzfile);
    
    //
    // Basic programs cannot be loaded at startup because the basic pointers are
    // overwritten when the rom is loaded
    //
    if (vzf.ftype == 0xF0)
    {
	VZMEM[0x78A4] = vzf.start_addrl;
	VZMEM[0x78A5] = vzf.start_addrh;
	VZMEM[0x78F9] = ((start_addr+n) & 0x00FF);
	VZMEM[0x78FA] = (((start_addr+n) & 0xFF00) >> 8);
    }


    if (vzf.ftype == 0xF1)
    {
	   Regs.PC.B.l = vzf.start_addrl;
	   Regs.PC.B.h = vzf.start_addrh;
    }
}


//----------------------------------------------------------------
//   Function: InitVZ();
//   Purpose:  Load the vz rom, install the font, set up colours 
//----------------------------------------------------------------

void InitVZ()
{
    FILE    *vzrom;
    FILE    *vzfnt;
    long    n;
	
    //
    // Set up the Z80
    //
    RAM = &VZMEM[0];
    Regs.IPeriod = 71590;               // 3.58 mhz
    ResetZ80(&Regs);

    //
    // Load the VZ rom
    //
    vzrom = fopen("vzrom.v20","rb");
    if (!vzrom)
    {
       exit(1);
    }
    for (n=0;n<16384;n++)
    {
	VZMEM[n] = fgetc(vzrom);
    }
    fclose(vzrom);

    //
    // Load the VZ dos rom if present
    //
    vzrom = fopen("vzdos.rom","rb");
    if (vzrom)
    {
       for (n=0;n<8192;n++)
       {
	    VZMEM[16384+n] = fgetc(vzrom);
       }
       fclose(vzrom);
    }
    

    //
    // Load the VZ font
    //
    vzfnt = fopen("vz200.fnt","rb");
    if (!vzfnt)
    {
	exit(1);
    }

    for (n=0;n<3072;n++)
    {
	   vzfont[n] = fgetc(vzfnt);
    }
    fclose(vzfnt);

    //
    // Setup the VZ colors - chop from 24 to 8 bit
    //
    VZ_BLACK        = makecol8(0,0,0);
    VZ_GREEN        = makecol8(0,255,0);     
    VZ_YELLOW       = makecol8(255,255,0);             
    VZ_BLUE         = makecol8(0,0,255);             
    VZ_RED          = makecol8(255,0,0);             
    VZ_BUFF         = makecol8(224,224,144);             
    VZ_CYAN         = makecol8(0,192,160);             
    VZ_MAGENTA      = makecol8(255,0,255);             
    VZ_ORANGE       = makecol8(240,112,0);             
    VZ_DK_GREEN     = makecol8(0,64,0);     
    VZ_BR_GREEN     = makecol8(0,224,24);     
    VZ_DK_ORANGE    = makecol8(64,16,0);     
    VZ_BR_ORANGE    = makecol8(255,196,24);     

    InstallGfx();

    //
    // Initial state - redraw all lines
    //
    for (n=0;n<64;n++) scanline[n] = 1;
	
}


//----------------------------------------------------------------
//   Function: DrawMode0();
//   Purpose:  Paint the VZ text mode
//----------------------------------------------------------------

void DrawMode0()
{
   int     x,y;
   int     srcx, srcy, destx, desty;
   int     foffs, offs, row, col;
   byte    ch;

   // redraw black border if coming back from mode 1
   if (redraw_screen)
   {
	   rectfill(screen,0,0,319,239,0);
   }

   // determine background color from latch
   if (vz_latch & 0x10) 
	foffs = 96;                                     // font with orange background starts here
   else
	foffs = 0;                                      // font with green background starts here

   for (y=0;y<16;y++)
   {
      if ((scanline[y]) || (redraw_screen))
      {
	     for (x=0;x<32;x++)
		 {
		offs = 32 * y + x;
		ch = RAM[28672+offs];
		row = ch; row = row / 32;
		col = ch; col = col - row * 32;
		srcx  = col*8;  srcy = 240+foffs+row*12;
		destx = 32+x*8; desty = 24+y*12;
		blit(screen,screen,srcx,srcy,destx,desty,8,12);
		 }
	     scanline[y] = 0;
      }
   }
   redraw_screen = 0;
}


//----------------------------------------------------------------
//   Function: DrawMode1();
//   Purpose:  Paint the VZ graphics mode
//----------------------------------------------------------------

void DrawMode1()
{
   int     x,y;
   int     srcx, srcy, destx, desty;
   int     foffs, offs, row, col;
   int     bgcolor;
   byte    ch;

   // determine background color from latch
   if (vz_latch & 0x10) 
   {
       foffs = 216;                            // buff background
       bgcolor = VZ_BUFF;
   }
   else
   {
       foffs = 192;                            // green background 
       bgcolor = VZ_GREEN;
   }

   // remove black border if coming from mode 0
   if (redraw_screen)
   {
      rectfill(screen,0,0,319,239,bgcolor); 
   }
   
   for (y=0;y<64;y++)
   {
      if ((scanline[y]) || (redraw_screen))
      {
	     for (x=0;x<32;x++)
		 {
		offs = 32 * y + x;
		ch = RAM[28672+offs];
		row = ch; row = row / 32;
		col = ch; col = col - row * 32;
		srcx  = col*8;  srcy = 240+foffs+row*3;
		destx = 32+x*8; desty = 24+y*3;
		blit(screen,screen,srcx,srcy,destx,desty,8,3);
		 }
	     scanline[y] = 0;
      }
   }
   redraw_screen = 0;
}




//----------------------------------------------------------------
//   Function: DrawVZScreen();
//   Purpose:  Duh
//----------------------------------------------------------------

void DrawVZScreen()
{
    if (vz_latch & 0x08)                            // graphics mode
	  DrawMode1();
    else
	  DrawMode0();                              // text mode
}


//----------------------------------------------------------------
//   Function: PlaySound();
//   Purpose:  Check if the sound needs updating and if so copies  
//             from the buffer    
//----------------------------------------------------------------

void PlaySound()
{
    static int spos = 0;
    byte *p;
    int n;

    p = get_audio_stream_buffer(stream);
    if (p)                                // finished playing, need more samples
    {
	for (n=0;n<SAMPLE_SIZE;n++)
	{
            p[n] = audio_buffer[n];
            audio_buffer[n] = 127;
	}
        free_audio_stream_buffer(stream);
        spkr_cycles = Regs.IPeriod;
        bpos = 0;
    }
}



//----------------------------------------------------------------
//   Function: EmulateVZ();
//   Purpose:  Run the Z80, check VZ hardware, draw the screen
//----------------------------------------------------------------

void EmulateVZ()
{
    word PC;

    PC = RunZ80(&Regs);                     // execute Z80 till next interrupt
    ScanKbrd();                             // Get key presses/releases                                      
    PlaySound();
    while (timer == 0);                     // wait for right timing
    retrace == 1;                           // video retrace
    timer = 0;                              // reset timer (set in interrupt handler)
    DrawVZScreen();                         // Paint it up
}

int main(int nArgs, char *chArgs[])
{
    word PC;
    int i;

    allegro_init();
    install_keyboard();
    install_timer();

    /* install sound driver */
    if (install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) != 0) 
    {
       printf("Error initialising sound system\n", allegro_error);
       return 1;
    }

    /* create an audio stream */
    stream = play_audio_stream(SAMPLE_SIZE, 8, FALSE, 11025, 255, 128);
    if (!stream)
    {
      printf("Error creating audio stream!\n");
      return 1;
    }

    if (set_gfx_mode(GFX_MODEX, 320, 240, 320, 480) != 0)
    {
	printf(" ! Fatal error setting video mode !\n");
	return 1;
    }
	
    LOCK_VARIABLE(timer);
    LOCK_FUNCTION(Synch);
    install_int(Synch,20);                  // 20ms = 50hz

    InitVZ();
    
    if (nArgs > 1)
    {
	// check switch: -f = file, -d = disk
	for (i=1;i<nArgs;i++)
	{
	    if (!strcmp(chArgs[i],"-f")) LoadVZFile(chArgs[i+1]);
	    if (!strcmp(chArgs[i],"-d")) vtech1_floppy_init(0,chArgs[i+1]);
            if (!strcmp(chArgs[i],"-t")) TRKSIZE_VZ = 0x9b0;
	}
    }
 
    running = 1;
    while (running)
    {
	EmulateVZ();
        if (debug) debugger(&Regs);
    }
    stop_audio_stream(stream);
    allegro_exit();
    return(0);
}

END_OF_MAIN();
