import java.applet.*;
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import java.io.*;
import java.util.*;

/**********************************************************************************
 *                   Microtan65 class by Fabrice Frances                          *
 *                                                                                *
 *  This simple computer extends the R6502 class and implements the behavior of a *
 * simple Microtan65 system (no VIA or ACIA on the TANEX board).                  *
 * This is really intended as a demonstration of use of the R6502 class, note how *
 * the MemRead, MemWrite and Synchronize methods are implemented to accurately    *
 * emulate the 8-cycles Microtan65 delayed NMI. Microtan's TANBUG monitor uses    *
 * this hardware to provide a single-step debugging feature.                      *
 *                                                                                *
 * Feel free to extend this emulator, just credit this work if you do.            *
 **********************************************************************************/

final class Microtan_board extends R6502 {
      byte keyboard_port=0;
	boolean keyboard_flag=false;
      boolean request_screen_repaint=false;
      byte[] ram;
      boolean[] graphics_bits=new boolean[1024];
      boolean[] dirty_graphics=new boolean[512];
      byte[] rom=new byte[16*1024];
      int RomMask,ramSize;
      boolean graphic_switch=false;
      int delayed_nmi_counter=0;
      Image[] mosaic_characters=new Image[256];
      Image[] charset=new Image[128];
      Container microtanContainer;
      Graphics screen;
      Color background, foreground;

      public Microtan_board(Container panel, Color b, Color f, InputStream charsetStream, int ramSize, InputStream romStream) throws IOException {
        super(750,50);   // R6502 runnning at 750 kHz, let's refresh screen every 50 ms.
        microtanContainer=panel;
        background=b; foreground=f;
        screen=panel.getGraphics();
        draw_mosaic_characters();
        draw_ascii_characters(charsetStream);
        if (ramSize<1) ramSize=1;
        if (ramSize>=48) ramSize=47; // keep 1K address space for I/O
        ram=new byte[ramSize*1024];
        this.ramSize = ramSize*1024;
        Random generator=new Random();
        generator.nextBytes(ram);
        for (int i=0;i<1024;i++) graphics_bits[i]=(generator.nextInt()&1)==0;   // nextBoolean() absent sur IExplorer!
        for (int i=0;i<512;i++) dirty_graphics[i]=true;
        int romSize=romStream.read(rom); // rom must be contiguous
        RomMask=romSize-1;   // rom size must be a power of 2 (i.e 1KB, 2KB, 4KB, 8KB, 16KB)
      }

      public void synchronize() {
        super.synchronize();
        screen_refresh();
      }

      public byte MemRead(int address) {
        cycles++;
        if (delayed_nmi_counter!=0) {
          delayed_nmi_counter--;
          if (delayed_nmi_counter==0) NMI();
        }
        if (address<ramSize) return ram[address];
        if (address<0xBFC0) return -1; // beware ! it should be 1K I/O space, but
						// then I/O should not be partially decoded like below
						// (otherwise a funny side-effect of the precise 6502 emulator is
						// a LDA BFDE,Y with Y>20 may cause dummy read to the graphic switch
						// This LDA BFDE,Y occurs in Basic65 when computing the address of functions !)
        if (address<0xC000) {
            byte val= -1;
            switch (address & 3) {
              case 0: graphic_switch=true;
              case 1: 
              case 2: break;
              case 3: val=(byte)(keyboard_port | (keyboard_flag ? 0x80 : 0));
            }
            return val;
        }
        return rom[address & RomMask];
      }
    
      public void MemWrite(int address, byte val) {
          cycles++;
          if (delayed_nmi_counter!=0) {
              delayed_nmi_counter--;
              if (delayed_nmi_counter==0) NMI();
          }
          if (address<ramSize) {
              if ((address & 0xFE00)==0x200) write_screen(address,val);
              else ram[address]=val;
              return;
          }
          if (address<0xBC00) return;
          if (address<0xC000) {
            switch (address & 3) {
              case 0: keyboard_flag=false; IRQ(false); break;
              case 1: delayed_nmi_counter=8; break;
              case 2: break; // write keypad pattern
              case 3: graphic_switch=false; break;
            }
          }
      }

      public void dumpMem(OutputStream s) throws IOException { 
		s.write(ram); 
		byte[] bitram = new byte[64];
		for (int i=0;i<64;i++)
			for (int bit=0;bit<8;bit++)
				if (graphics_bits[0x200+i*8+bit])
					bitram[i] += 1 << bit;
		s.write(bitram);
	}

      public void loadMem(InputStream s) throws IOException { 
		s.read(ram);
		byte[] bitram = new byte[64];
		s.read(bitram);
		for (int i=0;i<64;i++)
			for (int bit=0;bit<8;bit++)
				graphics_bits[0x200+i*8+bit]= (bitram[i] & (1<<bit))!=0;
      }

    void write_screen(int address, byte val) {
       if (graphic_switch!=graphics_bits[address] || ram[address]!=val)
         dirty_graphics[address&0x1FF]=true;
       ram[address]=val;
       graphics_bits[address]=graphic_switch;
    }

    public void paint() { request_screen_repaint=true; }

    void screen_refresh() {
      if (request_screen_repaint) {
        for (int address=0x1FF;address>=0;address--) dirty_graphics[address]=true;
        request_screen_repaint=false;
      }
      int address=0;
      for (int y=0; y<256; y+=16)
        for (int x=0; x<256; x+=8) {
          if (dirty_graphics[address]) {
            dirty_graphics[address]=false;
            if (graphics_bits[address+0x200])
              screen.drawImage(mosaic_characters[ram[address+0x200]&0xFF],x,y,null);
            else
              screen.drawImage(charset[ram[address+0x200]&0x7F],x,y,null);
          }
          address++;
        }
    }

    void keyboard_entry(byte c) {
        keyboard_port=c;
        keyboard_flag=true;
        IRQ(true);
    }

    void draw_mosaic_characters() {
        for(int i=0;i<256;i++) {
            mosaic_characters[i]=microtanContainer.createImage(8,16);
            Graphics g=mosaic_characters[i].getGraphics();
            for (int y=0;y<4;y++)
                for (int x=0;x<2;x++) {
                    int b=y*2+x;
                    if ( (i&(1<<b)) !=0 ) g.setColor(foreground);
                    else g.setColor(background);
                    g.fillRect(x*4,y*4,4,4);
                }
        }
    }

    void draw_ascii_characters(InputStream s) throws IOException {
        int adr=0;
        byte[] video_prom = new byte[2048];
        int charsetSize=s.read(video_prom)/16;
        for (int code=0;code<128;code++) {
            charset[code]=microtanContainer.createImage(8,16);
            Graphics g=charset[code].getGraphics();
            if (charsetSize<128)    // no lower-case kit: we only have codes 0x20-0x5F
                switch (code/32) {
                    case 0: case 2: adr = (code & 0x1f | 0x20)*16; break;
                    case 1: case 3: adr = (code & 0x1f)*16; break;
                }
            else adr = code * 16;
            for (int line=0;line<16;line++)
                for (int col=0;col<8;col++) {
                    int bit=video_prom[adr+line]&(1<<(7-col));
                    if (bit!=0) g.setColor(foreground);
                    else g.setColor(background);
                    g.fillRect(col,line,1,1);
                }
        }
    }

} // class Microtan_Board
