import java.io.*;

/**********************************************************************************
 *                   NMOS 6502 emulator by Fabrice Frances                        *
 *                                                                                *
 *  This R6502 class behaves like a real 6502 cpu from a black-box point of view: *
 * - every memory read|write cycle is present and accurate (read|write cycles are *
 *   based on Marko Makela's instruction timing), so precise driving of I/O is    *
 *   possible.                                                                    *
 * - as a nice side effect, synchronizing virtual emulation time with real time is*
 *   just a matter of counting read|write cycles.                                 *
 * - an example of use of this cycle-based emulator is provided with the microtan *
 *   emulator, which features the emulation of Microtan65's delayed NMI occuring  *
 *   exactly 8 cycles (3 flip-flops) after the activation of an I/O port. Microtan*
 *   implements a single-step debugging feature thanks to this delayed NMI.       *
 *                                                                                *
 * You are free to use it in your own emulators, just credit me if you do or if   *
 * you start your 6502 emulator from this work.                                   *
 *                                                                                *
 * How to use this class: you can either use it                                   *
 * - as is, since it provides its own 64K memory space. Build a R6502 object by   *
 *   providing the frequency of the cpu (in kHz) and how often you want the cpu   *
 *   synchronized with real time (in milliseconds). Initialize memory with public *
 *   method 'MemWrite'. Then let the cpu start using public method 'run'. You will*
 *   have to provide at least another thread if you want to see what happens, and *
 *   interact with the cpu with the public methods 'IRQ','NMI','SO','RESET'.      *
 * - or inherit from this class (the prefered way), and rewrite the 'MemRead' and *
 *   'MemWrite' methods to provide connection to your RAM/ROM/I-O addresses. Also,*
 *   rewrite 'synchronize' if you want to do something at every virtual time/real *
 *   time synchronization (a video refresh for example). Don't forget to increment*
 *   field 'cycles' in MemRead and MemWrite methods, and to call the synchronize  *
 *   method of this parent class.                                                 *
 *                                                                                *
 * Things I should add soon:                                                      *
 * - emulation of unofficial opcodes                                              *
 * - behavior of a NMI interrupting a BRK                                         *
 * - verify times when an interrupt is not handled before next instruction        *
 * - anything suggested... 8-)                                                    *
 **********************************************************************************/

class R6502 {

int           PSW;
boolean       N, D, I, Z, C;
volatile boolean V;
int           PC;
int           A, X, Y, S;
int           EA;
volatile boolean IRQ=false,NMI=false;
byte[]        state = new byte[7];
byte[]        Mem=new byte[65536];
int           synchroTimeMillis;
int           cyclesBeforeSynchro;
int           cycles;

public R6502(int freq, int synchroMillis) {
  cyclesBeforeSynchro=synchroMillis*freq;
  synchroTimeMillis=synchroMillis;
}

public byte MemRead(int adr) { cycles++; return Mem[adr]; }
public void MemWrite(int adr, byte value) { cycles++; Mem[adr]=value; }

/**********************************************************************************
 ********************************* Utility functions ******************************
 **********************************************************************************/
int  MemWord(int adr) { return (MemRead(adr) & 0xFF) + ((MemRead(adr+1) & 0xFF) << 8); }

void SetPSWNZ(byte val)  { N = (val<0); Z = (val==0); }
void SetFlagCarry(int val) { C = (val&0x100)!=0; }
void SetFlagBorrow(int val){ C = (val&0x100)==0; }

void CompressPSW() { 
    PSW = (N ? 0x80 : 0)
          | (V ? 0x40 : 0)
          | 0x30
          | (D ? 8 : 0)
          | (I ? 4 : 0)
          | (Z ? 2 : 0)
          | (C ? 1 : 0);
}

void UncompressPSW() {
    N = (PSW & 0x80)!=0;
    V = (PSW & 0x40)!=0;
    D = (PSW & 0x08)!=0;
    I = (PSW & 0x04)!=0;
    Z = (PSW & 0x02)!=0;
    C = (PSW & 0x01)!=0;
}

void PushPC() {
    MemWrite(S+0x100,(byte)(PC >> 8));
    S = (S-1) & 0xFF;
    MemWrite(S+0x100,(byte)PC);
    S = (S-1) & 0xFF;
}

void PopPC() {
    S = (S+1) & 0xFF;
    PC = MemRead(S+0x100) & 0xFF;
    S = (S+1) & 0xFF;
    PC += (MemRead(S+0x100) & 0xFF) << 8;
}

/**********************************************************************************
 ********************************** Addressing modes ******************************
 **********************************************************************************/

void Imp()   { MemRead(PC); } // dummy read
void Imm()   { EA = PC++; }
void Zero()  { EA = MemRead(PC++) & 0xFF; }
void ZeroX() { EA = (MemRead(PC++) + X) & 0xFF; } // dummy read
void ZeroY() { EA = (MemRead(PC++) + Y) & 0xFF; } // dummy read
void Abs()   { EA = MemWord(PC); PC+=2; }
void AbsX()  { 
	int EAL = (MemRead(PC++) & 0xFF) + X;
	int EAH = (MemRead(PC++) & 0xFF) << 8;
      if (EAL >= 0x100) MemRead(EAH + (EAL & 0xFF)); // dummy read, while address is fixed up
	EA = (EAH + EAL) & 0xFFFF; }
void AbsY()  { 
	int EAL = (MemRead(PC++) & 0xFF) + Y;
	int EAH = (MemRead(PC++) & 0xFF) << 8;
      if (EAL >= 0x100) MemRead(EAH + (EAL & 0xFF)); // dummy read, while address is fixed up
	EA = (EAH + EAL) & 0xFFFF; }
void Ind()   { 
	int PTRL = MemRead(PC++) & 0xFF; 
	int PTRH = (MemRead(PC++) & 0xFF) << 8; 
	EA = MemRead(PTRH+PTRL) & 0xFF; 
	PTRL = (PTRL+1) & 0xFF;	// NMOS 6502 bug
	EA += (MemRead(PTRH+PTRL) & 0xFF) << 8; }
void IndX()  { 
	int PTR = (X + MemRead(PC++)) & 0xFF; // we don't read dummy addresses in Zero page, no I/O there
	EA = MemRead(PTR);
      EA += (MemRead((PTR+1)&0xFF)&0xFF) << 8; }
void IndY()  { 
	int PTR = MemRead(PC++) & 0xFF; 
	int EAL = (MemRead(PTR) & 0xFF) + Y; 
	int EAH = (MemRead((PTR+1)&0xFF) & 0xFF) << 8;
	if (EAL >= 0x100) MemRead(EAH + (EAL & 0xFF)); // dummy read, while address is fixed up
	EA = (EAH + EAL) & 0xFFFF; }
void Rel()   { EA = MemRead(PC++); EA = (EA+PC) & 0xFFFF; } // added cycles only if branch taken

void WAbsX()  { // this is still AbsX mode, but a dummy read always occurs in a write or modify instruction
	int EAL = (MemRead(PC++) & 0xFF) + X;
	int EAH = (MemRead(PC++) & 0xFF) << 8;
	MemRead(EAH + (EAL & 0xFF)); // dummy read, while address is fixed up
	EA = (EAH + EAL) & 0xFFFF; }
void WAbsY()  { // this is still AbsY mode, but a dummy read always occurs in a write or modify instruction
	int EAL = (MemRead(PC++) & 0xFF) + Y;
	int EAH = (MemRead(PC++) & 0xFF) << 8;
	MemRead(EAH + (EAL & 0xFF)); // dummy read, while address is fixed up
	EA = (EAH + EAL) & 0xFFFF; }
void WIndY()  { // this is still IndY mode, but a dummy read always occurs in a write or modify instruction
	int PTR = MemRead(PC++) & 0xFF; 
	int EAL = (MemRead(PTR) & 0xFF) + Y; 
	int EAH = (MemRead((PTR+1)&0xFF) & 0xFF) << 8;
	MemRead(EAH + (EAL & 0xFF)); // dummy read, while address is fixed up
	EA = (EAH + EAL) & 0xFFFF; }
/**********************************************************************************
 ********************* Load/Store instructions ************************************
 **********************************************************************************/
void LDA() { A = MemRead(EA)&0xFF; SetPSWNZ((byte)A); }
void LDX() { X = MemRead(EA)&0xFF; SetPSWNZ((byte)X); }
void LDY() { Y = MemRead(EA)&0xFF; SetPSWNZ((byte)Y); }

void STA() { MemWrite(EA,(byte)A); }
void STX() { MemWrite(EA,(byte)X); }
void STY() { MemWrite(EA,(byte)Y); }

/**********************************************************************************
 ********************* Arithmetical and logical instructions **********************
 **********************************************************************************/

void ADC() {
  int Op1=A, Op2=MemRead(EA)&0xFF;
  if (D) {    // decimal mode behavior following Marko Makela's explanations
     int tmp;
     Z = ((Op1 + Op2 + (C ? 1 : 0)) & 0xFF) == 0;
     tmp = (Op1 & 0x0F) + (Op2 & 0x0F) + (C ? 1 : 0);
     A   = tmp < 0x0A ? tmp : tmp + 6;
     tmp = (Op1 & 0xF0) + (Op2 & 0xF0) + (tmp & 0xF0);
     N = ((byte)tmp) < 0;
     V = ((Op1 ^ tmp) & ~(Op1 ^ Op2) & 0x80)!=0;
     tmp = (A & 0x0F) | (tmp < 0xA0 ? tmp : tmp + 0x60);
     C = (tmp >= 0x100);
     A   = tmp & 0xFF;
  } else {       // binary mode 
     int tmp = Op1 + Op2 + (C ? 1 : 0);
     A = tmp & 0xFF;
     V = ((Op1 ^ A) & ~(Op1 ^ Op2) & 0x80)!=0;
     SetFlagCarry(tmp);
     SetPSWNZ((byte)A);
  }
} 						 

void SBC() {
  int Op1=A, Op2=MemRead(EA)&0xFF;
  if (D) {  // decimal mode behavior following Marko Makela's explanations
    int tmp;
    tmp = (Op1 & 0x0F) - (Op2 & 0x0F) - (C ? 0 : 1);
    A   = (tmp & 0x10)==0 ? tmp : tmp - 6;
    tmp = (Op1 & 0xF0) - (Op2 & 0xF0) - (A & 0x10);
    A = (A & 0x0F) | ((tmp & 0x100)==0 ? tmp : tmp - 0x60);
    tmp = Op1 - Op2 - (C ? 0 : 1);
    SetFlagBorrow(tmp);
    SetPSWNZ((byte)tmp);
  } else {	// binary mode
    int tmp;
    tmp = Op1 - Op2 - (C ? 0 : 1);
    A = tmp & 0xFF;
    V = ((Op1 ^ Op2) & (Op1 ^ A) & 0x80)!=0;
    SetFlagBorrow(tmp);
    SetPSWNZ((byte)A);
  }
}	
						 
void CMP() { int tmp = A - (MemRead(EA) & 0xFF); SetFlagBorrow(tmp); SetPSWNZ((byte)tmp); }
void CPX() { int tmp = X - (MemRead(EA) & 0xFF); SetFlagBorrow(tmp); SetPSWNZ((byte)tmp); }                
void CPY() { int tmp = Y - (MemRead(EA) & 0xFF); SetFlagBorrow(tmp); SetPSWNZ((byte)tmp); }



void AND() { A &= MemRead(EA)&0xFF; SetPSWNZ((byte)A); }
void ORA() { A |= MemRead(EA)&0xFF; SetPSWNZ((byte)A); }											                                  
void EOR() { A ^= MemRead(EA)&0xFF; SetPSWNZ((byte)A); }											                                  


void ASL() { 
  byte tmp = MemRead(EA);
  MemWrite(EA,tmp);  // additional dummy write
  C = tmp<0;
  tmp <<= 1;
  SetPSWNZ(tmp);
  MemWrite(EA,tmp);
}
void ASL_A() { int tmp = A<<1; A = tmp&0xFF; SetFlagCarry(tmp); SetPSWNZ((byte)A); }

void LSR() {
  byte tmp = MemRead(EA);
  MemWrite(EA,tmp);  // additional dummy write
  C = (tmp & 1)!=0;
  tmp = (byte)((tmp & 0xFF) >> 1);
  SetPSWNZ(tmp);
  MemWrite(EA,tmp);
}
void LSR_A() { C = (A & 1)!=0; A >>= 1; SetPSWNZ((byte)A); }

void ROL() { 
  byte tmp = MemRead(EA);
  MemWrite(EA,tmp);  // additional dummy write
  boolean newCarry = tmp < 0;
  tmp = (byte)(((tmp&0xFF) << 1) | (C ? 1 : 0));
  C = newCarry;
  SetPSWNZ(tmp);
  MemWrite(EA,tmp);
}
void ROL_A() { int tmp = (A<<1) | (C ? 1 : 0); A = tmp&0xFF; SetFlagCarry(tmp); SetPSWNZ((byte)A); }

void ROR() { 
  byte tmp = MemRead(EA);
  MemWrite(EA,tmp);  // additional dummy write
  boolean newCarry = (tmp & 1)!=0;
  tmp = (byte)(((tmp&0xFF) >> 1) | (C ? 0x80 : 0));
  C = newCarry;
  SetPSWNZ(tmp); 
  MemWrite(EA,tmp);
}
void ROR_A() { int tmp = A | (C ? 0x100 : 0); C = (A & 1)!=0; A = tmp>>1; SetPSWNZ((byte)A); }

/**********************************************************************************
 ****************************** Inc Dec instructions ******************************
 **********************************************************************************/

void INC() { byte tmp=MemRead(EA); MemWrite(EA,tmp); tmp++; SetPSWNZ(tmp); MemWrite(EA,tmp); }                
void DEC() { byte tmp=MemRead(EA); MemWrite(EA,tmp); tmp--; SetPSWNZ(tmp); MemWrite(EA,tmp); }

void INX() { X = (X+1)&0xFF; SetPSWNZ((byte)X); }
void INY() { Y = (Y+1)&0xFF; SetPSWNZ((byte)Y); }
void DEX() { X = (X-1)&0xFF; SetPSWNZ((byte)X); }
void DEY() { Y = (Y-1)&0xFF; SetPSWNZ((byte)Y); }

/**********************************************************************************
 *********************************** Bit tests ************************************
 **********************************************************************************/
void BIT() { byte tmp = MemRead(EA); V = (tmp & 0x40) != 0 ; N = (tmp<0); Z = (tmp & A)==0; }

/**********************************************************************************
 ****************************** Stack instructions ********************************
 **********************************************************************************/
void PHA()        { MemWrite(0x100+S,(byte)A); S = (S-1)&0xFF; }
void PHP()        { CompressPSW(); MemWrite(0x100+S,(byte)PSW); S = (S-1)&0xFF; }

void PLA()        { MemRead(S+0x100); S = (S+1)&0xFF; A = MemRead(S+0x100)&0xFF; SetPSWNZ((byte)A);}
void PLP()        { MemRead(S+0x100); S = (S+1)&0xFF; PSW = MemRead(S+0x100)&0xFF; UncompressPSW(); }

/**********************************************************************************
 ***************************** Software interruption ******************************
 **********************************************************************************/

void BRK()        { MemRead(EA); PushPC(); PHP(); I = true; PC = MemWord(0xFFFE); }
void RTI()        { MemRead(S+0x100); PLP(); PopPC(); }

/**********************************************************************************
 ************************************* Jumps **************************************
 **********************************************************************************/
void JMP()        { PC = EA; }          
void RTS()        { MemRead(S+0x100); PopPC(); MemRead(PC++); }

void JSR() { // special order for the fetch of the absolute address
    int EAL = MemRead(PC++) & 0xFF; 
    MemRead(S+0x100);
    PushPC();
    PC = EAL + ((MemRead(PC) & 0xFF) << 8);
}

void branch() {
    MemRead(PC); // dummy read of next opcode
    if ((PC&0xFF00)!=(EA&0xFF00)) { 
        MemRead(PC&0xFF00 | EA&0xFF);  // dummy read of wrong opcode
    }
    PC = EA; 
}

void BNE()		{ if (!Z) branch(); }
void BEQ()        { if (Z)  branch(); }
void BVC()        { if (!V) branch(); }
void BVS()        { if (V)  branch(); }
void BCC()        { if (!C) branch(); }
void BCS()        { if (C)  branch(); }
void BPL()        { if (!N) branch(); }
void BMI()        { if (N)  branch(); }

/**********************************************************************************
 ********************************** Transfers *************************************
 **********************************************************************************/
void TAX()      { X = A; SetPSWNZ((byte)A); }
void TXA()      { A = X; SetPSWNZ((byte)A); }
void TAY()      { Y = A; SetPSWNZ((byte)A); }
void TYA()      { A = Y; SetPSWNZ((byte)A); }
void TXS()      { S = X; }  // no affected PSW !
void TSX()      { X = S; SetPSWNZ((byte)X); }

/**********************************************************************************
 ************************************ PSW ***************************************
 **********************************************************************************/
void CLC()      { C = false; }
void SEC()      { C = true; }
void CLI()      { I = false; }
void SEI()      { I = true; }
void CLV()      { V = false; }
void CLD()      { D = false; }
void SED()      { D = true; }

/**********************************************************************************
 ********************************** NOP NOP NOP ***********************************
 **********************************************************************************/
void NOP()        {}

void Unoff()  {}
void Unoff1() {}
void Unoff2() { PC++; }
void Unoff3() { PC+=2; }
void Hang() { PC--; }

/**********************************************************************************
 ******************************** the interrupts **********************************
 **********************************************************************************/

public void IRQ(boolean state)
{
    IRQ=state;
}

public void NMI()
{
    NMI=true;
}

public void RESET()
{
  PC = MemWord(0xFFFC); 
  A=X=Y=0; N=V=D=I=Z=C=false; PSW=0x30; S=0xFF;
}

public void SO()
{
    V=true;
}

void HandleIRQ()
{
	PushPC();
	CompressPSW();
	MemWrite(0x100+S,(byte)(PSW & ~0x10));  // flag B not set
	S = (S-1)&0xFF;
	I = true;
	PC=MemWord(0xFFFE);
	cycles+=7;
}

void HandleNMI()
{
	PushPC();
	CompressPSW();
	MemWrite(0x100+S,(byte)(PSW & ~0x10));  // flag B not set
	S = (S-1)&0xFF;
	I = true; 
	NMI=false; 
	PC=MemWord(0xFFFA); 
	cycles+=7;
}

/**********************************************************************************
 ******************************** loading/saving state ****************************
 **********************************************************************************/

public void dumpState(OutputStream s) throws IOException
{
     CompressPSW();
     state[0]=(byte)PC; state[1]=(byte)(PC>>8);
     state[2]=(byte)PSW;
     state[3]=(byte)A;
     state[4]=(byte)X;
     state[5]=(byte)Y;
     state[6]=(byte)S;
     s.write(state);
}

public void loadState(InputStream s) throws IOException
{
     s.read(state);
     PC = (state[0]&0xFF) + ((state[1]&0xFF)<<8);
     PSW = state[2] & 0xFF;
     A = state[3] & 0xFF;
     X = state[4] & 0xFF;
     Y = state[5] & 0xFF;
     S = state[6] & 0xFF;
     UncompressPSW();
}

public void dumpMem(OutputStream s) throws IOException
{
     s.write(Mem);
}

public void loadMem(InputStream s) throws IOException
{
     s.read(Mem);
}

/**********************************************************************************
 ******************************** running in real-time ****************************
 **********************************************************************************/

long lastTime=System.currentTimeMillis();

public void synchronize() {
    int realTimeMillis=(int)(System.currentTimeMillis()-lastTime);
    int sleepMillis=synchroTimeMillis-realTimeMillis;
    if (sleepMillis<0) sleepMillis=5; /* always sleep a while for old browsers */
    try {
        Thread.sleep(sleepMillis);
    } catch (Exception e) {}
    lastTime=System.currentTimeMillis();
}

public void run() {
  while(true) {
    synchronize();
    cycles=0;
    while (cycles<cyclesBeforeSynchro) {
      if (!I && IRQ) HandleIRQ();
      if (NMI) HandleNMI();
      int opcode = MemRead(PC++) & 0xFF;
      switch (opcode) {

        case 0x00: Imm();   BRK();   break;
        case 0x01: IndX();  ORA();   break;
        case 0x02:                          Hang(); break;
        case 0x03:                          Unoff(); break;
        case 0x04:                          Unoff2(); break;
        case 0x05: Zero();  ORA();   break;
        case 0x06: Zero();  ASL();   break;
        case 0x07:                          Unoff(); break;
        case 0x08: Imp();   PHP();   break;
        case 0x09: Imm();   ORA();   break;
        case 0x0a: Imp();   ASL_A(); break;
        case 0x0b:                          Imm();   AND();   break;
        case 0x0c:                          Unoff3(); break;
        case 0x0d: Abs();   ORA();   break;
        case 0x0e: Abs();   ASL();   break;
        case 0x0f:                          Unoff(); break;

        case 0x10: Rel();   BPL();   break;
        case 0x11: IndY();  ORA();   break;
        case 0x12:                          Hang(); break;
        case 0x13:                          Unoff(); break;
        case 0x14:                          Unoff2(); break;
        case 0x15: ZeroX(); ORA();   break;
        case 0x16: ZeroX(); ASL();   break;
        case 0x17:                          Unoff(); break;
        case 0x18: Imp();   CLC();   break;
        case 0x19: AbsY();  ORA();   break;
        case 0x1a:                          Unoff1(); break;
        case 0x1b:                          Unoff(); break;
        case 0x1c:                          Unoff3(); break;
        case 0x1d: AbsX();  ORA();   break;
        case 0x1e: WAbsX(); ASL();   break;
        case 0x1f:                          Unoff(); break;

        case 0x20:          JSR();   break;     // special order for the read/write cycles
        case 0x21: IndX();  AND();   break;
        case 0x22:                          Hang(); break;
        case 0x23:                          Unoff(); break;
        case 0x24: Zero();  BIT();   break;
        case 0x25: Zero();  AND();   break;
        case 0x26: Zero();  ROL();   break;
        case 0x27:                          Unoff(); break;
        case 0x28: Imp();   PLP();   break;
        case 0x29: Imm();   AND();   break;
        case 0x2a: Imp();   ROL_A(); break;
        case 0x2b:                          Imm();   AND();   break;
        case 0x2c: Abs();   BIT();   break;
        case 0x2d: Abs();   AND();   break;
        case 0x2e: Abs();   ROL();   break;
        case 0x2f:                          Unoff(); break;

        case 0x30: Rel();   BMI();   break;
        case 0x31: IndY();  AND();   break;
        case 0x32:                          Hang(); break;
        case 0x33:                          Unoff(); break;
        case 0x34:                          Unoff2(); break;
        case 0x35: ZeroX(); AND();   break;
        case 0x36: ZeroX(); ROL();   break;
        case 0x37:                          Unoff(); break;
        case 0x38: Imp();   SEC();   break;
        case 0x39: AbsY();  AND();   break;
        case 0x3a:                          Unoff1(); break;
        case 0x3b:                          Unoff(); break;
        case 0x3c:                          Unoff3(); break;
        case 0x3d: AbsX();  AND();   break;
        case 0x3e: WAbsX(); ROL();   break;
        case 0x3f:                          Unoff(); break;

        case 0x40: Imp();   RTI();   break;
        case 0x41: IndX();  EOR();   break;
        case 0x42:                          Hang(); break;
        case 0x43:                          Unoff(); break;
        case 0x44:                          Unoff2(); break;
        case 0x45: Zero();  EOR();   break;
        case 0x46: Zero();  LSR();   break;
        case 0x47:                          Unoff(); break;
        case 0x48: Imp();   PHA();   break;
        case 0x49: Imm();   EOR();   break;
        case 0x4a: Imp();   LSR_A(); break;
        case 0x4b:                          Unoff(); break;
        case 0x4c: Abs();   JMP();   break;
        case 0x4d: Abs();   EOR();   break;
        case 0x4e: Abs();   LSR();   break;
        case 0x4f:                          Unoff(); break;

        case 0x50: Rel();   BVC();   break;
        case 0x51: IndY();  EOR();   break;
        case 0x52:                          Hang(); break;
        case 0x53:                          Unoff(); break;
        case 0x54:                          Unoff2(); break;
        case 0x55: ZeroX(); EOR();   break;
        case 0x56: ZeroX(); LSR();   break;
        case 0x57:                          Unoff(); break;
        case 0x58: Imp();   CLI();   break;
        case 0x59: AbsY();  EOR();   break;
        case 0x5a:                          Unoff1(); break;
        case 0x5b:                          Unoff(); break;
        case 0x5c:                          Unoff3(); break;
        case 0x5d: AbsX();  EOR();   break;
        case 0x5e: WAbsX(); LSR();   break;
        case 0x5f:                          Unoff(); break;

        case 0x60: Imp();   RTS();   break;
        case 0x61: IndX();  ADC();   break;
        case 0x62:                          Hang(); break;
        case 0x63:                          Unoff(); break;
        case 0x64:                          Unoff2(); break;
        case 0x65: Zero();  ADC();   break;
        case 0x66: Zero();  ROR();   break;
        case 0x67:                          Unoff(); break;
        case 0x68: Imp();   PLA();   break;
        case 0x69: Imm();   ADC();   break;
        case 0x6a: Imp();   ROR_A(); break;
        case 0x6b:                          Unoff(); break;
        case 0x6c: Ind();   JMP();   break;
        case 0x6d: Abs();   ADC();   break;
        case 0x6e: Abs();   ROR();   break;
        case 0x6f:                          Unoff(); break;

        case 0x70: Rel();   BVS();   break;
        case 0x71: IndY();  ADC();   break;
        case 0x72:                          Hang(); break;
        case 0x73:                          Unoff(); break;
        case 0x74:                          Unoff2(); break;
        case 0x75: ZeroX(); ADC();   break;
        case 0x76: ZeroX(); ROR();   break;
        case 0x77:                          Unoff(); break;
        case 0x78: Imp();   SEI();   break;
        case 0x79: AbsY();  ADC();   break;
        case 0x7a:                          Unoff1(); break;
        case 0x7b:                          Unoff(); break;
        case 0x7c:                          Unoff3(); break;
        case 0x7d: AbsX();  ADC();   break;
        case 0x7e: WAbsX(); ROR();   break;
        case 0x7f:                          Unoff(); break;

        case 0x80:                          Unoff2(); break;
        case 0x81: IndX();  STA();   break;
        case 0x82:                          Unoff2(); break;
        case 0x83:                          Unoff(); break;
        case 0x84: Zero();  STY();   break;
        case 0x85: Zero();  STA();   break;
        case 0x86: Zero();  STX();   break;
        case 0x87:                          Unoff(); break;
        case 0x88: Imp();   DEY();   break;
        case 0x89:                          Unoff2(); break;
        case 0x8a: Imp();   TXA();   break;
        case 0x8b:                          Unoff(); break;
        case 0x8c: Abs();   STY();   break;
        case 0x8d: Abs();   STA();   break;
        case 0x8e: Abs();   STX();   break;
        case 0x8f:                          Unoff(); break;

        case 0x90: Rel();   BCC();   break;
        case 0x91: WIndY(); STA();   break;
        case 0x92:                          Hang(); break;
        case 0x93:                          Unoff(); break;
        case 0x94: ZeroX(); STY();   break;
        case 0x95: ZeroX(); STA();   break;
        case 0x96: ZeroY(); STX();   break;
        case 0x97:                          Unoff(); break;
        case 0x98: Imp();   TYA();   break;
        case 0x99: WAbsY(); STA();   break;
        case 0x9a: Imp();   TXS();   break;
        case 0x9b:                          Unoff(); break;
        case 0x9c:                          Unoff(); break;
        case 0x9d: WAbsX(); STA();   break;
        case 0x9e:                          Unoff(); break;
        case 0x9f:                          Unoff(); break;

        case 0xa0: Imm();   LDY();   break;
        case 0xa1: IndX();  LDA();   break;
        case 0xa2: Imm();   LDX();   break;
        case 0xa3:                          Unoff(); break;
        case 0xa4: Zero();  LDY();   break;
        case 0xa5: Zero();  LDA();   break;
        case 0xa6: Zero();  LDX();   break;
        case 0xa7:                          Unoff(); break;
        case 0xa8: Imp();   TAY();   break;
        case 0xa9: Imm();   LDA();   break;
        case 0xaa: Imp();   TAX();   break;
        case 0xab:                          Unoff(); break;
        case 0xac: Abs();   LDY();   break;
        case 0xad: Abs();   LDA();   break;
        case 0xae: Abs();   LDX();   break;
        case 0xaf:                          Unoff(); break;

        case 0xb0: Rel();   BCS();   break;
        case 0xb1: IndY();  LDA();   break;
        case 0xb2:                          Hang(); break;
        case 0xb3:                          Unoff(); break;
        case 0xb4: ZeroX(); LDY();   break;
        case 0xb5: ZeroX(); LDA();   break;
        case 0xb6: ZeroY(); LDX();   break;
        case 0xb7:                          Unoff(); break;
        case 0xb8: Imp();   CLV();   break;
        case 0xb9: AbsY();  LDA();   break;
        case 0xba: Imp();   TSX();   break;
        case 0xbb:                          Unoff(); break;
        case 0xbc: AbsX();  LDY();   break;
        case 0xbd: AbsX();  LDA();   break;
        case 0xbe: AbsY();  LDX();   break;
        case 0xbf:                          Unoff(); break;

        case 0xc0: Imm();   CPY();   break;
        case 0xc1: IndX();  CMP();   break;
        case 0xc2:                          Unoff2(); break;
        case 0xc3:                          Unoff(); break;
        case 0xc4: Zero();  CPY();   break;
        case 0xc5: Zero();  CMP();   break;
        case 0xc6: Zero();  DEC();   break;
        case 0xc7:                          Unoff(); break;
        case 0xc8: Imp();   INY();   break;
        case 0xc9: Imm();   CMP();   break;
        case 0xca: Imp();   DEX();   break;
        case 0xcb:                          Unoff(); break;
        case 0xcc: Abs();   CPY();   break;
        case 0xcd: Abs();   CMP();   break;
        case 0xce: Abs();   DEC();   break;
        case 0xcf:                          Unoff(); break;

        case 0xd0: Rel();   BNE();   break;
        case 0xd1: IndY();  CMP();   break;
        case 0xd2:                          Hang(); break;
        case 0xd3:                          Unoff(); break;
        case 0xd4:                          Unoff2(); break;
        case 0xd5: ZeroX(); CMP();   break;
        case 0xd6: ZeroX(); DEC();   break;
        case 0xd7:                          Unoff(); break;
        case 0xd8: Imp();   CLD();   break;
        case 0xd9: AbsY();  CMP();   break;
        case 0xda:                          Unoff1();   break;
        case 0xdb:                          Unoff(); break;
        case 0xdc:                          Unoff3(); break;
        case 0xdd: AbsX();  CMP();   break;
        case 0xde: WAbsX(); DEC();   break;
        case 0xdf:                          Unoff(); break;

        case 0xe0: Imm();   CPX();   break;
        case 0xe1: IndX();  SBC();   break;
        case 0xe2:                          Unoff2(); break;
        case 0xe3:                          Unoff(); break;
        case 0xe4: Zero();  CPX();   break;
        case 0xe5: Zero();  SBC();   break;
        case 0xe6: Zero();  INC();   break;
        case 0xe7:                          Unoff(); break;
        case 0xe8: Imp();   INX();   break;
        case 0xe9: Imm();   SBC();   break;
        case 0xea: Imp();   NOP();   break;
        case 0xeb:                          Imm(); SBC(); break;
        case 0xec: Abs();   CPX();   break;
        case 0xed: Abs();   SBC();   break;
        case 0xee: Abs();   INC();   break;
        case 0xef:                          Unoff(); break;

        case 0xf0: Rel();   BEQ();   break;
        case 0xf1: IndY();  SBC();   break;
        case 0xf2:                          Hang(); break;
        case 0xf3:                          Unoff(); break;
        case 0xf4:                          Unoff2(); break;
        case 0xf5: ZeroX(); SBC();   break;
        case 0xf6: ZeroX(); INC();   break;
        case 0xf7:                          Unoff(); break;
        case 0xf8: Imp();   SED();   break;
        case 0xf9: AbsY();  SBC();   break;
        case 0xfa:                          Unoff1();   break;
        case 0xfb:                          Unoff(); break;
        case 0xfc:                          Unoff3(); break;
        case 0xfd: AbsX();  SBC();   break;
        case 0xfe: WAbsX(); INC();   break;
        case 0xff:                          Unoff(); break;
      }
    }
  }
}

} // class R6502