import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;

/**********************************************************************************
 *         Microtan65 Applet/Application emulator by Fabrice Frances              *
 *                                                                                *
 *  This class encapsulate the microtan emulator in applet and application form,  *
 * so that it can be run either in a browser (without Load/Save capability) or    *
 * locally as a Java application.                                                 *
 *                                                                                *
 * Parameters for the Applet, and their default values :                          *
 *  <param name="rom" value="microtan.rom" >                                      *
 *  <param name="ram" value=8 >   size of ram in kB                               *
 *  <param name="dump" value=<dump_filename> >  loads a dump on launch, no default*
 *  <param name="bg" value="black" >  background color                            *
 *  <param name="fg" value="white" >  foreground color                            *
 *                                                                                *
 * Command line usage for the Application (brackets denote optional arguments):   *
 * jre microtan [rom microtan.rom] [ram 8] [dump <dumpfile>] [bg black] [fg white]*
 *                                                                                *
 * It might need a bit of tidying up, this is my first Java program, just intended*
 * to experiment with precise emulation of computers in an object oriented form...*
 *                                                                                *
 * Feel free to extend this emulator, just credit this work if you do.            *
 **********************************************************************************/

public class microtan extends Applet 
                   implements KeyListener, WindowListener, Runnable
{
    Frame microtanFrame;
    Panel microtanPanel;
    FileDialog loadsaveDialog;
    Microtan_board microtan_board;
    Thread cpu;   
    InputStream romStream,charsetStream;
    Color background=Color.black;
    Color foreground=Color.white;
    int last_key=0, request=0;
    int ramSize=8;
    String romName="microtan.rom";
    String dumpFile=null;
    String[][] appletInfo = {   
        { "rom", "filename", "filename of ROM" },
        { "ram", "integer", "size of RAM (in KB)" },
        { "dump", "dumpfile", "dump file to load" },
        { "bg", "color", "color of background" },
        { "fg", "color", "color of foreground" }};

    public String[][] getParameterInfo() { return appletInfo; }

    public static void main(String args[]) {
      microtan m65 = new microtan();
      m65.run_application(args);
    }

    void run_application(String args[]) {
      background=Color.black;
      foreground=Color.white;
      parseArguments(args);
      microtanFrame = new Frame("Microtan 65");
	microtanFrame.addWindowListener(this);
      microtanPanel = this;
      microtanFrame.add(microtanPanel);
      microtanFrame.setSize(264,283);
      microtanFrame.show();
//      microtanFrame.setResizable(false);
      loadsaveDialog = new FileDialog(microtanFrame);

      String cwd=System.getProperty("user.dir");
      try {
        romStream = new FileInputStream(cwd+"/"+romName);
        charsetStream = new FileInputStream(cwd+"/charset.rom");
      } catch (Exception e) {
        System.out.println("cannot access microtan roms");
      }
      create_microtan();
      microtan_board.RESET();
      if (dumpFile!=null) loadDump(dumpFile);
      microtan_board.run();
    }

    public void init() {
        microtanFrame = new Frame("Microtan 65 emulator");
        microtanPanel = this;
        readAppletParameters();
        URL microtanURL = getDocumentBase();
        try {
          URL rom_url=new URL(microtanURL,romName);
          URLConnection rom_connection=rom_url.openConnection();
          rom_connection.connect();
          romStream=rom_connection.getInputStream();
          URL charset_url=new URL(microtanURL,"charset.rom");
          URLConnection charset_connection=charset_url.openConnection();
          charset_connection.connect();
          charsetStream=charset_connection.getInputStream();
        } catch (Exception e) {
          System.out.println("cannot access microtan roms");
        }
        create_microtan();
    }

    void create_microtan() {
        microtanPanel.addKeyListener(this);
        microtanPanel.requestFocus();

        try {
            microtan_board = new Microtan_board(
                                     microtanPanel, background, foreground,
                                     charsetStream,
                                     ramSize,
                                     romStream);
            charsetStream.close();
            romStream.close();
        } catch (IOException e) { System.out.println("cannot read microtan roms"); }
    }

    public void start() {
        cpu=new Thread(this);
        cpu.start();
    }

    public void run() {
        microtan_board.RESET();
        if (dumpFile!=null) loadDump(dumpFile);
        microtan_board.run();
    }

    public void update(Graphics g) { paint(g); }

    public void paint(Graphics g) {
        if (microtan_board!=null) 
            microtan_board.paint();
    }

    public void keyPressed(KeyEvent ev) {
        String name;
        last_key=ev.getKeyCode();
        switch (last_key) {
          case KeyEvent.VK_F2:
            if (loadsaveDialog==null) break;
		loadsaveDialog.setTitle("Load memory dump from file");
		loadsaveDialog.setMode(FileDialog.LOAD);
            loadsaveDialog.show();
            name=loadsaveDialog.getDirectory()+loadsaveDialog.getFile();
            if (loadsaveDialog.getFile()!=null) loadDump(name);
            break;
          case KeyEvent.VK_F3:
            if (loadsaveDialog==null) break;
		loadsaveDialog.setTitle("Dump memory to file...");
		loadsaveDialog.setMode(FileDialog.SAVE);
            loadsaveDialog.show();
            name=loadsaveDialog.getDirectory()+loadsaveDialog.getFile();
            if (loadsaveDialog.getFile()!=null) saveDump(name);
            break;
          case KeyEvent.VK_F8:
            microtan_board.RESET();
            break;
        }
        microtanPanel.requestFocus();
    }

    public void keyReleased(KeyEvent e) {;}

    public void keyTyped(KeyEvent e) {
        int c=e.getKeyChar();
        if (last_key==KeyEvent.VK_ENTER) {   // damn', the ENTER key doesn't return the same KeyChar on every platform...
          c=0x0D;				// let it be Carriage Return everywhere
          if (e.isShiftDown() || e.isControlDown()) c=0x0A;   // let Shift-RETURN and Ctrl-RETURN mean LineFeed
        } else if (last_key==KeyEvent.VK_BACK_SPACE) c=0x7F;  // map BackSpace to a DELETE
        e.consume();
        if (c<128) {
          microtan_board.keyboard_entry((byte)c);
        }
    }

    public void windowClosing(WindowEvent e) { System.exit(0); }
    public void windowActivated(WindowEvent e) { microtanPanel.requestFocus(); }
    public void windowClosed(WindowEvent e) {}
    public void windowDeactivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) { microtanPanel.requestFocus(); }
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) { microtanPanel.requestFocus(); }

    void parseArguments(String[] args) {
      try {
        for (int i=0;i<args.length;i++) {
            if (args[i].equals("ram")) {
                i++;
                ramSize=Integer.parseInt(args[i]);
            } else if (args[i].equals("rom")) {
                i++;
                romName=args[i];
            } else if (args[i].equals("bg")) {
                i++;
                background=getColor(args[i]);
            } else if (args[i].equals("fg")) {
                i++;
                foreground=getColor(args[i]);
            } else if (args[i].equals("dump")) {
                i++;
                dumpFile = args[i];
            }
         }
       } catch (Exception e) { 
         System.out.println("Bad arguments.");
         System.out.println("Usage: microtan [rom <filename>] [ram <kB size>] [bg <color name>] [fg <color name>]");
       }
     }

    void readAppletParameters() {
      try {
        String filename = getParameter("rom");
        if (filename!=null) romName=filename;
        String ram = getParameter("ram");
        if (ram!=null) ramSize=Integer.parseInt(ram);
        String bg = getParameter("bg");
        if (bg!=null) background=getColor(bg);
        String fg = getParameter("fg");
        if (fg!=null) foreground=getColor(fg);
        dumpFile = getParameter("dump");
      } catch (Exception e) {
        showStatus("bad parameters");
      }
    }

     Color getColor(String name) throws Exception {
       Color[] colors= { Color.black, Color.blue, Color.cyan, Color.darkGray, Color.green, Color.lightGray,
                         Color.magenta, Color.orange, Color.pink, Color.red, Color.white, Color.yellow };
       String[] names= { "black", "blue", "cyan", "darkgray", "green", "lightgray",
                         "magenta", "orange", "pink", "red", "white", "yellow" };
       for (int i=0; i<colors.length; i++)
                if (name.equals(names[i])) return colors[i];
       throw new Exception("color unknown");
     }

     void loadDump(String name) {
         try {
             FileInputStream s=new FileInputStream(name);
             microtan_board.loadMem(s);
             microtan_board.loadState(s);
             s.close();
         } catch (Exception e) {
             System.out.println("Error loading dump "+name);
         }
         repaint();
     }

     void saveDump(String name) {
         try {
             FileOutputStream s=new FileOutputStream(name);
             microtan_board.dumpMem(s);
             microtan_board.dumpState(s);
             s.close();
         } catch (SecurityException e) {
             System.out.println("Security check doesn't allow to create file");
         } catch (Exception e) {
             System.out.println("Error dumping to file "+name);
         }
     }
}
