/**
 * @file	nevent.c
 * @brief	Implementation of the event
 */

#include "compiler.h"
#include "nevent.h"
#include "cpucore.h"
#include "pccore.h"

#if 0
#undef	TRACEOUT
#define	TRACEOUT(s)	(void)(s)
static void trace_fmt_ex(const char* fmt, ...)
{
	char stmp[2048];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(stmp, fmt, ap);
	strcat(stmp, "\n");
	va_end(ap);
	OutputDebugStringA(stmp);
}
#define	TRACEOUT(s)	trace_fmt_ex s
#endif	/* 1 */

	_NEVENT g_nevent;

#if defined(SUPPORT_MULTITHREAD)
static int nevent_cs_initialized = 0;
static CRITICAL_SECTION nevent_cs;

static BOOL nevent_tryenter_criticalsection(void){
	if(!nevent_cs_initialized) return TRUE;
	return TryEnterCriticalSection(&nevent_cs);
}
static void nevent_enter_criticalsection(void){
	if(!nevent_cs_initialized) return;
	EnterCriticalSection(&nevent_cs);
}
static void nevent_leave_criticalsection(void){
	if(!nevent_cs_initialized) return;
	LeaveCriticalSection(&nevent_cs);
}
	
void nevent_initialize(void)
{
	/* NeBJZNV */
	if(!nevent_cs_initialized){
		memset(&nevent_cs, 0, sizeof(nevent_cs));
		InitializeCriticalSection(&nevent_cs);
		nevent_cs_initialized = 1;
	}
}
void nevent_shutdown(void)
{
	/* NeBJZNVj */
	if(nevent_cs_initialized){
		memset(&nevent_cs, 0, sizeof(nevent_cs));
		DeleteCriticalSection(&nevent_cs);
		nevent_cs_initialized = 0;
	}
}
#endif

void nevent_allreset(void)
{
	/* ׂĂZbg */
	memset(&g_nevent, 0, sizeof(g_nevent));
}

void nevent_get1stevent(void)
{
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	/* ŒZ̃Cxg̃NbNZbg */
	if (g_nevent.readyevents)
	{
		CPU_BASECLOCK = g_nevent.item[g_nevent.level[0]].clock;
	}
	else
	{
		/* CxgȂꍇ̃NbNZbg */
		CPU_BASECLOCK = NEVENT_MAXCLOCK;
	}

	/* JE^փZbg */
	CPU_REMCLOCK = CPU_BASECLOCK;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
}

static void nevent_execute(void)
{
	UINT nEvents;
	UINT i;
	NEVENTID id;
	NEVENTITEM item;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	nEvents = 0;
	for (i = 0; i < g_nevent.waitevents; i++)
	{
		id = g_nevent.waitevent[i];
		item = &g_nevent.item[id];

		/* R[obN̎s */
		if (item->proc != NULL)
		{
			item->proc(item);

			/* Ɏz̃Cxg̃`FbN */
			if (item->flag & NEVENT_WAIT)
			{
				g_nevent.waitevent[nEvents++] = id;
			}
		}
		else {
			item->flag &= ~(NEVENT_WAIT);
		}
		item->flag &= ~(NEVENT_SETEVENT);
	}
	g_nevent.waitevents = nEvents;
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
}

void nevent_progress(void)
{
	UINT nEvents;
	SINT32 nextbase;
	UINT i;
	NEVENTID id;
	NEVENTITEM item;
	UINT8 fevtchk = 0;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	CPU_CLOCK += CPU_BASECLOCK;
	nEvents = 0;
	nextbase = NEVENT_MAXCLOCK;
	for (i = 0; i < g_nevent.readyevents; i++)
	{
		id = g_nevent.level[i];
		item = &g_nevent.item[id];
		item->clock -= CPU_BASECLOCK;
		if (item->clock > 0)
		{
			/* Cxg҂ */
			g_nevent.level[nEvents++] = id;
			if (nextbase >= item->clock)
			{
				nextbase = item->clock;
			}
		}
		else
		{
			/* Cxg */
			if (!(item->flag & (NEVENT_SETEVENT | NEVENT_WAIT)))
			{
				g_nevent.waitevent[g_nevent.waitevents++] = id;
			}
			item->flag |= NEVENT_SETEVENT;
			item->flag &= ~(NEVENT_ENABLE);
//			TRACEOUT(("event = %x", id));
#if defined(SUPPORT_ASYNC_CPU)
			pccore_asynccpustat.screendisp = (id == NEVENT_FLAMES && g_nevent.item[NEVENT_FLAMES].proc == screendisp);
#endif
		}
		fevtchk |= (id==NEVENT_FLAMES ? 1 : 0);
	}
	g_nevent.readyevents = nEvents;
	CPU_BASECLOCK = nextbase;
	CPU_REMCLOCK += nextbase;
	nevent_execute();

	// NEVENT_FLAMESɎbΏ
	if(!fevtchk){
		//printf("NEVENT_FLAMES is missing!!\n");
		pcstat.screendispflag = 0;
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
//	TRACEOUT(("nextbase = %d (%d)", nextbase, CPU_REMCLOCK));
}

void nevent_changeclock(UINT32 oldclock, UINT32 newclock)
{
	UINT i;
	NEVENTID id;
	NEVENTITEM item;
	SINT32 baseClock;
	SINT32 remClock;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif

	if (oldclock > 0)
	{
		if (g_nevent.readyevents)
		{
			// Cxg̃NbNC
			for (i = 0; i < g_nevent.readyevents; i++)
			{
				id = g_nevent.level[i];
				item = &g_nevent.item[id];
				if (item->clock > 0)
				{
					SINT64 newClock = ((SINT64)item->clock * newclock + oldclock / 2) / oldclock;
					if (item->clock > 0 && newClock <= 0) newClock = 1;
					if (newClock > INT_MAX) newClock = INT_MAX;
					item->clock = (SINT32)newClock;
				}
			}

			// ̃NbNύX̃^C~O CPU_BASECLOCK==CPU_REMCLOCK ̃^C~OɂȂ悤ɒς
			if (CPU_BASECLOCK == CPU_REMCLOCK) {
				CPU_BASECLOCK = g_nevent.item[g_nevent.level[0]].clock;
				CPU_REMCLOCK = CPU_BASECLOCK;/* JE^փZbg */
			}
			else {
				// I/OoȐꍇ͂ĂꍇB̏ꍇ̓XP[
				CPU_BASECLOCK = ((SINT64)CPU_BASECLOCK * newclock + oldclock / 2) / oldclock;
				CPU_REMCLOCK = ((SINT64)CPU_REMCLOCK * newclock + oldclock / 2) / oldclock;
			}
		}
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif

}

void nevent_reset(NEVENTID id)
{
	UINT i;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	/* ݐisĂCxg */
	for (i = 0; i < g_nevent.readyevents; i++)
	{
		if (g_nevent.level[i] == id)
		{
			break;
		}
	}
	/* Cxg݂͑H */
	if (i < g_nevent.readyevents)
	{
		/* ݂Ă */
		g_nevent.readyevents--;
		for (; i < g_nevent.readyevents; i++)
		{
			g_nevent.level[i] = g_nevent.level[i + 1];
		}
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
}

void nevent_waitreset(NEVENTID id)
{
	UINT i;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	/* ݐisĂCxg */
	for (i = 0; i < g_nevent.waitevents; i++)
	{
		if (g_nevent.waitevent[i] == id)
		{
			break;
		}
	}
	/* Cxg݂͑H */
	if (i < g_nevent.waitevents)
	{
		/* ݂Ă */
		g_nevent.waitevents--;
		for (; i < g_nevent.waitevents; i++)
		{
			g_nevent.waitevent[i] = g_nevent.waitevent[i + 1];
		}
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
}

void nevent_set(NEVENTID id, SINT32 eventclock, NEVENTCB proc, NEVENTPOSITION absolute)
{
	SINT32 clk;
	NEVENTITEM item;
	UINT eventId;
	UINT i;

//	TRACEOUT(("event %d - %xclocks", id, eventclock));
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	clk = CPU_BASECLOCK - CPU_REMCLOCK;
	item = &g_nevent.item[id];
	item->proc = proc;
	item->flag = 0;
	if (absolute)
	{
		item->clock = eventclock + clk;
	}
	else
	{
		item->clock += eventclock;
	}
#if 0
	if (item->clock < clk)
	{
		item->clock = clk;
	}
#endif
	/* Cxg̍폜 */
	nevent_reset(id);

	/* Cxg̑}ʒǔ */
	for (eventId = 0; eventId < g_nevent.readyevents; eventId++)
	{
		if (item->clock < g_nevent.item[g_nevent.level[eventId]].clock)
		{
			break;
		}
	}

	/* Cxg̑} */
	for (i = g_nevent.readyevents; i > eventId; i--)
	{
		g_nevent.level[i] = g_nevent.level[i - 1];
	}
	g_nevent.level[eventId] = id;
	g_nevent.readyevents++;

	/* ŒZCxg... */
	if (eventId == 0)
	{
		clk = CPU_BASECLOCK - item->clock;
		CPU_BASECLOCK -= clk;
		CPU_REMCLOCK -= clk;
//		TRACEOUT(("reset nextbase -%d (%d)", clock, CPU_REMCLOCK));
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
}

void nevent_setbyms(NEVENTID id, SINT32 ms, NEVENTCB proc, NEVENTPOSITION absolute)
{
	UINT64 waittime = (UINT64)(pccore.realclock / 1000) * ms;
	if (waittime > INT_MAX - pccore.realclock)
	{
		nevent_set(id, INT_MAX - pccore.realclock, proc, absolute);
	}
	else
	{
		nevent_set(id, (SINT32)waittime, proc, absolute);
	}
}

BOOL nevent_iswork(NEVENTID id)
{
	UINT i;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	/* ݐisĂCxg */
	for (i = 0; i < g_nevent.readyevents; i++)
	{
		if (g_nevent.level[i] == id)
		{
#if defined(SUPPORT_MULTITHREAD)
			nevent_leave_criticalsection();
#endif
			return TRUE;
		}
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
	return FALSE;
}

void nevent_forceexecute(NEVENTID id)
{
	NEVENTITEM item;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	item = &g_nevent.item[id];

	/* R[obN̎s */
	if (item->proc != NULL)
	{
		item->proc(item);
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
}

SINT32 nevent_getremain(NEVENTID id)
{
	UINT i;
	
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	/* ݐisĂCxg */
	for (i = 0; i < g_nevent.readyevents; i++)
	{
		if (g_nevent.level[i] == id)
		{
			SINT32 result = (g_nevent.item[id].clock - (CPU_BASECLOCK - CPU_REMCLOCK));
#if defined(SUPPORT_MULTITHREAD)
			nevent_leave_criticalsection();
#endif
			return result;
		}
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
	return -1;
}

void nevent_forceexit(void)
{
#if defined(SUPPORT_MULTITHREAD)
	nevent_enter_criticalsection();
#endif
	if (CPU_REMCLOCK > 0)
	{
		CPU_BASECLOCK -= CPU_REMCLOCK;
		CPU_REMCLOCK = 0;
	}
#if defined(SUPPORT_MULTITHREAD)
	nevent_leave_criticalsection();
#endif
}
