/**
 * @file	ct1741c.c
 * @brief	Implementation of the Creative SoundBlaster16 CT1741 DSP
 */

#ifdef SUPPORT_SOUND_SB16

#include "compiler.h"
#include "ct1741.h"
#include "iocore.h"
#include "fmboard.h"
#include "dmac.h"
#include "cpucore.h"

#if defined(SUPPORT_MULTITHREAD)
static int ct1741_cs_initialized = 0;
static CRITICAL_SECTION ct1741_cs;

void ct1741cs_enter_criticalsection(void)
{
	if (!ct1741_cs_initialized) return;
	EnterCriticalSection(&ct1741_cs);
}
void ct1741cs_leave_criticalsection(void)
{
	if (!ct1741_cs_initialized) return;
	LeaveCriticalSection(&ct1741_cs);
}

void ct1741cs_initialize(void)
{
	/* NeBJZNV */
	if (!ct1741_cs_initialized)
	{
		memset(&ct1741_cs, 0, sizeof(ct1741_cs));
		InitializeCriticalSection(&ct1741_cs);
		ct1741_cs_initialized = 1;
	}
}
void ct1741cs_shutdown(void)
{
	/* NeBJZNVj */
	if (ct1741_cs_initialized)
	{
		memset(&ct1741_cs, 0, sizeof(ct1741_cs));
		DeleteCriticalSection(&ct1741_cs);
		ct1741_cs_initialized = 0;
	}
}
#endif

void ct1741_initialize(UINT rate) {
	g_sb16.dsp_info.dma.rate2 = ct1741_playinfo.playrate = rate;
}

void ct1741_setpicirq(void) {
	pic_setirq(g_sb16.dmairq);
}
void ct1741_resetpicirq(void) {
	pic_resetirq(g_sb16.dmairq);
}

void ct1741_set_dma_irq(UINT8 irq) {
	switch (irq) {
	case 1:
		g_sb16.dsp_info.dmairq = irq;
		g_sb16.dmairq = 3;
		break;
	case 8:
		g_sb16.dsp_info.dmairq = irq;
		g_sb16.dmairq = 5;
		break;
	case 2:
		g_sb16.dsp_info.dmairq = irq;
		g_sb16.dmairq = 10;
		break;
	case 4:
		g_sb16.dsp_info.dmairq = irq;
		g_sb16.dmairq = 12;
		break;
	}
}

UINT8 ct1741_get_dma_irq() {
	switch (g_sb16.dmairq) {
	case  3:
		return 1;
	case  5:
		return 8;
	case 10:
		return 2;
	case 12:
		return 4;
	}
	return 0x00;
}

void ct1741_set_dma_ch(UINT8 dmach) {
	g_sb16.dsp_info.dmachnum = dmach;
	if (dmach & 0x21) {
		if (g_sb16.dmachnum != 0) {
			g_sb16.dmachnum = 0;
			g_sb16.dsp_info.dma.dmach = dmac.dmach + g_sb16.dmachnum; // DMA蓖
			dmac_attach(DMADEV_CT1741, g_sb16.dmachnum);
		}
	}
	if (dmach & 0x42) {
		if (g_sb16.dmachnum != 3) {
			g_sb16.dmachnum = 3;
			g_sb16.dsp_info.dma.dmach = dmac.dmach + g_sb16.dmachnum; // DMA蓖
			dmac_attach(DMADEV_CT1741, g_sb16.dmachnum);
		}
	}
}

UINT8 ct1741_get_dma_ch() {
	switch (g_sb16.dmachnum) {
	case 0:
		return (g_sb16.dsp_info.dmachnum & 0xe0) ? 0x21 : 0x01;
	case 3:
		return (g_sb16.dsp_info.dmachnum & 0xe0) ? 0x42 : 0x02;
	}
	return 0x00;
}

void ct1741_dma(NEVENTITEM item)
{
	UINT	r;
	SINT32	cnt;
	UINT	rem;
	UINT	pos;
	UINT	irqsamples;
	UINT	irqsamplesleft;
	UINT8	dmabuf[CT1741_DMA_BUFSIZE];
	int i;
	int bytesPerSample; // 1TṽoCg

	if (item->flag & NEVENT_SETEVENT) {
		if (g_sb16.dmachnum != 0xff) {
			sound_sync();
			if (g_sb16.dsp_info.mode == CT1741_DSPMODE_DMA || g_sb16.dsp_info.mode == CT1741_DSPMODE_DMA2) {
				int isautoinit = g_sb16.dsp_info.dma.autoinit;

#if defined(SUPPORT_MULTITHREAD)
				ct1741cs_enter_criticalsection();
#endif
				// 1TṽoCgvZ
				bytesPerSample = CT1741_BUF_ALIGN[g_sb16.dsp_info.dma.mode | g_sb16.dsp_info.dma.stereo << 3];

				// ]\TCYvZ
				if (g_sb16.dsp_info.dma.bufsize < g_sb16.dsp_info.dma.bufdatas) {
					rem = 0;
				}
				else {
					rem = g_sb16.dsp_info.dma.bufsize - g_sb16.dsp_info.dma.bufdatas;
					if (rem > CT1741_DMA_READINTERVAL) {
						rem = CT1741_DMA_READINTERVAL; // xɓǂݎf[^ʂ𐧌
					}
				}

				// 荞ނ܂ł̃TvvZ
				irqsamples = (int)g_sb16.dsp_info.dma.total * (g_sb16.dsp_info.dma.last16mode ? 2 : 1);
				if (g_sb16.dsp_info.smpcounter + rem > irqsamples) {
					rem = irqsamples - g_sb16.dsp_info.smpcounter; // ̊荞݂܂ł̃Tvɂ
				}
				//rem = rem & ~(bytesPerSample - 1);

				// WORKAROUND: Win9xł͍ĐbusyɂȂȂƂȂ炵BWin2kłBUSYɂĂƉ̍Ō̕_uBWin2K̍Đŕ򂵂ĉ
				if (g_sb16.dsp_info.smpcounter < g_sb16.dsp_info.dma.dmach->startcount || g_sb16.dsp_info.dma.total != 65536) {
					g_sb16.dsp_info.wbusy = (g_sb16.dsp_info.dma.lastautoinit && !g_sb16.dsp_info.dma.autoinit) ? 0 : 1; 
				}

				// DMA]
				if (g_sb16.dsp_info.dma.dmach->ready) {
					r = dmac_getdatas(g_sb16.dsp_info.dma.dmach, dmabuf, rem);
				}
				else {
					r = 0;
				}

				// Đpobt@փRs[@Ȃobt@͂ݏođĂ邱Ƃ悤Ȃ̂g_sb16.dsp_info.dma.dmach->startcount͈͓̔]Ȃi͈͊Ô͎Ăj
				pos = (g_sb16.dsp_info.dma.bufpos + g_sb16.dsp_info.dma.bufdatas) & (CT1741_DMA_BUFSIZE - 1);
				for (i = 0; i < r; i++) {
					int dstpos = (pos + i) & (CT1741_DMA_BUFSIZE - 1);
					if (g_sb16.dsp_info.smpcounter2 < (g_sb16.dsp_info.dma.dmach->startcount & ~(bytesPerSample - 1))) {
						g_sb16.dsp_info.dma.buffer[dstpos] = dmabuf[i];
						g_sb16.dsp_info.smpcounter2++;
						g_sb16.dsp_info.dma.bufdatas++;
					}
				}

				// ĐJnfBCiobt@ɂȂ̂h~j
				if (ct1741_playinfo.playwaitcounter > 0) {
					ct1741_playinfo.playwaitcounter -= r;
					if (ct1741_playinfo.playwaitcounter < 0) {
						ct1741_playinfo.playwaitcounter = 0;
					}
				}
#if defined(SUPPORT_MULTITHREAD)
				ct1741cs_leave_criticalsection();
#endif

				// autoinit̎̃f[^ʂ]犄荞݂𔭐@
				g_sb16.dsp_info.smpcounter += r;
				if (g_sb16.dsp_info.smpcounter >= irqsamples) {
					if (isautoinit) {
						// autoinitȂJE^߂Ċ荞݁AłȂȂDMA]Ŋ荞
						g_sb16.dsp_info.smpcounter -= irqsamples;
						g_sb16.mixreg[0x82] |= (g_sb16.dsp_info.dma.last16mode ? 2 : 1);
						ct1741_setpicirq();
						g_sb16.dsp_info.wbusy = 0;
					}
				}
				if (g_sb16.dsp_info.smpcounter < irqsamples) {
					irqsamplesleft = bytesPerSample + irqsamples - g_sb16.dsp_info.smpcounter; // ̊荞݂𔭐f[^c@1Tv̓}[Wt
				}
				else {
					irqsamplesleft = bytesPerSample; // t]Ă1TvƂ
				}

				if (g_sb16.dsp_info.dma.dmach->ready && (g_sb16.dsp_info.dma.dmach->leng.w) && (g_sb16.dsp_info.freq) && g_sb16.dsp_info.smpcounter < irqsamples) {
					// ܂f[^̂ōēxCxgݒ 
					cnt = pccore.realclock / g_sb16.dsp_info.freq / bytesPerSample * min(min(g_sb16.dsp_info.dma.dmach->startcount / 4, CT1741_DMA_READINTERVAL / 2), irqsamplesleft); // obt@1/4NbN or ̊荞݃^C~O
					if (cnt != 0) {
						nevent_set(NEVENT_CT1741, cnt, ct1741_dma, NEVENT_RELATIVE);
					}
					else {
						nevent_setbyms(NEVENT_CT1741, 1, ct1741_dma, NEVENT_RELATIVE); // nevent0ŃZbgƔLt[Ŷŉi{͂ɂ͗Ȃ͂j
					}
				}
				else {
					// DMA]I
					g_sb16.dsp_info.wbusy = 0;
					if (g_sb16.dsp_info.freq) {
						// tO𗧂Ă芄荞݂肷B
						if (g_sb16.dsp_info.dma.dmach->ready && (isautoinit || g_sb16.dsp_info.smpcounter + 1 < irqsamples)) {
							// autoinit܂singleŋK萔]ĂȂȂDMA]JԂ
							g_sb16.dsp_info.dma.laststartaddr = g_sb16.dsp_info.dma.dmach->startaddr;
							g_sb16.dsp_info.dma.laststartcount = g_sb16.dsp_info.dma.dmach->startcount;
							if ((g_sb16.dsp_info.dmachnum & 0xe0) && g_sb16.dsp_info.dma.mode == CT1741_DMAMODE_16) {
								// Ȃ1rbg炵
								g_sb16.dsp_info.dma.dmach->adrs.d = (g_sb16.dsp_info.dma.dmach->startaddr & 0xffff0000) | ((g_sb16.dsp_info.dma.dmach->startaddr << 1) & 0xffff);
								g_sb16.dsp_info.dma.dmach->lastaddr = g_sb16.dsp_info.dma.dmach->adrs.d + g_sb16.dsp_info.dma.dmach->startcount;
								g_sb16.dsp_info.dma.last16mode = 1; // 16bit][h
							}
							else {
								g_sb16.dsp_info.dma.dmach->adrs.d = g_sb16.dsp_info.dma.dmach->startaddr; // ߂
							}
							g_sb16.dsp_info.dma.dmach->leng.w = g_sb16.dsp_info.dma.dmach->startcount; // ߂
							g_sb16.dsp_info.smpcounter2 = 0;
							if (isautoinit && g_sb16.dsp_info.smpcounter >= irqsamples - 16) {
								// ኱덷邪荞݂DMAɍ킹̂D悷
								g_sb16.dsp_info.smpcounter = 0;
								g_sb16.mixreg[0x82] |= (g_sb16.dsp_info.dma.last16mode ? 2 : 1);
								ct1741_setpicirq();
								g_sb16.dsp_info.wbusy = 0;
							}

							// ēxCxgݒ
							cnt = pccore.realclock / g_sb16.dsp_info.freq / bytesPerSample * min(min(g_sb16.dsp_info.dma.dmach->startcount / 4, CT1741_DMA_READINTERVAL) / 4, irqsamplesleft); // obt@1/16NbN
							if (cnt != 0) {
								nevent_set(NEVENT_CT1741, cnt, ct1741_dma, NEVENT_RELATIVE);
							}
							else {
								nevent_setbyms(NEVENT_CT1741, 1, ct1741_dma, NEVENT_RELATIVE);
							}
						}
						else {
							// singleȂ犄荞݂𑗏oDMA]I
							if (g_sb16.dsp_info.dma.lastautoinit) {
#if defined(SUPPORT_MULTITHREAD)
								ct1741cs_enter_criticalsection();
#endif
								// WORKAROUND: 肷 Win3.1
								if (g_sb16.dsp_info.dma.bufdatas > 512) {
									g_sb16.dsp_info.dma.bufdatas -= 512;
								}
								else {
									g_sb16.dsp_info.dma.bufdatas = 0;
								}
#if defined(SUPPORT_MULTITHREAD)
								ct1741cs_leave_criticalsection();
#endif
							}
							g_sb16.dsp_info.wbusy = 0;
							g_sb16.mixreg[0x82] |= (g_sb16.dsp_info.dma.last16mode ? 2 : 1);
							ct1741_setpicirq();
							if (g_sb16.dmachnum != 0xff) {
								dmac.stat |= (1 << g_sb16.dmachnum);
							}
							g_sb16.dsp_info.smpcounter2 = 0;
							g_sb16.dsp_info.smpcounter = 0;
							g_sb16.dsp_info.dma.dmach->ready = 0;
							if (g_sb16.dsp_info.dma.lastautoinit) {
								// WORKAROUND: autoinitIsingleɂȂĂꍇA_~[CxgŊ荞ݓԊuێ
								cnt = pccore.realclock / g_sb16.dsp_info.freq / bytesPerSample * min(min(g_sb16.dsp_info.dma.dmach->startcount / 4, CT1741_DMA_READINTERVAL) / 4, irqsamples);
								if (cnt != 0) {
									nevent_set(NEVENT_CT1741, cnt, ct1741_dma, NEVENT_RELATIVE);
								}
								else {
									nevent_setbyms(NEVENT_CT1741, 1, ct1741_dma, NEVENT_RELATIVE);
								}
							}
						}
					}
					else {
						// Ƃ肠񂵂Ă
						nevent_setbyms(NEVENT_CT1741, 1, ct1741_dma, NEVENT_RELATIVE);
					}
				}
			}
			else if (g_sb16.dsp_info.mode == CT1741_DSPMODE_DMA_IN) {
				// DMA^[h Ή̂߈ԊuŊ荞݂oɂ
				int isautoinit = g_sb16.dsp_info.dma.lastautoinit || g_sb16.dsp_info.dma.autoinit;

				// busy
				g_sb16.dsp_info.wbusy = 0;

				// 1TṽoCgvZ
				bytesPerSample = CT1741_BUF_ALIGN[g_sb16.dsp_info.dma.mode | g_sb16.dsp_info.dma.stereo << 3];

				// 荞݂𔭐oCgvZ@
				irqsamples = (int)g_sb16.dsp_info.dma.total * (g_sb16.dsp_info.dma.last16mode ? 2 : 1);

				// 荞
				g_sb16.mixreg[0x82] |= (g_sb16.dsp_info.dma.last16mode ? 2 : 1);
				ct1741_setpicirq();
				if (g_sb16.dmachnum != 0xff) {
					dmac.stat |= (1 << g_sb16.dmachnum);
				}

				if (isautoinit) {
					// autoinitȂ玟Cxgo^
					cnt = pccore.realclock / g_sb16.dsp_info.freq / bytesPerSample * irqsamples; // ̊荞݃^C~O
					if (cnt != 0) {
						nevent_set(NEVENT_CT1741, cnt, ct1741_dma, NEVENT_RELATIVE);
					}
					else {
						nevent_setbyms(NEVENT_CT1741, 1, ct1741_dma, NEVENT_RELATIVE); // nevent0ŃZbgƔLt[Ŷŉi{͂ɂ͗Ȃ͂j
					}
				}
				else {
					// singleȂDMA]I
					g_sb16.dsp_info.smpcounter2 = 0;
					g_sb16.dsp_info.smpcounter = 0;
					g_sb16.dsp_info.dma.dmach->ready = 0;
				}
			}
			else {
				// DMA]I
				g_sb16.dsp_info.wbusy = 0;
				g_sb16.mixreg[0x82] |= (g_sb16.dsp_info.dma.last16mode ? 2 : 1);
				ct1741_setpicirq();
				g_sb16.dsp_info.smpcounter2 = 0;
				g_sb16.dsp_info.smpcounter = 0;
				g_sb16.dsp_info.dma.dmach->ready = 0;
				if (g_sb16.dmachnum != 0xff) {
					dmac.stat |= (1 << g_sb16.dmachnum);
				}
			}
		}
	}

}

void ct1741_startdma()
{
#if defined(SUPPORT_MULTITHREAD)
	ct1741cs_enter_criticalsection();
#endif
	g_sb16.dsp_info.dma.laststartaddr = g_sb16.dsp_info.dma.dmach->startaddr;
	g_sb16.dsp_info.dma.laststartcount = g_sb16.dsp_info.dma.dmach->startcount;
	if ((g_sb16.dsp_info.dmachnum & 0xe0) && (g_sb16.dsp_info.dma.mode == CT1741_DMAMODE_16))
	{
		// 16bit DMA Ȃ1rbg炵
		g_sb16.dsp_info.dma.dmach->startcount *= 2; // f[^2{
		g_sb16.dsp_info.dma.dmach->adrs.d = (g_sb16.dsp_info.dma.dmach->startaddr & 0xffff0000) | ((g_sb16.dsp_info.dma.dmach->startaddr << 1) & 0xffff);
		g_sb16.dsp_info.dma.dmach->lastaddr = g_sb16.dsp_info.dma.dmach->adrs.d + g_sb16.dsp_info.dma.dmach->startcount;
		g_sb16.dsp_info.dma.dmach->leng.w = g_sb16.dsp_info.dma.dmach->startcount; // ߂
	}
	else {
		// 8bit DMA
		g_sb16.dsp_info.dma.dmach->adrs.d = g_sb16.dsp_info.dma.dmach->startaddr;
		g_sb16.dsp_info.dma.dmach->lastaddr = g_sb16.dsp_info.dma.dmach->adrs.d + g_sb16.dsp_info.dma.dmach->startcount;
		g_sb16.dsp_info.dma.dmach->leng.w = g_sb16.dsp_info.dma.dmach->startcount; // ߂
	}
	g_sb16.dsp_info.dma.last16mode = (g_sb16.dsp_info.dma.mode == CT1741_DMAMODE_16) ? 1 : 0; // 16bit][htO
	g_sb16.mixreg[0x82] &= ~3;
	ct1741_resetpicirq();
	g_sb16.dsp_info.wbusy = 0; // WORKAROUND: ]Jn̏uԂbusyȂWin3.1ŃmCY
	g_sb16.dsp_info.smpcounter = 0;// g_sb16.dsp_info.dma.total / 2;
	g_sb16.dsp_info.smpcounter2 = 0;
	g_sb16.dsp_info.dma.bufdatas = 0;
	g_sb16.dsp_info.dma.bufpos = 0;
	ct1741_playinfo.bufdatasrem = 0;
#if defined(SUPPORT_MULTITHREAD)
	ct1741cs_leave_criticalsection();
#endif
}

REG8 DMACCALL ct1741dmafunc(REG8 func)
{
	SINT32	cnt;
	int bytesPerSample; // 1TṽoCg

	switch (func) {
	case DMAEXT_START:
		// DMA]Jn
		TRACEOUT(("DMAEXT_START DMA_MODE=%d", g_sb16.dsp_info.dma.mode));
		ct1741_startdma();
		bytesPerSample = CT1741_BUF_ALIGN[g_sb16.dsp_info.dma.mode | g_sb16.dsp_info.dma.stereo << 3];
		ct1741_playinfo.playwaitcounter = CT1741_DMA_READINTERVAL;
		cnt = pccore.realclock / g_sb16.dsp_info.freq / bytesPerSample * min(min(g_sb16.dsp_info.dma.dmach->startcount / 4, CT1741_DMA_READINTERVAL) / 4, (int)g_sb16.dsp_info.dma.total * (g_sb16.dsp_info.dma.last16mode ? 2 : 1) / 4); // obt@NbN҂@ZƃmCY
		if (cnt != 0) {
			nevent_set(NEVENT_CT1741, cnt, ct1741_dma, NEVENT_RELATIVE);
		}
		else {
			nevent_setbyms(NEVENT_CT1741, 1, ct1741_dma, NEVENT_RELATIVE);
		}
		break;

	case DMAEXT_END:
		// DMA]B{͂Ŋ荞ނ̂ł́HƎv܂Ŋ荞ނƉȂ܂B
		break;

	case DMAEXT_BREAK:
		// DMA]f
		nevent_reset(NEVENT_CT1741);
		g_sb16.mixreg[0x82] &= ~3;
		ct1741_resetpicirq();
		g_sb16.dsp_info.wbusy = 0;
		if (g_sb16.dmachnum != 0xff) {
			dmac.stat |= (1 << g_sb16.dmachnum);
		}
		if (g_sb16.dsp_info.dma.last16mode) {
			g_sb16.dsp_info.dma.last16mode = 0;
			g_sb16.dsp_info.dma.dmach->startaddr = g_sb16.dsp_info.dma.laststartaddr;
			g_sb16.dsp_info.dma.dmach->startcount = g_sb16.dsp_info.dma.laststartcount;
		}

		break;
	}
	return(0);
}

#endif
