/****************************************************************************
   pallette.c

   Pallette hardware function implementation
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "defs.h"
#include "globals.h"
#include "pallette.h"

UInt16	 pal_flag;		  // flag to update VGA palette (has palette changed ?	1=yes, 0=no)
UInt16   pal_idx = 0;             // current palette 'index' in PCE hardware (ie. value at $0402/$0403)
ColEntry pce_palet[512];          // PCE's palette (what's stored in Hu6260 chip)
UInt16   pce_vga_pal_map[512];    // mapping from PCE's 512 colors to VGA's 256
VGAEntry vga_palet[256];	  // array of PC palette entries (stored in original 9-bit format)


// Initialize palette arrays:
void  Pallette_Init(void) {
   int i;
   for (i = 0; i < 512; i++) {
      pce_palet[i].byte_low  = 0;
      pce_palet[i].byte_high = 0;
      pce_vga_pal_map[i] = 0;
   }
   vga_palet[0].color     = 0;
   vga_palet[0].ref_count = 512;
   for (i = 1; i < 256; i++) {
      vga_palet[i].color     = -1;
      vga_palet[i].ref_count = 0;
   }
   pal_flag = 1;		  // flag to update VGA palette
}

void  VGA_Palet_Incref(UInt16 i) {
   vga_palet[i].ref_count++;
}

// Set a VGA palette entry to 0 (for later reallocation):
void  VGA_Palet_Decref(UInt16 i) {
   if (--vga_palet[i].ref_count == 0) {
      vga_palet[i].color = -1;
   }
}

// Set a VGA palette entry to a color:
void  VGA_Palet_Set(UInt16 i, Int16 col_new) {
   vga_palet[i].color = col_new;
   pal_flag = 1;
}

// Check to see if a VGA palette already contains
// a particular color:
Int16 VGA_Palet_Contains(Int16 col_test) {
   Int16 i;
   Int16 retval = -1;

   for (i = 0; i < 256; i++) {
      if (vga_palet[i].color == col_test) {
	 retval = i;
         break;
      }
   }
   return(retval);
}

// Allocate and set a VGA palette entry to
// a new unique color:
Int16 VGA_Palet_Alloc(Int16 col_new) {
   Int16  i;
   Int16  retval = -1;

   // first, look for an unused palet entry
   // (vga_palet entry 0 is always black, so skip it):

   for (i = 1; i < 256; i++) {
      if (vga_palet[i].ref_count == 0) { 	// not in use
	 VGA_Palet_Set(i, col_new);
	 retval = i;
         break;
      }
   }
   return(retval);
}

// Called when the PCE palette is changed;
// makes all necessary adjustments to the
// VGA palette, and the mapping from the
// PCE palette->VGA palette
void  VGA_Palet_Adjust(void) {
   Int16   col_new;
   Int16   vga_new_idx;
   Int16   i;
   Int16   temp;

   // decrease references to old color:
   VGA_Palet_Decref(pce_vga_pal_map[pal_idx]);

   // find value of new color (is it already allocated in VGA pallette ?)
   col_new = ((pce_palet[pal_idx].byte_high & 0x01) * 256)
	     + pce_palet[pal_idx].byte_low;

   vga_new_idx = VGA_Palet_Contains(col_new);

   if (vga_new_idx != -1) {

      // if new color already exists in palette, add another reference
      VGA_Palet_Incref(vga_new_idx);
      pce_vga_pal_map[pal_idx] = vga_new_idx;

   } else {

      // VGA palette entry does not exist - allocate one, and reference it
      temp = VGA_Palet_Alloc(col_new);
      if (temp != -1) {
         VGA_Palet_Incref(temp);
         pce_vga_pal_map[pal_idx] = temp;
      }
   }

}

/* HARDWARE STUFF: */

/* unsure of exact use of this address */

void  hw_put_0400(UChar byte) {
}


/* these two addresses set on-chip 'pallette index' */

void  hw_put_0402(UChar byte) {
   pal_idx = (pal_idx & 0xff00) + byte;
}

void  hw_put_0403(UChar byte) {
   pal_idx = ((byte & 0x01) * 256) + (pal_idx & 0x00ff);
}


/* these two addresses set pallette entry at current 'pallette index' */
/* Also decode the PC's corresponding color pallette */

void  hw_put_0404(UChar byte) {
   pce_palet[pal_idx].byte_low	= byte;
   VGA_Palet_Adjust();
}

void  hw_put_0405(UChar byte) {
   pce_palet[pal_idx].byte_high = (byte & 0x01);
   VGA_Palet_Adjust();
   pal_idx++;
   pal_idx = pal_idx & 0x01ff;
}



/* unsure of exact use of this address */

UChar hw_get_0400(void) {
   return 0;
}


/* these two addresses read on-chip 'pallette index' */

UChar hw_get_0402(void) {
   return(pal_idx & 0xff);
}

UChar hw_get_0403(void) {
   return(pal_idx>>8);
}


/* these two addresses set pallette entry at current 'pallette index' */

UChar hw_get_0404(void) {
   return(pce_palet[pal_idx].byte_low);
}

UChar hw_get_0405(void) {
   UChar byte = pce_palet[pal_idx].byte_high;
   pal_idx++;
   pal_idx = pal_idx & 0x01ff;
   return(byte);
}
