//  exe.c - Execute CPU instructions
//

#include <conio.h>
#include <stdio.h>
#include <stdlib.h>

#include "defs.h"
#include "globals.h"
#include "exe.h"
#include "grph.h"
#include "hardware.h"
#include "impl.h"
#include "interupt.h"
#include "pallette.h"
#include "timer.h"
#include "video.h"

// registers:

UInt16 reg_pc;
UChar  reg_a;
UChar  reg_x;
UChar  reg_y;
UChar  reg_p;
UChar  reg_s;

UInt16 halt_flag;
long   cycles;
long   frames=0;

// flag-value table (for speed)

UChar flnz_list[256] = {
FL_Z,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,    // 00-0F
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,       // 40-4F
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,       // 70-7F
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // 80-87
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // 90-97
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // A0-A7
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // B0-B7
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // C0-C7
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // D0-D7
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // E0-E7
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,  // F0-F7
FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N,FL_N
};

// Elementary operations:
// - defined here for clarity, brevity,
//   and reduced typographical errors

//Addressing modes:

#define abs_operand(x)	   get_8bit_addr(get_16bit_addr(x))
#define absx_operand(x)    get_8bit_addr(get_16bit_addr(x)+reg_x)
#define absy_operand(x)    get_8bit_addr(get_16bit_addr(x)+reg_y)
#define zp_operand(x)	   get_8bit_zp(imm_operand(x))
#define zpx_operand(x)	   get_8bit_zp(imm_operand(x)+reg_x)
#define zpy_operand(x)	   get_8bit_zp(imm_operand(x)+reg_y)
#define zpind_operand(x)   get_8bit_addr(get_16bit_zp(imm_operand(x)))
#define zpindx_operand(x)  get_8bit_addr(get_16bit_zp(imm_operand(x)+reg_x))
#define zpindy_operand(x)  get_8bit_addr(get_16bit_zp(imm_operand(x))+reg_y)

// Elementary flag check (flags 'N' and 'Z'):

#define chk_flnz_8bit(x) \
  reg_p = (reg_p & (~(FL_N|FL_T|FL_Z)) | flnz_list[x]);

// This code ignores hardware-segment accesses; it should only be used
// to access immediate data (hardware segment does not run code):
inline UChar EXE_NEAR imm_operand(UInt16 addr) {
  register UInt16 memreg = addr>>13;
  return( (UChar) *(mmr_base[memreg] + (addr & 0x1FFF)) );
}

// This is the more generalized access routine:
inline UChar EXE_NEAR get_8bit_addr(UInt16 addr) {
  register UInt16 memreg = addr>>13;
  if (mmr[memreg] == 0xFF)
    return(hardware_get(addr & 0x1FFF));
  else
    return((UChar) *(mmr_base[memreg] + (addr & 0x1FFF)));
}

inline int EXE_NEAR put_8bit_addr(UInt16 addr, UChar byte) {
  register UInt16 memreg = addr>>13;
  if (mmr[memreg] == 0xFF) {
    return(hardware_put((addr & 0x1FFF), byte));
  } else {
    *(mmr_base[memreg] + (addr & 0x1FFF)) = byte;
  }
  return(0);
}

inline UInt16 EXE_NEAR get_16bit_addr(UInt16 addr) {
  register UInt16 memreg = addr>>13;
  UInt16 ret_16bit = (UChar) *(mmr_base[memreg] + (addr & 0x1FFF));
  memreg = (++addr)>>13;
  ret_16bit += (UInt16) ((UChar) *(mmr_base[memreg] + (addr & 0x1FFF)) << 8);

  return(ret_16bit);
}

inline UChar EXE_NEAR get_8bit_zp(UChar zp_addr) {
  return((UChar) *(zp_base + zp_addr) );
}

inline UInt16 EXE_NEAR get_16bit_zp(UChar zp_addr) {
  UInt16 n = *(zp_base + zp_addr);
  n += (*(zp_base + (UChar)(zp_addr+1)) << 8);
  return(n);
}

inline void EXE_NEAR put_8bit_zp(UChar zp_addr, UChar byte) {
  *(zp_base + zp_addr) = byte;
}

inline void EXE_NEAR push_8bit(UChar byte) {
  *(sp_base + reg_s--) = byte;
}

inline UChar EXE_NEAR pull_8bit(void) {
  return((UChar) *(sp_base + ++reg_s) );
}

inline void EXE_NEAR push_16bit(UInt16 addr) {
  *(sp_base + reg_s--) = (UChar)(addr>>8);
  *(sp_base + reg_s--) = (UChar)(addr&0xFF);
  return;
}

inline UInt16 EXE_NEAR pull_16bit(void) {
  UInt16 n = (UChar) *(sp_base + ++reg_s);
  n += (UInt16)(((UChar) *(sp_base + ++reg_s)) << 8);
  return(n);
}

// Implementation of actual opcodes:
//
// Macro 'EXE_NEAR' is defined as 'near' for Intel
// (and other segmented) architectures, for speed.
// On other systems, it can be defined as null

UChar EXE_NEAR adc(UChar acc, UChar val) {
  Int16  sig  = (Char)acc;
  UInt16 usig = (UChar)acc;
  UInt16 temp, temp1;

  if (!(reg_p & FL_D)) {		/* binary mode */
    if (reg_p & FL_C) {
      usig++;
      sig++;
    }
    sig  += (Char)val;
    usig += (UChar)val;
    acc   = usig & 0xFF;

    reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z|FL_C))
	    | (((sig > 127) || (sig < -128)) ? FL_V:0)
	    | ((usig > 255) ? FL_C:0)
	    | flnz_list[acc];

  } else {				/* decimal mode */

// treatment of out-of-range accumulator
// and operand values (non-BCD) is not
// adequately defined.	Nor is overflow
// flag treatment.

    temp1 = (usig & 0xF0) + (val & 0xF0);
    temp  = (usig & 0x0F) + (val & 0x0F);

    if (reg_p & FL_C) { temp++; }

    temp1 += (temp/10);
    temp   = (temp % 10);

    acc    = (temp1 % 10) << 4 + temp;

    reg_p  = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	     | ((temp1 > 9) ? 0:FL_C)
	     | flnz_list[acc];

    cycles++;	/* decimal mode takes an extra cycle */
  }
  return(acc);
}

void EXE_NEAR sbc(UChar val) {
  Int16  sig  = (Char)reg_a;
  UInt16 usig = (UChar)reg_a;
  Int16  temp, temp1;

  if (!(reg_p & FL_D)) {		/* binary mode */
    if (!(reg_p & FL_C)) {
      usig--;
      sig--;
    }
    sig   -= (Char)val;
    usig  -= (UChar)val;
    reg_a  = (usig & 0xFF);
    reg_p  = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z|FL_C))
	     | (((sig > 127) || (sig < -128)) ? FL_V:0)
	     | ((usig > 255) ? 0:FL_C)
	     | flnz_list[reg_a];      /* FL_N, FL_Z */

  } else {				/* decimal mode */

// treatment of out-of-range accumulator
// and operand values (non-bcd) is not
// adequately defined.	Nor is overflow
// flag treatment.

    temp1 = (usig & 0xF0) - (val & 0xF0);
    temp  = (usig & 0x0F) - (val & 0x0F);

    if (!(reg_p & FL_C)) { temp--; }

    while (temp < 0) {
      temp1--;
      temp += 10;
    }

    reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	    | ((temp1 < 0) ? 0:FL_C);

    while (temp1 < 0) {
      temp1 += 10;
    }

    reg_a  = (temp1 % 10) << 4 + temp;
    chk_flnz_8bit(reg_a);

    // decimal mode takes an extra cycle:
    cycles++;
  }
}

int EXE_NEAR adc_abs(void) {
// if flag 'T' is set, use zero-page address specified by register 'X'
// as the accumulator...

  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), abs_operand(reg_pc+1)));
    cycles+=8;
  } else {
    reg_a = adc(reg_a, abs_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR adc_absx(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), absx_operand(reg_pc+1)));
    cycles+=8;
  } else {
    reg_a = adc(reg_a, absx_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR adc_absy(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), absy_operand(reg_pc+1)));
    cycles+=8;
  } else {
    reg_a = adc(reg_a, absy_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR adc_imm(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), imm_operand(reg_pc+1)));
    cycles+=5;
  } else {
    reg_a = adc(reg_a, imm_operand(reg_pc+1));
    cycles+=2;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR adc_zp(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), zp_operand(reg_pc+1)));
    cycles+=7;
  } else {
    reg_a = adc(reg_a, zp_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR adc_zpx(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), zpx_operand(reg_pc+1)));
    cycles+=7;
  } else {
    reg_a = adc(reg_a, zpx_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR adc_zpind(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), zpind_operand(reg_pc+1)));
    cycles+=10;
  } else {
    reg_a = adc(reg_a, zpind_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR adc_zpindx(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), zpindx_operand(reg_pc+1)));
    cycles+=10;
  } else {
    reg_a = adc(reg_a, zpindx_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR adc_zpindy(void) {
  if (reg_p & FL_T) {
    put_8bit_zp(reg_x, adc(get_8bit_zp(reg_x), zpindy_operand(reg_pc+1)));
    cycles+=10;
  } else {
    reg_a = adc(reg_a, zpindy_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR and_abs(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= abs_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a &= abs_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR and_absx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= absx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a &= absx_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR and_absy(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= absy_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a &= absy_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR and_imm(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= imm_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=5;

  } else {
    chk_flnz_8bit(reg_a &= imm_operand(reg_pc+1));
    cycles+=2;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR and_zp(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= zp_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=7;

  } else {
    chk_flnz_8bit(reg_a &= zp_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR and_zpx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= zpx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=7;

  } else {
    chk_flnz_8bit(reg_a &= zpx_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR and_zpind(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= zpind_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a &= zpind_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR and_zpindx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= zpindx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a &= zpindx_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR and_zpindy(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp &= zpindy_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a &= zpindy_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR asl_a(void) {
  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(reg_a<<=1)];
  cycles+=2;
  reg_pc++;
  return 0;
}

int EXE_NEAR asl_abs(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1);
  UChar  temp	   = get_8bit_addr(temp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp<<=1)];
  cycles+=7;

  if (put_8bit_addr(temp_addr,temp)) {	// if error writing to address,
    return 1;				// stop before advancing PC
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR asl_absx(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1)+reg_x;
  UChar  temp	   = get_8bit_addr(temp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp<<=1)];
  cycles+=7;

  if (put_8bit_addr(temp_addr,temp)) {	// if error writing to address,
    return 1;				// stop before advancing PC
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR asl_zp(void) {
  UChar zp_addr = imm_operand(reg_pc+1);
  UChar temp	= get_8bit_zp(zp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp<<=1)];
  cycles+=6;
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  return 0;
}

int EXE_NEAR asl_zpx(void) {
  UChar  zp_addr = imm_operand(reg_pc+1)+reg_x;
  UChar  temp	 = get_8bit_zp(zp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp<<=1)];
  cycles+=6;
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  return 0;
}

int EXE_NEAR bbr0(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x01) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbr1(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x02) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbr2(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x04) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbr3(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x08) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbr4(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x10) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbr5(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x20) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbr6(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x40) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbr7(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x80) {
    reg_pc+=3;
    cycles+=6;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  }
  return 0;
}

int EXE_NEAR bbs0(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x01) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bbs1(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x02) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bbs2(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x04) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bbs3(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x08) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bbs4(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x10) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bbs5(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x20) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bbs6(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x40) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bbs7(void) {
  reg_p &= ~FL_T;
  if (zp_operand(reg_pc+1)&0x80) {
    reg_pc+=(Char)imm_operand(reg_pc+2)+3;
    cycles+=8;
  } else {
    reg_pc+=3;
    cycles+=6;
  }
  return 0;
}

int EXE_NEAR bcc(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_C) {
    reg_pc+=2;
    cycles+=2;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  }
  return 0;
}

int EXE_NEAR bcs(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_C) {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  } else {
    reg_pc+=2;
    cycles+=2;
  }
  return 0;
}

int EXE_NEAR beq(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_Z) {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  } else {
    reg_pc+=2;
    cycles+=2;
  }
  return 0;
}

int EXE_NEAR bit_abs(void) {
  UChar temp = abs_operand(reg_pc+1);
  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80)  ? FL_N:0)
	  | ((temp&0x40)  ? FL_V:0)
	  | ((reg_a&temp) ? 0:FL_Z);
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR bit_absx(void) {
  UChar temp = absx_operand(reg_pc+1);
  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80)  ? FL_N:0)
	  | ((temp&0x40)  ? FL_V:0)
	  | ((reg_a&temp) ? 0:FL_Z);
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR bit_imm(void) {
// orig code (Eyes/Lichty said immediate mode did not affect
//            'N' and 'V' flags):
//reg_p = (reg_p & ~(FL_T|FL_Z))
//	  | ((reg_a & imm_operand(reg_pc+1)) ? 0:FL_Z);

  UChar temp = imm_operand(reg_pc+1);
  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80)  ? FL_N:0)
	  | ((temp&0x40)  ? FL_V:0)
	  | ((reg_a&temp) ? 0:FL_Z);
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR bit_zp(void) {
  UChar temp = zp_operand(reg_pc+1);
  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80)  ? FL_N:0)
	  | ((temp&0x40)  ? FL_V:0)
	  | ((reg_a&temp) ? 0:FL_Z);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR bit_zpx(void) {
  UChar temp = zpx_operand(reg_pc+1);
  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80)  ? FL_N:0)
	  | ((temp&0x40)  ? FL_V:0)
	  | ((reg_a&temp) ? 0:FL_Z);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR bmi(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_N) {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  } else {
    reg_pc+=2;
    cycles+=2;
  }
  return 0;
}

int EXE_NEAR bne(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_Z) {
    reg_pc+=2;
    cycles+=2;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  }
  return 0;
}

int EXE_NEAR bpl(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_N) {
    reg_pc+=2;
    cycles+=2;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  }
  return 0;
}

int EXE_NEAR bra(void) {
  reg_p &= ~FL_T;
  reg_pc+=(Char)imm_operand(reg_pc+1)+2;
  cycles+=4;
  return 0;
}

int EXE_NEAR brk(void) {
  push_16bit(reg_pc+2);
  reg_p &= ~FL_T;
  push_8bit(reg_p|FL_B);
  reg_p =(reg_p & ~FL_D) | FL_I;
  reg_pc=get_16bit_addr(0xFFF6);
  cycles+=8;
  return 0;
}

int EXE_NEAR bsr(void) {
  reg_p &= ~FL_T;
  push_16bit(reg_pc+1);
  reg_pc+=(Char)imm_operand(reg_pc+1)+2;
  cycles+=8;
  return 0;
}

int EXE_NEAR bvc(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_V) {
    reg_pc+=2;
    cycles+=2;
  } else {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  }
  return 0;
}

int EXE_NEAR bvs(void) {
  reg_p &= ~FL_T;
  if (reg_p & FL_V) {
    reg_pc+=(Char)imm_operand(reg_pc+1)+2;
    cycles+=4;
  } else {
    reg_pc+=2;
    cycles+=2;
  }
  return 0;
}

int EXE_NEAR cla(void) {
  reg_p &= ~FL_T;
  reg_a = 0;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR clc(void) {
  reg_p &= ~(FL_T|FL_C);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR cld(void) {
  reg_p &= ~(FL_T|FL_D);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR cli(void) {
  reg_p &= ~(FL_T|FL_I);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR clv(void) {
  reg_p &= ~(FL_V|FL_T);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR clx(void) {
  reg_p &= ~FL_T;
  reg_x = 0;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR cly(void) {
  reg_p &= ~FL_T;
  reg_y = 0;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR cmp_abs(void) {
  UChar temp = abs_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR cmp_absx(void) {
  UChar temp = absx_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR cmp_absy(void) {
  UChar temp = absy_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR cmp_imm(void) {
  UChar temp = imm_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR cmp_zp(void) {
  UChar temp = zp_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR cmp_zpx(void) {
  UChar temp = zpx_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR cmp_zpind(void) {
  UChar temp = zpind_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR cmp_zpindx(void) {
  UChar temp = zpindx_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR cmp_zpindy(void) {
  UChar temp = zpindy_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_a-temp)];
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR cpx_abs(void) {
  UChar temp = abs_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_x < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_x-temp)];
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR cpx_imm(void) {
  UChar temp = imm_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_x < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_x-temp)];
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR cpx_zp(void) {
  UChar temp = zp_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_x < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_x-temp)];
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR cpy_abs(void) {
  UChar temp = abs_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_y < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_y-temp)];
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR cpy_imm(void) {
  UChar temp = imm_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_y < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_y-temp)];
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR cpy_zp(void) {
  UChar temp = zp_operand(reg_pc+1);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_y < temp) ? 0:FL_C)
	  | flnz_list[(UChar)(reg_y-temp)];
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR dec_a(void) {
  chk_flnz_8bit(--reg_a);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR dec_abs(void) {
  UChar  temp;
  UInt16 temp_addr = get_16bit_addr(reg_pc+1);
  chk_flnz_8bit(temp = get_8bit_addr(temp_addr)-1);
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
     return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR dec_absx(void) {
  UChar  temp;
  UInt16 temp_addr = get_16bit_addr(reg_pc+1)+reg_x;
  chk_flnz_8bit(temp = get_8bit_addr(temp_addr)-1);
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR dec_zp(void) {
  UChar  temp;
  UChar  zp_addr = imm_operand(reg_pc+1);
  chk_flnz_8bit(temp = get_8bit_zp(zp_addr)-1);
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR dec_zpx(void) {
  UChar  temp;
  UChar  zp_addr = imm_operand(reg_pc+1)+reg_x;
  chk_flnz_8bit(temp = get_8bit_zp(zp_addr)-1);
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR dex(void) {
  chk_flnz_8bit(--reg_x);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR dey(void) {
  chk_flnz_8bit(--reg_y);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR eor_abs(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= abs_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a ^= abs_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR eor_absx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= absx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a ^= absx_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR eor_absy(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= absy_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a ^= absy_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR eor_imm(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= imm_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=5;

  } else {
    chk_flnz_8bit(reg_a ^= imm_operand(reg_pc+1));
    cycles+=2;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR eor_zp(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= zp_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=7;

  } else {
    chk_flnz_8bit(reg_a ^= zp_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR eor_zpx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= zpx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=7;

  } else {
    chk_flnz_8bit(reg_a ^= zpx_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR eor_zpind(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= zpind_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a ^= zpind_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR eor_zpindx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= zpindx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a ^= zpindx_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR eor_zpindy(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp ^= zpindy_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a ^= zpindy_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR halt(void) {
  return(1);
}

int EXE_NEAR inc_a(void) {
  chk_flnz_8bit(++reg_a);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR inc_abs(void) {
  UChar  temp;
  UInt16 temp_addr = get_16bit_addr(reg_pc+1);
  chk_flnz_8bit(temp = get_8bit_addr(temp_addr)+1);
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR inc_absx(void) {
  UChar  temp;
  UInt16 temp_addr = get_16bit_addr(reg_pc+1)+reg_x;
  chk_flnz_8bit(temp = get_8bit_addr(temp_addr)+1);
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR inc_zp(void) {
  UChar temp;
  UChar zp_addr = imm_operand(reg_pc+1);
  chk_flnz_8bit(temp = get_8bit_zp(zp_addr)+1);
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR inc_zpx(void) {
  UChar temp;
  UChar zp_addr = imm_operand(reg_pc+1)+reg_x;
  chk_flnz_8bit(temp = get_8bit_zp(zp_addr)+1);
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR inx(void) {
  chk_flnz_8bit(++reg_x);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR iny(void) {
  chk_flnz_8bit(++reg_y);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR jmp(void) {
  reg_p &= ~FL_T;
  reg_pc = get_16bit_addr(reg_pc+1);
  cycles+=4;
  return 0;
}

int EXE_NEAR jmp_absind(void) {
  reg_p &= ~FL_T;
  reg_pc = get_16bit_addr(get_16bit_addr(reg_pc+1));
  cycles+=7;
  return 0;
}

int EXE_NEAR jmp_absindx(void) {
  reg_p &= ~FL_T;
  reg_pc = get_16bit_addr(get_16bit_addr(reg_pc+1)+reg_x);
  cycles+=7;
  return 0;
}

int EXE_NEAR jsr(void) {
  reg_p &= ~FL_T;
  push_16bit(reg_pc+2);
  reg_pc = get_16bit_addr(reg_pc+1);
  cycles+=7;
  return 0;
}

int EXE_NEAR lda_abs(void) {
  chk_flnz_8bit(reg_a = abs_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR lda_absx(void) {
  chk_flnz_8bit(reg_a = absx_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR lda_absy(void) {
  chk_flnz_8bit(reg_a = absy_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR lda_imm(void) {
  chk_flnz_8bit(reg_a = imm_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR lda_zp(void) {
  chk_flnz_8bit(reg_a = zp_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR lda_zpx(void) {
  chk_flnz_8bit(reg_a = zpx_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR lda_zpind(void) {
  chk_flnz_8bit(reg_a = zpind_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR lda_zpindx(void) {
  chk_flnz_8bit(reg_a = zpindx_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR lda_zpindy(void) {
  chk_flnz_8bit(reg_a = zpindy_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR ldx_abs(void) {
  chk_flnz_8bit(reg_x = abs_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR ldx_absy(void) {
  chk_flnz_8bit(reg_x = absy_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR ldx_imm(void) {
  chk_flnz_8bit(reg_x = imm_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR ldx_zp(void) {
  chk_flnz_8bit(reg_x = zp_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR ldx_zpy(void) {
  chk_flnz_8bit(reg_x = zpy_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR ldy_abs(void) {
  chk_flnz_8bit(reg_y = abs_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR ldy_absx(void) {
  chk_flnz_8bit(reg_y = absx_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR ldy_imm(void) {
  chk_flnz_8bit(reg_y = imm_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR ldy_zp(void) {
  chk_flnz_8bit(reg_y = zp_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR ldy_zpx(void) {
  chk_flnz_8bit(reg_y = zpx_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR lsr_a(void) {
  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a&1) ? FL_C:0)
	  | flnz_list[(reg_a/=2)];
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR lsr_abs(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1);
  UChar  temp	   = get_8bit_addr(temp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp&1) ? FL_C:0)
	  | flnz_list[(temp/=2)];
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR lsr_absx(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1)+reg_x;
  UChar  temp	   = get_8bit_addr(temp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp&1) ? FL_C:0)
	  | flnz_list[(temp/=2)];
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR lsr_zp(void) {
  UChar  zp_addr = imm_operand(reg_pc+1);
  UChar  temp	 = get_8bit_zp(zp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp&1) ? FL_C:0)
	  | flnz_list[(temp/=2)];
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR lsr_zpx(void) {
  UChar  zp_addr = imm_operand(reg_pc+1)+reg_x;
  UChar  temp	 = get_8bit_zp(zp_addr);

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp&1) ? FL_C:0)
	  | flnz_list[(temp/=2)];
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR nop(void)	{
  reg_p &= ~FL_T;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR ora_abs(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= abs_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a |= abs_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR ora_absx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= absx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a |= absx_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR ora_absy(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= absy_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=8;

  } else {
    chk_flnz_8bit(reg_a |= absy_operand(reg_pc+1));
    cycles+=5;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR ora_imm(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= imm_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=5;

  } else {
    chk_flnz_8bit(reg_a |= imm_operand(reg_pc+1));
    cycles+=2;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR ora_zp(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= zp_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=7;

  } else {
    chk_flnz_8bit(reg_a |= zp_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR ora_zpx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= zpx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=7;

  } else {
    chk_flnz_8bit(reg_a |= zpx_operand(reg_pc+1));
    cycles+=4;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR ora_zpind(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= zpind_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a |= zpind_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR ora_zpindx(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= zpindx_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a |= zpindx_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR ora_zpindy(void) {
  if (reg_p & FL_T) {
    UChar temp = get_8bit_zp(reg_x);
    chk_flnz_8bit(temp |= zpindy_operand(reg_pc+1));
    put_8bit_zp(reg_x, temp);
    cycles+=10;

  } else {
    chk_flnz_8bit(reg_a |= zpindy_operand(reg_pc+1));
    cycles+=7;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR pha(void) {
  reg_p &= ~FL_T;
  push_8bit(reg_a);
  reg_pc++;
  cycles+=3;
  return 0;
}

int EXE_NEAR php(void) {
  reg_p &= ~FL_T;
  push_8bit(reg_p);
  reg_pc++;
  cycles+=3;
  return 0;
}

int EXE_NEAR phx(void) {
  reg_p &= ~FL_T;
  push_8bit(reg_x);
  reg_pc++;
  cycles+=3;
  return 0;
}

int EXE_NEAR phy(void) {
  reg_p &= ~FL_T;
  push_8bit(reg_y);
  reg_pc++;
  cycles+=3;
  return 0;
}

int EXE_NEAR pla(void) {
  chk_flnz_8bit(reg_a = pull_8bit());
  reg_pc++;
  cycles+=4;
  return 0;
}

int EXE_NEAR plp(void) {
  reg_p = pull_8bit();
  reg_pc++;
  cycles+=4;
  return 0;
}

int EXE_NEAR plx(void) {
  chk_flnz_8bit(reg_x = pull_8bit());
  reg_pc++;
  cycles+=4;
  return 0;
}

int EXE_NEAR ply(void) {
  chk_flnz_8bit(reg_y = pull_8bit());
  reg_pc++;
  cycles+=4;
  return 0;
}

int EXE_NEAR rmb0(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x01));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rmb1(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x02));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rmb2(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x04));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rmb3(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x08));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rmb4(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x10));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rmb5(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x20));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rmb6(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x40));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rmb7(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) & (~0x80));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR rol_a(void) {
  UChar flg_tmp = (reg_p & FL_C) ? 1:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(reg_a = (reg_a<<1)+flg_tmp)];
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR rol_abs(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1);
  UChar  temp	   = get_8bit_addr(temp_addr);
  UChar  flg_tmp   = (reg_p & FL_C) ? 1:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp<<1)+flg_tmp)];
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR rol_absx(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1)+reg_x;
  UChar  temp	   = get_8bit_addr(temp_addr);
  UChar  flg_tmp   = (reg_p & FL_C) ? 1:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp<<1)+flg_tmp)];
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR rol_zp(void) {
  UChar  zp_addr = imm_operand(reg_pc+1);
  UChar  temp	 = get_8bit_zp(zp_addr);
  UChar  flg_tmp = (reg_p & FL_C) ? 1:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp<<1)+flg_tmp)];
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR rol_zpx(void) {
  UChar  zp_addr = imm_operand(reg_pc+1)+reg_x;
  UChar  temp	 = get_8bit_zp(zp_addr);
  UChar  flg_tmp = (reg_p & FL_C) ? 1:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x80) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp<<1)+flg_tmp)];
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR ror_a(void) {
  UChar flg_tmp = (reg_p & FL_C) ? 0x80:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((reg_a & 0x01) ? FL_C:0)
	  | flnz_list[(UChar)(reg_a = (reg_a/2)+flg_tmp)];
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR ror_abs(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1);
  UChar  temp	   = get_8bit_addr(temp_addr);
  UChar  flg_tmp   = (reg_p & FL_C) ? 0x80:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x01) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp/2)+flg_tmp)];
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR ror_absx(void) {
  UInt16 temp_addr = get_16bit_addr(reg_pc+1)+reg_x;
  UChar  temp	   = get_8bit_addr(temp_addr);
  UChar  flg_tmp   = (reg_p & FL_C) ? 0x80:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x01) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp/2)+flg_tmp)];
  cycles+=7;
  if (put_8bit_addr(temp_addr, temp)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR ror_zp(void) {
  UChar  zp_addr = imm_operand(reg_pc+1);
  UChar  temp	 = get_8bit_zp(zp_addr);
  UChar  flg_tmp = (reg_p & FL_C) ? 0x80:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x01) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp/2)+flg_tmp)];
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR ror_zpx(void) {
  UChar  zp_addr = imm_operand(reg_pc+1)+reg_x;
  UChar  temp	 = get_8bit_zp(zp_addr);
  UChar  flg_tmp = (reg_p & FL_C) ? 0x80:0;

  reg_p = (reg_p & ~(FL_N|FL_T|FL_Z|FL_C))
	  | ((temp & 0x01) ? FL_C:0)
	  | flnz_list[(UChar)(temp = (temp/2)+flg_tmp)];
  put_8bit_zp(zp_addr, temp);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR rti(void) {
  /* FL_B reset in RTI */
  reg_p = pull_8bit() & ~FL_B;
  reg_pc = pull_16bit();
  cycles+=7;
  return 0;
}

int EXE_NEAR rts(void) {
  reg_p &= ~FL_T;
  reg_pc = pull_16bit()+1;
  cycles+=7;
  return 0;
}

int EXE_NEAR sax(void) {
  UChar temp = reg_x;
  reg_p &= ~FL_T;
  reg_x = reg_a;
  reg_a = temp;
  reg_pc++;
  cycles+=3;
  return 0;
}

int EXE_NEAR say(void) {
  UChar temp = reg_y;
  reg_p &= ~FL_T;
  reg_y = reg_a;
  reg_a = temp;
  reg_pc++;
  cycles+=3;
  return 0;
}

int EXE_NEAR sbc_abs(void) {
  sbc(abs_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR sbc_absx(void) {
  sbc(absx_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR sbc_absy(void) {
  sbc(absy_operand(reg_pc+1));
  reg_pc+=3;
  cycles+=5;
  return 0;
}

int EXE_NEAR sbc_imm(void) {
  sbc(imm_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=2;
  return 0;
}

int EXE_NEAR sbc_zp(void) {
  sbc(zp_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sbc_zpx(void) {
  sbc(zpx_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sbc_zpind(void) {
  sbc(zpind_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR sbc_zpindx(void) {
  sbc(zpindx_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR sbc_zpindy(void) {
  sbc(zpindy_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR sec(void) {
  reg_p = (reg_p|FL_C) & ~FL_T;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR sed(void) {
  reg_p = (reg_p|FL_D) & ~FL_T;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR sei(void) {
  reg_p = (reg_p|FL_I) & ~FL_T;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR set(void) {
  reg_p |= FL_T;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR smb0(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x01);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR smb1(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x02);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR smb2(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x04);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR smb3(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x08);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR smb4(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x10);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR smb5(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x20);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR smb6(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x40);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR smb7(void) {
  reg_p &= ~FL_T;
  UChar temp = imm_operand(reg_pc+1);
  put_8bit_zp(temp, get_8bit_zp(temp) | 0x80);
  reg_pc+=2;
  cycles+=7;
  return 0;
}

int EXE_NEAR st0(void) {
  reg_p &= ~FL_T;
  hw_put_0000(imm_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR st1(void) {
  reg_p &= ~FL_T;
  hw_put_0002(imm_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR st2(void) {
  reg_p &= ~FL_T;
  hw_put_0003(imm_operand(reg_pc+1));
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sta_abs(void) {
  reg_p &= ~FL_T;
  cycles+=5;
  if (put_8bit_addr(get_16bit_addr(reg_pc+1), reg_a)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR sta_absx(void) {
  reg_p &= ~FL_T;
  cycles+=5;
  if (put_8bit_addr(get_16bit_addr(reg_pc+1)+reg_x, reg_a)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR sta_absy(void) {
  reg_p &= ~FL_T;
  cycles+=5;
  if (put_8bit_addr(get_16bit_addr(reg_pc+1)+reg_y, reg_a)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR sta_zp(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1), reg_a);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sta_zpx(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1)+reg_x, reg_a);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sta_zpind(void) {
  reg_p &= ~FL_T;
  cycles+=7;
  if (put_8bit_addr(get_16bit_zp(imm_operand(reg_pc+1)), reg_a)) {
    return 1;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR sta_zpindx(void) {
  reg_p &= ~FL_T;
  cycles+=7;
  if (put_8bit_addr(get_16bit_zp(imm_operand(reg_pc+1)+reg_x), reg_a)) {
    return 1;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR sta_zpindy(void) {
  reg_p &= ~FL_T;
  cycles+=7;
  if (put_8bit_addr(get_16bit_zp(imm_operand(reg_pc+1))+reg_y, reg_a)) {
    return 1;
  }
  reg_pc+=2;
  return 0;
}

int EXE_NEAR stx_abs(void) {
  reg_p &= ~FL_T;
  cycles+=5;
  if (put_8bit_addr(get_16bit_addr(reg_pc+1), reg_x)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR stx_zp(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1), reg_x);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR stx_zpy(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1)+reg_y, reg_x);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sty_abs(void) {
  reg_p &= ~FL_T;
  cycles+=5;
  if (put_8bit_addr(get_16bit_addr(reg_pc+1), reg_y)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR sty_zp(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1), reg_y);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sty_zpx(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1)+reg_x, reg_y);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR stz_abs(void) {
  reg_p &= ~FL_T;
  cycles+=5;
  if (put_8bit_addr(get_16bit_addr(reg_pc+1), 0)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR stz_absx(void) {
  reg_p &= ~FL_T;
  cycles+=5;
  if (put_8bit_addr((get_16bit_addr(reg_pc+1)+reg_x), 0)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR stz_zp(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1), 0);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR stz_zpx(void) {
  reg_p &= ~FL_T;
  put_8bit_zp(imm_operand(reg_pc+1)+reg_x, 0);
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR sxy(void) {
  UChar temp = reg_y;
  reg_p &= ~FL_T;
  reg_y = reg_x;
  reg_x = temp;
  reg_pc++;
  cycles+=3;
  return 0;
}

int EXE_NEAR tai(void) {
  UInt16 from, to, len, alternate;
  int err = 0;

  reg_p &= ~FL_T;
  from = get_16bit_addr(reg_pc+1);
  to   = get_16bit_addr(reg_pc+3);
  len  = get_16bit_addr(reg_pc+5);
  alternate = 0;

  cycles+=(6 * len) + 17;
  while ((len-- != 0) && (err == 0)) {
    err = put_8bit_addr(to++, get_8bit_addr(from+alternate));
    alternate ^= 1;
  }
  if (!err) reg_pc+=7;
  return(err);
}

int EXE_NEAR tam(void) {
  UInt16 i;
  UChar bitfld = imm_operand(reg_pc+1);

  for (i = 0; i < 8; i++) {
    if (bitfld & (1 << i)) {
      mmr[i] = reg_a;
      mmr_base[i] = base[reg_a];
    }
  }
  reg_p &= ~FL_T;
  reg_pc+=2;
  cycles+=5;
  return 0;
}

int EXE_NEAR tax(void) {
  chk_flnz_8bit(reg_x = reg_a);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR tay(void) {
  chk_flnz_8bit(reg_y = reg_a);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR tdd(void) {
  UInt16 from, to, len;
  int err = 0;

  reg_p &= ~FL_T;
  from = get_16bit_addr(reg_pc+1);
  to   = get_16bit_addr(reg_pc+3);
  len  = get_16bit_addr(reg_pc+5);

  cycles+=(6 * len) + 17;
  while ((len-- != 0) && (err == 0)) {
    err = put_8bit_addr(to--, get_8bit_addr(from--));
  }
  if (!err) reg_pc+=7;
  return(err);
}

int EXE_NEAR tia(void) {
  UInt16 from, to, len, alternate;
  int err = 0;

  reg_p &= ~FL_T;
  from = get_16bit_addr(reg_pc+1);
  to   = get_16bit_addr(reg_pc+3);
  len  = get_16bit_addr(reg_pc+5);
  alternate = 0;

  cycles+=(6 * len) + 17;
  while ((len-- != 0) && (err == 0)) {
    err = put_8bit_addr(to+alternate, get_8bit_addr(from++));
    alternate ^= 1;
  }
  if (!err) reg_pc+=7;
  return(err);
}

int EXE_NEAR tii(void) {
  UInt16 from, to, len;
  int err = 0;

  reg_p &= ~FL_T;
  from = get_16bit_addr(reg_pc+1);
  to   = get_16bit_addr(reg_pc+3);
  len  = get_16bit_addr(reg_pc+5);

  cycles+=(6 * len) + 17;
  while ((len-- != 0) && (err == 0)) {
    err = put_8bit_addr(to++, get_8bit_addr(from++));
  }
  if (!err) reg_pc+=7;
  return(err);
}

int EXE_NEAR tin(void) {
  UInt16 from, to, len;
  int err = 0;

  reg_p &= ~FL_T;
  from = get_16bit_addr(reg_pc+1);
  to   = get_16bit_addr(reg_pc+3);
  len  = get_16bit_addr(reg_pc+5);

  cycles+=(6 * len) + 17;
  while ((len-- != 0) && (err == 0)) {
    err = put_8bit_addr(to, get_8bit_addr(from++));
  }
  if (!err) reg_pc+=7;
  return(err);
}

int EXE_NEAR tma(void) {
  UInt16 i;
  UChar bitfld = imm_operand(reg_pc+1);

  for (i = 0; i < 8; i++) {
    if (bitfld & (1 << i)) {
      reg_a = mmr[i];
    }
  }
  reg_p &= ~FL_T;
  reg_pc+=2;
  cycles+=4;
  return 0;
}

int EXE_NEAR trb_abs(void) {
  UInt16 abs_addr = get_16bit_addr(reg_pc+1);
  UChar  temp	  = get_8bit_addr(abs_addr);
  UChar  temp1	  = (~reg_a) & temp;

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp1&0x80) ? FL_N:0)
	  | ((temp1&0x40) ? FL_V:0)
	  | ((temp & reg_a) ? 0:FL_Z);
  cycles+=7;
  if (put_8bit_addr(abs_addr, temp1)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR trb_zp(void) {
  UChar zp_addr  = imm_operand(reg_pc+1);
  UChar temp	 = get_8bit_zp(zp_addr);
  UChar temp1	 = (~reg_a) & temp;

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp1&0x80) ? FL_N:0)
	  | ((temp1&0x40) ? FL_V:0)
	  | ((temp & reg_a) ? 0:FL_Z);
  put_8bit_zp(zp_addr, temp1);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR tsb_abs(void) {
  UInt16 abs_addr = get_16bit_addr(reg_pc+1);
  UChar  temp	  = get_8bit_addr(abs_addr);
  UChar  temp1	  = reg_a | temp;

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp1&0x80) ? FL_N:0)
	  | ((temp1&0x40) ? FL_V:0)
	  | ((temp & reg_a) ? 0:FL_Z);
  cycles+=7;
  if (put_8bit_addr(abs_addr, temp1)) {
    return 1;
  }
  reg_pc+=3;
  return 0;
}

int EXE_NEAR tsb_zp(void) {
  UChar zp_addr  = imm_operand(reg_pc+1);
  UChar temp	 = get_8bit_zp(zp_addr);
  UChar temp1	 = reg_a | temp;

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp1&0x80) ? FL_N:0)
	  | ((temp1&0x40) ? FL_V:0)
	  | ((temp & reg_a) ? 0:FL_Z);
  put_8bit_zp(zp_addr, temp1);
  reg_pc+=2;
  cycles+=6;
  return 0;
}

int EXE_NEAR tstins_abs(void) {
  UChar  imm	  = imm_operand(reg_pc+1);
  UChar  temp	  = abs_operand(reg_pc+2);

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80) ? FL_N:0)
	  | ((temp&0x40) ? FL_V:0)
	  | ((temp&imm)  ? 0:FL_Z);
  cycles+=8;
  reg_pc+=4;
  return 0;
}

int EXE_NEAR tstins_absx(void) {
  UChar  imm	  = imm_operand(reg_pc+1);
  UChar  temp	  = absx_operand(reg_pc+2);

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80) ? FL_N:0)
	  | ((temp&0x40) ? FL_V:0)
	  | ((temp&imm)  ? 0:FL_Z);
  cycles+=8;
  reg_pc+=4;
  return 0;
}

int EXE_NEAR tstins_zp(void) {
  UChar imm	= imm_operand(reg_pc+1);
  UChar temp	= zp_operand(reg_pc+2);

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80) ? FL_N:0)
	  | ((temp&0x40) ? FL_V:0)
	  | ((temp&imm)  ? 0:FL_Z);
  cycles+=7;
  reg_pc+=3;
  return 0;
}

int EXE_NEAR tstins_zpx(void) {
  UChar imm	= imm_operand(reg_pc+1);
  UChar temp	= zpx_operand(reg_pc+2);

  reg_p = (reg_p & ~(FL_N|FL_V|FL_T|FL_Z))
	  | ((temp&0x80) ? FL_N:0)
	  | ((temp&0x40) ? FL_V:0)
	  | ((temp&imm)  ? 0:FL_Z);
  cycles+=7;
  reg_pc+=3;
  return 0;
}


int EXE_NEAR tsx(void) {
  chk_flnz_8bit(reg_x = reg_s);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR txa(void) {
  chk_flnz_8bit(reg_a = reg_x);
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR txs(void) {
  reg_p &= ~FL_T;
  reg_s = reg_x;
  reg_pc++;
  cycles+=2;
  return 0;
}

int EXE_NEAR tya(void) {
  chk_flnz_8bit(reg_a = reg_y);
  reg_pc++;
  cycles+=2;
  return 0;
}

// perform machine operations for IRQ2:

int EXE_NEAR int_irq2 (void) {
  if ((irq_mask & FL_IRQ2) != 0) {   // interrupt disabled
     return 0;
  }
//if ((irq_register & FL_IRQ2) == 0) {	 // interrupt disabled ?
//   return 0;
//}
  cycles+=7;
  push_16bit(reg_pc);
  push_8bit(reg_p);
  reg_p = (reg_p & ~FL_D) | FL_I;
  reg_pc = get_16bit_addr(0xFFF6);
  return 0;
}

// perform machine operations for IRQ1 (video interrupt):

int EXE_NEAR int_irq1 (void) {
  if ((irq_mask & FL_IRQ1) != 0) {   // interrupt disabled
     return 0;
  }
//if ((irq_register & FL_IRQ1) == 0) {	 // interrupt disabled ?
//   return 0;
//}
  cycles+=7;
  push_16bit(reg_pc);
  push_8bit(reg_p);
  reg_p = (reg_p & ~FL_D) | FL_I;
  reg_pc = get_16bit_addr(0xFFF8);
  return 0;
}

// perform machine operations for timer interrupt:

int EXE_NEAR int_tiq(void) {
  if ((irq_mask & FL_TIQ) != 0) {    // interrupt disabled
     return 0;
  }
//if ((irq_register & FL_TIQ) == 0) {	// interrupt disabled ?
//   return 0;
//}
  cycles+=7;
  push_16bit(reg_pc);
  push_8bit(reg_p);
  reg_p = (reg_p & ~FL_D) | FL_I;
  reg_pc = get_16bit_addr(0xFFFA);
  return 0;
}

// Execute a single instruction :

void EXE_FAR exe_instruct(void) {
  (*optable[*(mmr_base[reg_pc>>13]+(reg_pc&0x1FFF))].func_exe)();
}

// Execute instructions as a machine would, including all
// important (known) interrupts, hardware functions, and
// actual video display on the hardware
//
// Until the following happens:
// (1) An unknown instruction is to be executed
// (2) An unknown hardware access is performed
// (3) <ESC> key is hit

void EXE_FAR exe_go(void) {
  int err = 0;

  init_pckeyint();       // grab PC's keyboard-read interrupt
  init_pctimerint();     // grab PC's timer interrupt

  if (gr_mode != 0) set_palette();
  while (!err) {
    err = (*optable[*(mmr_base[reg_pc>>13]+(reg_pc&0x1FFF))].func_exe)();

    // TIMER stuff:
    if (tiq_flag) {
       tiq_flag = 0;
       if (!(reg_p & FL_I)) int_tiq();
    }

    // HSYNC stuff:
    if (cycles > 455) {
      cycles -= 455;
      lines++;

      // VSYNC stuff:
//    if (lines == 260) {
      if (vsync_flag) {
	vsync_flag = 0;
	lines = 0;
	frames++;
	read_pckeyjoys();
	if ((gr_mode != 0) && ((frames & 3) == 1)) {
	  if (pal_flag == 1) set_palette();
	  put_graphics();
	}
	stat_6270 = 0x20;
	if (key_array[1] != 0) err = 1;
//	if (VSync_Int)	   // as if...
	   if (!(reg_p & FL_I)) int_irq1();

      } else {

	// HSync-only stuff (NOT VSync...):

//	HLine_Draw(lines);
	if (lines < 3) {
	   stat_6270 = 0x24;  // VSYNC *and* HSYNC
	} else {
	   stat_6270 = 0x04;
	}
	if (HSync_Int) {
	   if (!(reg_p & FL_I)) int_irq1();
	}
      }
    }
  }
  exit_pctimerint();     // release PC's timer interrupt
  exit_pckeyint();       // release PC's keyboard-read interrupt
}

