/****************************************************************************
   format.c
   line-formatting routines for disassembler
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "defs.h"
#include "format.h"


/* lineprinters, to keep code clean. */

void lineprint1(Char *outf, long ctr, UChar * op, Char *outstring)
{
   sprintf(outf, "%04lX: %02X              %s", ctr, *op, outstring);
}

void lineprint2(Char *outf, long ctr, UChar * op, Char *outstring)
{
   sprintf(outf, "%04lX: %02X %02X           %s", ctr, *op, *(op+1), outstring);
}

void lineprint3(Char *outf, long ctr, UChar * op, Char *outstring)
{
   sprintf(outf, "%04lX: %02X %02X %02X        %s", ctr, *op, *(op+1), *(op+2), outstring);
}

void lineprint4(Char *outf, long ctr, UChar * op, Char *outstring)
{
   sprintf(outf, "%04lX: %02X %02X %02X %02X     %s", ctr, *op, *(op+1), *(op+2), *(op+3), outstring);
}

void lineprint7(Char *outf, long ctr, UChar * op, Char *outstring)
{
   sprintf(outf, "%04lX: %02X%02X%02X%02X%02X%02X%02X  %s",
                 ctr, *op, *(op+1), *(op+2), *(op+3), *(op+4),
                      *(op+5), *(op+6), outstring);
}


/* common addressing-mode formatters */
/* look/act as wrappers around lineprint functions */

void implicit(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s", str);
   lineprint1(outf, ctr, op, buf);
}

void immed(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  #$%02X", str, *(op+1));
   lineprint2(outf, ctr, op, buf);
}

void relative(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   long newadd;
   Int16 offset = (int) *(op+1);


   if (offset >= 128)
      offset -= 256;

   newadd = (ctr+2) + offset;

   sprintf(buf, "%-4s  $%04lX", str, newadd);
   lineprint2(outf, ctr, op, buf);
}

void ind_zp(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  $%02X", str, *(op+1));
   lineprint2(outf, ctr, op, buf);
}

void ind_zpx(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  $%02X,X", str, *(op+1));
   lineprint2(outf, ctr, op, buf);
}

void ind_zpy(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  $%02X,Y", str, *(op+1));
   lineprint2(outf, ctr, op, buf);
}

void ind_zpind(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  ($%02X)", str, *(op+1));
   lineprint2(outf, ctr, op, buf);
}

void ind_zpix(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  ($%02X,X)", str, *(op+1));
   lineprint2(outf, ctr, op, buf);
}

void ind_zpiy(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  ($%02X),Y", str, *(op+1));
   lineprint2(outf, ctr, op, buf);
}

void absol(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  $%02X%02X", str, *(op+2), *(op+1));
   lineprint3(outf, ctr, op, buf);
}

void absind(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  ($%02X%02X)", str, *(op+2), *(op+1));
   lineprint3(outf, ctr, op, buf);
}

void absindx(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  ($%02X%02X,X)", str, *(op+2), *(op+1));
   lineprint3(outf, ctr, op, buf);
}

void absx(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  $%02X%02X,X", str, *(op+2), *(op+1));
   lineprint3(outf, ctr, op, buf);
}

void absy(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   sprintf(buf, "%-4s  $%02X%02X,Y", str, *(op+2), *(op+1));
   lineprint3(outf, ctr, op, buf);
}

void pseudorel(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];
   long newadd;
   Int16  offset = (int) *(op+2);

   if (offset >= 128)
      offset -= 256;

   newadd = (ctr+3) + offset;

   sprintf(buf, "%-4s  $%02X, $%04lX", str, *(op+1), newadd);
   lineprint3(outf, ctr, op, buf);
}

void tst_zp(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];

   sprintf(buf, "%-4s  $%02X, $%02X", str, *(op+1), *(op+2));
   lineprint3(outf, ctr, op, buf);
}

void tst_abs(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];

   sprintf(buf, "%-4s  $%02X, $%02X%02X", str, *(op+1), *(op+3), *(op+2));
   lineprint4(outf, ctr, op, buf);
}

void tst_zpx(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];

   sprintf(buf, "%-4s  $%02X, $%02X,X", str, *(op+1), *(op+2));
   lineprint3(outf, ctr, op, buf);
}

void tst_absx(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];

   sprintf(buf, "%-4s  $%02X, $%02X%02X,X", str, *(op+1), *(op+3), *(op+2));
   lineprint4(outf, ctr, op, buf);
}

void xfer(Char *outf, long ctr, UChar * op, Char *str)
{
   Char buf[256];

   sprintf(buf, "%-4s  $%02X%02X, $%02X%02X, $%02X%02X",
                str, *(op+2), *(op+1), *(op+4), *(op+3), *(op+6), *(op+5));
   lineprint7(outf, ctr, op, buf);
}
