/****************************************************************************
 * simtg.c - TurboGrafx simulation environment
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <math.h>
#include <sys/stat.h>
#include <errno.h>
#include <conio.h>
#include <bios.h>

#include "defs.h"
#include "format.h"
#include "h6280.h"
#include "dis.h"
#include "display.h"
#include "grph.h"
#include "impl.h"
#include "video.h"
#include "pallette.h"
#include "psg.h"
/* #include "joy.h" */


#define _BIOS_CLK_TCK  18.2


/***********/
/* GLOBALS */
/***********/

/* PROCESSOR stuff */

UChar * pce_memory;
UChar * base[256];
UChar * zp_base;
UChar * sp_base;
UChar * mmr_base[8];

UChar	mmr[8];

/* misc */

Char	buf[128];
UInt16	bkpt;


long cvtnum(char * string)
{
   long   value = 0;
   char * c = string;

   while (*c != '\0') {

     value *= 16;

     if ((*c >= '0') && (*c <= '9')) {
       value += (*c - '0');

     } else if ((*c >= 'A') && (*c <= 'F')) {
       value += (*c - 'A' + 10);

     } else if ((*c >= 'a') && (*c <= 'f')) {
       value += (*c - 'a' + 10);

     } else {

       return(-1);
     }
     c++;
   }
   return(value);
}


void inputnum(char * buffer, int len)
{
   char  ch;

   while (len) {
      ch = getch();
      if ((ch >= '0' && ch <= '9') ||
          (ch >= 'A' && ch <= 'F') ||
          (ch >= 'a' && ch <= 'f')) {

	 *(buffer++) = (char)ch;
	 len--;
      }
   }
   *buffer = '\0';
}


int main(int argc, char *argv[])
{
   FILE *inf;
   FILE *outf;
   char linebuf[256];
   char filename[256];
   char *  ptr;
   UInt16  addr;
   UInt16  vidaddr;
   UChar   op;
   long length;
   int	i, j;
   int	temp, break_out;
   int	memreg;
   UChar hold;
   register void (*func)(void);
   register unsigned int reg;
   unsigned long starttime, endtime;
   unsigned ret;
   float duration, vmhz;

   if (argc < 2) {
      fprintf(stderr, "Usage:  %s <infile>\n", argv[0]);
      exit(1);
   }

   /* Open input and output files. */
/*   strcpy(filename, ".\\roms\\"); */
/*   strcat(filename, argv[1]);  */
   strcpy(filename, argv[1]);
   if ((inf = fopen(filename, "rb")) == NULL) {
      fprintf(stderr, "%s: can't open %s\n", argv[0], filename);
      exit(1);
   }

/* initialize PC structures */
   impl_init();

   init_graphics();
   PSG_Init();

/* initialize video subsystem */

   Video_Init();
   Pallette_Init();

/******************************************/
/* Here, we allocate memory in a loop.	  */
/* The memory is allocated in 8K chunks   */
/* to be used as base addresses for	  */
/* segments.				  */
/*					  */
/* The index 'i' is the value which would */
/* be contained in an MMR register for an */
/* operation on this memory.		  */
/*					  */
/* The size and locations of TG memory	  */
/* allocated in this section are enough   */
/* for a 2Mbit card game		  */
/*					  */
/* Card Memory is at	$00-$0F 	  */
/* NVRAM is at		$F7 (?) 	  */
/* Scratchpad RAM is at $F8		  */
/* Hardware I/O is at	$FF		  */
/*					  */
/* All of these areas must be allocated   */
/* because access attempts at unallocated */
/* areas of memory may cause SEGV's       */
/******************************************/

   pce_memory = malloc(256*8192);

   for (i = 0; i < 256; i++) {
      base[i] = (UChar *)(pce_memory + i*8192);
   }
   zp_base = base[0xf8];		  /* set zero page base address  */
   sp_base = base[0xf8] + 0x100;	  /* set stack page base address */

   for (i = 0; i < 8; i++) {             /* init mmr var's */
      mmr[i] = 0;
      mmr_base[i] = base[0];
   }

   /* fill buffers */
   break_out = 0;

   for (i = 0; (i < 128) && (!break_out); i++) {

      ptr = base[i];

      for (j = 0; j < 0x2000; j++) {
	 if ((temp = fgetc(inf)) == EOF) {
            printf("EOF at block %d, offset %d\n", temp, i, j);
            break_out = 1;
	    break;
	 }
	 *ptr = (char)temp;
	 ptr++;
      }
   }

   reg_pc = ((UChar)(*(base[0] + 0x1fff)) * 256) +
	     (UChar)(*(base[0] + 0x1ffe));

   clrscr();				 /* init screen */
   put_frame();


/* loop until escape key hit */

   while (1) {
      put_regs();
      put_disasm(reg_pc);
      put_cmdprompt();

      i = getch();			  /* get keyboard hit */

      switch (i) {

        case 27:                          /* if escape, break 'switch' stmt */
	   break;

        case 'A':                         /* Time video display 1000 frames */
        case 'a':
	   cycles = 0;

	   switch_gfx_mode();
	   set_gr_palette();

//	   starttime = biostime(0,0);
           ret = _bios_timeofday(_TIME_GETCLOCK, &starttime);
	   for (i = 0; i < 10000; i++) {
	      put_graphics();
	   }
//	   endtime = biostime(0,0);
           ret = _bios_timeofday(_TIME_GETCLOCK, &endtime);

// temporary:
//	   while (getch() == NULL);
//
	   set_text();
	   put_frame();

	   duration = (endtime-starttime);
	   if (duration != 0) {
              printf("\n\n\n10000 frames, Duration = %f\n",
                      (float)(duration/_BIOS_CLK_TCK));
           }
	   while (getch() == NULL);
	   break;

        case 'B':                         /* set breakpoint */
        case 'b':
           /* implement 'breakpoint' by inserting a non-opcode */
	   /* in the breakpoint location, and recovering later */

	   inputnum(buf, 4);
	   bkpt = cvtnum(buf);
	   hold = *(mmr_base[bkpt>>13] + (bkpt & 0x1FFF));
	   *(mmr_base[bkpt>>13] + (bkpt & 0x1FFF)) = 0x0B;
	   break;

        case 'F':                         /* Dump NVRam to File */
        case 'f':
           if ((outf = fopen("nvram.000", "wb")) == NULL) {
              fprintf(stderr, "%s: can't open %s\n", argv[0], argv[1]);
	      exit(1);
	   }
	   for (i = 0; i < 0x0800; i++) {
	      fputc((int) *(base[0xf7] + i), outf);
	   }
	   fclose(outf);
	   break;

        case 'G':                         /* 'Go' execute inst.'s */
        case 'g':
	   cycles = 0;

           kbd_init();

	   switch_gfx_mode();

//	   starttime = biostime(0,0);
           ret = _bios_timeofday(_TIME_GETCLOCK, &starttime);
	   exe_go();
//	   endtime = biostime(0,0);
           ret = _bios_timeofday(_TIME_GETCLOCK, &endtime);

           kbd_exit();

	   set_text();
	   put_frame();

	   if (reg_pc == bkpt) {
	      *(mmr_base[bkpt>>13] + (bkpt & 0x1FFF)) = hold;
	   }
	   duration = (endtime-starttime);
	   if (duration != 0) {
	      vmhz = ((((frames*262)+lines)*455+cycles)/(duration/_BIOS_CLK_TCK))/1000000;
              printf("\n\n\n\n\n\n\nDuration = %f\nVirtual MHz = %f\n", duration, vmhz);
	   }
           while (getch() == NULL);
	   break;

        case 'N':                         /* Show NVRam */
        case 'n':
	   hexdump_nvram();
	   i = getch();
	   break;

        case 'P':                         /* Display Picture */
        case 'p':
	   switch_gfx_mode();
	   set_gr_palette();
	   put_sprites();
	   put_graphics();
	   while (getch() == NULL);
	   set_text();
	   put_frame();
	   break;

        case 'Q':                         /* 'Q'uit */
        case 'q':
	   i = 27;
	   break;

        case 'R':                         /* Show 'SATB' Sprite descriptors */
        case 'r':
	   hexdump_SATB();
	   i = getch();
	   break;

        case 'S':                         /* skip (don't execute) instruction */
        case 's':
	   op	   = *(mmr_base[reg_pc>>13] + (reg_pc & 0x1FFF));
	   reg_pc += addr_info[optable[op].addr_mode].size;
	   break;

        case 'T':                         /* Display Stack */
        case 't':
	   hexdump_sp();
	   i = getch();
	   break;

        case 'V':                         /* display vram @ pointer */
        case 'v':
	   /* ask for vram address to display */
	   inputnum(buf, 4);
	   vidaddr = cvtnum(buf);
	   hexdump_vram(vidaddr);
	   i = getch();
	   break;

        case 'X':                         /* execute instruction */
        case 'x':
	   exe_instruct();
	   break;

// "Debug" code...
//        case 'Y':                         /* Dump Sound info to File */
//        case 'y':
//           if ((outf = fopen("sound.000", "w")) == NULL) {
//              fprintf(stderr, "%s: can't open %s\n", argv[0], argv[1]);
//	      exit(1);
//	   }
//           fprintf(outf, "psg_voice = %d\n", (int)psg_voicenum);
//           fprintf(outf, "psg_mainvol: left = %d, right = %d\n",
//                          (int)psg_mainvol_l, (int)psg_mainvol_r);
//           fprintf(outf, "psg_lfo_freq = %.2x\n", (int)psg_lfo_freq);
//           fprintf(outf, "psg_lfo_ctrl = %.2x\n", (int)psg_lfo_ctrl);
//           fprintf(outf, "\n");
//	   for (i = 0; i < 6; i++) {
//              fprintf(outf, "psg_freq[%d]       = %.4X\n",
//                             i, (int)psg_voice[i].freq);
//              fprintf(outf, "psg_dda_ctrl[%d]   = %.2X\n",
//                             i, psg_voice[i].dda_ctrl);
//              fprintf(outf, "psg_pan[%d]   left = %d, right = %d\n",
//                             i, psg_voice[i].pan_l, psg_voice[i].pan_r);
//              fprintf(outf, "psg_noise_ctrl[%d] = %.2X\n",
//                             i, psg_voice[i].noise_ctrl);
//              fprintf(outf, "\n");
//	   }
//	   fclose(outf);
//	   break;

        case 'Z':                         /* Display Zero-Page */
        case 'z':
	   hexdump_zp();
	   i = getch();
	   break;

	default:
	   break;
      }

      if (i == 27)                      /* break 'while' stmt */
	 break;
   }

   return(0);				/* return value to appease compiler */
}
