/****************************************************************************
   video.c

   Video controller hardware function implementation
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "defs.h"
#include "globals.h"
#include "video.h"


// These are all global variables, used mostly by
// the graphics-generation routines:

VidByte  *pce_vidram;			  // pointer to PCE video RAM
UInt16	 satb_val;			  // SAT address, within PCE video RAM
UInt16   vid_bat_x_vis;                   // # visible BAT 'blocks' x-axis
UInt16   vid_bat_x_virt, vid_bat_y_virt;  // # virtual BAT 'blocks' x/y
UInt16   HSync_Int;                       // Flag whether HSYNC generates INT
UInt16   VSync_Int;                       // Flag whether VSYNC generates INT

// These are all internal variables, indicating states
// of various hardware registers:
UInt16	 lines;
UChar	 stat_6270;
UChar	 vid_reg;

Word	 wr_addr;
Word	 rd_addr;
Word	 vidreg[21];


//
// This 'publishes' some shared global values, when the video registers
// are updated.
//

void BAT_Set_Globals(void) {

   /* visual x size: */
   vid_bat_x_vis = (vidreg[11].w & 0x7F) + 1;
   if (vid_bat_x_vis > 40) vid_bat_x_vis = 40;


   /*  (& 0x70) -> bit 0x40 = virtY = 64 */
   /*		   bit 0x10 = virtX = 64 */

   if ((vidreg[9].w & 0x30) == 0x10) {
      vid_bat_x_virt = 64;
   } else if ((vidreg[9].w & 0x30) == 0x20) {
      vid_bat_x_virt = 128;
   } else if ((vidreg[9].w & 0x30) == 0x30) {
      vid_bat_x_virt = 128;
   } else {
      vid_bat_x_virt = vid_bat_x_vis;
   }

   if ((vidreg[9].w & 0x40) == 0x40) {
      vid_bat_y_virt = 64;
   } else {
      vid_bat_y_virt = 32;
   }

   set_CGline_globals();
}


/* This routine allocates video memory, and will eventually
   initialize video registers properly.  */

void Video_Init() {
   long int i;
   VidByte  *ptr;

   pce_vidram = (VidByte *) malloc((unsigned long)(32768*sizeof(VidByte)));
   if (pce_vidram == NULL) {
      printf("Couldn't allocate Video RAM\n");
      exit(1);
   }

   /* initalize video memory */
   ptr = pce_vidram;
   for (i = 0; i < 32768; i++) {
      ptr->w = 0;
      ptr++;
   }

   HSync_Int = 0;     // No HSync-generated interrupt
   VSync_Int = 0;     // No VSync-generated interrupt

   wr_addr.w = 0;
   rd_addr.w = 0;
   vid_reg   = 0;
   stat_6270 = 0;
   lines     = 0;
   for (i = 0; i < 21; i++) {
      vidreg[i].w = 0;
   }
   BAT_Set_Globals();
}


/* this is the video register; it determines what the other
   two bytes' function are.  Unsure of exact use of most modes */

UChar hw_put_0000(UChar byte) {
   vid_reg = byte;
   return(0);
}


/* these two addresses set register/video RAM information */

UChar hw_put_0002(UChar byte) {
   UChar retval = 0;

   switch(vid_reg) {
     case 0:	     // MAWR - memory address write
       wr_addr.b.bl = byte;
       break;
     case 1:	     // MARR - memory address read
       rd_addr.b.bl = byte;
       break;
     case 2:	     // VWR - VRAM write
       (pce_vidram + wr_addr.w)->b.bl = byte;
       break;

     case 3:	     // illegal registers
     case 4:
       retval = 1;
       break;

     case 5:	     // CR - Control register
       vidreg[vid_reg].b.bl = byte;
       HSync_Int = (vidreg[vid_reg].w & 0x04) ? 1:0;
       VSync_Int = (vidreg[vid_reg].w & 0x08) ? 1:0;
       break;

     case 6:	     // RCR - Raster counter register
       vidreg[vid_reg].b.bl = byte;
       break;

     case 7:	     // BGX - Background X scroll register
       vidreg[vid_reg].b.bl = byte;
       break;
     case 8:	     // BGY - Background Y scroll register
       vidreg[vid_reg].b.bl = byte;
       break;

     case 9:	     // MWR - Memory dot space setup
       vidreg[vid_reg].b.bl = byte;
       BAT_Set_Globals();
       break;

     case 10:	     // HSR - HSync setup
       vidreg[vid_reg].b.bl = (byte & 0x1f);
       break;
     case 11:	     // HDR - HSync setup
       vidreg[vid_reg].b.bl = (byte & 0x7f);
       BAT_Set_Globals();
       break;

     case 12:	     // VPR - VSync setup
       vidreg[vid_reg].b.bl = (byte & 0x1f);
       break;
     case 13:	     // VDR - VSync setup
       vidreg[vid_reg].b.bl = byte;
       break;
     case 14:	     // VCR - VSync setup
       vidreg[vid_reg].b.bl = byte;
       break;

     case 15:	     // DCR - DMA Control
       vidreg[vid_reg].b.bl = (byte & 0x1f);
       break;
     case 16:	     // SOUR - DMA Source register
       vidreg[vid_reg].b.bl = byte;
       break;
     case 17:	     // DESR - DMA Destination register
       vidreg[vid_reg].b.bl = byte;
       break;
     case 18:	     // LENR - DMA Length register
       vidreg[vid_reg].b.bl = byte;
       break;

     case 19:	     // DVSSR (Sprite Attribute Table addr)
       vidreg[19].b.bl = byte;
       satb_val = vidreg[19].w;
       break;

     default:
       vidreg[vid_reg].b.bl = byte;
       break;
   }
   if ((vid_reg >= 16) && (vid_reg != 19)) {
       retval = 1;
   }
   return(retval);
}

UChar  hw_put_0003(UChar byte) {
   UChar retval = 0;

   switch(vid_reg) {
     case 0:	     // MAWR - memory address write
       wr_addr.b.bh = byte;
       break;
     case 1:	     // MARR - memory address read
       rd_addr.b.bh = byte;
       break;
     case 2:	     // VWR - VRAM write
       (pce_vidram + wr_addr.w)->b.bh = byte;
       wr_addr.w++;
       break;

     case 3:	     // illegal registers
     case 4:
       retval = 1;
       break;

     case 5:	     // CR - Control register
       vidreg[vid_reg].b.bh = (byte & 0x1f);
       break;

     case 6:	     // RCR - Raster counter register
       vidreg[vid_reg].b.bh = (byte & 0x03);
       break;

     case 7:	     // BGX - Background X scroll register
       vidreg[vid_reg].b.bh = (byte & 0x03);
       BAT_Set_Globals();
       break;
     case 8:	     // BGY - Background Y scroll register
       vidreg[vid_reg].b.bh = (byte & 0x01);
       break;

     case 9:	     // MWR - Memory dot space setup
       vidreg[vid_reg].b.bh = byte;
       BAT_Set_Globals();
       break;

     case 10:	     // HSR - HSync setup
       vidreg[vid_reg].b.bh = (byte & 0x7f);
       break;
     case 11:	     // HDR - HSync setup
       vidreg[vid_reg].b.bh = (byte & 0x7f);
       BAT_Set_Globals();
       break;

     case 12:	     // VPR - VSync setup
       vidreg[vid_reg].b.bh = byte;
       break;
     case 13:	     // VDR - VSync setup
       vidreg[vid_reg].b.bh = (byte & 0x01);
       break;
     case 14:	     // VCR - VSync setup
       vidreg[vid_reg].b.bh = byte;
       break;

     case 15:	     // DCR - DMA Control
       vidreg[vid_reg].b.bh = byte;
       break;

     case 16:	     // SOUR - DMA Source register
       vidreg[vid_reg].b.bh = byte;
       break;
     case 17:	     // DESR - DMA Destination register
       vidreg[vid_reg].b.bh = byte;
       break;
     case 18:	     // LENR - DMA Length register
       vidreg[vid_reg].b.bh = byte;
       break;

     case 19:	     // DVSSR (Sprite Attribute Table addr)
       vidreg[19].b.bh = byte;
       satb_val = vidreg[19].w;
       break;

     default:
       vidreg[vid_reg].b.bh = byte;
       break;
   }
   if ((vid_reg >= 16) && (vid_reg != 19)) {
       retval = 1;
   }
   return(retval);
}


/* unsure of return values from this address */

UChar hw_get_0000(void) {
   return stat_6270;
}


/* these two addresses are the 'read ports' as well */

UChar hw_get_0002(void) {
   UChar byte;

   switch(vid_reg) {
     case 0:	     // MAWR - memory address write
       byte = wr_addr.b.bl;
       break;
     case 1:	     // MARR - memory address read
       byte = rd_addr.b.bl;
       break;
     case 2:	     // VRR - VRAM read
       byte = (pce_vidram + rd_addr.w)->b.bl;
       break;
     default:
       byte = vidreg[vid_reg].b.bl;
       break;
   }

   return(byte);
}

UChar hw_get_0003(void) {
   UChar byte;

   switch(vid_reg) {
     case 0:	     // MAWR - memory address write
       byte = wr_addr.b.bh;
       break;
     case 1:	     // MARR - memory address read
       byte = rd_addr.b.bh;
       break;
     case 2:	     // VRR - VRAM read
       byte = (pce_vidram + rd_addr.w)->b.bh;
       rd_addr.w++;
       break;
     default:
       byte = vidreg[vid_reg].b.bh;
       break;
   }

   return(byte);
}
