
-------------------------------------------------------
-- Simulator for Elliott 900 series 18 bit Machines. --
-------------------------------------------------------
-- Terry Froggatt, 15 December 1992, Meridian Ada.   --
-- Don Hunter reduced the store size to 16384 (MEMORY_INDEX)
-- on 30-DEC-1994.
-- Don Hunter made the input come from EIN.DAT in local
-- directory on 25-MAR-1996.
-- And filled the store initially from M0.DAT and M1.DAT
-- on 10-APR-1996.
-- Also made OPERAND_ADDRESS mod 65536.
-- 18-APR-1996 Made it accept RLB stuff.
-- 21-APR-1996 Cleared Q register on a "/" instruction.
-- 02-MAY-1996 Check for excessive execution of 15 7168 on a faulty
-- program.
-- 18-MAY-1996 Renamed to SIM900AL.
-- 27-SEP-1997 Amended to do library operations.
-- 05-APR-1998 Amended to compile after library installation
--             and correct "12" instruction multiplying by 
--             "-1.0".
-- 29-AUG-1998 Change constant 242144 to 262144 and amend code too !
-- 
-------------------------------------------------------

with TEXT_IO; use TEXT_IO;
-- Just PUT(CHAR) NEW_LINE GET(CHAR) SKIP_LINE END_OF_LINE END_ERROR.
-- Could use ADA_IO or TTY, with Meridian Ada?
with ARRAY_OBJECT;
-- Provides the ability to declare array objects larger than 64K.
with BIT_OPS;
-- Only used so far in the Collate instruction.
with iio;
-- 31-MAR-1996
procedure SIM900AL is

  FATAL_EXCEPTION, TROUBLE: exception;
  BATCH_MODE_RATHER_THAN_INTERACTIVE: BOOLEAN := TRUE;

  CRLF: constant CHARACTER := CHARACTER'VAL(10); -- ASCII.LF.
  HALT: constant CHARACTER := CHARACTER'VAL(20); -- ASCII.DC4.

--
-- New declarations on 25-SEP-1997
--
  lib: file_type;
  library: constant string := "LIBRARY.DAT";
  lib_flag1: boolean := FALSE;
  lib_flag2: boolean := FALSE;
  has_run:   boolean := FALSE;
--

  f: file_type;
  ein: constant string := "EIN.DAT";
  MODULE: file_type;
  m0: constant string := "M0.DAT";
  m1: constant string := "M1.DAT";
  top, bot, int_level: integer;
  run_after_lib: integer;

  INPUT_BUFFER: array (0..255) of CHARACTER;
  INPUT_COLUMN: INTEGER := -1;
  OUTPUT_COLUMN: INTEGER := 0;

  type INPUT_OUTPUT_MODES is
    (ASCII_MODE, BINARY_MODE, LEGIBLE_MODE, SINK_MODE);
  INPUT_MODE, OUTPUT_MODE: INPUT_OUTPUT_MODES := BINARY_MODE;

  type OP_KIND is
    (TRANSLATE, RUN, ADD_RLB);
  OP: OP_KIND := TRANSLATE;

  TRACING_ON, AMENDING_ON: BOOLEAN := FALSE;
  AMEND_ADDRESS: LONG_INTEGER := 8;
  AMEND_WORD, AMEND_FIELD: LONG_INTEGER;
  DISPLAY_FROM, DISPLAY_TO: LONG_INTEGER;
  BREAKPOINT_ADDRESS: LONG_INTEGER := -1;

  type MACHINE_STATES is
    (UNDEFINED_STATE, RUNNABLE_STATE, INPUT_HUNG_STATE);
  MACHINE_STATE: MACHINE_STATES := UNDEFINED_STATE;

  ACCUMULATOR:                   LONG_INTEGER := 0;
  Q_REGISTER:                    LONG_INTEGER := 0;
  B_REGISTER_ADDRESS:            LONG_INTEGER := 1;
  SEQUENCE_CONTROL_REGISTER:     LONG_INTEGER := 0;
  OLD_SEQUENCE_CONTROL_REGISTER: LONG_INTEGER := 0;
  INSTRUCTION:                   LONG_INTEGER := 0;
  OPERAND_ADDRESS:               LONG_INTEGER := 0;
  MEMORY_BUFFER:                 LONG_INTEGER := 0;
  X_BITS:                        LONG_INTEGER := 0;
  PERIPHERAL_DATA:               LONG_INTEGER := 0;

  -- The next line had 0..65535 in it before 30-DEC-1994
  -- And it has again on 11-MAY-1996
  subtype MEMORY_INDEX is LONG_INTEGER range 0..65535;
  package MEMORY_ARRAY is new ARRAY_OBJECT
    (MEMORY_INDEX, LONG_INTEGER);
  procedure SET_MEMORY (I: MEMORY_INDEX; TO: LONG_INTEGER)
    renames MEMORY_ARRAY.SET;
  function MEMORY (I: MEMORY_INDEX) return LONG_INTEGER
    renames MEMORY_ARRAY.GET;

  procedure INCREMENT ( I: in out INTEGER ) is
  begin
    I := INTEGER'SUCC (I);
  end INCREMENT;

  procedure INCREMENT ( I: in out LONG_INTEGER ) is
  begin
    I := LONG_INTEGER'SUCC (I);
  end INCREMENT;

  procedure CHARACTER_PUT ( ITEM: CHARACTER ) is
  begin
    if ITEM = CRLF
    then NEW_LINE; OUTPUT_COLUMN := 0;
    else PUT (ITEM=>ITEM); INCREMENT (OUTPUT_COLUMN);
    end if;
  end CHARACTER_PUT;

  procedure ENSURE_NEW_LINE is
  begin
    if OUTPUT_COLUMN > 0 then CHARACTER_PUT (CRLF); end if;
  end ENSURE_NEW_LINE;

  procedure ENSURE_NEW_FIELD is
  begin
    loop
      CHARACTER_PUT (' '); -- Always at least one.
      exit when OUTPUT_COLUMN mod 15 = 0;
    end loop;
  end ENSURE_NEW_FIELD;

  procedure STRING_PUT ( ITEM: STRING ) is
  begin
    for I in ITEM'RANGE loop CHARACTER_PUT ( ITEM(I) ); end loop;
  end STRING_PUT;

  procedure MESSAGE_PUT ( ITEM: STRING ) is
  begin
    ENSURE_NEW_LINE;
    STRING_PUT ("SIM900AL:  ");
    STRING_PUT (ITEM);
    ENSURE_NEW_LINE;
  end MESSAGE_PUT;

  procedure ERROR_PUT ( ITEM: STRING ) is
  begin
    MESSAGE_PUT (ITEM);
    if BATCH_MODE_RATHER_THAN_INTERACTIVE then raise FATAL_EXCEPTION; end if;
  end ERROR_PUT;

  procedure NATURAL_PUT ( ITEM: LONG_INTEGER ) is
  begin
    if ITEM >= 10 then NATURAL_PUT (ITEM / 10); end if;
    CHARACTER_PUT ( CHARACTER'VAL ( INTEGER(ITEM rem 10) + CHARACTER'POS('0') ) );
  end NATURAL_PUT;

  procedure WORD_PUT ( NAME: CHARACTER; ITEM: LONG_INTEGER ) is
  begin
    ENSURE_NEW_FIELD;
    if NAME /= ' ' then
      CHARACTER_PUT (NAME); CHARACTER_PUT (' ');
      CHARACTER_PUT ('='); CHARACTER_PUT (' ');
    end if;
    if ITEM >= 131072
    then CHARACTER_PUT ('-'); NATURAL_PUT (262144 - ITEM);
    else CHARACTER_PUT ('+'); NATURAL_PUT (ITEM);
    end if;
  end WORD_PUT;

  procedure INFORMATION_PUT (
    ADDRESS: LONG_INTEGER;
    DISPLAY_REGISTERS_RATHER_THAN_FORMATS: BOOLEAN )
  is
    WORD: LONG_INTEGER; -- Not constant.
  begin
    if ADDRESS > 16383 then raise TROUBLE; 
    end if;
    WORD := MEMORY(ADDRESS);
    ENSURE_NEW_LINE;
    CHARACTER_PUT ('('); NATURAL_PUT (ADDRESS mod 8192);
    if ADDRESS >= 8192 then
      CHARACTER_PUT ('^'); NATURAL_PUT (ADDRESS / 8192);
    end if;
    CHARACTER_PUT (')');
    ENSURE_NEW_FIELD;
    if WORD >= 131072 then CHARACTER_PUT ('/'); end if;
    NATURAL_PUT ( (WORD / 8192) mod 16 ); CHARACTER_PUT (' ');
    NATURAL_PUT (WORD mod 8192); CHARACTER_PUT (';');
    if DISPLAY_REGISTERS_RATHER_THAN_FORMATS then
      WORD_PUT ( 'A', ACCUMULATOR );
      WORD_PUT ( 'Q', Q_REGISTER );
      WORD_PUT ( 'B', MEMORY (B_REGISTER_ADDRESS) );
    else
      WORD_PUT ( ' ', WORD );
      ENSURE_NEW_FIELD;
      CHARACTER_PUT ('&');
      for I in 1..6 loop -- OCTAL_PUT unsigned with leading zeros.
        CHARACTER_PUT (
          CHARACTER'VAL( (WORD / 32768) + CHARACTER'POS('0') ) );
        WORD := (WORD * 8) mod 262144;
      end loop;
      WORD := MEMORY (ADDRESS);
      ENSURE_NEW_FIELD;
      CHARACTER_PUT ('#');
      while WORD /= 0 loop -- ALPHANUMERIC_PUT.
        CHARACTER_PUT (
          CHARACTER'VAL( (WORD / 4096) + CHARACTER'POS(' ') ) );
        WORD := (WORD * 64) mod 262144;
      end loop;
    end if;
    ENSURE_NEW_LINE;
  end INFORMATION_PUT;

  function CHARACTER_LOOK return CHARACTER is
  begin
    if INPUT_COLUMN = -1 then
      if BATCH_MODE_RATHER_THAN_INTERACTIVE
-- 02-MAY-1996 was ENSURE_NEW_LINE instead of null;
      then null; else MESSAGE_PUT ( "What Next?" ); end if;
      loop
        INCREMENT (INPUT_COLUMN);
        if END_OF_LINE then
          begin SKIP_LINE;
            INPUT_BUFFER (INPUT_COLUMN) := CRLF;
          exception when END_ERROR =>
            INPUT_BUFFER (INPUT_COLUMN) := HALT;
          end; exit;
        end if;
        GET (ITEM => INPUT_BUFFER (INPUT_COLUMN) );
      end loop;
      INPUT_COLUMN := 0;
    end if;
    return INPUT_BUFFER (INPUT_COLUMN);
  end CHARACTER_LOOK;

  procedure CHARACTER_GET is
  begin
    if CHARACTER_LOOK = HALT then
      BATCH_MODE_RATHER_THAN_INTERACTIVE := TRUE;
      ERROR_PUT ( "Read Beyond File" );
    elsif CHARACTER_LOOK = CRLF then
      INPUT_COLUMN := -1; -- But don't read the line yet,
      -- as this might give trouble in interactive mode.
    else
      INCREMENT (INPUT_COLUMN);
    end if;
  end CHARACTER_GET;

  function CHARACTER_MATCH ( ITEM: CHARACTER ) return BOOLEAN is
  begin
    if CHARACTER_LOOK = ITEM
    then CHARACTER_GET; return TRUE;
    else return FALSE;
    end if;
  end CHARACTER_MATCH;

  function STRING_MATCH ( ITEM: STRING ) return BOOLEAN is
    -- Where ITEM must be in upper-case.
    -- Could produce a version (for Perspective)
    -- where ITEM is mainly in lower-case but
    -- with the abbreviation letters in upper-case.
    BACKTRACK_COLUMN: INTEGER := INPUT_COLUMN;
    DIFFERENCE_IN_POS: INTEGER;
  begin
    for I in ITEM'RANGE loop
      DIFFERENCE_IN_POS :=
        CHARACTER'POS(CHARACTER_LOOK) - CHARACTER'POS(ITEM(I));
      if DIFFERENCE_IN_POS /= 0 and then DIFFERENCE_IN_POS /= 32
      then
        if BACKTRACK_COLUMN = -1 then INPUT_COLUMN := 0;
        else INPUT_COLUMN := BACKTRACK_COLUMN; end if;
        return FALSE;
      end if;
      CHARACTER_GET;
    end loop;
    return TRUE;
  end STRING_MATCH;

  function NATURAL_GET ( BASE: LONG_INTEGER := 10 ) return LONG_INTEGER is
    ITEM: LONG_INTEGER := 0;
    DIGIT: LONG_INTEGER;
    ERROR: BOOLEAN := TRUE;
  begin
    loop
      DIGIT := LONG_INTEGER ( CHARACTER'POS(CHARACTER_LOOK) - CHARACTER'POS('0') );
      exit when DIGIT not in 0..BASE-1;
      ITEM := ITEM * BASE + DIGIT; CHARACTER_GET; ERROR := FALSE;
    end loop;
    if ERROR then ERROR_PUT ( "Bad Parameter" ); end if;
    return ITEM;
  end NATURAL_GET;

  function PARAMETER_GET return LONG_INTEGER is
  begin
    while CHARACTER_LOOK <= ' ' loop CHARACTER_GET; end loop;
    return NATURAL_GET;
  end PARAMETER_GET;

  procedure IGNORE_COMMENTS is
  begin
    loop
      if CHARACTER_MATCH ('(') then IGNORE_COMMENTS;
      elsif CHARACTER_MATCH (')') then exit;
      else CHARACTER_GET; end if;
    end loop;
  end IGNORE_COMMENTS;

  procedure CLEAR_STORE ( LAST: LONG_INTEGER := MEMORY_INDEX'LAST ) is
  begin
    for MEMORY_ADDRESS in MEMORY_INDEX'FIRST..LAST loop
      SET_MEMORY (MEMORY_ADDRESS, 0);
    end loop;
  end CLEAR_STORE;

  procedure ESTABLISH_INITIAL_INSTRUCTIONS is
  begin
    SET_MEMORY (8180, 262144 - 3);
    SET_MEMORY (8181, 8192 *  0 + 8180);
    SET_MEMORY (8182, 8192 *  4 + 8189);
    SET_MEMORY (8183, 8192 * 15 + 2048);
    SET_MEMORY (8184, 8192 *  9 + 8186);
    SET_MEMORY (8185, 8192 *  8 + 8183);
    SET_MEMORY (8186, 8192 * 15 + 2048);
    SET_MEMORY (8187, 8192 *  5 + 8180 + 131072);
    SET_MEMORY (8188, 8192 * 10 +    1);
    SET_MEMORY (8189, 8192 *  4 +    1);
    SET_MEMORY (8190, 8192 *  9 + 8182);
    SET_MEMORY (8191, 8192 *  8 + 8177);
  end ESTABLISH_INITIAL_INSTRUCTIONS;

  procedure STORE_IN_MEMORY ( ITEM: LONG_INTEGER ) is
  begin
    SET_MEMORY (AMEND_ADDRESS, ITEM mod 262144);
    INCREMENT (AMEND_ADDRESS);
  end STORE_IN_MEMORY;

  procedure PREPARE_TO_JUMP ( TO: LONG_INTEGER ) is
  begin
--
-- 27-SEP-1997 This bit removed because it clutters the output
--
--    if MACHINE_STATE /= UNDEFINED_STATE then
--      MESSAGE_PUT ( "Machine Reset" );
--    end if;
--
    SET_MEMORY (B_REGISTER_ADDRESS - 1, SEQUENCE_CONTROL_REGISTER);
    B_REGISTER_ADDRESS := 1;
    SEQUENCE_CONTROL_REGISTER := TO;
--
--  05-APR-1998
--
    if TO = 18 then run_after_lib := 1;
               else run_after_lib := 0;
    end if;
--        
    MACHINE_STATE := RUNNABLE_STATE;
  end PREPARE_TO_JUMP;

  procedure COMBINED_LEFT_SHIFT is
  begin
    if Q_REGISTER >= 131072 then
      ACCUMULATOR := (ACCUMULATOR * 2) mod 262144 + 1;
    else
      ACCUMULATOR := (ACCUMULATOR * 2) mod 262144;
    end if;
    Q_REGISTER := (Q_REGISTER * 2) mod 262144;
  end COMBINED_LEFT_SHIFT;

  procedure COMBINED_RIGHT_SHIFT is
  begin
    if ACCUMULATOR mod 2 = 1 then
      Q_REGISTER := Q_REGISTER / 2 + 131072;
    else
      Q_REGISTER := Q_REGISTER / 2;
    end if;
    if ACCUMULATOR >= 131072 then
      ACCUMULATOR := ACCUMULATOR / 2 + 131072;
    else
      ACCUMULATOR := ACCUMULATOR / 2;
    end if;
  end COMBINED_RIGHT_SHIFT;

  function ODD_PARITY return BOOLEAN is
    SHIFT: LONG_INTEGER := PERIPHERAL_DATA;
    TOTAL: BOOLEAN := FALSE;
  begin
    while SHIFT > 0 loop
      if SHIFT mod 2 = 1 then TOTAL := not TOTAL; end if;
      SHIFT := SHIFT / 2;
    end loop;
    return TOTAL;
  end ODD_PARITY;

begin --SIM900AL

  CLEAR_STORE;
--  ESTABLISH_INITIAL_INSTRUCTIONS;

  MESSAGE_PUT("Version of 29-AUG-1998");

  open(file => MODULE, name => m0, mode => in_file);
  set_input(MODULE);
  set_line_length(72);
  for I in 0..8191 loop
    iio.get(top);
    iio.get(bot);
    OPERAND_ADDRESS := long_integer(I); 
    ACCUMULATOR := long_integer(top) * 8192 + 
                   long_integer(bot);
    SET_MEMORY(OPERAND_ADDRESS, ACCUMULATOR);
  end loop;
  close(file => MODULE);

  open(file => MODULE, name => m1, mode => in_file);
  set_input(MODULE);
  for I in 8192..16383 loop
    iio.get(top);
    iio.get(bot);           
    OPERAND_ADDRESS := long_integer(I);
    ACCUMULATOR := long_integer(top) * 8192 + 
                   long_integer(bot);
    SET_MEMORY(OPERAND_ADDRESS, ACCUMULATOR);
  end loop;
  close(file => MODULE);

  open(file => f, name => ein, mode => in_file);
  set_input(f);
  int_level := 1;

  MESSAGE_PUT ( "Simulation Starting" );
  loop

    AMENDING_ON := FALSE;
    loop

      if STRING_MATCH ("KEYB")  then
        set_input(standard_input);
      elsif STRING_MATCH("EIN") then
        set_input(f);
      elsif STRING_MATCH("ADRLB") then
        OP := ADD_RLB;
--
-- 25-SEP-1997. New directive ADLIB.
--
      elsif STRING_MATCH("ADLIB") then
        lib_flag1 := TRUE;
      elsif STRING_MATCH ("IPBIN") then
        INPUT_MODE := BINARY_MODE;
      elsif INPUT_MODE = ASCII_MODE or else
      ( AMENDING_ON = FALSE and then CHARACTER_LOOK in '0'..'9' ) then
        if INPUT_MODE = ASCII_MODE then
          PERIPHERAL_DATA := CHARACTER'POS(CHARACTER_LOOK); CHARACTER_GET;
          if ODD_PARITY then
            PERIPHERAL_DATA := PERIPHERAL_DATA + 128;
          end if;
        else
          PERIPHERAL_DATA := NATURAL_GET;
          if PERIPHERAL_DATA >= 256 then
            ERROR_PUT ( "Bad Peripheral Data" );
          end if;
        end if;
        if MACHINE_STATE = INPUT_HUNG_STATE then
          exit;
        elsif PERIPHERAL_DATA = 0 then
          MESSAGE_PUT ( "Blank Ignored" );
        else
          ERROR_PUT ( "Character Ignored" );
        end if;
      elsif STRING_MATCH ("IPASC") then
        -- Without ignoring the following CRLF or anything before it.
        INPUT_MODE := ASCII_MODE;

      elsif CHARACTER_LOOK <= ' ' then CHARACTER_GET;
      elsif CHARACTER_MATCH ('(') then IGNORE_COMMENTS;

      elsif CHARACTER_MATCH ('*') then
        CLEAR_STORE ( ( NATURAL_GET / 8192 ) * 8192 - 1 );
      elsif CHARACTER_MATCH ('^') then
        AMENDING_ON := TRUE;
        AMEND_ADDRESS := NATURAL_GET;
      elsif CHARACTER_MATCH ('>') then
        AMENDING_ON := TRUE;
        AMEND_ADDRESS := AMEND_ADDRESS + NATURAL_GET;
      elsif CHARACTER_MATCH ('<') then
        PREPARE_TO_JUMP (AMEND_ADDRESS);
      elsif CHARACTER_MATCH ('&') then
        STORE_IN_MEMORY ( NATURAL_GET (BASE=>8) );
      elsif CHARACTER_MATCH ('+') then
        STORE_IN_MEMORY ( NATURAL_GET );
      elsif CHARACTER_MATCH ('-') then
        STORE_IN_MEMORY ( 262144 - NATURAL_GET );

      elsif CHARACTER_LOOK = '/' or else CHARACTER_LOOK in '0'..'9' then
        if CHARACTER_MATCH ('/') then AMEND_WORD := 131072;
        else AMEND_WORD := 0; end if;
        AMEND_FIELD := NATURAL_GET;
        if AMEND_FIELD >= 16 then ERROR_PUT ( "Bad Function" ); end if;
        AMEND_WORD := AMEND_WORD + ( AMEND_FIELD mod 16 ) * 8192;
        while CHARACTER_LOOK <= ' ' loop CHARACTER_GET; end loop;
        if CHARACTER_MATCH (';') then
          if CHARACTER_MATCH ('+') then
            AMEND_FIELD := AMEND_ADDRESS + NATURAL_GET;
          elsif CHARACTER_MATCH ('-') then
            AMEND_FIELD := AMEND_ADDRESS - NATURAL_GET;
          else ERROR_PUT ( "Bad Address" ); end if;
        else
          AMEND_FIELD := NATURAL_GET;
        end if;
        STORE_IN_MEMORY ( AMEND_WORD + AMEND_FIELD mod 8192 );

      elsif STRING_MATCH ("LOAD") then
        ESTABLISH_INITIAL_INSTRUCTIONS;
        PREPARE_TO_JUMP (8181); exit;
      elsif STRING_MATCH ("JUMP") then
        PREPARE_TO_JUMP ( PARAMETER_GET ); exit;
      elsif STRING_MATCH ("GO") then
        if MACHINE_STATE = UNDEFINED_STATE then
          ERROR_PUT ( "No Continuation" );
        else -- Have no PERIPHERAL_DATA so must cancel any INPUT_HUNG_STATE.
          MACHINE_STATE := RUNNABLE_STATE; exit;
        end if;
      elsif STRING_MATCH ("QUIT") then
        PREPARE_TO_JUMP (0); -- Just to check MACHINE_STATE.
        MESSAGE_PUT ( "Simulation Completed" );
        return;

      elsif STRING_MATCH ("OPASC") then
        OUTPUT_MODE := ASCII_MODE;
      elsif STRING_MATCH ("OPBIN") then
        OUTPUT_MODE := BINARY_MODE;
      elsif STRING_MATCH ("OPLEG") then
        OUTPUT_MODE := LEGIBLE_MODE;
      elsif STRING_MATCH ("OPSIN") then
        OUTPUT_MODE := SINK_MODE;
      elsif STRING_MATCH ("COPY") then
        while CHARACTER_MATCH (' ') loop null; end loop;
        ENSURE_NEW_LINE;
        while CHARACTER_LOOK /= CRLF loop
          CHARACTER_PUT ( CHARACTER_LOOK ); CHARACTER_GET;
        end loop;
        ENSURE_NEW_LINE;

      elsif STRING_MATCH ("DISPLAY") then
        DISPLAY_FROM := PARAMETER_GET; DISPLAY_TO := PARAMETER_GET;
        for DISPLAY_ADDRESS in DISPLAY_FROM..DISPLAY_TO loop
          INFORMATION_PUT ( DISPLAY_ADDRESS,
            DISPLAY_REGISTERS_RATHER_THAN_FORMATS => FALSE );
        end loop;
      elsif STRING_MATCH ("TRACE") then
        TRACING_ON := TRUE;
      elsif STRING_MATCH ("TROFF") then
        TRACING_ON := FALSE;
      elsif STRING_MATCH ("BREAK") then
        BREAKPOINT_ADDRESS := PARAMETER_GET;
      elsif STRING_MATCH ("BROFF") then
        BREAKPOINT_ADDRESS := -1;
      elsif STRING_MATCH ("BATCH") then
        BATCH_MODE_RATHER_THAN_INTERACTIVE := TRUE;
      elsif STRING_MATCH ("INTER") then
        BATCH_MODE_RATHER_THAN_INTERACTIVE := FALSE;

      else
        ERROR_PUT ( "Unknown Command" );
        CHARACTER_GET;
      end if;

    end loop;

    loop

      INSTRUCTION := MEMORY (SEQUENCE_CONTROL_REGISTER);

      OPERAND_ADDRESS := (INSTRUCTION mod 8192) +
        ( (SEQUENCE_CONTROL_REGISTER / 8192) * 8192 );
      if INSTRUCTION >= 131072 then
-- 21-APR-1996
        Q_REGISTER := 0;
-- 11-MAY-1996. Should be mod 131072 but 65536 covers up for a bit of
--              bad coding in Call By Name.
        OPERAND_ADDRESS :=
       ( OPERAND_ADDRESS + MEMORY (B_REGISTER_ADDRESS) ) mod 65536;

      end if;

      OLD_SEQUENCE_CONTROL_REGISTER := SEQUENCE_CONTROL_REGISTER;
      SEQUENCE_CONTROL_REGISTER := SEQUENCE_CONTROL_REGISTER + 1;

      case (INSTRUCTION / 8192) mod 16 is

      when 0 => -- Set B-register.
        Q_REGISTER := MEMORY (OPERAND_ADDRESS);
        SET_MEMORY (B_REGISTER_ADDRESS, Q_REGISTER);

      when 1 => -- Add.
        ACCUMULATOR :=
          ( MEMORY (OPERAND_ADDRESS) + ACCUMULATOR ) mod 262144;

      when 2 => -- Negate & add.
        Q_REGISTER := MEMORY (OPERAND_ADDRESS);
        ACCUMULATOR := (Q_REGISTER - ACCUMULATOR) mod 262144;

      when 3 => -- Store Q-register.
        SET_MEMORY (OPERAND_ADDRESS, Q_REGISTER / 2);

      when 4 => -- Read.
        ACCUMULATOR := MEMORY (OPERAND_ADDRESS);

      when 5 => -- Write.
        SET_MEMORY (OPERAND_ADDRESS, ACCUMULATOR);

      when 6 => -- Collate.
        ACCUMULATOR :=
          BIT_OPS."and" ( MEMORY (OPERAND_ADDRESS), ACCUMULATOR );
--      MEMORY_BUFFER := MEMORY (OPERAND_ADDRESS);
--      for PROCESS_COUNTER in 1..18 loop
--        if ACCUMULATOR mod 2 = 1 and then MEMORY_BUFFER mod 2 = 1 then
--          ACCUMULATOR := ACCUMULATOR / 2 + 131072;
--        else
--          ACCUMULATOR := ACCUMULATOR / 2;
--        end if;
--        MEMORY_BUFFER := MEMORY_BUFFER / 2;
--      end loop;

      when 7 => -- Jump if zero.
        if ACCUMULATOR = 0 then
          SEQUENCE_CONTROL_REGISTER := OPERAND_ADDRESS;
        end if;

      when 8 => -- Jump.
        SEQUENCE_CONTROL_REGISTER := OPERAND_ADDRESS;

      when 9 => -- Jump if negative.
        if ACCUMULATOR >= 131072 then
          SEQUENCE_CONTROL_REGISTER := OPERAND_ADDRESS;
        end if;

      when 10 => -- Count in store.
        SET_MEMORY (OPERAND_ADDRESS,
          ( MEMORY (OPERAND_ADDRESS) + 1 ) mod 262144);

      when 11 => -- Store SCR.
        Q_REGISTER := (SEQUENCE_CONTROL_REGISTER / 8192) * 8192;
        SET_MEMORY (OPERAND_ADDRESS, SEQUENCE_CONTROL_REGISTER mod 8192);

      when 12 => -- Multiply.
        MEMORY_BUFFER := MEMORY (OPERAND_ADDRESS);
-- 05-APR-1998
        if MEMORY_BUFFER = 131072
        then
           if ACCUMULATOR >= 131072
           then
              Q_REGISTER := 1;
           else
              Q_REGISTER := 0;
           end if;
-- 29-AUG-1998
           if ACCUMULATOR /= 0 then
              ACCUMULATOR := 262144 - ACCUMULATOR;
           end if;
        else
           for PROCESS_COUNTER in 1..18 loop
             if PROCESS_COUNTER = 1 then
               Q_REGISTER := ACCUMULATOR; ACCUMULATOR := 0;
               X_BITS := Q_REGISTER * 2;
             else
               X_BITS := Q_REGISTER;
               COMBINED_RIGHT_SHIFT;
             end if;
             case X_BITS mod 4 is
             when 1 =>
               ACCUMULATOR := (ACCUMULATOR + MEMORY_BUFFER) mod 262144;
             when 2 =>
               ACCUMULATOR := (ACCUMULATOR - MEMORY_BUFFER) mod 262144;
             when others =>
               null;
             end case;
           end loop;
        end if;
      when 13 => -- Divide.
        MEMORY_BUFFER := MEMORY (OPERAND_ADDRESS);
        Q_REGISTER := (Q_REGISTER / 2) * 2;
        X_BITS := ACCUMULATOR;
        for PROCESS_COUNTER in 1..18 loop
          if ( X_BITS >= 131072 ) = ( MEMORY_BUFFER >= 131072 ) then
            Q_REGISTER := Q_REGISTER + 1;
            ACCUMULATOR := (ACCUMULATOR - MEMORY_BUFFER) mod 262144;
          else
            ACCUMULATOR := (ACCUMULATOR + MEMORY_BUFFER) mod 262144;
          end if;
          X_BITS := ACCUMULATOR;
          COMBINED_LEFT_SHIFT;
        end loop;
        ACCUMULATOR := Q_REGISTER + 1;

      when 14 => -- Shift, etc..
        OPERAND_ADDRESS := OPERAND_ADDRESS mod 8192;
        if OPERAND_ADDRESS < 4096
        then -- Left shift.
          for PROCESS_COUNTER in 1..OPERAND_ADDRESS loop
            COMBINED_LEFT_SHIFT;
          end loop;
        else -- Right shift.
          for PROCESS_COUNTER in OPERAND_ADDRESS..8191 loop
            COMBINED_RIGHT_SHIFT;
          end loop;
        end if;

      when 15 => -- Input/output, etc..
        OPERAND_ADDRESS := OPERAND_ADDRESS mod 8192;
        if OPERAND_ADDRESS = 7168
        then -- Level terminate.
          SET_MEMORY (B_REGISTER_ADDRESS - 1, SEQUENCE_CONTROL_REGISTER);
          B_REGISTER_ADDRESS := 7;
          SEQUENCE_CONTROL_REGISTER := MEMORY (B_REGISTER_ADDRESS - 1);
          int_level := int_level + 1;
        if int_level > 4 then raise TROUBLE; end if;
        elsif OPERAND_ADDRESS < 4096
        then -- Input.
          if MACHINE_STATE = RUNNABLE_STATE then
            SEQUENCE_CONTROL_REGISTER := OLD_SEQUENCE_CONTROL_REGISTER;
            MACHINE_STATE := INPUT_HUNG_STATE; exit;
          elsif ACCUMULATOR mod 2 = 1 then
            PERIPHERAL_DATA := PERIPHERAL_DATA mod 128;
          else
            PERIPHERAL_DATA := PERIPHERAL_DATA mod 256;
          end if;
          -- MACHINE_STATE = INPUT_HUNG_STATE.
          ACCUMULATOR := ( ACCUMULATOR * 128 + PERIPHERAL_DATA ) mod 262144;
          MACHINE_STATE := RUNNABLE_STATE;
        else -- Output.
          PERIPHERAL_DATA := ACCUMULATOR mod 256;
          case OUTPUT_MODE is
          when BINARY_MODE =>
            if OUTPUT_COLUMN > 60 then CHARACTER_PUT (CRLF);
            elsif OUTPUT_COLUMN /= 0 then CHARACTER_PUT (' ');
            end if;
            NATURAL_PUT (PERIPHERAL_DATA);
          when ASCII_MODE =>
            if ODD_PARITY then ERROR_PUT ( "Parity Error" ); end if;
            PERIPHERAL_DATA := PERIPHERAL_DATA mod 128;
            case PERIPHERAL_DATA is
            when 0 | 13 | 127 => null;
            when others => CHARACTER_PUT ( CHARACTER'VAL (PERIPHERAL_DATA) );
            end case;
          when LEGIBLE_MODE =>
            ENSURE_NEW_LINE;
            for I in 1..8 loop
              if PERIPHERAL_DATA >= 128
              then CHARACTER_PUT ('O'); else CHARACTER_PUT ('.'); end if;
              if I = 5 then CHARACTER_PUT ('o'); end if;
              PERIPHERAL_DATA := (PERIPHERAL_DATA * 2) mod 256;
            end loop;
            ENSURE_NEW_LINE;
          when others => null; end case; -- SINK_MODE.
        end if;

      when others => null; end case; -- Impossible.

      if TRACING_ON then
        INFORMATION_PUT ( OLD_SEQUENCE_CONTROL_REGISTER,
          DISPLAY_REGISTERS_RATHER_THAN_FORMATS => TRUE );
      end if;

--
--  This whole section has changed on 27-SEP-1997.
--    
      
      if SEQUENCE_CONTROL_REGISTER = OLD_SEQUENCE_CONTROL_REGISTER 
      and SEQUENCE_CONTROL_REGISTER /= 7943 then
         if SEQUENCE_CONTROL_REGISTER = 1727
         and (lib_flag1 or (OP = ADD_RLB))
         and not has_run then
            INPUT_MODE := BINARY_MODE;
            PREPARE_TO_JUMP(9);
            if lib_flag1 then
               open(file => lib, name => library, mode => in_file);
               set_input(lib);
               lib_flag1 := FALSE;
               lib_flag2 := TRUE;
            else
               OP := TRANSLATE;
            end if;
         else
            MACHINE_STATE := UNDEFINED_STATE;
            CHARACTER_PUT(CRLF); CHARACTER_PUT(CRLF);
            if not has_run then
               MESSAGE_PUT("Add an ADLIB directive");
            end if;
            STRING_PUT("Stop at");
            WORD_PUT(' ', SEQUENCE_CONTROL_REGISTER);
            exit;
         end if;
      elsif SEQUENCE_CONTROL_REGISTER = 7943 then
         if OP = TRANSLATE then           
            INPUT_MODE := ASCII_MODE;
--          
            if run_after_lib = 1 then
               has_run := FALSE;
               OP := TRANSLATE;   
               run_after_lib := 0;
               PREPARE_TO_JUMP(8);
            elsif run_after_lib = 0 then
--          
               OP := RUN;
               if lib_flag2 then
                  close(file => lib);
                  set_input(f);
                  lib_flag2 := FALSE;
               end if;
               has_run := TRUE;
               PREPARE_TO_JUMP(10);
            end if;
         elsif OP = ADD_RLB then
            INPUT_MODE := BINARY_MODE;
            OP := TRANSLATE;
            if lib_flag2 then
               close(file => lib);
               set_input(f);
               lib_flag2 := FALSE;
            end if;
            PREPARE_TO_JUMP(14);
         end if;            
      elsif SEQUENCE_CONTROL_REGISTER = BREAKPOINT_ADDRESS then
        MESSAGE_PUT ( "Hit Breakpoint" ); exit;
      end if;
      
    end loop;

  end loop;

  close(file => f);

  exception
  when TROUBLE =>
     for I in 0..7 loop
        OPERAND_ADDRESS := long_integer(I);
        INFORMATION_PUT(OPERAND_ADDRESS,
            DISPLAY_REGISTERS_RATHER_THAN_FORMATS => FALSE);
     end loop;
     INFORMATION_PUT(135, TRUE);
     return;

   when fatal_exception =>
        PREPARE_TO_JUMP (0); -- Just to check MACHINE_STATE.
        MESSAGE_PUT ( "Simulation Completed" );
        return;

end SIM900AL;

-- Add ENTER and OBEY ?
-- Add 3 Interrupts (and "hardware" trace) ?
-- Add fuller Strobe & Trap on Instruction or Operand or Peripheral ?
-- Separate out the Teleprinter & Paper Tape channels, & add Echoing ?
-- Add the single word & block input/output ?
-- Add 905/920C extra instructions & addressing modes ?
-- Add machine-distinguishing side-effects ?
-- Add an INCLUDE and an IPASC_INCLUDE_IPBIN facility ?
-- Make prompting when interactive appropriate for MACHINE_STATE ?
-- Add stepping by a given number of instructions ?
-- Add timing analysis ?
-- Do proper checking of Jumping & Amending Addresses ?
-- Make memory size bigger or dynamic ?
-- Unroll the Collate, Multiply, & Divide loops ?
-- Use a case for Shift instructions ?
-- Invoke OPLEG automatically somehow ?

