package sorcererII;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;

public class SorcererAudioOutputStream extends OutputStream {

  private final ByteArrayOutputStream _bos = new ByteArrayOutputStream();
  private final boolean _trace = true;
  private final int _vol = 100 * 256;
  private final float _sampleRate = 44100.0f;
  private final File _fileOut;
  private final int _bitsPerSample = 16;
  private boolean _state = false;
  private double _samplesPerBit;

  public SorcererAudioOutputStream(final File fileOut, final double bitFrequency) {
    _fileOut = fileOut;
    _samplesPerBit = (_sampleRate / bitFrequency  );
  }
  
  @Override
  public void write(final int arg0) throws IOException {
    _bos.write(arg0);
  }

  private void writeSample(final ByteArrayOutputStream sampleBytes) {
    final int s = _state ? _vol : -_vol;
    sampleBytes.write(s & 0xff);
    sampleBytes.write((s>>8) & 0xff);
  }
  
  private void writeHalf(final ByteArrayOutputStream sampleBytes){
    
    final int p0 = (int)(_samplesPerBit * 0.25);
    final int p1 = (int)(_samplesPerBit * 0.5);
    final int p2 = (int)(_samplesPerBit *0.75);
     
    for(int i = 0; i < _samplesPerBit; ++i) {
      writeSample(sampleBytes);
      if(i == p0 || i == p1 || i == p2 ) _state = !_state;
    }
    _state = !_state;
  } 
  
  private void writeHalfPadding(final ByteArrayOutputStream sampleBytes) {
    for(int i = 0; i < 800; ++i) {
      writeHalf(sampleBytes);      
    }
  }
  
  private void writePadding(final ByteArrayOutputStream sampleBytes) {
    for(int i = 0; i < 800; ++i) {
      writeOne(sampleBytes);      
    }
  }
  
  private void writeStart(final ByteArrayOutputStream sampleBytes){
    writeOne(sampleBytes);
    writeOne(sampleBytes);
    writeZero(sampleBytes);
  }
  
  private void writeOne(final ByteArrayOutputStream sampleBytes){
    final int p = (int)(_samplesPerBit / 2);
    for(int i = 0; i < _samplesPerBit; ++i) {
      writeSample(sampleBytes);
      if(i == p) _state = !_state;
    }
    _state = !_state;
  } 
  
  private void writeZero(final ByteArrayOutputStream sampleBytes){
    for(int i = 0; i < _samplesPerBit; ++i) {
      writeSample(sampleBytes);
    }
    _state = !_state;
  }
  
  private void writeByte(final byte b, final ByteArrayOutputStream sampleBytes) {
    writeStart(sampleBytes);
    final int v = (int)b & 0xff;
    if(_trace) System.out.println("Data: " + Integer.toHexString(v));
    for(int i = 0; i < 8; ++i) {
      if((v & (1 << i)) == 0) {
        writeZero(sampleBytes);
      }
      else {
        writeOne(sampleBytes);
      }
    }
  }
  
  @Override
  public void close() throws IOException {
    final ByteArrayOutputStream sampleBytes = new ByteArrayOutputStream();
    final byte[] outputButes = _bos.toByteArray();
    writeHalfPadding(sampleBytes);
    writePadding(sampleBytes);
    for(int i = 0; i < outputButes.length; ++i){
      writeByte(outputButes[i], sampleBytes);
    }
    writePadding(sampleBytes);
    writeHalfPadding(sampleBytes);
    final byte[] _audioData = sampleBytes.toByteArray();    
    final InputStream sampleInputStream = new ByteArrayInputStream(_audioData);  
    final AudioFormat audioFormat = new AudioFormat(AudioFormat.Encoding.PCM_SIGNED, _sampleRate, _bitsPerSample, 1, 2, _sampleRate, false);
    final AudioInputStream audioInputStream = new AudioInputStream(sampleInputStream, audioFormat, _audioData.length/2);
    AudioSystem.write(audioInputStream, AudioFileFormat.Type.WAVE, _fileOut);
  }
}
