package sorcererII;

public class SorcererDiskSystem
{
  private SorcererDiskDrive[] _drives = new SorcererDiskDrive[ 4 ];
  private SorcererDiskDrive _activeDrive = null;
  private int _activeDriveNumber = 0x40;

  public SorcererDiskSystem()
  {
    for( int i =0; i < _drives.length; ++i )
    {
      _drives[ i ] = new SorcererDiskDrive(i);
    }
  }
  
  public SorcererDiskDrive getSorcererDiskDrive(int drive) {
    return _drives[ drive ];
  }

  public final void insertDisk( SorcererDisk disk, int drive )
  {
    _drives[ drive ].insertDisk( disk );
  }

  private final boolean dataReady()
  {
    if( _activeDrive != null )
    {
      return _activeDrive.dataReady();
    }
    else
    {
      return false;
    }
  }

  private final boolean home()
  {
    if( _activeDrive != null )
    {
      return _activeDrive.home();
    }
    else
    {
      return false;
    }
  }

  private final void stepForward()
  {
    if( _activeDrive != null )
    {
      _activeDrive.stepForward();
    }
  }

  private final void stepBackward()
  {
    if( _activeDrive != null )
    {
      _activeDrive.stepBackward();
    }
  }

  private final void readyWrite()
  {
    if( _activeDrive != null )
    {
      _activeDrive.readyWrite();
    }
  }

  private final void activate( int drive )
  {
    _activeDriveNumber = drive;
    _activeDrive = _drives[ drive ];
    _activeDrive.activate();
  }

  private final boolean active()
  {
    return _activeDrive != null;
  }

  private final void writeReg0( int b )
  {
    switch( b )
    {
      case 0xA0:                 break;
      case 0x20: activate( 0 );  break;
      case 0x21: activate( 1 );  break;
      case 0x22: activate( 2 );  break;
      case 0x23: activate( 3 );  break;
      case 0x60: stepBackward(); break;
      case 0x61: stepForward();  break;
      case 0x80: readyWrite();   break;
    }
  }

  private final void writeReg1( int b )
  {
    switch( b )
    {
      case 0xA0:                 break;
      case 0x20: activate( 0 );  break;
      case 0x21: activate( 1 );  break;
      case 0x22: activate( 2 );  break;
      case 0x23: activate( 3 );  break;
      case 0x60: stepBackward(); break;
      case 0x61: stepForward();  break;
      case 0x80: readyWrite();   break;
    }
  }

  private final void writeReg2( int b )
  {
    if( _activeDrive != null )
    {
      _activeDrive.writeReg2( b );
    }
  }

  private final int readReg0()
  {
    if( _activeDrive != null )
    {
      return _activeDrive.readReg0();
    }
    else
    {
      return 0;
    }
  }

  private final int readReg1()
  {
    int r = _activeDriveNumber;

    if( active() ) r |= 0x20;
    if( home() ) r |= 0x08;
    if( dataReady() ) r |= 0x80;

    return r;
  }

  private final int readReg2()
  {
    if( _activeDrive != null )
    {
      return _activeDrive.readReg2();
    }
    else
    {
      return 0;
    }
  }

  public final void write( int address, int b )
  {
    if( _activeDrive != null ) _activeDrive.activate();

    //System.out.println( "A: Writing " + Z80.toHexByte( b ) + " to " + Z80.toHexWord( address ) );
    switch( address )
    {
      case 0: writeReg0( b ); break;
      case 1: writeReg1( b ); break;
      case 2: writeReg2( b ); break;
    }
  }

  public final int read( int address )
  {
    if( _activeDrive != null ) _activeDrive.activate();

    int r = 0;
    switch( address )
    {
      case 0: r = readReg0(); break;
      case 1: r = readReg1(); break;
      case 2: r = readReg2(); break;
    }
    //System.out.println( "A: Reading " + Z80.toHexByte( r ) + " from " + Z80.toHexWord( address ) );
    return r;
  }

  public final void tick()
  {
    for( int i =0; i < _drives.length; ++i )
    {
      _drives[ i ].tick();
    }
    if( _activeDrive != null )
    {
      if( _activeDrive.active() == false )
      {
        _activeDrive = null;
        _activeDriveNumber = 0x40;
      }
    }
  }
}