; EXPORT.MAC
; ~~~~~~~~~~
;
; Contents:
; 	Z80 assembler program for CP/M.
;
; Purpose:
;	Export a file from the xtrs emulator to unix.
;
; Usage:
;	EXPORT cpmfileref [$[T][L]]
;
; Parameters:
;	cpmfileref - name of the file in CP/M to export (wildcards allowed).
;	$ indicates options to export:
;		T - treat files as text (cr,lf -> lf, stop at sub)
;		L - make lowercase UNIX filenames.
;
; Notes:
;	The CP/M CCP converts all command line arguments to upper
;	case, therefore the file will be exported as an all upper case
;	file name (unless $L is used).
;
;	Also, CP/M does not keep an accurate file size.  Binary files
;	are always multiples of 128 byte blocks; text file end at the
;	first sub character.
;
;	There are bdos calls to locate matching file references - Find
;	First and Find Next.  It would be simple to uses these calls
;	to find each file and export each file as it is found.  Wrong!
;	In the Find Next bdos call, my CP/M bible states "No other
;	file operation should be done between two search requests
;	because the BDOS may lose its position in the directory.  In
;	some version of the system this request can use used following
;	Close File to find the next file after the one closed.  This
;	technique is not recommended".  I tested this and after
;	opening the first file, Find Next returns file not found.  So
;	I first create a list of matching files and then export each
;	file in the list.  A "STAT DSK:" reports that my disks can
;	hold 128 directory entries.  At 11 bytes a file this would
;	require less than 1.5KBytes - just be careful if you have a
;	hard disk with *HEAPS* of directory entries.
;
;	If you do have a hard disk and are using it with Montezuma
;	CP/M, I'd like to know about it as I'd really like to emulate a hard
;	disk.
;
; Author(s):
;	RKG	Roland Gerlach <Roland@rkga.com.au>
;
; Amendment History:
;	10MAY98	RKG Work started.
;	14SEP98	RKG Initial release (after a really long break).
;	10OCT98	RKG Clean up code.
;	21JUL99 RKG Fix binary export bug.
;	14AUG99 RKG Rewrite to cope with wildcard CP/M filerefs
;	15AUG99	RKG Release (note new usage)
;
;       28SEP01 JTB Transliterated code to 8080 form for use by Solace

;==============================================================================
; Macro Libraries
;==============================================================================

		maclib	CPM
		maclib	EMTDEF

;==============================================================================
; Messages
;==============================================================================

		org	0100h
		jmp	main

m$usage:	db	cr,lf,'Usage:',tab
		db	'EXPORT cpmfileref [$'
m$usage1:	db	'[T][L]]'
m$crlf:	db	cr,lf,'$'

m$invalid:	db	'Invalid option "'
invalid:	db	'x'
		db	'"',cr,lf,'$'

m$copen:	db	'Unable to open CP/M file$'
m$cread:	db	'Unable to read CP/M file$'

m$uopen:	db	'Unable to open PC file: $'
m$uwrite:	db	'Unable to write PC file: $'
m$uclose:	db	'Unable to close PC file: $'
m$uunkerr:	db	'Unknown PC error',cr,lf,'$'

;==============================================================================
; Storage
;==============================================================================

textflag:	db	0		; 0=>binary, else text
caseflag:	db	0		; 0=>uppercase, else lowercase

fcb$search:	; fcb used for searching
fcb$drive:	db	0
fcb$filename:	db	'????????'
fcb$type:	db	'???'
		db	0		; extent
		db	0,0,0		; s1, s2, record count
		dw	0,0,0,0,0,0,0,0	; data map
		dw	0,0		; current record and direct address

unixfn:	ds	13		; unix filename
unixfd:	dw	0		; unix file descriptor

unixbuf:	ds	128		; space for unix ascii buffer
lastchar:	db	0		; last character processed in text file

		; space for the stack
		ds	32
stack$top:	dw	0	; stack grows down from this (wasted)

;==============================================================================
; Code
;==============================================================================

main:		; set up the stack
		lxi sp,stack$top

		; check for command line arguments
		lda c$fcb1$fname	; start of command args
		cpi blank
		jz usage		; no args? show usage

		lxi h,c$fcb2$fname	; second arg
		mvi b,8
main1:		mov a,m
		inx h
		cpi '$'			; ignore '$'
		jz main9
		cpi blank		; ignore blanks
		jz main9

		cpi 'T'			; text?
		jnz main2
		sta textflag		; set text flag
		jmp main9

main2:		cpi 'L'			; lower case?
		jnz main3
		sta caseflag		; set lower case flag
		jmp main9

main3:		sta invalid		; store character in message
		bdos b$couts,m$invalid
		jmp usage
		
main9:		dcr b ! jnz main1

		call mklist		; make list of filenames to export

		call prlist		; process the list

done:		jmp c$boot

usage:
;------------------------------------------------------------------------------
; Display program usage
;	exit:	does not return
;------------------------------------------------------------------------------
		bdos b$couts,m$usage
		mvi e,'$'
		bdos b$coutb
		bdos b$couts,m$usage1
		jmp done		; that's all folks	

mklist:
;------------------------------------------------------------------------------
; Make a list of filenames to export
;------------------------------------------------------------------------------
		; copy fcb1 to searching fcb
		lxi d,fcb$search	; destination
		lxi h,c$fcb1		; source
		lxi b,12		; length; only want drive, filename, & type
		call ldir		; copy it

		; find first matching file
		bdos b$ffirst,fcb$search

		lxi d,filelist		; point to area for next entry in list

mklist1:	; check for file not found
		ora a			; not found if a=FFh
		jm mklist2

					; a=0,1,2,3 file number in buffer
;		sla a			; a=0,2,4,6
;		sla a			; a=0,4,8,12
;		sla a			; a=0,8,16,24
;		sla a			; a=0,16,32,48
;		sla a			; a=0,32,64,96 offset in buffer
		add a ! add a ! add a ! add a ! add a

		mvi b,0
		mov c,a			; bc=offset
		lxi h,c$defbuf
		dad b
		inx h			; first char of filename
		
		lxi b,11		; length; filename & type
		call ldir		; copy it

		push d		
		bdos b$fnext,fcb$search
		pop d

		jmp mklist1

mklist2:	; append a null file to list.
		mvi a,0
		stax d
		ret


prlist:
;------------------------------------------------------------------------------
; Process files in the list
;------------------------------------------------------------------------------
		lxi h,filelist

prlist1:	mov a,m
		ora a
		rz

		; copy filename and type to fcb
		lxi d,c$fcb1$fname	; destination
		lxi b,11		; length; filename & type
		call ldir		; copy it
		push h

		mvi a,0			; zero remainder of fcb
		mvi b,24
prlist2:	stax d
		inx d
		dcr b ! jnz prlist2

		call uname		; build UNIX name

		lda (caseflag)
		ora a
		cnz ulower		; convert UNIX name to lower case

		call copen		; open CP/M file
		jnz skip

		call uopen		; open UNIX file
		jnz skip

		call export		; export CP/M->UNIX

		; no need to close CP/M file it was only used for input		
		call uclose		; close UNIX file

skip:		bdos b$couts,m$crlf

		pop h
		jmp prlist1

uname:
;------------------------------------------------------------------------------
; Show filename in fcb1 and store it in unixfn
;
;	exit:	all registers trashed
;------------------------------------------------------------------------------
		lxi h,c$fcb1$drive	; drive
		mov a,m
		inx h
		ora a
		jz uname1
		adi 'A'
		dcr a
		push h
		mov e,a
		bdos b$coutb
		mvi e,':'
		bdos b$coutb
		pop h

uname1:		lxi d,unixfn		; unix filename

; prefix name with an exclamation mark for testing
;	mvi a,21h
;	stax d
;	inx d

		mvi b,8			; show filename
		call uname2

		mvi a,'.'		; a '.' after filename
		stax d			; store in unix filename
		inx d
		push h
		push d
		mov e,a
		bdos b$coutb
		pop d
		pop h

		mvi b,3			; type
		call uname2

		mvi a,0			; terminate unix filename
		stax d

		mvi e,blank		; show a blank
		bdos b$coutb
		
		ret

uname2:		mov a,m
		inx h
		cpi blank
		jz uname3		; ignore blanks

		ani 7Fh			; strip top bit
		stax d			; save in unix filename
		inx d

 		push h
		push b
		push d
		mov e,a
		bdos b$coutb
		pop d
		pop b
		pop h

uname3:		dcr b ! jnz uname2
		ret

ulower:
;------------------------------------------------------------------------------
; convert unix file name to lower case
;------------------------------------------------------------------------------
		lxi h,unixfn
ulower1:	mov a,m
		ora a
		rz
		call lc
		mov m,a
		inx h
		jmp ulower1

lc:		cpi 'A'
		rc			; before 'A'
		cpi 'Z'+1
		rnc			; after 'Z'
		adi 'a'-'A'		; convert
		ret

export:
;==============================================================================
; Export
;==============================================================================
		lda textflag
		ora a
		jnz export1
						
		jmp binary		; binary export

export1:	jmp ascii		; text export

binary:
;------------------------------------------------------------------------------
; export a binary file
;	entry:	(c_fcb1) contains an open CP/M file
;		(unixfd) contains an open unix file
;	exit:	z indicates a successful export
;		all registers trashed
;------------------------------------------------------------------------------
		lxi h,c$iobuf		; all i/o thru cp/m io buffer
		mvi b,c$iobufsize	; fixed block size

binary1:			; read cp/m block
		call cread
		jnz binary2		; eof

				; write unix block
		call uwrite
		rnz			; error
		jmp binary1

binary2:	xra a
		ret

ascii:
;------------------------------------------------------------------------------
; export an ascii file (convert cr lf to lf and stop at sub character)
;	during: (hl) unix buffer
;		(de) cpm buffer
;		b bytes remaining in unix buffer
;		c bytes remaining in cp/m buffer
;	exit:	z indicates a successful export
;		all registers trashed
;------------------------------------------------------------------------------
		lxi h,unixbuf		; unix buffer
		mvi b,c$iobufsize

		lxi d,c$iobuf		; cp/m buffer
		mvi c,0
			
ascii1:		mov a,c
		ora a
		jnz ascii2		; cp/m buffer not empty

		lxi d,c$iobuf		; read block from cp/m file
		mvi c,c$iobufsize
		call cread

ascii2:		; check for end of file
		ldax d
		cpi ctrlz		; sub is eof for ascii file
		jz ascii6		; done

		; was last character a cr?
		lda lastchar
		cpi cr
		jnz ascii3		; no

		; current character a lf?
		ldax d			; get current char
		cpi lf
		jz ascii4		; yes

		; store the cr which was not followed by lf
		mvi a,cr
		call asciiput
ascii3:		ldax d			; get current char

		; current character a cr?
		cpi cr
		jz ascii5		; yes

ascii4:		; store current character
		push psw
		call asciiput
		pop psw

ascii5:		; finished with current character
		sta lastchar		; remember it as last character
		inx d			; bump pointer
		dcr c			; one less char in buffer
		jmp ascii1

ascii6:		; cp/m eof - anything in unix buffer
		mvi a,c$iobufsize
		sub b			; a has bytes used in unix buffer
		rz			; buffer empty

		; flush unix buffer
		lxi h,unixbuf		; buffer start
		mov b,a			; bytes to write
		call uwrite

		ret

asciiput:	; put a character into the unix buffer
		mov m,a
		inx h			; bump pointer
		dcr b			; dec count
		mov a,b
		ora a
		rnz			; buffer not full yet
		lxi h,unixbuf		; reset pointer and counter
		mvi b,c$iobufsize	; bytes to write
		call uwrite
		ret

;==============================================================================
; cp/m file routines
;==============================================================================

copen:
;------------------------------------------------------------------------------
; open the cp/m filename in fcb1
;	entry:	(c_fcb1) contains cp/m fcb
;	exit:	z indicates a successful open
;		all registers trashed
;------------------------------------------------------------------------------
		bdos b$fopen,c$fcb1	; open existing file (a=FFh is error)
		ora a
		jm copen1
		xra a
		ret			; open ok

copen1:		bdos b$couts,m$copen
		xra a
		dcr a
		ret			; ret nz
		
cread:
;------------------------------------------------------------------------------
; read a block from the cp/m file
;	entry:	(c_fcb1) contains the cp/m fcb
;	exit:	z indicates a successful read
;		af trashed (bc,de,hl saved)
;------------------------------------------------------------------------------
		push b
		push d
		push h
		bdos b$frseq,c$fcb1	; a=0 is ok, a=0FFh is eof
		pop h
		pop d
		pop b
		ora a
		ret			; z=>0, nz=>eof

;==============================================================================
; utility routines
;==============================================================================

;------------------------------------------------------------------------------
; substitute for z80 ldir instruction
;	entry:	hl points to source
;		de points to destination
;		bc is # of bytes to transfer
;	exit:	z indicates a sucessful close
;		all registers trashed
;------------------------------------------------------------------------------
		
ldir:		sta	ldir$a		; tuck it away for the duration

ldir2:		mov	a,m
		stax	d
		inx	h
		inx	d
		dcx	b
		mov	a,c
		ora	b
		jnz	ldir2

		lda	ldir$a		; restore a but leave flags set
		ret

ldir$a:	ds	1

;==============================================================================
; unix file routines
;==============================================================================

uerror:
;------------------------------------------------------------------------------
; display the error message for a unix errno
;	entry:	a=error no
;	exit:	all registers trashed
;------------------------------------------------------------------------------
		lxi d,c$iobuf		; buffer to store msg
		push d
		pop h			; ld hl,de
		lxi b,c$iobufsize
		dw emt$strerror
		jnz uerror2		; another error?!
		dad b
		mvi m,lf		; add lf
		inx h
		mvi m,'$'		; add terminator
		bdos b$couts,c$iobuf
uerror1:	xra a
		dcr a
		ret			; ret nz

uerror2:	bdos b$couts,m$uunkerr
		jmp uerror1

uopen:
;------------------------------------------------------------------------------
; open unix file
;	entry:	(unixfn) contains unix file name
;	exit:	z indicates successful open
;		(unixfd) contains unix file descriptor
;		all registers trashed.
;------------------------------------------------------------------------------
		lxi h,unixfn
		lxi b,eo$wronly + eo$creat + eo$trunc
		lxi d,0666q		; permissions
		dw emt$open
;		ld (unixfd),de		; save unix fd
		xchg ! shld unixfd ! xchg
		rz			; ok

		push psw		; save error number
		bdos b$couts,m$uopen
		pop psw			; restore error number
		jmp uerror

uwrite:
;------------------------------------------------------------------------------
; write a block to the unix file
;	entry:	(hl) start of buffer
;		b contains the number of bytes to write
;		(unixfd) contains unix file descriptor
;	exit:	z indicates sucessful write
;		af trashed (bc,de,hl saved)
;------------------------------------------------------------------------------
		push h
		push d
		push b
		mov c,b			; length in bc
		mvi b,0
;		ld de,(unixfd)		; get unix fd
		xchg ! lhld unixfd ! xchg
		dw emt$write	
		pop b			; restore registers
		pop d
		pop h
		rz			; ok

		push psw		; save error number
		bdos b$couts,m$uwrite
		pop psw			; restore error number
		jmp uerror

uclose:
;------------------------------------------------------------------------------
; close the unix file
;	entry:	(unixfd) contains unix file descriptor
;	exit:	z indicates a sucessful close
;		all registers trashed
;------------------------------------------------------------------------------
;		ld de,(unixfd)		; get unix fd
		xchg ! lhld unixfd ! xchg
		dw emt$close
		rz			; ok
		push psw		; save error number
		bdos b$couts,m$uclose
		pop psw			; restore error number
		jmp uerror

filelist:	db	0		; build list of files up from here


end:		end main

