; import.mac
; ~~~~~~~~~~
;
; Contents:
;	Z80 assembler program for CP/M.
;
; Purpose:
;	Import a file from unix into the xtrs emulator.
;
; Usage:
;	import [-n] [unixfilename [cpmfilename]]
;
; Parameters:
;	-n converts unix lf to CP/M cr lf;
;	unixfilename - name of the unix file to import; and
;	cpmfilename - name of the file in CP/M (must not exist);
;
; Notes:
;	When started with no arguments, import will prompt for all args.
;	If only the unixfilename is specified, the cpmfilename will be 
;	assumed to have the same name.
;
;	Also note that the CP/M CCP converts all command line arguments to
;	upper case, therefore if unixfilename is specified on the command
;	line then it will be taken as all upper case.
;
; Author(s):
;	RKG	Roland Gerlach <Roland@rkga.com.au>
;
; Amendment History:
;	11MAR98	 RKG  Work started.
;	15MAR98	 RKG  Initial release.
;	10OCT98	 RKG  Clean up code.
;       02NOV98  RKG  Fix bug in "copen" which was triggered when attempting
;                     to import to an existing CP/M file.
;
;       28SEP01  JTB  Transliterated code to 8080 form for use by Solace
;       27NOV01  JTB  Changed to use 1 KB buffer -- about 4x faster
;
;==============================================================================
; Macro Libraries
;==============================================================================

		maclib	CPM
		maclib	EMTDEF

bufsize		equ	1024	; must be multiple of 128
*eject
;==============================================================================
; messages
;==============================================================================

		org 0100h
		jmp program

m$nl:		db	cr,lf,'$'

m$usage:	db	cr,lf,'Usage:',tab
		db	'import [[-n] pcfn [cpmfn]]',cr,lf,'$'
m$complete:	db	cr,lf,'import complete',cr,lf,'$'

m$f$prompt:	db	'Convert lf to cr lf.........: $'
m$f$invalid:	db	'Invalid flag',cr,lf,'$'

m$u$prompt:	db	'PC filename.................: $'
m$u$open:	db	'Unable to open PC file: ','$'
m$u$read:	db	'Unable to read PC file: ','$'
m$u$close:	db	'Unable to close PC file: ','$'
m$u$unkerr:	db	'Unknown PC error',cr,lf,'$'

m$c$prompt:	db	'CP/M filename...............: $'
m$c$exists:	db	'CP/M file already exists',cr,lf,'$'
m$c$open:	db	'Unable to open CP/M file',cr,lf,'$'
m$c$write:	db	'Unable to write CP/M file',cr,lf,'$'
m$c$close:	db	'Unable to close CP/M file',cr,lf,'$'

*eject
import:
;==============================================================================
; import [[-n] unixfn [cpmfn]]
;==============================================================================

program:
		lxi sp,stack$top	; set up the stack

				; check for command line arguments
		lxi h,c$cmdstr		; start of command args
		call skipblanks

				; parse or prompt?
		jnz import1		; cmd line arguments
		call prompt		; prompt for arguments
		jmp import2
import1:	call parse		; parse command line arguments

import2:			; binary or ascii?
		mvi a,1
		jz import3		; ascii import
		dcr a			; 0=binary import
import3:	sta asciimode
		call xfer

import4:			; export complete
		jnz endpgm		; import was bad
		call uclose		; close unix file
		call cclose		; close cp/m file
		bdos b$couts,m$complete

endpgm:		jmp c$boot		; all done

usage:
;==============================================================================
; display program usage
;	exit:	does not return
;==============================================================================

		bdos b$couts,m$usage
		jmp endpgm		; that's all folks
*eject
parse:
;==============================================================================
; parse the command line arguments and open both the unix and cp/m files
;	entry:	(hl) start of command line args
;		a contains first character of arguments (from skipblanks)
;	exit:	z flag indicates ascii import (i.e. conversion)
;		(c_fcb1) contains the cp/m fcb
;		(unixfd) contains the unix file descriptor
;		all registers trashed
;==============================================================================

				; conversion flag
		cpi '-'			; check first character
		jnz parse3		; no '-'
		inx h
		mov a,m		; check second character
		cpi 'N'
		jz parse2		; "-N"

parse1:					; bad flag
		bdos b$couts,m$f$invalid
		jmp usage
parse2:		inx h
		mov a,m		; check third character
		cpi ' '
		jnz parse1		; no space after "-N"
		xra a			; zero (i.e. ascii import)
parse3:		ora a			; set flags
		push psw		; save conversion flag

				; unix filename
		call skipblanks
		jz usage		; no unix filename
		push h			; save start of unix filename
parse4:		inx h			; find end
		mov a,m
		cpi null		; end of args?
		jz parse5
		cpi blank
		jnz parse4
		mvi m,null		; terminate unix filename
		inx h
parse5:		xthl			; save place and get start of fn
		push h			; save start address
		call uopen		; open unix filename
		pop b			; restore start of unix filename
		jnz usage
		pop h			; retore place in cmd line

				; cpm filename
		call skipblanks
		jnz parse6
		push b			; start of unix filename
		pop h			; use the same name
parse6:		call cfillfcb
		call copen
		jnz usage

		pop psw			; conversion flag
		ret
*eject
prompt:
;==============================================================================
; prompt for arguments
;	exit:	z flag indicates ascii import (i.e. conversion)
;		(c_fcb1) contains the cp/m fcb
;		(unixfd) contains the unix file descriptor
;		all registers trashed
;==============================================================================

				; conversion flag
		mvi a,blank		; default answer
		sta (c$iobuf+2)
prompt1:	bdos b$couts,m$f$prompt
		lxi d,c$iobuf		; storage buffer
		mvi a,4			; max chars to read
		stax d
		bdos b$cins
		bdos b$couts,m$nl
		lda (c$iobuf+2)		; first char of answer
		call upcase
		cpi 'N'
		jz prompt2
		cpi 'Y'
		jnz prompt1		; ask again
		xra a			; zero
prompt2:	ora a			; set flags
		push psw		; save conversion flag

prompt3:			; unix filename
		bdos b$couts,m$u$prompt
		lxi d,c$iobuf		; storage buffer
		mvi a,70		; max number of characters to read
		stax d
		bdos b$cins
		bdos b$couts,m$nl
		lxi h,c$iobuf+1		; point to length
		call termstring
		call skipblanks
		call uopen
		jnz prompt3		; ask again

prompt4:			; cpm filename
		bdos b$couts,m$c$prompt
		lxi d,c$iobuf		; storage buffer
		mvi a,15		; max chars to read
		stax d
		bdos b$cins
		bdos b$couts,m$nl
		lxi h,c$iobuf+1
		call termstring
		call skipblanks
		jz prompt4		; no filename
		call cfillfcb
		call copen
		cnz prompt4		; ask again

		pop psw			; conversion flag
		ret
*eject
;==============================================================================
; unix file routines
;==============================================================================

uopen:
;------------------------------------------------------------------------------
; open unix file
;	entry:	(hl) start of unix filename
;	exit:	z indicates successful open
;		(unixfd) contains unix file descriptor
;		af,bc,de trashed.
;------------------------------------------------------------------------------

		lxi b,eo$rdonly
		lxi d,0
		dw emt$open
;		ld (unixfd),de		; save unix fd
		xchg ! shld unixfd ! xchg ; save unix fd

		rz			; ok
		push psw		; save error number
		bdos b$couts,m$u$open
		pop psw			; restore error number
		jmp uerror

uread:
;------------------------------------------------------------------------------
; read a block from the unix file
;	entry:	(hl) start of buffer (bufsize length)
;		(unixfd) contains unix file descriptor
;	exit:	z indicates sucessful read
;		bc contains number of bytes read
;		af trashed (de,hl saved)
;------------------------------------------------------------------------------

		push h
		push d

		lxi b,bufsize		; length
;		ld de,(unixfd)		; get unix fd
		xchg ! lhld unixfd ! xchg ; get unix fd
		dw emt$read		; bytes read left in bc

				; show action
		push b
		mvi e,'r'
		bdos b$coutb
		pop b

		pop d
		pop h
		rz

		push psw		; save error number
		bdos b$couts,m$u$read
		pop psw			; restore error number
		jmp uerror

uclose:
;------------------------------------------------------------------------------
; close the unix file
;	entry:	(unixfd) contains unix file descriptor
;	exit:	z indicates a sucessful close
;		all registers trashed
;------------------------------------------------------------------------------

		lhld unixfd
		xchg			; de = unix fd
		dw emt$close
		rz			; ok
		push psw		; save error number
		bdos b$couts,m$u$close
		pop psw			; restore error number
		jmp uerror

uerror:
;------------------------------------------------------------------------------
; display the error message for a unix errno
;	entry:	a contains the error no
;	exit:	all registers trashed
;		a is non-zero (indicating an error)
;------------------------------------------------------------------------------

		lxi d,c$iobuf		; buffer to store msg
		push d
		pop h			; ld hl,de
		lxi b,c$iobufsize
		dw emt$strerror
		jnz uerror1		; another error?!
		dad b
		mvi m,lf		; add lf
		inx h
		mvi m,'$'		; add terminator
		bdos b$couts,c$iobuf
		jmp retnz
uerror1:	bdos b$couts,m$u$unkerr
		jmp retnz
*eject
;==============================================================================
; cp/m file routines
;==============================================================================

cfillfcb:
;------------------------------------------------------------------------------
; fill fcb1 with a cp/m fileref
;	entry:	(hl) start of cp/m fileref
;	exit:	(hl) first char after cp/m filename
;		(c_fcb1) contains the cp/m fcb
;		af,b,de trashed
;------------------------------------------------------------------------------

		lxi d,c$fcb1		; fcb to fill

				; drive code (optional)
		inx h			; test 2nd character
		mov a,m
		cpi ':'			; a ':' indicates a drive code
		dcx h			; back to the start
		jz cfillfcb1		; found a ':', must be a drive code

				; no drive code
		mvi a,0			; zero indicates current drive
		stax d
		jmp cfillfcb2

cfillfcb1:			; a drive code
		mov a,m
		call upcase		; convert to upper case
		sui 'A'-1		; 'A' becomes 1
		stax d			; store drive code
		inx h			; skip over the ':'
		inx h

cfillfcb2:			; filename
		inx d
		mvi b,8
		call cgetname		; 8 char filename
		mvi b,3
		call cgetname		; 3 char extension
		ret

cgetname:
;------------------------------------------------------------------------------
; transribe bytes and pad with spaces until a nul, space, or a '.' found
;	entry:	(hl) source
;		(de) destination
;		b bytes to transribe/pad
;	exit:	(hl) next byte in source (nul or space), or byte after '.'
;		(de) next byte in destination
;		af trashed
;------------------------------------------------------------------------------

		mov a,m		; get character
		cpi null
		jz cgetshort		; stop if nul
		cpi blank
		jz cgetshort		; stop if blank
		cpi '.'
		jz cgetshort		; stop if '.'
		call upcase
		stax d			; store character
		inx h			; bump pointers
		inx d
		dcr b ! jnz cgetname

cgetskip:			; ignore extra chars in source
		mov a,m
		cpi null
		rz			; stop on nul
		cpi blank
		rz			; stop on blank
		inx h			; move to next character
		cpi '.'
		rz			; stop after '.'
		jmp cgetskip

cgetshort:			; fill remainder of destination with blanks
		mvi a,blank
cgetshort1:	stax d
		inx d
		dcr b ! jnz cgetshort1
		mov a,m			; check last character copied
		cpi '.'
		rnz			; stop on non '.' character
		inx h
		ret			; stop after '.' character

copen:
;------------------------------------------------------------------------------
; open the cp/m filename in fcb1
;	entry:	(c_fcb1) contains cp/m fcb
;	exit:	z indicates a successful open
;		af,bd,de,hl trashed
;------------------------------------------------------------------------------

		bdos b$fopen,c$fcb1	; open existing file (a=0ffh is error)
		inr a
		jz copen1		; file doesn't exist (good)
		bdos b$fclose,c$fcb1    ; close the existing file
		bdos b$couts,m$c$exists
		jmp retnz
copen1:		bdos b$fnew,c$fcb1	; make new file (a=0FFh is error)
		inr a
		jz copen2
		xra a
		ret
copen2:		bdos b$couts,m$c$open
		jmp retnz

cwrite:
;------------------------------------------------------------------------------
; write a block to the cp/m file
;	entry:	(c_fcb1) contains the cp/m fcb
;               (cpmcount) contains the number of bytes to write
;               cpmbuf contains the data
;	exit:	z indicates a successful write
;		af trashed (bc,de,hl saved)
;------------------------------------------------------------------------------

		push b
		push d
		push h

		lxi d,cpmbuf		; de points to buffer

				; turn byte count into sector count
		lhld cpmcount
		lxi b,007Fh
		dad b			; round up
		dad h
		mov a,h			; see if we are done already
		ora a
		jz cwrite2
		mov b,h			; B now has the # of 128b sectors

				; transfer B 128B sectors
cwrite1:	push b
		push d
		bdos b$setdma		; de point to next block
		bdos b$fwseq,c$fcb1	; do the sector write
		pop d
		pop b
		jnz cwrite3		; error detected
		lxi h,0080h		; point to next 128 bytes
		dad d
		xchg
		dcr b			; decrement sector count
		jnz cwrite1

				; show action
cwrite2:	mvi e,'w'
		bdos b$coutb

cwrite3:	pop h
		pop d
		pop b
		rz
		bdos b$couts,m$c$write
		jmp retnz

cclose:
;------------------------------------------------------------------------------
; close the cp/m filename in fcb1
;	entry:	(c_fcb1) contains cp/m fcb
;	exit:	z indicates a sucessful close
;		all registers trashed
;------------------------------------------------------------------------------

		bdos b$fclose,c$fcb1
		rz
		bdos b$couts,m$c$close
		jmp retnz
*eject
xfer:
;==============================================================================
; import a file
;	during: (hl) unix buffer
;		(de) cpm buffer
;		bc bytes remaining in host buffer
;		(cpmcount) count of bytes in cp/m buffer
;		(cpmleft)  count of space in cp/m buffer
;		(asciimode) flag nonzero if transferring ascii file
;	exit:	z indicates a successful export
;		all registers trashed
;==============================================================================

				; setup cpm buffer
		mvi a,1
		sta cpmempty		; mark cp/m buffer as empty

				; read a block from unix file
xfer1:		lxi h,unixbuf
		call uread
		mov a,b
		ora c
		jz xfer4		; nothing was read

				; scan thru all bytes in unix buffer
xfer2:		mov a,m
		cpi lf
		jnz xfer3
		lda asciimode		; see if we are doing conversion
		ora a
		mvi a,cr
		cnz putbyte
		mvi a,lf
xfer3:		call putbyte
		inx h
		dcx b
		mov a,b
		ora c
		jnz xfer2		; continue processing this block
		jmp xfer1		; done with block -- get next block

xfer4:				; end of file reached
		lda asciimode		; see if we are doing conversion
		ora a
		mvi a,ctrlz		; add a sub character
		cnz putbyte

				; fill end of cpm sector with nul chars
		lda cpmleft		; get ls byte of count
		ani 7Fh			; cp/m sectors are 128 bytes long
		jz xfer6		; already done
		xchg			; hl now points to cp/m buffer
xfer5:		mvi m,null
		inx h
		dcr a
		jnz xfer5

xfer6:		lda cpmempty		; check if the buffer is empty
		dcr a
		rz			; yes, done
		call cwrite		; write final buffer
		ret


				; put a character into the cp/m buffer
				; exit: bc,hl are preserved
				;       de is updated
putbyte:	push h
		push psw
				; see if the buffer needs to be set up
		lxi h,cpmempty
		mov a,m
		ora a
		jz putbyte1
		mvi m,0			; indicate not empty
		lxi d,cpmbuf
		lxi h,bufsize
		shld cpmleft
		lxi h,0
		shld cpmcount

putbyte1:	pop psw
		stax d
		inx d			; bump pointer
		lhld cpmcount		; inc byte count
		inx h
		shld cpmcount
		lhld cpmleft		; dec remaining count
		dcx h
		shld cpmleft
		mov a,h
		ora l
		jnz putbyte2

		inr a		; buffer is full -- flush it
		sta cpmempty
		call cwrite		; buffer is full
		jz putbyte2
		pop h
		jmp retnz		; indicate error

putbyte2:	pop h
		xra a			; success
		ret
*eject
;==============================================================================
; misc routines
;==============================================================================

retnz:
;------------------------------------------------------------------------------
; load a with non-zero and sets flags
;	exit:	a set to 1 and flags set
;------------------------------------------------------------------------------

		xra a
		inr a
		ret


upcase:
;------------------------------------------------------------------------------
; convert character to upper case
;	entry:	a is the character to convert
;	exit:	a is now upper case
;------------------------------------------------------------------------------

		cpi 'a'
		rc			; smaller than 'a'
		cpi 'z'+1
		rnc			; larger than 'z'
		sui 'a'-'A'		; convert '[a-z]' to '[A-Z]'
		ret


skipblanks:
;------------------------------------------------------------------------------
; skip blanks at the front of a buffer
;	entry:	(hl) start of buffer
;	exit:	(hl) first non blank
;		z flag set if a nul character was found
;------------------------------------------------------------------------------

		mov a,m
		cpi null
		rz			; found nul
		cpi blank
		rnz			; found non-space
		inx h
		jmp skipblanks


termstring:
;------------------------------------------------------------------------------
; append a nul to the end of a bdos console input line buffer
;	entry:	(hl) 1 byte length field followed by string
;	exit:	(hl) first byte of string
;		f,bc trashed
;------------------------------------------------------------------------------

		mvi b,0
		mov c,m			; bc has length
		inx h			; point to first byte
		push h			; save start of response
		dad b
		mvi m,null		; terminate input
		pop h			; restore start of response
		ret


*eject
;==============================================================================
; storage
;==============================================================================

				; buffer to hold the unix file descriptor
unixfd:		dw	0000h

				; space for the stack
		ds	32
stack$top	equ	$		; stack grows down from here

asciimode	ds	1		; flag if transfer does cr/lf munging
cpmempty	ds	1		; flag if buffer is empty
cpmcount	ds	2		; # of bytes in buffer
cpmleft		ds	2		; # of bytes left in buffer

cpmbuf		ds	bufsize		; space for cp/m buffer
unixbuf		ds	bufsize		; space for unix buffer

		end import

