// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2005

// This file contains structures and function prototypes for
// manipulating virtual helios disks for Solace, the Sol emulator.

#ifndef _VDISK_HELIOS_H_
#define _VDISK_HELIOS_H_

#include "vdisk_svdlib.h"

// ===================================================================
//               Logical Layer Access Functions
//
// These functions are used to access the virtual disk independently
// of the controller hardware.
// ===================================================================

// given a filename, get information about a disk.
// returns SVD_OK if it worked, something else if it didn't.
int svh_read_disk_info( char *filename,
			int  *filetype,
			int  *sides,   int *tracks,       int  *sectors,
			int  *density, int *writeprotect, char *label,
			int  *maxsectorsize);

// write a disk header to the specified file.
// returns SVD_OK if it worked, something else if it didn't.
int svh_write_disk_info( char *filename,
			 int   sides, int tracks, int sectors,
			 int   density, int writeprotect, char *label);

// create a blank virtual disk of the specified type
// return 0 if OK, otherwise some error indication.
// does not check if file already exists.
int svh_format(char *filename, int writeprotect, char *label);

// install boot tracks on an already formatted disk.
int svh_make_bootable(char *filename, uint8 *image, int tracks);

// debugging aid -- dump contents of virtual disk file
int svh_dump_disk(char *diskfilename, char *dumpfilename);


// ===================================================================
//                       Emulation Layer
//
// These functions are used to manipulate the emulated hardware --
// the controller and virtual disk drives.
// ===================================================================

// called once at power up
void svh_emu_init(void);

// must be called on reset
void svh_board_reset(void);

// install the card in the system
int svh_install_driver(int base_address);

// this gives the driver a chance to clean up, such as flushing
// any buffers to disk.  returning 0 means OK, 1 means request denied.
int svh_request_uninstall_driver(void);

// pull the card from the system
int svh_uninstall_driver(void);

// put a disk into a drive
int svh_insert_disk(int drive, char *filename);

// remove a disk from a drive
void svh_remove_disk(int drive);


// virtual disk controller properties
// later add more props for debug mode control options
enum { VDCPROP_INSTALLED = 1,	// controller installed or not
       VDCPROP_BASEADDR  = 2,	// where in S-100 space
       VDCPROP_DRIVES    = 3,	// number of disk drives
       VDCPROP_DEBUGMODE = 4	// complain when SW driver errs
     };
// these return 0 on success, non-zero on failure
enum {	VDCPROP_OK      = 0,	// success
	VDCPROP_BADPROP = 1,	// no such property
	VDCPROP_CANTSET = 2,	// not allowed to set this property
	VDCPROP_NOTNOW  = 3	// this property can be set, just not now
     };

int GetDiskControllerProp(int prop, int *value);
int SetDiskControllerProp(int prop, int  value);


// this interface allows querying and in some cases setting
// the state of a given disk or disk drive.
// properties #1 to #7 are read-only
enum { VDPROP_EMPTY    = 1,	// read-only; drive state
       VDPROP_SELECTED = 2,	// read-only; drive state
       VDPROP_SIDES    = 3,	// read-only; disk state
       VDPROP_TRACKS   = 4,	// read-only; disk state
       VDPROP_SECTORS  = 5,	// read-only; disk state
       VDPROP_DENSITY  = 6,	// read-only; disk state
       VDPROP_READONLY = 7,	// disk state (write protect tab)
       VDPROP_LABEL    = 8,	// disk state
       VDPROP_FILENAME = 9 };

enum {	VDPROP_OK       = 0,	// success
	VDPROP_BADPROP  = 1,	// no such property
	VDPROP_CANTSET  = 2,	// not allowed to set this property
	VDPROP_BADDRIVE = 3,	// not a legal disk drive
	VDPROP_NODISK   = 4	// no disk in drive
     };

// these return 0 on success, non-zero on failure
int GetDiskProp(int unit, int prop, int *value);
int SetDiskProp(int unit, int prop, int  value);


// return 1 if the specified virtual disk file is already in use,
// return 0 otherwise.
int DiskInUse(char *filename);

#endif // ifdef _VDISK_HELIOS_H_
