// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2005

/*
   This file adds support to Solace for emulating a Northstar dual density
   floppy disk controller.  This "driver" can be installed and uninstalled
   dynamically for the convenience of the user.  Also, the location it is
   mapped to in memory can be changed to any 1 KB boundary.
 */

#include <stdio.h>
#include <stdarg.h>	// for varargs
#include <string.h>
#include <malloc.h>	// for alloca()
#include <memory.h>	// for memcpy()
#include <sys/stat.h>	// for stat()

#include "vdisk_svnlib.h"
#include "solace_intf.h"	// required for Host* functions

// --------------- local macros -----------------

#define SYNCBYTE (0xFB)

/* -------------------------------------------------------------------
     The following docs are lifted more or less verbatim from
     NorthStar document "Micro-disk system MDS-A-D Double Density"

     DISK CONTROLLER DATA FORMAT

     Each diskette has 35 tracks of data.  Each track is divided into
     10 sectors.  The rotational position of the beginning of the
     sectors is marked by sector holes in the diskette.  Each sector
     is recorded using the following format.  This information is
     recorded starting about 96 microseconds after the sector hole is
     detected.

                           Single Density  Double Density
            Zeros             16 bytes        32 bytes
            Sync Char(FB)      1 byte          2 bytes
            Data             256 bytes       512 bytes
            Check Char         1 byte          1 byte

     [ NOTE: a bit further on the docs claim 15/31 leading 00H
             bytes for each sector, not 16/32. ]

     The check character is computed iteratively by setting it to zero
     and then exclusive ORing each successive data byte value with the
     current value of the check character and left cycling the result.

     The sync character that marks the end of the header and
     the start of data is 0FBH.
   ------------------------------------------------------------------- */


#define SVD_SNG_SECSIZE (256+56)
#define SVD_DBL_SECSIZE (512+112)

#define SVD_SECTOR_OVERHEAD 4
#define SVD_SNG_BLKSIZE (SVD_SNG_SECSIZE + SVD_SECTOR_OVERHEAD)
#define SVD_DBL_BLKSIZE (SVD_DBL_SECSIZE + SVD_SECTOR_OVERHEAD)


// this function is used to compute a byte offset within the real
// disk file that represents a given position in the emulated disk.
// if there are two sides, we store the tracks next to each other on disk
// side is 1 or 2.
// tracks count up from 0.
// sectors count up from 0.
static int
compute_sector_pos(svd_t *svd, int side, int track, int sector)
{
    int block;
    int blksize = (svd->density == 1) ? SVD_SNG_BLKSIZE : SVD_DBL_BLKSIZE;

    ASSERT(svd    != NULL);
    ASSERT(side   >=0 && side   < svd->sides);
    ASSERT(track  >=0 && track  < svd->tracks);
    ASSERT(sector >=0 && sector < svd->sectors);

    block = track * (svd->sectors)*(svd->sides)
	  + side  * (svd->sectors)
	  + sector;

    return ((svd->cached) ? 0 : SVD_HDR_SIZE) +
	   block*blksize;
}


// create a blank virtual disk of the specified type.
// returns SVD_OK if successful, otherwise some error SVD_* indication.
// it does not check if file already exists.  the file is not actually
// written until svnlib_flush() is called using the returned svd_t pointer.
int
svnlib_create_disk(const char *filename,
		   int sides, int tracks, int sectors,
		   int density, int writeprotect, char *label, char fillchar,
		   svd_t *svd)
{
    char secbuf[SVD_DBL_SECSIZE];
    int side, track, sector, block_size;
    int i, stat, checksum, disksize;
    int fillbyte = ((int)fillchar) & 0xFF;
    svd_t disk;

    ASSERT(filename != NULL);
    ASSERT(strlen(filename) < SVD_FILENAME_SIZE);
    ASSERT(label != NULL);
    ASSERT(strlen(label) <= SVD_LABEL_SIZE);

    disksize = sides * tracks * sectors * 
	       ((density==1) ? SVD_SNG_BLKSIZE : SVD_DBL_BLKSIZE);
    disk.data = (char*)malloc(disksize);
    if (disk.data == NULL)
	return SVD_INTERROR;

    strcpy(disk.filename, filename);
    disk.format    = SVD_FORMAT_SVN;
    disk.cached    = 1;
    disk.dirty     = 1;
    disk.sides     = sides;
    disk.tracks    = tracks;
    disk.sectors   = sectors;
    disk.density   = density;
    disk.writeprot = writeprotect;
    strcpy(disk.label, label);

    // create dummy sector data block, filled with 0xE5 bytes.
    // directory entries on the disk are 32 byte structures.
    // the first byte indicates which USER number the file belongs
    // to.  0xE5 is a special value indicating empty entry.
    block_size = (density==1) ? 256 : 512;
    checksum = 0x00;
    for(i=0; i<block_size; i++) {
	secbuf[i] = fillbyte;
	checksum = (fillbyte ^ checksum);
	checksum = 0xFF & ((checksum<<1) + (checksum>>7));
    }
    secbuf[i] = (char)checksum;

    // initialize all dead blocks with sector data
    for(side=0; side<sides; side++) {
	for(track=0; track<tracks; track++) {
	    for(sector=0; sector<sectors; sector++) {
		stat = svnlib_write_sector(&disk, side, track, sector,
					   density, block_size+1, secbuf);
		if (stat != SVD_OK)
		    return stat;
	    }
	}
    }

    *svd = disk;
    return SVD_OK;
}


// this function returns a handle to a virtual disk file in svd*.
// the function returns SVD_OK on success, otherwise some SVD_* error.
// if the parameter "cached" is non-zero, the entire disk image is read
// into memory and subsequent modifications (via the read/write sector
// routines) are done on the in-memory image; the accumulated changes
// are finally commited to memory on svnlib_flush().  if "cached" is
// zero, svd just caches the disk metadata and subsequent sector read/
// write operations happen immediately; svnlib_flush should still be
// called after all changes are done.
int
svnlib_open(const char *filename, int cached, svd_t *svd)
{
    svd_t disk;
    FILE *fd;
    int stat;
    int block_size, num_blocks, blocks_read;

    ASSERT(filename != NULL);
    ASSERT(svd != NULL);

    stat = svdlib_read_file_header(filename, &disk);
    if (stat != SVD_OK)
	return stat;

    disk.cached = cached;
    disk.dirty  = 0;
    disk.data   = NULL;
    if (!cached) {
	*svd = disk;
	return SVD_OK;
    }

    fd = fopen(disk.filename, "rb");
    if (fd == NULL)
	return SVD_OPENERROR;
    if (fseek(fd, SVD_HDR_SIZE, SEEK_SET) != 0) {
	fclose(fd);
	return SVD_ACCESSERROR;
    }

    block_size = ((disk.density==1) ? SVD_SNG_BLKSIZE : SVD_DBL_BLKSIZE);
    num_blocks = disk.sides * disk.tracks * disk.sectors;

    disk.data = (unsigned char *)malloc(num_blocks * block_size);
    if (disk.data == NULL)
	return SVD_INTERROR;

    // earlier versions of Solace didn't write a full maximum sized block
    // of data to the last sector.  a full block (for double density)
    // is 512B payload + 112B pre/postamble + 4B overhead = 628 B/block.
    // Old solace was writing 32B preamble + 512B payload + 1B checksum + 4B overhead = 549 bytes.
    // Thus, when we read disks from the older versions of Solace, we must be
    // ready to handle less than the full maximum sized sector.
    blocks_read = fread(disk.data, block_size, num_blocks, fd);
    if (blocks_read < num_blocks-1)
	return SVD_ACCESSERROR;
    fclose(fd);

    *svd = disk;
    return SVD_OK;
}


// this function is called on an svd_t* handle after changes have been
// made to a disk image.  the function returns SVD_OK on success, otherwise
// some other SVD_* error status is returned.  
int
svnlib_flush(svd_t *svd)
{
    FILE *fd = NULL;
    char *filepath = NULL,	// path to file
         *tempname = NULL,	// name of temporary file
         *realname = NULL;	// final name of file
    int disksize;
    int overwriting;		// true if we are overwriting an existing file
    int status;

    ASSERT(svd != NULL);
    if (!svd->cached) {
	// everything is alredy written to the disk
	ASSERT(!svd->dirty);
	return SVD_OK;
    }
    if (!svd->dirty)
	return SVD_OK;
    svd->dirty = 0;	// not after we're done writing anyway

    // where is the current file directory?
    {
	char *fpath = HostExtractPath(svd->filename);
	if (fpath == NULL)
	    return SVD_INTERROR;
	filepath = alloca(strlen(fpath)+1);
	if (filepath == NULL) {
	    free(fpath);
	    return SVD_INTERROR;
	}
	strcpy(filepath, fpath);
	free(fpath);
    }

    // determine if the file already exists
    {
	struct stat st;
	status = stat(svd->filename, &st);
	if (status != 0) {
	    overwriting = 0;
	} else {
	    if (!(st.st_mode & S_IFREG))
		return SVD_BADFILE;	// not a regular file
	    if (!(st.st_mode & S_IWRITE))
		return SVD_BADFILE;	// not writable
	    overwriting = 1;
	}
    }

    // create a temporary filename
    if (overwriting) {
	char *tname = HostUniqueFilename(filepath, "svn");
	if (tname == NULL)
	    return SVD_INTERROR;
	tempname = alloca(strlen(tname)+1);
	if (tempname == NULL) {
	    free(tname);
	    return SVD_INTERROR;
	}
	strcpy(tempname, tname);
	free(tname);
    }

    // because other routines accept an svd_t*, we have to do this ugly
    // switcheroo in the svd_t structure
    if (overwriting) {
	realname = alloca(strlen(svd->filename)+1);
	if (realname == NULL)
	    return SVD_INTERROR;
	strcpy(realname, svd->filename);
	strcpy(svd->filename, tempname);
    } else {
	// create an empty file
	fd = fopen(svd->filename, "wb");
	if (fd == NULL)
	    return SVD_OPENERROR;
	fclose(fd);
    }

    // write the disk header
    status = svdlib_write_file_header(svd);
    if (status != SVD_OK)
	return status;

    // reopen file for writing
    fd = fopen(svd->filename, "rb+");	// r/w binary
    if (fd == NULL)
	return SVD_OPENERROR;

    // write the disk sectors to the file
    disksize = svd->sides * svd->tracks * svd->sectors
	     * ((svd->density==1) ? SVD_SNG_BLKSIZE : SVD_DBL_BLKSIZE);

    if (fseek(fd, SVD_HDR_SIZE, SEEK_SET) != 0)
	status = SVD_ACCESSERROR;
    else if (fwrite(svd->data, disksize, 1, fd) != 1)
	status = SVD_ACCESSERROR;
    else
	status = SVD_OK;

    fclose(fd);

    if (overwriting) {
	// restore the disk filename that we swapped out
	strcpy(svd->filename, realname);

	if (status == SVD_OK) {
	    // delete the "real" name and rename the temp to be the real name
	    status = svdlib_replace_file(realname, tempname);
	}
    }

    return status;
}


// read a sector: preamble, data, and any checksum are returned
int
svnlib_read_raw_sector(svd_t *svd, int side, int track, int sector,
		       int *density, int *bytes, unsigned char *secbuf)
{
    int offset = compute_sector_pos(svd, side, track, sector);
    int stat;
    unsigned char overhead[SVD_SECTOR_OVERHEAD];
    FILE *fd;

    if (svd->cached) {
	memcpy(overhead, &(svd->data[offset]), SVD_SECTOR_OVERHEAD);
    } else {
	fd = fopen(svd->filename, "rb");
	if (fd == NULL)
	    return SVD_OPENERROR;

	// jump to start of specified sector
	stat = fseek(fd, offset, SEEK_SET);
	if (stat != 0)
	    return SVD_ACCESSERROR;

	if (fread(overhead, SVD_SECTOR_OVERHEAD, 1, fd) != 1) {
	    fclose(fd);
	    return SVD_ACCESSERROR;
	}
    }

    *density = (int)overhead[0];
    ASSERT(*density == 1 || *density == 2);

    *bytes = (int)overhead[1] + 256*(int)overhead[2];	// # bytes actually in sector
    ASSERT(*bytes > 256 || *bytes < 1024);

    if (svd->cached) {
	memcpy(secbuf, &(svd->data[offset+SVD_SECTOR_OVERHEAD]), *bytes);
    } else {
	stat = fread(secbuf, *bytes, 1, fd);
	fclose(fd);
	if (stat != 1)
	    return SVD_ACCESSERROR;
    }

    return SVD_OK;
}


// read a sector, skipping the preamble; checksum is returned, as that
// is part of software-visible state
int
svnlib_read_sector(svd_t *svd, int side, int track, int sector,
		   int *density, int *bytes, unsigned char *secbuf)
{
    int dens, rv, i;
    int prebytes, rawbytes;
    uint8 rawbuff[SVD_DBL_BLKSIZE];

    rv = svnlib_read_raw_sector(svd, side, track, sector,
				density, &rawbytes, rawbuff);
    if (rv != SVD_OK)
	return rv;

    dens = min(*density, svd->density);
    prebytes = (dens==1) ? 20 : 40;

    if (rawbytes < prebytes + 10)	// arbitrary lower bound
	return SVD_BADFORMAT;		// don't bother trying

    for(i=0; i<prebytes; i++) {
	if (rawbuff[i] != 0x00)
	    break;
    }
    if (i == prebytes)
	return SVD_BADFORMAT;	// sync byte not found

    if (rawbuff[i++] != SYNCBYTE)
	return SVD_BADFORMAT;	// sync byte not found
    if (dens == 2)
	if (rawbuff[i++] != SYNCBYTE)
	    return SVD_BADFORMAT;	// sync byte not found

    // now 'i' is pointing at the start of data
    *bytes = rawbytes - i;
    memcpy(secbuf, &rawbuff[i], *bytes);

    return SVD_OK;
}


// write a sector: preamble, data, and any checksum must be supplied
int
svnlib_write_raw_sector(svd_t *svd, int side, int track, int sector,
		        int density, int bytes, unsigned char *secbuf)
{
    int offset = compute_sector_pos(svd, side, track, sector);
    int blksize, dens;
    int stat;
    unsigned char overhead[SVD_SECTOR_OVERHEAD];
    FILE *fd;

    ASSERT(density==1 || density==2);	// single or double

    // each sector contains some metadata
    overhead[0] = density;
    overhead[1] = bytes%256;
    overhead[2] = bytes/256;
    overhead[3] = 0x00;

    // although a disk is prepared to hold a certain amount of data, less may be written
    dens = min(density, svd->density);
    blksize = (dens==1) ? SVD_SNG_SECSIZE : SVD_DBL_SECSIZE;
    blksize = min(blksize, bytes);

    if (svd->cached) {
	// in-memory disk image

	memcpy(&(svd->data[offset+0]), overhead, SVD_SECTOR_OVERHEAD);
	memcpy(&(svd->data[offset+SVD_SECTOR_OVERHEAD]), secbuf, blksize);
	svd->dirty = 1;

    } else {
	// manipulate file directly

	fd = fopen(svd->filename, "rb+");	// open for update
	if (fd == NULL)
	    return SVD_OPENERROR;

	// jump to start of specified sector
	stat = fseek(fd, offset, SEEK_SET);
	if (stat != 0) {
	    fclose(fd);
	    return SVD_ACCESSERROR;
	}

	if (fwrite(overhead, SVD_SECTOR_OVERHEAD, 1, fd) != 1) {
	    fclose(fd);
	    return SVD_ACCESSERROR;
	}

	stat = fwrite((char*)secbuf, blksize, 1, fd);
	fclose(fd);
	if (stat != 1)
	    return SVD_ACCESSERROR;
    }

    return SVD_OK;
}


// write a sector, provided only the "payload" data
int
svnlib_write_sector(svd_t *svd, int side, int track, int sector,
		    int density, int bytes, unsigned char *secbuf)
{
    int blksize, dens, prebytes;
    int rv, i;
    uint8 *rawbuff;

    ASSERT(density==1 || density==2);	// single or double

    dens = min(density, svd->density);
    blksize  = (dens==1) ? SVD_SNG_SECSIZE : SVD_DBL_SECSIZE;
    prebytes = (dens==1) ? 15 : 30;

    rawbuff = (uint8*)alloca(blksize);	// allocate on stack
    ASSERT(rawbuff != NULL);

    // write leading premable 0x00 bytes
    for(i=0; i<prebytes; i++)
	rawbuff[i] = 0x00;
    // write one or two preamble sync bytes
    for(i=prebytes; i<prebytes+dens; i++)
	rawbuff[i] = SYNCBYTE;

    // protect against attempting to write too many bytes
    bytes += prebytes + dens;	// dens == # of sync characters
    bytes = min(blksize, bytes);

    memcpy(&rawbuff[prebytes+dens], secbuf, bytes - (prebytes+dens));

    rv = svnlib_write_raw_sector(svd, side, track, sector,
				 density, bytes, rawbuff);

    return rv;
}
