// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000

// ===============================================================
// The following section contains code for the register window.
// ===============================================================

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <windows.h>
#include <windowsx.h>	// message cracker macros

#include "z80.h"
#include "solace_intf.h"
#include "wingui.h"
#include "windbg_sub.h"
#include "resource.h"


static struct {
    int win_x;		// width of display, in pixels
    int win_y;		// height of display, in pixels
    int chars_x;	// width of display, in characters
    int chars_y;	// height of display, in characters
    // FIXME:
    //   in time, we could store the value of the regs from the last
    //   time we updated and use this information to print the
    //   differing register colors in red.
} regstate;


static BOOL
OnCreateReg(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    return TRUE;
}


// handle window resize events
static void
OnSizeReg(HWND hwnd, UINT state, int cx, int cy)
{
    regstate.win_x = cx;
    regstate.win_y = cy;

    regstate.chars_x = (cx + dbgstate.fix_x - 1) / dbgstate.fix_x;
    regstate.chars_y = (cy + dbgstate.fix_y - 1) / dbgstate.fix_y;
}


static void
OnPaintReg(HWND hwnd)
{
    const int xoff = 3;	// how many pixels from left edge of window
    PAINTSTRUCT	ps;
    HDC hdc;
    char buf[80];
    int saved, len, r;

    hdc = BeginPaint(hwnd, &ps);
    saved = SaveDC(hdc);

    if (!dbgstate.active) {

	// gray out the surface
	GrayOutWindow(hwnd, hdc);

    } else {

	// in order to reduce flashing on single step operation, when
	// we create the window we tell windows not to automatically
	// erase invalidated regions to the background color.  instead,
	// we just draw opaque text.  however, we have to be careful to
	// erase the border too.  that is what we do here.
	// erase strip down the left, bottom, right sides.
	// all the sprintf()'s below format strings 13 chars wide.
	COLORREF bg = RGB(0xFF, 0xFF, 0xFF);
	HBRUSH hBrush = CreateSolidBrush(bg);
	HPEN hPen     = CreatePen(PS_SOLID, 1, bg);
	SelectObject(hdc, hBrush);
	SelectObject(hdc, hPen);
	Rectangle(hdc, 0,0, xoff, regstate.win_y);	// L,T,R,B
	Rectangle(hdc, xoff, 7*dbgstate.fix_y, regstate.win_x, regstate.win_y);
	Rectangle(hdc, xoff + 13*dbgstate.fix_x, 0, regstate.win_x, regstate.win_y);
	DeleteObject(hBrush);
	DeleteObject(hPen);

	SelectFont(hdc, dbgstate.fixed_font);

	r = CPU_RegRead(CPU_REG_A);
	len = sprintf(buf, " A=%02X=%03d    ", r, r);
	TextOut(hdc, xoff, 0, buf, len);

	r = CPU_RegRead(CPU_REG_F);
	len = sprintf(buf, " F=%02X=%c%c%c%c%c%c ", r,
			((r) & 0x80) ? 'S' : ' ',
			((r) & 0x40) ? 'Z' : ' ',
			((r) & 0x10) ? 'H' : ' ',
			((r) & 0x04) ? 'P' : ' ',
			((r) & 0x02) ? 'N' : ' ',
			((r) & 0x01) ? 'C' : ' '
		    );
	TextOut(hdc, xoff, 1*dbgstate.fix_y, buf, len);

	r = CPU_RegRead(CPU_REG_BC);
	len = sprintf(buf, "BC=%04X=%5d", r, r);
	TextOut(hdc, xoff, 2*dbgstate.fix_y, buf, len);
	
	r = CPU_RegRead(CPU_REG_DE);
	len = sprintf(buf, "DE=%04X=%5d", r, r);
	TextOut(hdc, xoff, 3*dbgstate.fix_y, buf, len);

	r = CPU_RegRead(CPU_REG_HL);
	len = sprintf(buf, "HL=%04X=%5d", r, r);
	TextOut(hdc, xoff, 4*dbgstate.fix_y, buf, len);

	r = CPU_RegRead(CPU_REG_SP);
	len = sprintf(buf, "SP=%04X      ", r);
	TextOut(hdc, xoff, 5*dbgstate.fix_y, buf, len);
	
	r = CPU_RegRead(CPU_REG_PC);
	len = sprintf(buf, "PC=%04X      ", r);
	TextOut(hdc, xoff, 6*dbgstate.fix_y, buf, len);
	
//	len = sprintf(buf, "TICK=%08X", (long)0);	// FIXME
//	TextOut(hdc, xoff, 7*dbgstate.fix_y, buf, len);
    }

    RestoreDC(hdc, saved);
    EndPaint(hwnd, &ps);
}


static void
OnDestroyReg(HWND hwnd)
{
    dbgstate.hRegWnd = NULL;
}


// main window handler
static LRESULT CALLBACK
WndProcReg(HWND hWnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {
	HANDLE_MSG(hWnd, WM_CREATE,  OnCreateReg);
	HANDLE_MSG(hWnd, WM_SIZE,    OnSizeReg);
	HANDLE_MSG(hWnd, WM_PAINT,   OnPaintReg);
	HANDLE_MSG(hWnd, WM_DESTROY, OnDestroyReg);
    }

    return DefWindowProc(hWnd, iMsg, wParam, lParam);
}


void
RegisterRegClass()
{
    WNDCLASSEX  wndclass;

    wndclass.cbSize        = sizeof(wndclass);
    wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
    wndclass.lpfnWndProc   = WndProcReg;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = 0;
    wndclass.hInstance     = winstate.hInst;
    wndclass.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
    wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground = NULL; // (HBRUSH)(COLOR_WINDOW + 1); don't auto erase background
    wndclass.lpszMenuName  = NULL;
    wndclass.lpszClassName = "solacereg";
    wndclass.hIconSm       = LoadIcon(NULL, IDI_APPLICATION);

    RegisterClassEx(&wndclass);
}


HWND
CreateRegWindow(HWND hwnd)
{
    DWORD winstyle = WS_CHILD | WS_VISIBLE;
#if 1
    DWORD extstyle = WS_EX_CLIENTEDGE;
#else
    // non-editable windows are supposed to have this style,
    // but it doesn't look right to me
    DWORD extstyle = WS_EX_STATICEDGE;
#endif
    HWND  hwndreg;

    hwndreg = CreateWindowEx(
		extstyle,		// extended style
		"solacereg",		// window class name
		0,			// window caption
		winstyle,		// window style
		0,0,			// initial x,y position
		0,0,			// initial x,y size
		hwnd,			// parent window handle
		NULL,			// window menu handle
		winstate.hInst,		// program instance handle
		NULL);			// creation parameters

    ASSERT(hwndreg != NULL);

    return hwndreg;
}


