// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001

// Create a modal dialog box for some management of virtual disks.
// Here is where virtual disks can be created, properties of existing
// disks can be viewed, labels can be edited, or disks can be copied.

#include <stdlib.h>
#include <stdio.h>
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>

#include "solace_intf.h"
#include "vdisk_svn.h"
#include "wingui.h"
#include "resource.h"


#define NUMTABS 3

static struct {
    HMENU hMenu;		// handle to menu for creating new disks
    HMENU hMenu2;		// handle to menu for inspecting existing disks
    HWND  hTabCtl;		// handle to tab control
    HWND  hTabDlg[NUMTABS];	// handle to each subwindow

    BOOL placement_valid;	// is window position set?
    WINDOWPLACEMENT placement;	// size, position, open/close state of dialog
} state;


// holder for disk we are working with
struct {

    // overall dialog state
    int	modified;		// 1=user has changed something, 0=unchanged
    int	newdisk;		// 1=unsaved, mutable, 0=loaded from somewhere

    // state for 1st tab sheet
    char path[SVD_FILENAME_SIZE];	// path to file on disk
    int format;			// VDFORMAT_NORTHSTAR or VDFORMAT_HELIOS
    int sides;			// sides per disk
    int tracks;			// tracks per side
    int sectors;		// sectors per track
    int density;		// single or double density
    int writeprotect;		// 1=read only disk

    int bootoption;		// boot tracks to be installed?

    // state for 2nd tab sheet
    int allow_edit;		// not part of disk state
    char label[SVD_LABEL_SIZE];	// diskette label

} vdisk;


// this is a list of the various OS boot images built into Solace

extern byte vns_cpm_image[];
extern byte vns_nsdos52dq_image[];

typedef struct {
    char *name;
    char *image;
    int  tracks;
    char fillchar;
} boot_image_t;

// first one listed is the default
static boot_image_t boot_image[] = {
    { "None",		NULL,			0,	(char)0xE5 },
    { "CP/M 2.2",	vns_cpm_image,		2,	(char)0xE5 },
    { "NS DOS 5.2DQ",	vns_nsdos52dq_image,	1,	(char)0x20 },
};


// forward declarations
static int LoadVDisk(char *filename);
static int PerformDiskSave(HWND hwnd, int saveas);


// ========================================================================
// common code
// ========================================================================

// set the vdisk state to northstar format,
// but leave as much as possible alone
static void
set_vdisk_to_northstar(void)
{
    vdisk.format       = VDFORMAT_NORTHSTAR;
    vdisk.modified     = 1;
    vdisk.tracks       = 35;
    vdisk.sectors      = 10;
}


// set the vdisk state to helios format,
// but leave as much as possible alone
static void
set_vdisk_to_helios(void)
{
    vdisk.format       = VDFORMAT_HELIOS;
    vdisk.modified     = 1;
    vdisk.sides        = 1;
    vdisk.tracks       = 77;
    vdisk.sectors      = 16;
    vdisk.density      = 1;
}


// set the vdisk state to the default:
//   double sided, double density NS disk
static void
set_vdisk_to_new(void)
{
    set_vdisk_to_northstar();

    vdisk.modified     = 0;
    vdisk.newdisk      = 1;

    vdisk.path[0]      = '\0';
    vdisk.sides        = 2;
    vdisk.density      = 2;
    vdisk.writeprotect = 0;
    vdisk.bootoption   = 0;

    vdisk.label[0]   = '\0';
    vdisk.allow_edit = 1;

    strcpy(vdisk.label, "Enter your own disk description here.\r\n");
}


// set the vdisk state to inspect the specified drive
static void
set_vdisk_to_inspect(char *filename)
{
    int stat;

    ASSERT(filename != NULL);

    stat = LoadVDisk(filename);
    if (stat != SVD_OK) {
	UI_Alert("Error reading vdisk '%s', %s", filename,
		    vdisk_errstring(stat));
	set_vdisk_to_new();
	return;
    }
    strcpy(vdisk.path,filename);

    vdisk.newdisk    = 0;
    vdisk.modified   = 0;
    vdisk.bootoption = 0;
    vdisk.allow_edit = 0;
}


// ========================================================================
// The following code handle setting up and managing disk properties 
// sub-dialog.
// ========================================================================

static HWND
UpdateNewDiskPropDialog(void)
{
    HWND hDlgPage = state.hTabDlg[0];
    HWND hBut, hBootCtl;
    int enable;

    hBut = GetDlgItem(hDlgPage, IDC_DISK_PATH);
    SetWindowText(hBut, "-- new, not saved --");

    hBut = GetDlgItem(hDlgPage, IDC_DISK_TYPE_NS);
    Button_SetCheck(hBut, (vdisk.format == VDFORMAT_NORTHSTAR) ? BST_CHECKED : BST_UNCHECKED);
    hBut = GetDlgItem(hDlgPage, IDC_DISK_TYPE_HELIOS);
    Button_SetCheck(hBut, (vdisk.format == VDFORMAT_HELIOS)    ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(hBut, FALSE);	// for now

    hBut = GetDlgItem(hDlgPage, IDC_DISK_DENSITY_SINGLE);
    Button_SetCheck(hBut, (vdisk.density == 1) ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(hBut, (vdisk.format != VDFORMAT_HELIOS) && vdisk.newdisk);
    hBut = GetDlgItem(hDlgPage, IDC_DISK_DENSITY_DOUBLE);
    Button_SetCheck(hBut, (vdisk.density == 2) ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(hBut, (vdisk.format != VDFORMAT_HELIOS) && vdisk.newdisk);

    hBut = GetDlgItem(hDlgPage, IDC_DISK_SIDES_SINGLE);
    Button_SetCheck(hBut, (vdisk.sides == 1) ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(hBut, (vdisk.format != VDFORMAT_HELIOS) && vdisk.newdisk);
    hBut = GetDlgItem(hDlgPage, IDC_DISK_SIDES_DOUBLE);
    Button_SetCheck(hBut, (vdisk.sides == 2) ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(hBut, (vdisk.format != VDFORMAT_HELIOS) && vdisk.newdisk);

    enable = (vdisk.format != VDFORMAT_HELIOS)
	  && (vdisk.sides == 2)
	  && (vdisk.density == 2);
    hBootCtl = GetDlgItem(hDlgPage, IDC_DISK_BOOT);
    ComboBox_SetCurSel(hBootCtl, vdisk.bootoption);
    EnableWindow(hBootCtl, enable);
    hBootCtl = GetDlgItem(hDlgPage, IDC_STATIC_BOOT);
    EnableWindow(hBootCtl, enable);

    return hDlgPage;
}


static HWND
UpdateOldDiskPropDialog(void)
{
    HWND hDlgPage = state.hTabDlg[2];
    HWND hBut;
    char *str;

    hBut = GetDlgItem(hDlgPage, IDC_DISK_PATH);
    SetWindowText(hBut, vdisk.path);

    hBut = GetDlgItem(hDlgPage, IDC_VDINSPECT_FORMAT);
    switch (vdisk.format) {
	case VDFORMAT_NORTHSTAR: str = "Northstar"; break;
	case VDFORMAT_HELIOS:    str = "Helios";    break;
	default: ASSERT(0);
    }
    SetWindowText(hBut, str);

    hBut = GetDlgItem(hDlgPage, IDC_VDINSPECT_DENSITY);
    switch (vdisk.density) {
	case 1: str = "Single (256B/sector)"; break;
	case 2: str = "Double (512B/sector)"; break;
	default: ASSERT(0);
    }
    SetWindowText(hBut, str);

    hBut = GetDlgItem(hDlgPage, IDC_VDINSPECT_SIDES);
    switch (vdisk.sides) {
	case 1: str = "Single"; break;
	case 2: str = "Double"; break;
	default: ASSERT(0);
    }
    SetWindowText(hBut, str);

    return hDlgPage;
}


static void
UpdateDiskPropDialog(void)
{
    HWND hDlgPage, hBut;
    char buff[20];


    if (vdisk.newdisk)
	hDlgPage = UpdateNewDiskPropDialog();
    else
	hDlgPage = UpdateOldDiskPropDialog();


    // updated common parts

    hBut = GetDlgItem(hDlgPage, IDC_DISK_TRACKS);
    sprintf(buff, "%d", vdisk.tracks);
    SetWindowText(hBut, buff);
    //EnableWindow(hBut, FALSE);

    hBut = GetDlgItem(hDlgPage, IDC_DISK_SECTORS);
    sprintf(buff, "%d", vdisk.sectors);
    SetWindowText(hBut, buff);
    //EnableWindow(hBut, FALSE);

    hBut = GetDlgItem(hDlgPage, IDC_DISK_WP);
    Button_SetCheck(hBut, (vdisk.writeprotect) ? BST_CHECKED : BST_UNCHECKED);
}


static BOOL CALLBACK
DiskPropDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HWND hBootCtl = GetDlgItem(hwndDlg, IDC_DISK_BOOT);
    HWND hBut;
    int bst;
    int i;

    switch (msg) {

	case WM_INITDIALOG:
	    for(i=0; i<sizeof(boot_image)/sizeof(boot_image_t); i++)
		ComboBox_InsertString(hBootCtl, i, (LPARAM)boot_image[i].name);
	    ComboBox_SetCurSel(hBootCtl, 0);
	    return TRUE;

	case WM_COMMAND:

	    // check for combo box, then check for all clickable things
	    switch (LOWORD(wParam)) {

		case IDC_DISK_BOOT:
		    switch (HIWORD(wParam)) {
			case CBN_SELENDOK:
			    vdisk.bootoption = ComboBox_GetCurSel(hBootCtl);
			    vdisk.modified = 1;
			    return TRUE;
			default:
			    break;
		    }
		    return FALSE;

		default:
		    break;
	    }

	    if (HIWORD(wParam) == BN_CLICKED) {

		hBut = GetDlgItem(hwndDlg, LOWORD(wParam));
		bst = Button_GetState(hBut) & 0x0003;

		switch (LOWORD(wParam)) {

		    // dialog buttons and such
		    case IDC_DISK_TYPE_NS:
			if (vdisk.format != VDFORMAT_NORTHSTAR) {
			    set_vdisk_to_northstar();
			    UpdateDiskPropDialog();
			}
			return TRUE;
		    case IDC_DISK_TYPE_HELIOS:
			if (vdisk.format != VDFORMAT_HELIOS) {
			    set_vdisk_to_helios();
			    UpdateDiskPropDialog();
			}
			return TRUE;
		    case IDC_DISK_DENSITY_SINGLE:
			vdisk.density  = 1;
			vdisk.modified = 1;
			UpdateDiskPropDialog();
			return TRUE;
		    case IDC_DISK_DENSITY_DOUBLE:
			vdisk.density  = 2;
			vdisk.modified = 1;
			UpdateDiskPropDialog();
			return TRUE;
		    case IDC_DISK_SIDES_SINGLE:
			vdisk.sides    = 1;
			vdisk.modified = 1;
			UpdateDiskPropDialog();
			return TRUE;
		    case IDC_DISK_SIDES_DOUBLE:
			vdisk.sides    = 2;
			vdisk.modified = 1;
			UpdateDiskPropDialog();
			return TRUE;
		    case IDC_DISK_WP:
			vdisk.writeprotect = (bst == BST_CHECKED);
			vdisk.modified = 1;
			return TRUE;

		    default:
			break;
		}
	    }
	    return FALSE;	// message not handled


	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


// ========================================================================
// If the user exits or otherwise aborts an edit in progress,
// ask what to do.
// ========================================================================

static enum { DFCONFIRM_CANCEL = 0, DFCONFIRM_FORGET=1, DFCONFIRM_SAVE=2 };

static LRESULT CALLBACK
ConfirmDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    HWND hwndIcon;
    HICON hIcon;

    switch (iMsg) {

	case WM_INITDIALOG:
	    hIcon = LoadIcon(NULL, IDI_EXCLAMATION);
	    hwndIcon = GetDlgItem(hDlg, IDC_DFCONFIRM_ICON);
	    Static_SetIcon(hwndIcon, hIcon);
	    return TRUE;

	case WM_COMMAND:
	    if (HIWORD(wParam) == BN_CLICKED) {
		switch (LOWORD(wParam)) {
		    case ID_DFCONFIRM_SAVE:
			EndDialog(hDlg, DFCONFIRM_SAVE);
			return TRUE;
		    case ID_DFCONFIRM_FORGET:
			EndDialog(hDlg, DFCONFIRM_FORGET);
			return TRUE;
		    case IDCANCEL:
			EndDialog(hDlg, DFCONFIRM_CANCEL);
			return TRUE;
		    default:
			break;
		}
	    }
	    break;

	default:
	    break;

    } // switch (iMsg)

    return FALSE;	// message not handled
}


// check if the dialog has user changes.  if so, ask if they
// want to abandon their current changes or not.
// return 1 of OK to change, 0 if cancel.
static int
CheckOKtoChange(HWND hwnd)
{
    int confirm;

    if (!vdisk.modified)
	return 1;

#if 0
    confirm = MessageBox(hwnd,
	"You have made some changes to this disk.\n" \
	"Do want to forget those changes?",
	"Confirm", MB_OKCANCEL | MB_ICONEXCLAMATION);
    return (confirm == IDOK);
#else
    confirm = DialogBox(winstate.hInst, MAKEINTRESOURCE(IDD_DF_CONFIRM),
			hwnd, (DLGPROC)ConfirmDlgProc);

    switch (confirm) {
	case DFCONFIRM_SAVE:   return PerformDiskSave(hwnd, vdisk.newdisk);
	case DFCONFIRM_FORGET: return 1;
	case DFCONFIRM_CANCEL: return 0;
	default: ASSERT(0);
    }

    return 0;	// keep lint happy
#endif
}


// ========================================================================
// The following code handle setting up and managing disk label sub-dialog.
// ========================================================================

// update the tab containing the label.
// if 'newlabel' is TRUE, then the dialog is set from vdisk.label[],
// otherwise it just retains its old value
static void
UpdateDiskLabelDialog(int newlabel)
{
    HWND hDlgPage    = state.hTabDlg[1];
    HWND hEditBox    = GetDlgItem(hDlgPage, IDC_DISKLABEL_EDIT);
    HWND hEditEnable = GetDlgItem(hDlgPage, IDC_DISKLABEL_ENABLE);

    if (newlabel)
	SetWindowText(hEditBox, vdisk.label);

    Edit_SetReadOnly(hEditBox, !vdisk.allow_edit);
    Button_SetCheck(hEditEnable, (vdisk.allow_edit) ? BST_CHECKED : BST_UNCHECKED);
}


static BOOL CALLBACK
DiskLabelDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HWND hEditEnable = GetDlgItem(hwndDlg, IDC_DISKLABEL_ENABLE);

    switch (msg) {

	case WM_INITDIALOG:
	    UpdateDiskLabelDialog(TRUE);
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD(wParam)) {

		case IDC_DISKLABEL_ENABLE:
		    if (HIWORD(wParam) == BN_CLICKED) {
			int bst = Button_GetState(hEditEnable) & 0x0003;
			vdisk.allow_edit = (bst == BST_CHECKED);
			UpdateDiskLabelDialog(FALSE);
			return TRUE;
		    }
		    break;

		case IDC_DISKLABEL_EDIT:
		    if (HIWORD(wParam) == EN_CHANGE) {
			// the user modified the edit box in some way
			vdisk.modified = 1;
			return TRUE;
		    }
		    break;

		default:
		    break;

	    }
	    return FALSE;	// shouldn't ever happen

	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


// ========================================================================
// The following code handle setting up and managing the container
// dialog.
// ========================================================================

static void
AddMenu(HWND hDlg, int inspectmode)
{
    int i;

    state.hMenu  = LoadMenu(winstate.hInst, MAKEINTRESOURCE(IDR_MENU_DISK));
    state.hMenu2 = LoadMenu(winstate.hInst, MAKEINTRESOURCE(IDR_MENU_DISK2));
    i = SetMenu(hDlg, (inspectmode) ? state.hMenu2 : state.hMenu);
    ASSERT(i != 0);
}


static void
UpdateMenu(HWND hDlg)
{
    int stat;
    stat = SetMenu(hDlg, (vdisk.newdisk) ? state.hMenu : state.hMenu2);
    ASSERT(stat != 0);
}


// change to tabsheet n, taking into account which type of page
// to have for tab 0, based on create/inspect mode
static void
SetTabsheet(int n)
{
    int i;

    ASSERT(0 <= n && n < NUMTABS);

    // hide all controls
    for(i=0; i<NUMTABS; i++)
	ShowWindow(state.hTabDlg[i], SW_HIDE);

    TabCtrl_SetCurSel(state.hTabCtl, n);

    switch (n) {
	case 0:
	    if (vdisk.newdisk)
		ShowWindow(state.hTabDlg[0], SW_SHOW);
	    else
		ShowWindow(state.hTabDlg[2], SW_SHOW);
	    break;
	case 1:
	    ShowWindow(state.hTabDlg[1], SW_SHOW);
	    break;
    }
}


// if the tab currently selected is 0, update the display,
// otherwise leave it as-is
static void
UpdateTabsheet(void)
{
    int page;

    page = TabCtrl_GetCurSel(state.hTabCtl);
    SetTabsheet(page);
}


static void
AddTabSheets(HWND hDlg, int inspectmode)
{
    DWORD winstyle = WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE;
    HWND hTabCtl;
    TCITEM tcItem[NUMTABS];
    RECT rc;
    int i, w, h;

    GetClientRect(hDlg, &rc);
    w = rc.right - rc.left;
    h = rc.bottom - rc.top;

    // create tab control at top of the window
    hTabCtl = CreateWindow(
			WC_TABCONTROL,	// class name for tab control
			NULL,		// caption (title)
			winstyle,
			rc.left,	// initial x position
			rc.top,		// initial y position
			w,		// initial x size
			h,		// initial y size

			hDlg,		// parent
			(HMENU)1234,	// child window identifier
			winstate.hInst,
			NULL);		// no WM_CREATE parameter

    state.hTabCtl = hTabCtl;

    // add the two tabs
    tcItem[0].mask    = TCIF_TEXT;
    tcItem[0].pszText = "Disk Format";
    tcItem[0].iImage  = -1;
    TabCtrl_InsertItem(hTabCtl, 0, &tcItem[0]);

    tcItem[1].mask    = TCIF_TEXT;
    tcItem[1].pszText = "Label";
    tcItem[1].iImage  = -1;
    TabCtrl_InsertItem(hTabCtl, 1, &tcItem[1]);

    // create two subordinate dialogs
    state.hTabDlg[0] = CreateDialog(winstate.hInst,
				    MAKEINTRESOURCE(IDD_DISKPROPS),
				    hTabCtl,
				    DiskPropDlgProc);
    ASSERT(state.hTabDlg[0] != NULL);

    state.hTabDlg[1] = CreateDialog(winstate.hInst,
				    MAKEINTRESOURCE(IDD_DISKLABEL),
				    hTabCtl,
				    DiskLabelDlgProc);
    ASSERT(state.hTabDlg[1] != NULL);

    state.hTabDlg[2] = CreateDialog(winstate.hInst,
				    MAKEINTRESOURCE(IDD_DISKPROPS2),
				    hTabCtl,
				    DiskPropDlgProc);
    ASSERT(state.hTabDlg[2] != NULL);


    // how big is the tab control?
    (void)TabCtrl_GetItemRect(hTabCtl, 0, &rc);

    // move dialogs down just below the row of tabs
    for(i=0; i<NUMTABS; i++)
	SetWindowPos(state.hTabDlg[i], HWND_TOP, rc.left, rc.bottom+2,
			 0, 0, SWP_NOSIZE);
}


// return SVD_OK if it worked, otherwise it is an error code
static int
LoadVDisk(char *filename)
{
    int stat, maxsectorsize;

    ASSERT(filename != NULL);
    strcpy(vdisk.path, filename);

    stat = svdlib_read_disk_info(vdisk.path,
				&vdisk.format,
				&vdisk.sides,
				&vdisk.tracks,
				&vdisk.sectors,
				&vdisk.density,
				&vdisk.writeprotect,
				&vdisk.label[0],
				&maxsectorsize);
    return stat;
}


// return SVD_OK if it worked, otherwise it is an error code
static int
SaveVDisk(char *filename)
{
    int stat;

    if (vdisk.newdisk) {

	// create a new disk (potentially overwriting the old one)
	stat = svn_format(vdisk.path,
		    vdisk.sides, vdisk.tracks, vdisk.sectors,
		    vdisk.density,
		    vdisk.writeprotect,
		    vdisk.label,
		    boot_image[vdisk.bootoption].fillchar
		);
	if (stat != SVD_OK)
	    return stat;

	// install boot tracks if requested to do so
	if (vdisk.newdisk) {
	    byte *image = boot_image[vdisk.bootoption].image;
	    if (image != NULL)
		stat = svn_make_bootable(vdisk.path, image,
				    boot_image[vdisk.bootoption].tracks);
	}

    } else {

	// save back to an existing disk
	stat = svn_write_disk_info( vdisk.path,
				    vdisk.sides, vdisk.tracks, vdisk.sectors,
				    vdisk.density, vdisk.writeprotect,
				    vdisk.label);
    }

    return stat;
}


// returns TRUE if saved, FALSE if cancelled
static int
PerformDiskSave(HWND hwnd, int saveas)
{
    int stat;

    if (saveas) {

	char *newpath;
	int flag = GetSaveFileName(&winstate.ofn[FILE_VDISK]);
	if (!flag)
	    return FALSE;	// aborted

	// if we succeed in opening a file, we want to remember what
	// directory to start the search in next time.
	strcpy(vdisk.path, (char*)winstate.ofn[FILE_VDISK].lpstrFile);
	newpath = HostExtractPath(vdisk.path);
	strcpy((char*)winstate.ofn[FILE_VDISK].lpstrInitialDir, newpath);
	free(newpath);
    }

    // normal save or continuation of saveas
    GetDlgItemText(state.hTabDlg[1], IDC_DISKLABEL_EDIT, vdisk.label, SVD_LABEL_SIZE);
    stat = SaveVDisk(vdisk.path);
    vdisk.modified = 0;
    if (stat != SVD_OK)
	UI_Alert("Error attempting to create new disk '%s', %s",
		    vdisk.path, vdisk_errstring(stat));

    if (saveas) {
	// convert create-disk mode to inspect-disk mode
	vdisk.newdisk    = 0;
	vdisk.allow_edit = 0;
	stat = SetMenu(hwnd, state.hMenu2);
	ASSERT(stat != 0);
	UpdateTabsheet();
	UpdateMenu(hwnd);
	UpdateDiskPropDialog();
	UpdateDiskLabelDialog(FALSE);
    }

    return TRUE;
}


static int
DumpVDisk(char *filename)
{
    char dumpname[MAX_PATH];
    char *p;
    int stat;

    // start with current path and filename
    strcpy(dumpname, vdisk.path);

    // back up and change suffix
    p = &dumpname[strlen(dumpname)-1];
    while (p > dumpname && (*p != '.') && (*p != '\\'))
	p--;
    if (*p != '.')
	strcpy(p, "vdisk.dump");	// just in case name is odd somehow
    else
	strcpy(p+1, "txt");		// eg c:\this\that\thing.svn
					// is c:\this\that\thing.txt

    stat = svn_dump_disk(vdisk.path, dumpname);
    if (stat != SVD_OK)
	UI_Alert("Error attempting to dump disk to '%s', %s",
		    dumpname, vdisk_errstring(stat));

    return stat;
}


static LRESULT CALLBACK
ContainerDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    HMENU hMenu;
    NMHDR *pnmhdr;
    int inspectmode;	// start up inspecting a disk instead of create mode
    char *filename;	// if inspectmode, which file to inspect
    int i, stat;

    switch (iMsg) {

	case WM_INITDIALOG:
	    filename = (char*)lParam;
	    inspectmode = (filename != NULL);
	    AddMenu(hDlg, inspectmode);
	    AddTabSheets(hDlg, 0);
	    if (inspectmode) set_vdisk_to_inspect(filename);
			else set_vdisk_to_new();
	    SetTabsheet(0);

	    // if it was opened & closed previously, use the size/location
	    // from when it was last closed.
	    if (state.placement_valid)
		SetWindowPlacement(hDlg, &state.placement);

	    UpdateMenu(hDlg);
	    UpdateDiskPropDialog();
	    UpdateDiskLabelDialog(TRUE);

	    return TRUE;


	case WM_INITMENUPOPUP:
	    // based on vdisk state, enable/disable "save" button
	    hMenu = GetMenu(hDlg);
	    EnableMenuItem(hMenu, IDM_DISKFILE_SAVE, (vdisk.modified) ? MFS_ENABLED  : MFS_DISABLED);
	    EnableMenuItem(hMenu, IDM_DISKFILE_DUMP, (vdisk.modified) ? MFS_DISABLED : MFS_ENABLED);
	    return TRUE;


	case WM_COMMAND:
	    switch (LOWORD(wParam)) {

		case IDM_DISKFILE_NEW:
		    if (CheckOKtoChange(hDlg)) {
			set_vdisk_to_new();
			SetTabsheet(0);
			UpdateMenu(hDlg);
			UpdateDiskPropDialog();
			UpdateDiskLabelDialog(TRUE);
		    }
		    return TRUE;	// handled


		case IDM_DISKFILE_INSPECT:
		    if (CheckOKtoChange(hDlg)) {
			if (FileOpenDlg(hDlg, FILE_VDISK)) {

			    set_vdisk_to_inspect(winstate.ofn[FILE_VDISK].lpstrFile);

			    SetTabsheet(0);
			    UpdateMenu(hDlg);
			    UpdateDiskPropDialog();
			    UpdateDiskLabelDialog(TRUE);
			}
		    }
		    return TRUE;	// handled

		case IDM_DISKFILE_SAVEAS:
		    PerformDiskSave(hDlg, TRUE);
		    return TRUE;

		case IDM_DISKFILE_SAVE:
		    PerformDiskSave(hDlg, FALSE);
		    return TRUE;	// handled

		case IDM_DISKFILE_DUMP:
		    stat = DumpVDisk(vdisk.path);
		    return TRUE;	// handled

		case IDM_DISKFILE_EXIT:
		    SendMessage(hDlg, WM_CLOSE, 0, 0L);
		    return TRUE;	// handled

		// there is no CANCEL button on the dialog, but we get this
		// message when the user clicks the X on the titlebar
		case IDCANCEL:
		    EndDialog(hDlg, LOWORD(wParam));
		    return TRUE;	// handled

		default:
		    break;
	    }
	    break;


	case WM_NOTIFY:
	    pnmhdr = (NMHDR*)lParam;
	    switch (pnmhdr->code) {
		case TCN_SELCHANGE:
		    i = TabCtrl_GetCurSel((HWND)pnmhdr->hwndFrom);
		    SetTabsheet(i);
		    break;
		default:
		    break;
	    }
	    break;


	case WM_CLOSE:
	    if (!CheckOKtoChange(hDlg))
		return TRUE;	// ignore the request
	    break;		// default handler causes WM_DESTROY


	case WM_DESTROY:

	    // save position
	    state.placement.length = sizeof(WINDOWPLACEMENT);
	    GetWindowPlacement(hDlg, &state.placement);
	    state.placement_valid = TRUE;

	    DestroyMenu(state.hMenu);
	    DestroyMenu(state.hMenu2);
	    state.hMenu = state.hMenu2 = 0;

	    // kill subordinate dialogs
	    for(i=0; i<NUMTABS; i++) {
		DestroyWindow(state.hTabDlg[i]);
		state.hTabDlg[i] = NULL;
	    }
	    state.hTabCtl = NULL;

	    return TRUE;


	default:
	    break;

    } // switch (iMsg)

    return FALSE;	// message not handled
}


// create the container modal dialog
void
CreateDiskPropsDialog(int inspectmode, int unit)
{
    static int firsttime = 1;	// first time called?
    char *filename;

    if (firsttime) {
	firsttime = 0;
	state.placement_valid = 0;
    }

    if (inspectmode) {
	int stat;
	ASSERT(1 <= unit && unit <= 4);
	// check that there is a disk in the drive
	stat = GetDiskProp(unit, VDPROP_FILENAME, (int*)(&filename));
	if (stat == VDPROP_NODISK) {
	    UI_Alert("Can't inspect: there is no disk in that drive");
	    return;
	} else
	    ASSERT(stat == VDPROP_OK);
    } else
	filename = (char*)NULL;

    (void)DialogBoxParam(
		winstate.hInst, MAKEINTRESOURCE(IDD_DISKCONTAINER),
		winstate.hWnd, (DLGPROC)ContainerDlgProc,
		(LPARAM)filename);
}

