// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001

// This include file contains declarations and objects that are shared
// amongst the Windows GUI files but aren't accessed by the core emulator.


#ifndef _WINGUI_H_
#define _WINGUI_H_

#include <windows.h>


#if 1
#include <assert.h>
#define ASSERT(x) assert(x)
#else
#define ASSERT(x)
#endif


// pick fixed or variable char generator
typedef enum {
    MAINONLY,
    MAINNOW,
    ALTNOW
} fontmode_t;

// pick Sol character generator
typedef enum {
    CHARGEN_6574,
    CHARGEN_6575,
    CHARGEN_DZONKHA,
    CHARGEN_DEVANAGRI,
} chargen_t;

// pick Sol monitor color
typedef enum {
    DISP_WHITE,
    DISP_GREEN,
    DISP_AMBER
} disp_color_t;


// display state -- used by winmain.c and windbg.c
typedef struct {

    HINSTANCE hInst;	// application instance

    // because the DC is private, we don't need to keep getting the
    // window DC, doing a makecompatibleDC, selecting the bitmap,
    // then deleting the DC for every character.  it is way too slow.
    // instead, we do it once then just use this copy of it.
    HDC  hdcWin;	// = GetDC(winstate.hWnd);

    HWND hWnd;		// handle to sol screen
    int  exitcode;	// what WM_QUIT returned (ugly hack)

    HWND hToolbar;	// handle to tool bar
    HANDLE hStatus;	// handle to status bar

    HWND hDbgWnd;	// handle to debugger window
    HWND hOpts;		// handle to options propsheet

    BOOL useHPTimer;	// high-precision timer is available
    BOOL useLPTimer;	// high-precision timer is available
    double fTickFreq;	// timer frequency as a double
    double fTickPer;	// timer period as a double
    LARGE_INTEGER startTime;	// time simulation began
    
    int hidemouse;	// hide mouse if it is in client area
    int show_stats;	// show CPU stats on statusbar
    int holdmsg;	// # time slices to not update titlebar

    // used for searching for files
    //   [0] is used for searching out ROMs (.rom)
    //   [1] is used for searching out programs (.ent)
    //   [2] is used for searching out scripts (.scr)
    //   [3] is used for searching out virtual disk (.svn,.svh)
    //   [4] is used for saving sampled audio (.wav)
#define FILE_ROM       (0)
#define FILE_PROGRAM   (1)
#define FILE_SCRIPT    (2)
#define FILE_VDISK     (3)
#define FILE_WAVOUT    (4)
#define FILE_NUMTYPES  (5)	// number of FILE_* types

    OPENFILENAME ofn[FILE_NUMTYPES];
    char basedir[MAX_PATH];	// directory we launched from

    HCURSOR hNullCursor;
    HCURSOR hArrowCursor;

    // ------ the following params are screen geometry information ------

    int client_w;	// current window width
    int client_h;	// current window height

    int toolbar_h;	// toolbar height

    int status_h;	// status bar height

    int xoff;		// sol screen origin, in pixels
    int yoff;		// sol screen origin, in pixels

} winstate_t;


// ============== exported by winmain.c ================

extern winstate_t winstate;

// open a file assuming one of a few default directories
BOOL FileOpenDlg(HWND hWnd, int filetype);

// return the size of the emulated Sol screen.
// this doesn't include any windows window dressing.
void GetSolScreenSize(HWND hwnd, int *pixels, int *rows);

// resize the main window such that the emulated screen is the specified size.
// don't move origin, though.
void SetSolScreenSize(HWND hwnd, int pixels, int rows);

// reset the emulated computer
void WinResetSim(int warm_reset);

// send a message to the status bar
void StatusBarMsg(int subwin, char *msg);

// filename functions
char *just_filename(char *fullname);

// ============== exported by winmain_screen.c ================

// called once at the beginning of time
void InitScreen(void);
int SolScreenWidth(void);
int SolScreenHeight(void);

void SetSolFontMode(fontmode_t fontmode);
fontmode_t GetSolFontMode(void);
void SetSolMainFont(chargen_t font);
chargen_t GetSolMainFont(void);
void SetSolAltFont(chargen_t font);
chargen_t GetSolAltFont(void);
char *SolMainFontName(void);
char *SolAltFontName(void);

disp_color_t GetSolDispColor(void);
void SetSolDispColor(disp_color_t color);

// event handlers
void OnPaintMain(HWND hwnd);
void OnTimerMain(HWND hwnd, UINT id);

// change the color of the emulated display tube
void SetDispColors(void);

// change the display character generator
void CreateCharMap(void);

// copy screen to a file.  on each successive call, a new
// filename is generate by simply incrementing a counter.
void CreateSnapshot(void);
int NextSnapshot(void);

// if the cursor toggle flag has been seen since the last
// time slice, redraw the appropriate parts of the screen.
// if the alternate font mode is active, there is no update
// because there are no inverse characters.
// also, the status message area gets blanked after a while.
void UpdateCursorBlinkState(void);

// cause redraw of entire Sol screen
void InvalidateScreen(void);

// called every time OUT 0FFH is performed
int WinOutFFTickle(void);

// called if OUT to scrollbase/scrollshade is made
void WinUpdateScrollBase(int scrollshade, int scrollbase);

void CreateCharMap(void);

// set screen fg/bg colors
void SetDispColors(void);


// ============== exported by winmain_kb.c ================

void OnKeyDownMain(HWND hwnd, UINT vk, BOOL down, int repeat, UINT flags);
void OnSetFocusMain(HWND hwnd, HWND oldwin);
void OnKillFocusMain(HWND hwnd, HWND newwin);

void WinInitKB(void);
int My_Handle_WM_CHAR(WPARAM wParam, LPARAM lParam);


// ============== exported by winopts.c ================

void InitOpts(void);
void CreateOptsDialog(void);
int  OptionsWindowVisible(void);
void ForceOption(int n);

void LoadOptions(void);
void SaveOptions(void);


// ============== exported by winmain_toolbar.c ================

#define ToolbarID 4321  // just a unique ID for later identification

// create the toolbar
void BuildToolBar(HWND hwnd);

// handle messages for the toolbar
void ToolbarMessage(int id);

// return a statically allocated string describing button for tooltip
char *ToolbarToolTipStr(int id);

// something on the toolbar had a right click
void ToolbarToolRClick(int id);

// something changed that the toolbar should know about
void WinUpdateToolBar(void);


// ============== exported by wintape.c ================

// get option setting for tape dialog
void GetSVTOpt(int unit, int opt, int *val);

// set option for tape dialog
void SetSVTOpt(int unit, int opt, int val);

#define SVTOpt_MOTENB 0		// motor enable button


// ============== exported by windisk.c ================

// create the container modal dialog
void CreateDiskPropsDialog(int inspectmode, int unit);


// ============== exported by windbg.c ================

void InitDebugger(void);
void DestroyDebugger(void);
BOOL CALLBACK DbgWinProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam);

int DebuggerWindowVisible(void);

// get the window size and placement.
// return 0 if it isn't valid, 1 if it is valid.
int DebuggerWindowGetSize(RECT *rc, int *visible);

// set the window size and placement.
void DebuggerWindowSetSize(RECT *rc);


// ============== exported by winaudio.c ================

// audio channel interface:
void WinAudioInit(void);
int  WinAudioOpen(void);
void WinAudioClose(void);

// wave capture interface:
int  SaveWaveOpen(char *name);
void SaveWaveClose(void);

// property codes
enum { WA_PROP_BITSPERSAMPLE=1,
       WA_PROP_SAMPLERATE=2,
       WA_PROP_CAPTURING=3 };

// return codes
enum { WA_STAT_OK=0,
       WA_STAT_BADPROP=1,
       WA_STAT_BADVALUE=2,
       WA_STAT_NOAUDIO=3 };

int  WinAudioSet(int prop, int value);
int  WinAudioGet(int prop, int *value);


#endif // ifdef _WINGUI_H_

