// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000

// This file implements the keyboard interface to Solace.
// As it stands, the implementation misses a lot of the odd
// behavior of the Sol keyboard, but hits most of the important
// things.  The current behavior is desribed here; the various bits
// of the real Sol keyboard behavior are documented at the end.

// the Sol doesn't have typematic (repeat) action unless the
// REPEAT key is held down simultaneously.  This has been mapped
// to the <F1> key.
//
// a few key values need to be remapped as well.  Here is a list of
// all the odd keys on a Sol:
//    KEY    Sol   PC
//    -----  ----  --------
//    MODE   0x80  113 <F2>
//    UP     0x97   38 EXT
//    DOWN   0x9A   40 EXT
//    LEFT   0x81   37 EXT
//    RIGHT  0x93   39 EXT
//    CLEAR  0x8B  114 <F3>
//    HOME   0x8E   36 EXT
//    DEL    0x7F   46 EXT  (no shift)
//    DEL    '_'    46 EXT  (with shift)
//    LOAD   0x8C  115 <F4>
//
// also, <CAPSLOCK><REPEAT> causes a machine reset.

#include <stdlib.h>
#include <stdio.h>
#include <windows.h>
#include <windowsx.h>	// message cracker macros

#include "solace_intf.h"
#include "wingui.h"


// ========== configuration ============

// when this option is 1:
//    Solace detects when focus enters the Solace window and then
//    forces the Numlock=ON interpretation of the number pad keys.
//    When focus leaves the window, the numlock state is returned
//    to whatever it was on entry, unless the user has actually
//    changed the numlock state.  perhaps too clever for its own good.
// when this option is 0:
//    don't fiddle with numlock state at all.
#define DIDDLE_NUMLOCK 0


// ========== declarations ============

// keyboard emulation state
struct {
    BOOL orig_numlock;	// whether numlock was true or false
    byte lastkeycode;	// last keyboard character
    int  lastvkeycode;	// virtual keycode that generated it
} keystate;


// called once at start of sim
void
WinInitKB(void)
{
    keystate.lastkeycode  =  0;	// last keyboard character
    keystate.lastvkeycode = -1;	// virtual keycode that generated it
}


// this sets the Numlock state to the specified value, and returns what
// the previous state of numlock was
static BOOL
SetNumLock(BOOL bNumLock)
{
    BYTE keyState[256];
    int numlock_state;

    GetKeyboardState(keyState);
    numlock_state = keyState[VK_NUMLOCK] & 1;

    if (bNumLock ^ numlock_state) {
	keyState[VK_NUMLOCK] = keyState[VK_NUMLOCK]^1;
#if DIDDLE_NUMLOCK
	SetKeyboardState(keyState);
#endif
    }

    return numlock_state;	// original state
}


void
OnKeyDownMain(HWND hwnd, UINT vk, BOOL down, int repeat, UINT flags)
{
    // we ignore typematic repeats unless <F1> is pressed
    if ( (flags & 0x4000) && 		// it was down last time
         (GetKeyState(VK_F1) >= 0))	// <F1> isn't depressed
	return;


    // handle emulated keyboard
    switch (vk) {

	case VK_F1:			// simulated REPEAT key
	    if ((GetKeyState(VK_CAPITAL) < 0) && (~flags & 0x4000)) {
		WinResetSim(TRUE);
		return;
	    }
	    // see if we pressed <F1> after a normal key, in which
	    // case repeat that old key.
	    if ((keystate.lastvkeycode >= 0) &&
		    GetKeyState(keystate.lastvkeycode) < 0) {
		Sys_Keypress(keystate.lastkeycode);
	    } else {
		keystate.lastvkeycode = -1;	// disable this check next time
	    }
	    break;

	case VK_CAPITAL:
	    if ((GetKeyState(VK_F1) < 0) &&  // simulated REPEAT key
		    (~flags & 0x4000)) {
		WinResetSim(TRUE);
		return;
	    }
	    break;

	case VK_F2:
	    keystate.lastkeycode  = (byte)0x80;
	    keystate.lastvkeycode = VK_F2;
	    Sys_Keypress((byte)0x80);	// simulated MODE key
	    return;

	case VK_F3:
	    keystate.lastkeycode  = (byte)0x8B;
	    keystate.lastvkeycode = VK_F3;
	    Sys_Keypress((byte)0x8B);	// simulated CLEAR key
	    return;

	case VK_F4:
	    keystate.lastkeycode  = (byte)0x8C;
	    keystate.lastvkeycode = VK_F4;
	    Sys_Keypress((byte)0x8C);	// simulated LOAD key
	    return;

	case VK_F5:
	    if (~flags & 0x4000) {		// this is a new keystroke
		// invoke/destroy debugger
		Sys_DbgNotify(RUN_KEY, 0);
	    }
	    return;

	case VK_UP:
	    keystate.lastkeycode  = (byte)0x97;
	    keystate.lastvkeycode = VK_UP;
	    Sys_Keypress((byte)0x97);
	    return;

	case VK_DOWN:
	    keystate.lastkeycode  = (byte)0x9A;
	    keystate.lastvkeycode = VK_DOWN;
	    Sys_Keypress((byte)0x9A);
	    return;

	case VK_LEFT:
	    keystate.lastkeycode  = (byte)0x81;
	    keystate.lastvkeycode = VK_LEFT;
	    Sys_Keypress((byte)0x81);
	    return;

	case VK_RIGHT:
	    keystate.lastkeycode  = (byte)0x93;
	    keystate.lastvkeycode = VK_RIGHT;
	    Sys_Keypress((byte)0x93);
	    return;

	case VK_HOME:
	    keystate.lastkeycode  = (byte)0x8E;
	    keystate.lastvkeycode = VK_HOME;
	    Sys_Keypress((byte)0x8E);
	    return;

	case VK_DELETE:
	    keystate.lastvkeycode = VK_DELETE;
	    if (GetKeyState(VK_SHIFT) < 0)
		keystate.lastkeycode = (byte)0x8E;
	    else
		keystate.lastkeycode = (byte)0x7F;
	    Sys_Keypress(keystate.lastkeycode);
	    return;

	// there is no WM_CHAR message for CTRL-@ (which is CTRL-SHIFT-2)
	// so we detect the keydown message for '2' and then check the
	// state of the SHIFT and CTRL keys.
	case '2':
	    if ((GetKeyState(VK_SHIFT)   < 0) &&
		(GetKeyState(VK_CONTROL) < 0)) {
		Sys_Keypress((byte)0x00);
	    }
	    return;

	default:
	    break;
    }

    // use default message handler
    FORWARD_WM_KEYDOWN(hwnd, vk, repeat, flags, DefWindowProc);
}


// we can't use a standard cracker because it doesn't give us
// the transition state we need to defeat auto repeat
int
My_Handle_WM_CHAR(WPARAM wParam, LPARAM lParam)
{
    if ((~lParam & 0x40000000) ||	// it is a new keystroke
	(GetKeyState(VK_F1) < 0)) {	// the repeat key is down
	keystate.lastkeycode  = (byte)wParam;
	keystate.lastvkeycode = VkKeyScanA((CHAR)wParam);
	Sys_Keypress((byte)wParam);
    }
    return 0;
}


// poll the keystate to see if we are still forcing reset
int
UI_KBResetState(void)
{
    return ((GetKeyState(VK_F1) < 0) && (GetKeyState(VK_CAPITAL) < 0))
	|| (Sys_GetDipswitch(1) & 1);
}


void
OnSetFocusMain(HWND hwnd, HWND oldwin)
{
    // if numlock is turned off, turn it on
    keystate.orig_numlock = SetNumLock(TRUE);
}


void
OnKillFocusMain(HWND hwnd, HWND newwin)
{
    BOOL bNumLock;

    // restore numstate to be what it was before we started
    bNumLock = SetNumLock(keystate.orig_numlock);

    // but if the user had turned NUMLOCK off while we had focus,
    // leave it off
    if (bNumLock != TRUE)
	bNumLock = SetNumLock(FALSE);
}

// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Details on the sometimes strange details of the real Sol keyboard.
/*

Keyboard emulation inadequacies, to name a few:

The Windows keyboard driver makes too many assumptions.  for instance,
if you hold <a><shift> then release <shift>, there are no more
WM_CHAR or WM_KEYDOWN messages.  The Sol keyboard doesn't behave
that way.

Also, the Windows keyboard driver seems to have limited key roll-over.
Not a big deal.

Sol keyboard behavior:

  1) There is infinite rollover.  You can press and hold down
     as many keys as you want and no decoding glitches occur.
     Each char just comes out as it is pressed.

  2) If multiple keystrokes occur before the software has had
     a chance to read the kb port, the last key pressed is the
     one that registers.

  3) There is no autorepeat; instead, the REPT key must be held
     down simultaneously with some other key (REPT is emulated by
     <F1> on the PC).  REPT or the main key can be held down first.

  4) The Sol has both <UPPERCASE> and <SHIFT LOCK> keys.  The
     <SHIFT LOCK> key is like that of a mechanical typewriter;
     it shifts everything, including number keys.  They <UPPERCASE>
     key is more like the modern PC <CAPS LOCK> key; only letters
     get shifted, but unlike the PC, pressing SHIFT doesn't
     reverse the case of the characters, instead they are always
     uppercase.

     In a break from emulation faithfulness, they <SHIFT LOCK>
     key won't be emulated, as it is highly useless as it shifts
     numbers too.  Mainly, there is no natural key to map it to,
     and mapping it to a function key would only make things more
     confusing, I think.

  5) Holding down <REPT> and <SHIFT LOCK> simultaneously causes
     the Sol to be reset.  It stays in a reset state until
     either one of them is released.

  6) Here are a number of odd things that happen, determined
     empirically:

     <REPT><A><S><D> --> A's, <!A>->S's, <!S>->nothing (no D).
     Two key's worth of rollover.

     <REPT><a><SHIFT> --> a's only.  shift doesn't affect train in progress.
     same with control

     <REPT><A><S><!S><!A> produces no S

     <A><S><D><REPT>, you'll keep getting <D> as long as any of those
     keys are down.  If you release one of them and then press it
     again, then when all the other held down ones are released,
     then re-pressed one will start repeating.  eg:
	    ASDF<REPT>-->D's, <!A><A>-->D's, <!F><!D><!S>->A's

     <REPT><a><S>->a's,<!a>->s's (not S's)
     <REPT><A><s>->A's,<!A>->s's (this time shift was obeyed!)
     similar behavior with ctrl

     <REPT><A><B><C> ->A's, <!A> -> B's, but occasionally CBBBB...

     <REPT><C><B><A> -> C's, <!C> -> A's!  aha, not the order of
     keyed in, but alphabetical order?  Scan order?


     After doing this, I went and read the keyboard theory of
     operation and perused the schematic a bit.  Some of the
     stuff above isn't really intended and is just a byproduct
     of the design.  Here is what I think is going on in that kb:

     The kb scans the keyboard one key at a time, in a matrix of 256
     virtual keys.  The kb maintains a couple bits of state per key,
     and is used to detect changes of state and to filter out glitches
     (false triggers).  The break, shift, shift lock, all caps,
     control, and local keys are all part of one column; their state
     is latched after each pass, and it is this latched state which
     is used to modify other key detections to product the appropriate
     code.  A 256x4 ROM modifies the four msb of the scan code by
     the last latched state of shift, control, and all caps.

     Because there is an edge detect done on the state and the state
     is kept per-key, there is infinite roll-over.

     When the REPT key is held down, it disables detecting any new
     keys, which has a couple of implications.  Actually, it prevents
     it so long as the original key is still held down.  However, the
     design doesn't really consider what happens when there are multiple
     keys held down.  Anyway, this is why toggling shift or control
     doesn't make any difference while repeating.  Also, it appears that
     as long as any of the originally down keys are still down, it
     prevents a new key from being recognized.  Once all of them are
     released, the circuit starts scanning the KB again.  It would
     appear then that if some more keys have been pressed while the
     repeat action was occuring, which one is recognized depends on
     the scan order of the keyboard.

  ---------------------------------------------------------------------

    Effect of shift/control on keypress (empirical):

    key	normal	shift	ctl	both

    0	30	20( )	00	00
    1	31	21(!)	01	01
    2	32	22(")	02	02
    3	33	23(#)	03	03
    4	34	24($)	04	04
    5	35	25(%)	05	05
    6	36	26(&)	06	06
    7	37	27(')	07	07
    8	38	28(()	08	08
    9	39	29())	09	09

    -	2D(-)	3D(=)	0D	0D
    [	5B([)	7B({)	1B	1B
    \	5C(\)	7C(|)	1C	1C
    ]	5D(])	7D(})	1D	1D
    ^	5E(^)	7E(~)	1E	1E

    a	61	41(A)	01	01
    ... etc ...
    q	71	51(Q)	11	11
    ... etc ...

    space	20	20	00	00
    :	3A	2A(*)	0A	0A
    ;	3B	2B(+)	0B	0B
    @	40	60(`)	00	00
    ,	2C	3C(<)	0C	0C
    .	2E	3E(>)	0E	0E
    /	2F	3F(?)	0F	0F
    del	7F	5F(_)	1F	1F

    tab	09	09	09	09
    LF	0A	0A	0A	0A
    return	0D	0D	0D	0D
    esc	1B	1B	1B	1B

    LOAD	8C	8C	8C	8C
    MODE	80	80	80	80
    up		97	97	97	97
    left	81	81	81	81
    right	93	93	93	93
    down	9A	9A	9A	9A
    home	8E	8E	8E	8E
    clear	8B	8B	8B	8B

    on keypad, numbers and symbols show up
    unaffected by shift & control.

    other nit-picking shortcomings:

     - a number of keys don't correctly take into account the
       SHIFT and CTRL keys.  For example, on the Sol, keys 0-9
       in combination with CTRL produce hex 00-09.  Not in Solace.

     - UPPER CASE/SHIFT LOCK are not accurately emulated.
       On the Sol, SHIFT LOCK is like the shift lock of a
       mechanical typewriter, causing even the number keys
       to be forced into a shift.  Pressing and releasing
       SHIFT removes SHIFT LOCK mode.  UPPER CASE shifts
       only a-z characters, and the SHIFT key has no effect
       on those characters.

       The modern PC keyboard has an intelligent CAPS LOCK
       that shifts just the letters, not the numbers/symbols.
       Its behavior seems more useful and is closer to the
       meaning of the UPPER CASE key on the sol.

       Finally, it would be too confusing to add yet another
       function key for this bit of arcania.

*/

