// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001

// The following stuff is overkill, but ascii drawings are so much fun...
// The tape player layout is like so:
//
//     -----------------------------------------
//     |\       \       \       \       \       \
//     | \  rec  \  stop \  play \  fwd  \  rew  \
//      \ \   O   \  []   \   >   \  >>   \  <<   \
//       \ \       \       \       \       \       \
//        \ +-------+-------+-------+-------+-------|
//         \|_______|_______|_______|_______|_______|
//
// Because I don't know what will look right, this code is
// parameterized to generate various looks.  In all cases,
// the 3D look is an orthographic projection.  The paramters are:
//      butt_up_height:   in pixels, how tall when not depressed
//      butt_down_height: in pixels, how tall when depressed
//      butt_width:       in pixels, left to right size
//      butt_depth:       in pixels, top to bottom
//      butt_skew:        in pixels, horizontal offset
//
//  or, for a given button, if there is no skew, then the button top
//  looks like this:
//
//                <-width-->
//
//          ^     +--------+
//          |     |        |
//          |     |        |
//        depth   |        |
//          |     |        |
//          |     |        |
//          V     +--------+
//
// skew slants it:
//
//                <-width-->
//
//          ^     +--------+
//          |      \        \
//          |       \        \
//        depth      \        \
//          |         \        \
//          |          \        \
//          V           +--------+
//                <skew>
//
// Note that width and depth don't change.
//
// For each button, there are three faces to be drawn.  They are
// the top, the side, and the front:
//
//                +--------+
//                |\        \
//                |s\        \
//                 \i\  top   \
//                  \d\        \
//                   \e\        \
//                    \ +--------+   \__ height
//                     \| front  |   /
//                      +--------+
//
// We draw the buttons from right to left to make sure the proper priority
// is maintained.  Also, in case the skew is low or zero, we draw the side
// first, then the front, then the top, so if one of the faces reduces to
// just slivers, the top will wipe them out.

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <windows.h>
#include <windowsx.h>	// message cracker macros
#include <direct.h>	// needed for _getcwd() etc

#include "solace_intf.h"
#include "wingui.h"
#include "wintape.h"	// interface to virtual tape drives
#include "resource.h"	// needed for meny definition

// this should probably be defined somewhere else...
#define IDC_MOTBTN 1999


// ==================================================================
//                      file global variables
// ==================================================================

static HINSTANCE g_hInst;	// instance of program
static HWND      g_mainwnd;	// main window handle

// file-global variables; shared by two possible tape units
static HBITMAP g_hDigits;	// handle to digit image map
static HDC     g_hDigitDC;	// handle to DC containing image map

// we attempt to use courier font, but failing that, we use ANSI_FIXED_FONT.
// thus, fixed_font is always defined, but cour_font may or may not be.
static HFONT g_hCourFont;	// courier new
static HFONT g_hFixedFont;	// chosen fixed font
static int   g_FixX;		// chosen font width (fixed pitch)
static int   g_FixY;		// chosen font height (fixed pitch)

static RECT  g_rcTime;		// location of timer digits
static RECT  g_rcMotorEn;	// location of motor enable switch

// this is used to tag a unit number to the window
static const char *g_TapePropName = "tapeunit";
static ATOM        g_TapeUnitAtom;

// window width and height: fixed
static const int g_width  = 320;
static const int g_height = 140;


// windows tape state struct
static struct
{
    HWND hWnd;			// handle to tape window

    HWND  hMotBtn;		// motor enable button
    int   show_motbtn;		// show motor button or not

    WINDOWPLACEMENT placement;	// size, position, open/close state of dialog
    BOOL  placement_valid;	// is window position set?

    int   showing;		// 1=visible, 0=hidden
    int   curtime;		// current time, in 10ths of seconds
    OPENFILENAME ofn;		// file dialog struct

} wtstate[2];


// ==================================================================
//               logging routines for debugging
// ==================================================================

#if 0
static char logfilename[256];

static void
OpenLog(char *filename)
{
    FILE *fp;
    strcpy(logfilename, filename);
    fp = fopen(filename, "w");
    fclose(fp);
}

static void
Log(char *str) {
    FILE *fp;
    fp = fopen(logfilename, "a");
    fprintf(fp, str);
    fclose(fp);
}
#endif


// ==================================================================
//                 routines local to this file
// ==================================================================

// figure polygons that define a key.
// used both to paint the buttons and to perform hit testing
static void
compute_key_geometry(int unit, int n, int depressed,
		     POINT *top, POINT *side, POINT *front,
		     POINT *textoff, char *str)
{
    POINT origin = { 10, 20 };	// upper left of key 0
    POINT keyorig;
    int style = 0;	// 0=pivot, 1=push down

    int butt_up_height   = 14;
    int butt_down_height =  4*style;
    int butt_width       = 40;
    int butt_depth       = (butt_width*3)/2;
    int butt_skew        = (butt_width*3)/4;

    int y_off_bot = (depressed) ? butt_down_height : butt_up_height;
    int y_off_top = (style == 1) ? y_off_bot : butt_up_height;

    keyorig.x = origin.x + n*butt_width;
    keyorig.y = origin.y + 0;

    // clockwise from top left
    top[0].x = keyorig.x;
    top[0].y = keyorig.y - y_off_top;
    top[1].x = top[0].x + butt_width;
    top[1].y = top[0].y;
    top[3].x = keyorig.x + butt_skew;
    top[3].y = keyorig.y + butt_depth - y_off_bot;
    top[2].x = top[3].x + butt_width;
    top[2].y = top[3].y;

    // clockwise, from top
    if (side != NULL) {
	side[0].x = top[0].x;
	side[0].y = top[0].y;
	side[1].x = top[3].x;
	side[1].y = top[3].y;
	side[2].x = side[1].x;
	side[2].y = side[1].y + y_off_bot;
	side[3].x = side[0].x;
	side[3].y = side[0].y + y_off_top;
    }

    if (front != NULL) {
	// clockwise, from top left
	front[0].x = top[3].x;
	front[0].y = top[3].y;
	front[1].x = top[2].x;
	front[1].y = top[2].y;
	front[2].x = front[1].x;
	front[2].y = front[1].y + y_off_bot;
	front[3].x = front[0].x;
	front[3].y = front[0].y + y_off_bot;
    }

    // center text on button top
    if (textoff != NULL && str != NULL) {
	textoff->x = top[0].x + 6 + (butt_width - strlen(str)*g_FixX)/2 + 2*depressed*style;
	textoff->y = top[0].y + 5;
    }
}


static void
draw_button(int unit, HDC hdc, int n, int depressed)
{
    POINT textoff;
    POINT top[4], side[4], front[4];
    COLORREF bg;
    HBRUSH hBrushFace, hBrushSide;
    HPEN hPen = (HPEN)GetStockObject(BLACK_PEN);
    int saved;
    char *str;

    switch (n) {
	 case 0: str = "REC";  break;
	 case 1: str = "STOP"; break;
	 case 2: str = "PLAY"; break;
	 case 3: str = "FWD";  break;
	default: str = "REW";  break;
    }
// FIXME: pause button?

    // figure out polygons
    compute_key_geometry(unit, n, depressed, top, side, front, &textoff, str);

    // draw the faces

    saved = SaveDC(hdc);
    SelectObject(hdc, hPen);

    bg = (n == 0) ? RGB(0x80, 0x00, 0x00) : RGB(0x50, 0x50, 0x50);
    hBrushSide = CreateSolidBrush(bg);
    SelectObject(hdc, hBrushSide);
    (void)Polygon(hdc, side,  4);

    bg = (n == 0) ? RGB(0xB0, 0x00, 0x00) : RGB(0x80, 0x80, 0x80);
    hBrushFace = CreateSolidBrush(bg);
    SelectObject(hdc, hBrushFace);
    (void)Polygon(hdc, front, 4);
    (void)Polygon(hdc, top,   4);

    SelectFont(hdc, g_hFixedFont);
    SetBkColor(hdc, bg);
    TextOut(hdc, textoff.x, textoff.y, str, strlen(str));

    RestoreDC(hdc, saved);

    DeleteObject(hBrushSide);
    DeleteObject(hBrushFace);
}


// the unnormalized cross product is used to determine side-of-halfplane
#define CROSS(a,b,c) ((b.x - a.x)*(c.y-a.y) - (b.y - a.y)*(c.x-a.x))

BOOL
hittest_button(int unit, POINT pt, int n, int depressed)
{
    POINT top[5];
    int i;

    // key geometry of keycap
    compute_key_geometry(unit, n, depressed, top, NULL, NULL, NULL, NULL);

    // hit test -- because we have a trapezoid, we must do a cross-product
    // of the point with each of the four half planes defined by the edges.
    // the polygon winds clockwise.
    //           -          |          +           ^
    //        ------>     + | -    <--------     - | +
    //           +          V          -           |
    top[4] = top[0];	// close the loop
    for(i=0; i<4; i++) {
	int sign = CROSS(top[i], top[i+1], pt);
	if (sign < 0)
	    return FALSE;
    }

    return TRUE;
}


// return name of tape in drive
char *
GetTapeFilename(int unit)
{
    int n;

    ASSERT(0 <= unit && unit <= 1);

    Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &n);
    if (n == EMPTY)
	return "";
    else if (strlen(wtstate[unit].ofn.lpstrFile) == 0)
	return "<unnamed>";
    else
	return wtstate[unit].ofn.lpstrFile;
}


static void
UpdateTitleBar(HWND hwnd)
{
    int unit = (int)GetProp(hwnd, (LPCTSTR)g_TapeUnitAtom);
    HMENU hMenu = GetMenu(hwnd);
    char buff[50];
    int n, playstate;

    sprintf(buff, "Unit %d: ", unit ? 1 : 2);
    Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &n);
    if (n == EMPTY)
	strcat(buff, "-- empty --");
    else if (strlen(wtstate[unit].ofn.lpstrFile) == 0)
	strcat(buff, "-- unnamed --");
    else {
#if 0
	// extract name of file from full path
	// we don't use lpstrFileTitle because it isn't
	// always coherent (eg due to LoadOptions)
	char *p, *lp;
	for(p=lp=wtstate[unit].ofn.lpstrFile; *p; *p++) {
	    if (*p == '\\')
		lp = p+1;
	}
#else
	char *lp = just_filename(wtstate[unit].ofn.lpstrFile);
#endif
	strcat(buff, lp);
    }

    SetWindowText(hwnd, buff);

    // also update the menu item for viewing the label
    Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &playstate);
    EnableMenuItem(hMenu, 1, MF_BYPOSITION |
		    ((playstate != EMPTY) ? MFS_ENABLED : MFS_DISABLED) );
    DrawMenuBar(hwnd);
}


static BOOL
OnCreateTape(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    int unit = (int)(lpCreateStruct->lpCreateParams);
    DWORD style;

    // if it was opened & closed previously, use the size/location
    // from when it was last closed.
    if (wtstate[unit].placement_valid) {
	SetWindowPlacement(hwnd, &wtstate[unit].placement);
	wtstate[unit].showing = (wtstate[unit].placement.showCmd == SW_SHOW);
    } else {
	// guess a "good" initial size; leave position, but change size
	SetWindowPos(hwnd, HWND_TOP, 0, 0, g_width, g_height, SWP_NOMOVE);
	ShowWindow(hwnd, SW_SHOW);
	wtstate[unit].showing = 1;
    }

    style = WS_CHILD | BS_AUTOCHECKBOX | BS_LEFT;
    if (wtstate[unit].show_motbtn)
	style |= WS_VISIBLE;
    wtstate[unit].hMotBtn = CreateWindow(
			    "button",
			    "motor",
			    style,
			    g_rcMotorEn.left,
			    g_rcMotorEn.top,
			    g_rcMotorEn.right - g_rcMotorEn.left,
			    g_rcMotorEn.bottom - g_rcMotorEn.top,
			    hwnd,
			    (HMENU)IDC_MOTBTN,
			    g_hInst,
			    NULL
			    );
//    ShowWindow(wtstate[unit].hMotBtn,
//	      (wtstate[unit].show_motbtn) ? SW_SHOW : SW_HIDE);

    return TRUE;
}


static int
map_playstate_to_buttons(int unit)
{
    int buttons, playstate;
    Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &playstate);
    switch (playstate) {
	case EMPTY:  buttons = 0x00; break;
	case STOP:   buttons = 0x02; break;
	case RECORD: buttons = 0x05; break;
	case PLAY:   buttons = 0x04; break;
	case FWD:    buttons = 0x08; break;
	case REW:    buttons = 0x10; break;
    }
    return buttons;
}


// change to timer state, in 10th's of seconds
static void
UpdateTime(int unit, int time, int force)
{
    if (force || (wtstate[unit].curtime != time)) {
	HDC hdc = GetDC(wtstate[unit].hWnd);
#if 0
	HFONT hOldFont = SelectFont(hdc, g_hFixedFont);
	char buf[10];
	sprintf(buf, "%04d.%1d", time/10, time%10);
	TextOut(hdc, g_rcTime.left, g_rcTime.top, buf, 6);
	SelectFont(hdc, hOldFont);
#else
	int i;
	int phase = time % 10;
	int secs = time / 10;

	RECT rc = g_rcTime;
	// draw frame around timer digits
	InflateRect(&rc, 4, 4);
	DrawEdge(hdc, &rc, EDGE_SUNKEN, BF_RECT);

	for(i=0; i<4; i++) {
	    static const int powers[] = { 1000, 100, 10, 1 };
	    int pow = powers[i];
	    int digit = (secs / pow) % 10;
	    int thisphase = ((i == 4) || ((secs+1)%pow == 0)) ? phase : 0;
	    if (thisphase == 0) {
		// digit is centered perfectly
		BitBlt( hdc,			// dest DC
			g_rcTime.left+9*i,	// dest left
			g_rcTime.top,		// dest upper
			9, 20,			// width, height
			g_hDigitDC,		// src DC
			9*digit, 0,		// src upper left (x,y)
			SRCCOPY );		// raster op
	    } else {
		// digit is rolling -- need parts of two digits
		BitBlt( hdc,			// dest DC
			g_rcTime.left+9*i,	// dest left
			g_rcTime.top,		// dest upper
			9, 20-2*thisphase,	// width, height
			g_hDigitDC,		// src DC
			9*digit, 2*thisphase,	// src upper left (x,y)
			SRCCOPY );		// raster op
		BitBlt( hdc,			// dest DC
			g_rcTime.left+9*i,	// dest left
			g_rcTime.bottom-2*thisphase,	// dest upper
			9, 2*thisphase,		// width, height
			g_hDigitDC,		// src DC
			9*((digit+1)%10), 0,	// src upper left (x,y)
			SRCCOPY );		// raster op
	    }
	}
#endif
	wtstate[unit].curtime = time;
    }
}


static void
OnPaintTape(HWND hwnd)
{
    PAINTSTRUCT	ps;
    HDC hdc;
    int saved, i;
    int unit = (int)GetProp(hwnd, (LPCTSTR)g_TapeUnitAtom);
    int buttons = map_playstate_to_buttons(unit);

    hdc = BeginPaint(hwnd, &ps);
    saved = SaveDC(hdc);

    // draw all buttons
    for(i=4; i>=0; i--)
	draw_button(unit, hdc, i, (buttons>>i)&1 );

    // draw counter digits
    UpdateTime(unit, wtstate[unit].curtime, 1);

    RestoreDC(hdc, saved);
    EndPaint(hwnd, &ps);
}


// just like UI_Alert, but the parent dialog is supplied
void
WinTapeAlert(HWND hwnd, char *msg)
{
    (void)MessageBox(hwnd, msg, "Warning", MB_OK | MB_ICONEXCLAMATION);
}


// if we control click on a line, we cycle between three states:
//     no breakpoint, enabled breakpoint, disabled breakpoint
static void
OnLButtonUpTape(HWND hwnd, int xClient, int yClient, UINT keyflags)
{
    POINT pt = { xClient, yClient };
    int unit = (int)GetProp(hwnd, (LPCTSTR)g_TapeUnitAtom);
    int buttons = map_playstate_to_buttons(unit);
    int i, playstate, curstate, legal, stat;

    // see if user clicked on timer digits
    if (PtInRect(&g_rcTime, pt)) {
	Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &curstate);
	if (curstate == EMPTY)
	    return;
	FORWARD_WM_COMMAND(hwnd, IDM_SVT_TIME, 0, 1, SendMessage);
	return;
    }

    for(i=0; i<5; i++) {
	BOOL b = hittest_button(unit, pt, i, (buttons>>i)&1);
	if (b) {

	    switch (i) {
		case 0: playstate = RECORD; break;
		case 1: playstate = STOP;   break;
		case 2: playstate = PLAY;   break;
		case 3: playstate = FWD;    break;
		case 4: playstate = REW;    break;
		default: ASSERT(0);
	    }

	    // check for "legality" of button press
	    Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &curstate);
	    legal = 1;
	    if (curstate == EMPTY && playstate == STOP) {
		// map to File/Load Tape...
		FORWARD_WM_COMMAND(hwnd, IDM_SVT_LOAD, 0, 1, SendMessage);
		legal = 0;
	    } else if (curstate == EMPTY) {
		WinTapeAlert(hwnd, "Error: tape player is empty");
		legal = 0;
	    } else if (curstate == RECORD && playstate != STOP) {
		WinTapeAlert(hwnd, "Please stop recording first");
		legal = 0;
	    } else if (playstate == RECORD && curstate != STOP) {
		WinTapeAlert(hwnd, "Please stop tape before starting to record");
		legal = 0;
	    } else if (playstate == RECORD) {
		Sys_GetTapeProp(unit, TPROP_WRITEPROTECT, &stat);
		if (stat) {
		    WinTapeAlert(hwnd, "This tape is write protected");
		    legal = 0;
		}
	    }

	    if (legal) {
		Sys_SetTapeProp(unit, TPROP_PLAYSTATE, playstate);
		InvalidateRect(hwnd, NULL, TRUE);
	    }

	    // we must break and not continue the loop because if button N
	    // is up and button N+1 is down, there is an overlap of the
	    // projection of those two buttontops.  we should let button
	    // N have priority over button N+1 in that case.
	    break;

	} // if (b)
    } // for(i)
}


// the user has requested that the tape window be closed
static void
OnCloseTape(HWND hwnd)
{
    int unit = (int)GetProp(hwnd, (LPCTSTR)g_TapeUnitAtom);

    // just hide the window, don't destroy it
    ShowWindow(hwnd, SW_HIDE);
    wtstate[unit].showing = 0;

    WinUpdateToolBar();
}


// destroy the tape window
// this may be called twice, once for each possible window.
// there are some global resources that get freed up here as well.
static void
OnDestroyTape(HWND hwnd)
{
    int unit = (int)GetProp(hwnd, (LPCTSTR)g_TapeUnitAtom);
    RemoveProp(hwnd, (LPCTSTR)g_TapeUnitAtom);

    if (wtstate[unit].ofn.lpstrFile) {
	free(wtstate[unit].ofn.lpstrFile);
	wtstate[unit].ofn.lpstrFile = NULL;
    }
    if (wtstate[unit].ofn.lpstrFileTitle) {
	free(wtstate[unit].ofn.lpstrFileTitle);
	wtstate[unit].ofn.lpstrFileTitle = NULL;
    }
    if (wtstate[unit].ofn.lpstrInitialDir) {
	free((void*)wtstate[unit].ofn.lpstrInitialDir);
	wtstate[unit].ofn.lpstrInitialDir = NULL;
    }
}


// we maintain state for search out ROMs and for searching out programs.
// it is possible that they exist in two different places.
static void
TapeFileDlgInit(int unit)
{
    char *fullpath  = (char*)malloc(MAX_PATH);
    char *initdir   = (char*)malloc(MAX_PATH);
    char *filetitle = (char*)malloc(_MAX_FNAME + _MAX_EXT + 1);
    ASSERT(fullpath  != NULL);
    ASSERT(initdir   != NULL);
    ASSERT(filetitle != NULL);

    memset(&wtstate[unit].ofn, 0x00, sizeof(OPENFILENAME));

    wtstate[unit].ofn.lStructSize     = sizeof(OPENFILENAME);
    wtstate[unit].ofn.lpstrFilter     = "SVT Files (*.svt)\0*.SVT\0" \
					"All Files (*.*)\0*.*\0\0";
    wtstate[unit].ofn.nFilterIndex    = 1;
    wtstate[unit].ofn.lpstrFile       = fullpath;
    wtstate[unit].ofn.nMaxFile        = MAX_PATH;
    wtstate[unit].ofn.lpstrFileTitle  = filetitle;
    wtstate[unit].ofn.nMaxFileTitle   = _MAX_FNAME + _MAX_EXT;
    wtstate[unit].ofn.lpstrInitialDir = initdir;
    wtstate[unit].ofn.lpstrDefExt     = "svt";

    // set the default search directory to where we launched from
    fullpath[0]  = '\0';
    filetitle[0] = '\0';
    strcpy(initdir, winstate.basedir);
    strcat(initdir,"\\binaries");
}


// type: 0=load, 1=save as
static BOOL
TapeFileDlg(int unit, int type)
{
    HWND hwnd = wtstate[unit].hWnd;
    char *prevfile = strdup(wtstate[unit].ofn.lpstrFile);
    char *p;
    BOOL flag;

    wtstate[unit].ofn.hwndOwner = hwnd;
    wtstate[unit].ofn.lpstrFile[0] = '\0';

    wtstate[unit].ofn.Flags = OFN_HIDEREADONLY;
    if (type == 0)	// load
	wtstate[unit].ofn.Flags |= OFN_FILEMUSTEXIST;

    if (type == 0)
	flag = GetOpenFileName(&wtstate[unit].ofn);
    else
	flag = GetSaveFileName(&wtstate[unit].ofn);

    // if we succeed in opening a file, we want to remember what
    // directory to start the search in next time.
    if (flag) {
	int len = strlen(wtstate[unit].ofn.lpstrFile);
	// copy the current filename into the initial dir...
	strcpy((char *)wtstate[unit].ofn.lpstrInitialDir, (const char*)wtstate[unit].ofn.lpstrFile);
	// ...then erase the text after the last backslash.
	p = (char*)wtstate[unit].ofn.lpstrInitialDir + len - 1;
	while (*p != '\\') {
	    ASSERT(p > wtstate[unit].ofn.lpstrInitialDir); // at least one backslash
	    p--;
	}
	*p = '\0';	// wipe out tail
    } else {
	// restore it to what it was in case user hit CANCEL button
	if (prevfile != NULL)
	    strcpy(wtstate[unit].ofn.lpstrFile, prevfile);
    }

    free(prevfile);
    return flag;
}


// if there is a tape in the player and it has been modified,
// ask the user if he wants to save it, abandon it, or cancel.
// if he wants to save it, we do so.  unless the user wants to
// cancel, we destroy the existing tape image.
// we return 0 normally, 1 if cancel.
static int
CheckForDirtyTape(int unit, int canceloption)
{
    int flags = (canceloption) ? MB_YESNOCANCEL : MB_YESNO;
    int reply, n;
    char buf[256];

    Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &n);
    if (n == EMPTY)
	return 0;

    Sys_GetTapeProp(unit, TPROP_DIRTY, &n);
    if (!n)
	return 0;

    // there is a modified tape in there
    sprintf(buf, "The virtual tape #%d has been modified.\n" \
		 "Should it be written to disk?", (unit==0) ? 2 : unit);
    reply = MessageBox(wtstate[unit].hWnd, buf, "Confirm", flags);

    if (reply == IDCANCEL)
	return 1;
    if (reply == IDNO)
	return 0;

    if (strlen(wtstate[unit].ofn.lpstrFile) == 0) {
	// do SaveAs function
	HWND hwnd = wtstate[unit].hWnd;
	// use SendMessage to trigger SaveAs behavior
	FORWARD_WM_COMMAND(hwnd, IDM_SVT_SAVEAS, 0, 1, SendMessage);
	if (strlen(wtstate[unit].ofn.lpstrFile) == 0)
	    return 1;	// SaveAs was cancelled
    }
    n = Sys_TapeFile(unit, VTF_SaveTape, wtstate[unit].ofn.lpstrFile);
    return n;
}


// update checks on menu items
// if we don't have timer control, grey out the speed choices
static void
update_menu_tape(HWND hwnd)
{
    HMENU hMenu = GetMenu(hwnd);
    int unit    = (int)GetProp(hwnd, (LPCTSTR)g_TapeUnitAtom);
    int noname  = (strlen(wtstate[unit].ofn.lpstrFile) == 0);
    int playstate, wp;

    Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &playstate);
    Sys_GetTapeProp(unit, TPROP_WRITEPROTECT, &wp);

    EnableMenuItem(hMenu, IDM_SVT_NEW,    (playstate == EMPTY || playstate == STOP) ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_LOAD,   (playstate == EMPTY || playstate == STOP) ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_SAVEAS, (playstate != EMPTY)                      ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_SAVE,   (playstate != EMPTY && !noname)           ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_TIME,   (playstate != EMPTY)                      ? MFS_ENABLED : MFS_DISABLED);

    EnableMenuItem(hMenu, IDM_SVT_PLAY,   (playstate == STOP)                       ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_STOP,   (playstate != EMPTY && playstate != STOP) ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_FWD,    (playstate != EMPTY && playstate != FWD && playstate != RECORD)  ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_REW,    (playstate != EMPTY && playstate != REW && playstate != RECORD)  ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_EJECT,  (playstate == STOP)                       ? MFS_ENABLED : MFS_DISABLED);
    EnableMenuItem(hMenu, IDM_SVT_RECORD, (playstate != EMPTY && playstate == STOP && !wp) ? MFS_ENABLED : MFS_DISABLED);
}


// update the checkmarks, etc, of the popup menus
static void
OnInitMenuPopupTape(HWND hwnd, HMENU hmenu, UINT item, BOOL sysmenu)
{
    if (!sysmenu)
	update_menu_tape(hwnd);
}


LRESULT CALLBACK
SVTTimeDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    HWND hEdit = GetDlgItem(hDlg, IDC_SVT_TIME);
    char buf1[40], buf2[40];
    float ftime;
    int unit;

    unit = (iMsg == WM_INITDIALOG) ? lParam
				   : (int)GetProp(hDlg, (LPCTSTR)g_TapeUnitAtom);
    sprintf(buf2, "%d.%1d", wtstate[unit].curtime/10, wtstate[unit].curtime%10);

    switch (iMsg) {

	case WM_INITDIALOG:
	    SetProp(hDlg, (LPCTSTR)g_TapeUnitAtom, (HANDLE)unit);
	    sprintf(buf1, "Set time for unit %d", lParam);
	    SetWindowText(hDlg, buf1);
	    // give edit box current time value
	    SetDlgItemText(hDlg, IDC_SVT_TIME, buf2);
	    // select contents of edit box
	    SendMessage(hEdit, EM_SETSEL, 0, -1);
	    SetFocus(hEdit);
	    return FALSE;	// FALSE because we want to set focus ourselves

	case WM_COMMAND:
	    switch (LOWORD(wParam)) {
		case IDOK:
		    GetDlgItemText(hDlg, IDC_SVT_TIME, buf1, sizeof(buf1));
		    if (sscanf(buf1, "%f", &ftime) == 1) {
			int newtime = (int)(ftime * 10.0);
			if (newtime >= 0 && newtime < 30*60*10) {
			    EndDialog(hDlg, newtime);	// >=0 represents new time in 10ths
			    return TRUE;
			}
		    }
		    WinTapeAlert(hDlg, "You entered an illegal time value.\nTry again.");
		    sprintf(buf1, "%d.%1d", wtstate[1].curtime/10, wtstate[1].curtime%10);
		    SetDlgItemText(hDlg, IDC_SVT_TIME, buf2);
		    SetFocus(hDlg);
		    return TRUE;
		case IDCANCEL:
		    EndDialog(hDlg, -1);  // -1 indicates cancel
		    return TRUE;
		default:
		    break;
	    }
	    break;

	case WM_DESTROY:
	    RemoveProp(hDlg, (LPCTSTR)g_TapeUnitAtom);
	    return TRUE;

	default:
	    break;

    } // switch (iMsg)

    return FALSE;	// message not handled
}


LRESULT CALLBACK
SVTTapeLabelDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    HWND hEdit = GetDlgItem(hDlg, IDC_SVT_TAPELABEL);
    HWND hWP   = GetDlgItem(hDlg, IDC_CHECK_WRITEPROTECT);
    char buf1[40], buf2[8192], *orig;
    int unit, n, playstate;

    unit = (iMsg == WM_INITDIALOG) ? lParam
				   : (int)GetProp(hDlg, (LPCTSTR)g_TapeUnitAtom);

    switch (iMsg) {

	case WM_INITDIALOG:
	    SetProp(hDlg, (LPCTSTR)g_TapeUnitAtom, (HANDLE)unit);
	    sprintf(buf1, "Label for tape on unit %d", lParam);
	    SetWindowText(hDlg, buf1);
	    // give edit box current time value
	    Sys_GetTapeProp(unit, TPROP_LABEL, (int*)(&orig));
	    SetDlgItemText(hDlg, IDC_SVT_TAPELABEL, orig);
	    // by default, don't allow editing unless explicitly asked
	    Edit_SetReadOnly(GetDlgItem(hDlg, IDC_SVT_TAPELABEL), TRUE);
	    // set default checkmark on write protect tab
	    Sys_GetTapeProp(unit, TPROP_WRITEPROTECT, &n);
	    Button_SetCheck(hWP, (n) ? BST_CHECKED : BST_UNCHECKED);
	    SetWindowFont(hEdit, g_hFixedFont, FALSE);
	    return FALSE;	// FALSE because we want to set focus ourselves

	case WM_COMMAND:
	    switch (LOWORD(wParam)) {

		case IDOK:
		    Sys_GetTapeProp(unit, TPROP_LABEL, (int*)(&orig));
		    GetDlgItemText(hDlg, IDC_SVT_TAPELABEL, buf2, sizeof(buf2));
		    if (  (strlen(buf2) != strlen(orig)) ||
		          (strncmp(buf2, orig, sizeof(buf2))) ) {
			Sys_SetTapeProp(unit, TPROP_LABEL, (int)buf2);
		    }
		    EndDialog(hDlg, 0);	// 0 means OK to caller
		    return TRUE;

		case IDCANCEL:
		    EndDialog(hDlg, -1);  // -1 indicates cancel
		    return TRUE;

		case IDC_CHECK_EDITTAPELABEL:
		    if (HIWORD(wParam) == BN_CLICKED) {
			int bst = Button_GetState(GetDlgItem(hDlg, IDC_CHECK_EDITTAPELABEL)) & 0x0003;
			Edit_SetReadOnly(GetDlgItem(hDlg, IDC_SVT_TAPELABEL), bst != BST_CHECKED);
			return TRUE;
		    }
		    break;

		case IDC_CHECK_WRITEPROTECT:
		    if (HIWORD(wParam) == BN_CLICKED) {
			int bst = Button_GetState(GetDlgItem(hDlg, IDC_CHECK_WRITEPROTECT)) & 0x0003;
			int checked = (bst == BST_CHECKED);
			Sys_SetTapeProp(unit, TPROP_WRITEPROTECT, checked);
			Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &playstate);
			// if currently recording, stop it
			if (playstate == RECORD && checked)
			    Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
			return TRUE;
		    }
		    break;

		default:
		    break;
	    }
	    break;

	case WM_DESTROY:
	    RemoveProp(hDlg, (LPCTSTR)g_TapeUnitAtom);
	    return TRUE;

	default:
	    break;

    } // switch (iMsg)

    return FALSE;	// message not handled
}


// get option setting for tape dialog
void
GetSVTOpt(int unit, int opt, int *val)
{
    switch (opt) {
	case SVTOpt_MOTENB:
	    *val = wtstate[unit].show_motbtn;
	    break;
	default:
	    ASSERT(0);
	    break;
    }
}


// set option for tape dialog
void
SetSVTOpt(int unit, int opt, int val)
{
    switch (opt) {
	case SVTOpt_MOTENB:
	    wtstate[unit].show_motbtn = val;
	    ShowWindow(wtstate[unit].hMotBtn, (val) ? SW_SHOW : SW_HIDE);
	    break;
	default:
	    ASSERT(0);
    }
}


static void
OnCommandTape(HWND hwnd, int id, HWND hctl, UINT codenotify)
{
    int unit = (int)GetProp(hwnd, (LPCTSTR)g_TapeUnitAtom);
    int stat;
    char buf[300];

    switch (id) {

    // --- File submenu ---

    case IDM_SVT_NEW:
	stat = CheckForDirtyTape(unit, TRUE);
	if (!stat) {
	    strcpy(wtstate[unit].ofn.lpstrFile, "");
	    stat = Sys_TapeFile(unit, VTF_NewTape, NULL);
	    if (stat != 0)
		WinTapeAlert(hwnd, "Couldn't create a blank tape");
	}
	return;

    case IDM_SVT_LOAD:
	stat = CheckForDirtyTape(unit, TRUE);
	if (stat)
	    return;
	stat = TapeFileDlg(unit, 0);
	if (stat) {
	    stat = Sys_TapeFile(unit, VTF_LoadTape, wtstate[unit].ofn.lpstrFile);
	    if (stat != 0) {
		sprintf(buf, "Error loading tape '%s'", wtstate[unit].ofn.lpstrFile);
		WinTapeAlert(hwnd, buf);
	    }
	}
	return;

    case IDM_SVT_SAVEAS:
	stat = TapeFileDlg(unit, 1);
	if (!stat)
	    return;
	// fall through to IDM_SVT_SAVE
    case IDM_SVT_SAVE:
	stat = Sys_TapeFile(unit, VTF_SaveTape, wtstate[unit].ofn.lpstrFile);
	if (stat != 0) {
	    sprintf(buf, "Error loading tape '%s'", wtstate[unit].ofn.lpstrFile);
	    WinTapeAlert(hwnd, buf);
	}
	return;

    case IDM_SVT_TIME:
	stat = DialogBoxParam(g_hInst, MAKEINTRESOURCE(IDD_SVTTIME),
			      hwnd, (DLGPROC)SVTTimeDlgProc, unit);
	if (stat >= 0) {
	    Sys_SetTapeProp(unit, TPROP_POSITION, stat);
	    Sys_GetTapeProp(unit, TPROP_POSITION, &stat);
	    UpdateTime(unit, stat, 1);
	}
	break;

    case IDM_SVT_PLAY:
	Sys_SetTapeProp(unit, TPROP_PLAYSTATE, PLAY);
	break;

    case IDM_SVT_STOP:
	Sys_SetTapeProp(unit, TPROP_PLAYSTATE, STOP);
	break;

    case IDM_SVT_FWD:
	Sys_SetTapeProp(unit, TPROP_PLAYSTATE, FWD);
	break;

    case IDM_SVT_REW:
	Sys_SetTapeProp(unit, TPROP_PLAYSTATE, REW);
	break;

    case IDM_SVT_EJECT:
	stat = CheckForDirtyTape(unit, TRUE);
	if (stat)
	    return;
	stat = Sys_TapeFile(unit, VTF_EjectTape, NULL);
	if (stat != 0)
	    WinTapeAlert(hwnd, "Error ejecting tape");
	break;

    case IDM_SVT_RECORD:
	Sys_SetTapeProp(unit, TPROP_PLAYSTATE, RECORD);
	break;

    // --- Label submenu ---

    case IDM_SVT_LABEL:
	stat = DialogBoxParam(g_hInst, MAKEINTRESOURCE(IDD_SVTTAPELABEL),
			      hwnd, (DLGPROC)SVTTapeLabelDlgProc, unit);
	break;

    // --- Motor enable button ---

    case IDC_MOTBTN:
	if ((codenotify == BN_CLICKED) &&
	    (hctl == wtstate[unit].hMotBtn)) {
	    stat = Button_GetState(hctl) & 0x0003;
	    Sys_SetTapeProp(unit, TPROP_FORCEMOTOR, (stat == 1));
	}
	break;

    default:
	break;
    }

    // we aren't handling it, pass it through to default handler
    FORWARD_WM_COMMAND(hwnd, id, hctl, codenotify, DefWindowProc);
}


// tape recorder window handler
static LRESULT CALLBACK
WndProcTape(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {
	HANDLE_MSG(hwnd, WM_CREATE,        OnCreateTape);
	HANDLE_MSG(hwnd, WM_PAINT,         OnPaintTape);
	HANDLE_MSG(hwnd, WM_CLOSE,         OnCloseTape);
	HANDLE_MSG(hwnd, WM_LBUTTONUP,     OnLButtonUpTape);
	HANDLE_MSG(hwnd, WM_COMMAND,       OnCommandTape);
	HANDLE_MSG(hwnd, WM_DESTROY,       OnDestroyTape);
	HANDLE_MSG(hwnd, WM_INITMENUPOPUP, OnInitMenuPopupTape);

	// if someone types something while focus is on the
	// virtual tape dialog, change focus to main window
	// and redirect this keystroke there too.
	case WM_CHAR:
	    SetFocus(winstate.hWnd);
	    SendMessage(winstate.hWnd, WM_CHAR, wParam, lParam);
	    break;
    }

    return DefWindowProc(hwnd, iMsg, wParam, lParam);
}


static void
RegisterTapeClass(void)
{
    WNDCLASSEX wndclass;

    wndclass.cbSize        = sizeof(wndclass);
    wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
    wndclass.lpfnWndProc   = WndProcTape;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = 0;
    wndclass.hInstance     = g_hInst;
    wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground = (HBRUSH)(COLOR_3DFACE + 1);
    wndclass.lpszMenuName  = MAKEINTRESOURCE(IDR_MENU_TAPE);
    wndclass.lpszClassName = "SolaceTapeWindow";
    wndclass.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
    wndclass.hIconSm       = LoadIcon(g_hInst, MAKEINTRESOURCE(IDI_ICON2));

    RegisterClassEx(&wndclass);
}


// ==================================================================
//                  exported routines follow
// ==================================================================

// this is called once at program start-up time
void
WinTapeInit(HINSTANCE hInst, HWND mainwnd)
{
    int unit;
    TEXTMETRIC fix_tm;		// info about the fixed font

    // create handle to a fixed-pitch font
    HDC hdc = GetDC(mainwnd);
    int saved = SaveDC(hdc);
    int PointSize = 8;
    int nHeight = -MulDiv(PointSize, GetDeviceCaps(hdc, LOGPIXELSY), 72);

    g_hCourFont = CreateFont(
		    nHeight,		// height
		    0,			// width (default)
		    0,0,		// escapement, orientation
		    FW_REGULAR,		// weight
		    0,0,0,		// italic,underline,strikeout
		    ANSI_CHARSET,	// maybe DEFAULT_CHARSET
		    OUT_RASTER_PRECIS,
		    CLIP_DEFAULT_PRECIS,
		    DEFAULT_QUALITY,
		    FIXED_PITCH | FF_MODERN,
		    "Courier New"
		    );

    g_hFixedFont = (g_hCourFont == NULL) ? GetStockFont(ANSI_FIXED_FONT)
					 : g_hCourFont;

    // get some basic information about the font we are using
    if (g_hFixedFont != NULL)
	SelectFont(hdc, g_hFixedFont);
    GetTextMetrics(hdc, &fix_tm);
    g_FixX = fix_tm.tmAveCharWidth;
    g_FixY = fix_tm.tmHeight + fix_tm.tmExternalLeading;


    // tuck away copies for later
    g_hInst   = hInst;
    g_mainwnd = mainwnd;

    // create a DC that is used for sourcing pixels when drawing digits
    g_hDigits = LoadBitmap(g_hInst, MAKEINTRESOURCE(IDB_DIGITS));
    g_hDigitDC = CreateCompatibleDC(hdc);
    SelectBitmap(g_hDigitDC, g_hDigits);

    RestoreDC(hdc, saved);
    ReleaseDC(mainwnd, hdc);


    RegisterTapeClass();	// register tape interface type

    g_TapeUnitAtom = GlobalAddAtom(g_TapePropName);

    for(unit=0; unit<2; unit++) {
	wtstate[unit].placement_valid = 0;
	wtstate[unit].placement.length = sizeof(WINDOWPLACEMENT);
	wtstate[unit].hWnd = NULL;
	wtstate[unit].curtime  = 0;
	TapeFileDlgInit(unit);
    }

    // location of timer digits
    g_rcTime.top    =  45;
    g_rcTime.left   = 260;
    g_rcTime.bottom = g_rcTime.top  + 20;	// 20 pixels high
    g_rcTime.right  = g_rcTime.left + 9*4;	// 4 digits 9 pixels wide

    // location of timer digits
    g_rcMotorEn.top    =   0;
    g_rcMotorEn.left   = 240;
    g_rcMotorEn.bottom = g_rcMotorEn.top  + 20;	 // 20 pixels high
    g_rcMotorEn.right  = g_rcMotorEn.left + 100;
}


// called as a request to close up shop.
// returns 0 if OK, 1 if not OK.
int
WinTapeClose(void)
{
    int unit;

    for(unit=0; unit<2; unit++) {
	if (wtstate[unit].hWnd == NULL)
	    continue;
	if (CheckForDirtyTape(unit, TRUE))
	    return 1;
    }

    return 0;
}


// free resources when program exits
void
WinTapeDestroy(void)
{
    int unit;

    for(unit=0; unit<2; unit++) {
	if (wtstate[unit].hWnd != NULL) {
	    DestroyWindow(wtstate[unit].hWnd);
	    wtstate[unit].hWnd = NULL;
	}
    }

    if (g_hDigitDC != NULL) {
	DeleteDC(g_hDigitDC);
	g_hDigitDC = NULL;
    }
    if (g_hDigits != NULL) {
	DeleteObject(g_hDigits);
	g_hDigits = NULL;
    }
    if (g_hCourFont != NULL) {
	DeleteObject(g_hCourFont);
	g_hCourFont = NULL;
    }
    // we don't delete g_hFixedFont because either it is
    // a copy of g_hCourFont, or it is a standard font.
}


// return status of whether tape controller window is showing or not
int
TapeWindowVisible(int unit)
{
    unit &= 1;

    return IsWindow(wtstate[unit].hWnd) &&
           IsWindowVisible(wtstate[unit].hWnd);
}


// create or restore a tape unit interface.
// if the unit exists, toggle the window visibility.
void
CreateTapeWindow(int unit)
{
    // typical windows window
    DWORD winstyle = WS_CAPTION | WS_MINIMIZEBOX
		   | WS_SYSMENU
		   | WS_CLIPCHILDREN;
    HWND hwnd;

    unit &= 1;

    if (IsWindow(wtstate[unit].hWnd)) {
	// it exists already
	wtstate[unit].showing = !IsWindowVisible(wtstate[unit].hWnd);
	ShowWindow(wtstate[unit].hWnd, wtstate[unit].showing);
	if (wtstate[unit].showing)
	    SetFocus(wtstate[unit].hWnd);
	WinUpdateToolBar();
	return;
    }


    // open up debugging container window
    hwnd = CreateWindow(
		"SolaceTapeWindow",	// window class name
		"Virtual Tape",		// window caption
		winstyle,		// window style
		CW_USEDEFAULT,		// initial x position
		0,			// initial y position
		CW_USEDEFAULT,		// initial x size
		0,			// initial y size
		NULL,			// parent window handle
		NULL,			// window menu handle
		winstate.hInst,		// program instance handle
		(LPVOID)unit);		// creation parameters

    ASSERT(hwnd != NULL);

    wtstate[unit].hWnd = hwnd;
    SetProp(hwnd, (LPCTSTR)g_TapeUnitAtom, (HANDLE)unit);

    UpdateTitleBar(hwnd);
    WinUpdateToolBar();
}


// callback to notify interface of changes
void
WinTapeNotify(int unit, int change, int param)
{
    switch (change) {

	case WT_EOF:
#if 0
	    {
		char buf[80];
		sprintf(buf, "Notice: Unit/%d just hit the end of tape", unit);
		UI_Alert(buf);
	    }
#endif
	    InvalidateRect(wtstate[unit].hWnd, NULL, TRUE);
	    break;

	case WT_TIME:
	    UpdateTime(unit, param, 0);
	    break;

	case WT_PLAYSTATE:
	    UpdateTitleBar(wtstate[unit].hWnd);
	    InvalidateRect(wtstate[unit].hWnd, NULL, TRUE);
	    break;

	case WT_FILESTATE:
	    UpdateTitleBar(wtstate[unit].hWnd);
	    break;

	default:
	    ASSERT(0);
    }
}


// get configuration from .ini file
// only happens at start up so windows aren't created yet
void
WinTapeLoadOptions(char *inifile)
{
    int xunit, n;
    char key[30];
    char buf[10+MAX_PATH];

    // global options
    GetPrivateProfileString("tapeopt", "ignorebaud", "0", buf, sizeof(buf), inifile);
    if (sscanf(buf, "%d", &n) == 1)
	Sys_SetTapeProp(0, TPROP_IGNOREBAUD, n);

    GetPrivateProfileString("tapeopt", "realtime", "1", buf, sizeof(buf), inifile);
    if (sscanf(buf, "%d", &n) == 1)
	Sys_SetTapeProp(0, TPROP_REALTIME, n);

    GetPrivateProfileString("tapeopt", "motorenb", "0", buf, sizeof(buf), inifile);
    if (sscanf(buf, "%d", &n) == 1) {
	SetSVTOpt(0, SVTOpt_MOTENB, n);
	SetSVTOpt(1, SVTOpt_MOTENB, n);
    }


    // per-tape options
    for(xunit=1; xunit<3; xunit++) {

	int unit = xunit & 1;

	sprintf(key, "tape%d", xunit);

	// retrieve window placement
	GetPrivateProfileString("windows", key, "", buf, sizeof(buf), inifile);
	if (strlen(buf) == 0) {
	    wtstate[unit].placement_valid = 0;
	} else {
	    RECT rc;
	    int showing;
	    if (sscanf(buf, "%d:%d,%d", &showing, &rc.top, &rc.left) == 3) {
		rc.bottom = rc.top + g_height;
		rc.right  = rc.left + g_width;
		wtstate[unit].placement_valid = 1;
		wtstate[unit].placement.showCmd = (showing) ? SW_SHOW : SW_HIDE;
		wtstate[unit].placement.flags   = 0;
		wtstate[unit].placement.rcNormalPosition = rc;
		// create window
		CreateTapeWindow(unit);
	    }
	}

	// restore tape file
	GetPrivateProfileString("files", key, "", buf, sizeof(buf), inifile);
	if (strlen(buf) > 0) {
	    int stat;
	    strcpy(wtstate[unit].ofn.lpstrFile, buf);
	    stat = Sys_TapeFile(unit, VTF_LoadTape, wtstate[unit].ofn.lpstrFile);
	    if (stat != 0) {
		sprintf(buf, "Error loading tape '%s'", wtstate[unit].ofn.lpstrFile);
		WinTapeAlert(g_mainwnd, buf);
		strcpy(wtstate[unit].ofn.lpstrFile, "");
	    }
	}
    }
}


// save configuration to .ini file
void
WinTapeSaveOptions(char *inifile)
{
    int xunit, playstate, n;
    char key[30];
    char buf[MAX_PATH];
    RECT rc;

    // global options
    Sys_GetTapeProp(0, TPROP_IGNOREBAUD, &n);
    sprintf(buf, "%d", n);
    WritePrivateProfileString("tapeopt", "ignorebaud", buf, inifile);

    Sys_GetTapeProp(0, TPROP_REALTIME, &n);
    sprintf(buf, "%d", n);
    WritePrivateProfileString("tapeopt", "realtime", buf, inifile);

    GetSVTOpt(0, SVTOpt_MOTENB, &n);
    sprintf(buf, "%d", n);
    WritePrivateProfileString("tapeopt", "motorenb", buf, inifile);

    // per-tape options
    for(xunit=1; xunit<3; xunit++) {

	int unit = xunit & 1;

	sprintf(key, "tape%d", xunit);

	// save window placement
	if (wtstate[unit].hWnd != NULL) {
	    GetWindowRect(wtstate[unit].hWnd, &rc);
	    sprintf(buf, "%d:%d,%d", wtstate[unit].showing, rc.top, rc.left);
	} else if (wtstate[unit].placement_valid) {
	    sprintf(buf, "%d:%d,%d", wtstate[unit].showing,
		    wtstate[unit].placement.rcNormalPosition.top,
		    wtstate[unit].placement.rcNormalPosition.left);
	} else
	    strcpy(buf, "");	// non existant
	WritePrivateProfileString("windows", key, buf, inifile);

	// save filename
	strcpy(buf, "");
	Sys_GetTapeProp(unit, TPROP_PLAYSTATE, &playstate);
	if (playstate != EMPTY) {
	    if (strlen(wtstate[unit].ofn.lpstrFile) > 0)
		strcpy(buf, wtstate[unit].ofn.lpstrFile);
	    // else
		// FIXME: prompt for SaveAs?
	}
	WritePrivateProfileString("files", key, buf, inifile);
    }
}

