// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2002, 2005

// P.T. sold a very cheap S-100 plug-in card that simply filtered the 8080's
// interrupt enable line via a passive single-pole RC lowpass filter.
// This module is notified every time the 8080 performs an EI or DI
// instruction and produces a series of samples out at the rate of the
// audio card.

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

#include "solace.h"
#include "solace_intf.h"
#include "wav.h"	// wav file description

// all state accessible from outside of this block
static struct {

    // controls:

    int  enabled;		// 0=disable the whole thing

    int  inten_volume;		// 0-100 volume control slider %-age
    int  fan_volume;		// 0-100 volume control slider %-age
    int  disk_volume;		// 0-100 volume control slider %-age

    float inten_logvol;		// companded volume
    float fan_logvol;		// companded volume
    float disk_logvol;		// companded volume

    // state:

    uint8 *fan1_wav;	// starting up sound
    int  fan1_samples;	// # of samples in audio
    uint8 *fan2_wav;	// running loop
    int  fan2_samples;	// # of samples in audio
    enum { FAN_OFF, FAN_START, FAN_ON } fan;
    int  fan_phase;	// sample pointer

    uint8 *step_wav;
    int  step_samples;	// # of samples in audio
    int  step_phase;	// -1 = off, otherwise it's a sample pointer

    uint8 *disk1_wav;
    int  disk1_samples;	// # of samples in audio
    uint8 *disk2_wav;
    int  disk2_samples;	// # of samples in audio
    enum { DISK_OFF, DISK_START, DISK_ON } disk;
    int  disk_phase;	// sample pointer

    uint8 *door1_wav;	// door opening sound
    int  door1_samples;	// # of samples in audio
    uint8 *door2_wav;	// door closing sound
    int  door2_samples;	// # of samples in audio
    enum { DOOR_NONE, DOOR_OPEN, DOOR_CLOSE } door;
    int  door_phase;	// sample pointer

} audiostate;


// given an integer 0-100, return an exponentially increasing value
static float
audlog(int n) {
    const float octave = 15.0f;		// # of units per doubling of amplitude
    const float maxval = (float)pow(2.0f, 100.0f/octave);
    if (n == 0)
	return 0.0f;	// truly cut off
    else
	return (float)(pow(2.0f, n/octave) / maxval);	// 2^(-octave)
}


void
Sys_AudioSet(int prop, int value)
{
    switch (prop) {

	case SFXP_ENABLE:
	    audiostate.enabled = !!value;
	    break;
	case SFXP_VOL_INTEN:
	    audiostate.inten_volume = value;
	    audiostate.inten_logvol = audlog(value);
	    break;
	case SFXP_VOL_FAN:
	    audiostate.fan_volume = value;
	    audiostate.fan_logvol = audlog(value);
	    break;
	case SFXP_VOL_DISK:
	    audiostate.disk_volume = value;
	    audiostate.disk_logvol = audlog(value);
	    break;
	default:
	    ASSERT(0);
	    break;
    }
}

void
Sys_AudioGet(int prop, int *value)
{
    switch (prop) {
	case SFXP_ENABLE:        *value = audiostate.enabled;      break;
	case SFXP_VOL_INTEN:     *value = audiostate.inten_volume; break;
	case SFXP_VOL_FAN:       *value = audiostate.fan_volume;   break;
	case SFXP_VOL_DISK:      *value = audiostate.disk_volume;  break;
	default: ASSERT(0); break;
    }
}


// this gets called whenever something that makes noise happens
void
Sys_AudioEvent(int event)
{
    switch (event) {

	case SFXE_FAN_ON:
	    if (audiostate.fan == FAN_OFF) {
		audiostate.fan       = FAN_START;
		audiostate.fan_phase = 0;
	    }
	    break;
	case SFXE_FAN_OFF:
	    // FIXME: needs fixing if we support spin-down of fan
	    audiostate.fan = FAN_OFF;
	    break;

	case SFXE_DISK_STEP:
#if 0
	    if (audiostate.step_phase < 0)	// don't step too fast
#endif
		audiostate.step_phase = 0;
	    break;

	case SFXE_MOTOR_ON:
	    if (audiostate.disk == DISK_OFF) {	// ignore redundant kicks
		audiostate.disk  = DISK_START;
		audiostate.disk_phase = 0;
	    }
	    break;
	case SFXE_MOTOR_OFF:
	    audiostate.disk = DISK_OFF;
	    audiostate.disk_phase = 0;
	    break;

	case SFXE_DISK_DOOR_OPEN:
	    audiostate.door = DOOR_OPEN;
	    audiostate.door_phase = 0;
	    break;
	case SFXE_DISK_DOOR_CLOSE:
	    audiostate.door = DOOR_CLOSE;
	    audiostate.door_phase = 0;
	    break;

	default:
	    ASSERT(0);
	    break;
    }
}


// everything on the Sol is slaved off of a 14.3 MHz crystal,
// although the CPU cycle time is a submultiple of that.
#define SAMPRATE_8080  14318180

// normally we generate output at 44100, but as an experiment to see
// if MUSIC was generating any high frequencies that were aliasing
// down and causing noise, I tried oversampling, filtering, then
// decimating.  It had no effect at all, and of course, costs cycles.
// That is why there are a few lowpass filters at the end that are
// set up for 88.2 KHz.
#define SAMPRATE_AUDIO (1*44100)	// fixed sampling rate


// # of samples to ever produce in one event
// FIXME: the proper figure depends on the timeslice size.
#define MAX_SAMPLES (500*(SAMPRATE_8080/SAMPRATE_AUDIO))

// the 3db point of a simple RC filter is 1/(2*pi*R*C), which
// in this case is a pitiable 1750 Hz.
#define R 910			// RC filter constant (ohms)
#define C 0.1e-6		// RC filter constant (farads)
#define F ((float)SAMPRATE_8080)
#define K (-1.0f/(F*R*C))	// time constant and scale

static int samprate_out;		// frequency (in Hz) of audio sample generation
static int samprate_ratio;		// ratio of SAMPRATE_AUDIO/samprate_out
static timer_t sample_last_time;	// time of last change in driver
static float   sample_last_value;	// filter output at time of last change
static int     sample_last_state;	// new driving value at time of last change
static uint32  sample_phase;		// audio sample DDA phase

#define DECAY_TABLESIZE (330*2)	// 326 for 44.1KHz, *2 for 22050
static float decay[DECAY_TABLESIZE];

// forward declaration
static void get_wav(char *filename, uint8 **wav, int *num_samples);
static int  WavHeaderOK(FILE *fd, int *samples);
static void intaudio_filter(float v);
static void intaudio_HP_filter(float v);
static void intaudio_downsample(float v);
static void mixer(float inten_sample);
static float upsample_sfx(void);
static float next_sfx_sample(void);


void
intaudio_init(void)
{
    int t;

    for(t=0; t<DECAY_TABLESIZE; t++)
	decay[t] = 1.0f - (float)exp(t*K);

    sample_last_time  = 0;
    sample_last_state = 0;
    sample_last_value = 0.0f;
    sample_phase      = 0;

    audiostate.fan        = FAN_OFF;
    audiostate.door       = DOOR_NONE;
    audiostate.disk       = DISK_OFF;
    audiostate.step_phase = -1;

    // try and read the samples
    get_wav("fan1.wav",  &audiostate.fan1_wav,  &audiostate.fan1_samples);
    get_wav("fan2.wav",  &audiostate.fan2_wav,  &audiostate.fan2_samples);
    get_wav("step.wav",  &audiostate.step_wav,  &audiostate.step_samples);
    get_wav("disk1.wav", &audiostate.disk1_wav, &audiostate.disk1_samples);
    get_wav("disk2.wav", &audiostate.disk2_wav, &audiostate.disk2_samples);
    get_wav("door1.wav", &audiostate.door1_wav, &audiostate.door1_samples);
    get_wav("door2.wav", &audiostate.door2_wav, &audiostate.door2_samples);
}

// specify the rate to produce samples
void
intaudio_samplerate(int rate)
{
    samprate_out = rate;

    if (samprate_out > 0) {
	samprate_ratio = SAMPRATE_AUDIO/samprate_out;
	// must be an integer multiple
	ASSERT(samprate_out*samprate_ratio == SAMPRATE_AUDIO);
    } else {
	samprate_ratio = 1;
    }
}


// called every time EI or DI is performed
void
intaudio_event(int inten)
{
    // filter out non-events
    if (inten == sample_last_state)
	return;

    // compute RC decay and spit out new samples as we go
    intaudio_advance_time();

    sample_last_state = inten;
}


#define KEEP_STATS 0
#if KEEP_STATS
long stat1 = 0;	//   58629  == 1
long stat2 = 0;	// 1516028  == 25
#endif

// advance the time to the specified time,
// generating any audio samples as necessary.
// directly emulate the simple RC filter of the original board.
void
intaudio_advance_time(void)
{
    float driver;
    timer_t now, diff;
    uint32 tdiff;

    if (!audiostate.enabled)
	return;	// not making noise

    driver = (sample_last_state) ? 1.0f : 0.0f;
    now  = TimeAbs();
    diff = TimeDiff(sample_last_time);

    // if there has been no audio activity for a long time,
    // limit how many samples we produce
    if (diff > MAX_SAMPLES)
	tdiff = MAX_SAMPLES;
    else
	tdiff = (uint32)diff;

#if KEEP_STATS
    stat1++;
#endif

    // use a DDA to maintain proper ratio of sample rates
    while (tdiff > 0) {

	// directly compute the next event
    // FIXME: there are ways to do this division less frequently
    //        eg, there will always be either (n) or (n+1) ticks
    //        between each audio sample.  We can compute this at
    //        the start and after each wraparound and just decrement
    //        this quotient until it is exhausted.
	uint32 n1 = (SAMPRATE_8080-1 - sample_phase) / SAMPRATE_AUDIO + 1;
	uint32 ticks = (n1 < tdiff) ? n1 : tdiff;	// pick min
	sample_phase += ticks*SAMPRATE_AUDIO;
	tdiff -= ticks;

	sample_last_value += (driver - sample_last_value) * decay[ticks];

	// either we generate an audio sample or we're done
	if (sample_phase >= SAMPRATE_8080) {
	    sample_phase -= SAMPRATE_8080;
	    // generate a new audio sample
	    ASSERT(ticks < DECAY_TABLESIZE);
	    intaudio_filter(sample_last_value);
	} else {
	    ASSERT(tdiff == 0);
	}

#if KEEP_STATS
	stat2++;
#endif
    } // (tdiff > 0)

    sample_last_time = now;
}


// the filters here were designed by a web-hosted filter designing
// program written by a Mr. Tony Fisher, located here as of March 2002:
// http://www-users.cs.york.ac.uk/~fisher/mkfilter/trad.html

#if 1
// 4th order butterworth lowpass filter, 4410Hz (44.1KHz/10)
// the input varies between 0.0 and 1.0
void
intaudio_filter(float v)
{
    float vin = v - 0.5f;
    float vout;
#define NZEROS 4
#define NPOLES 4
#define GAIN   2.072820954e+02f

    static float xv[NZEROS+1], yv[NPOLES+1];
    static int first_time = 1;
    if (first_time) {
	int i;
	for(i=0; i<NZEROS+1; i++)
	    xv[i] = yv[i] = 0.0f;
	first_time = 0;
    }

    xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3]; xv[3] = xv[4];
    xv[4] = vin / GAIN;
    yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3]; yv[3] = yv[4];
    yv[4] =   (xv[0] + xv[4]) + 4 * (xv[1] + xv[3]) + 6 * xv[2]
		 + ( -0.1873794924f * yv[0]) + (  1.0546654059f * yv[1])
		 + ( -2.3139884144f * yv[2]) + (  2.3695130072f * yv[3]);
    vout = yv[4];

    intaudio_HP_filter(vout);
}
#endif
#if 0
// 6th order butterworth lowpass filter, 5512.5Hz (44.1KHz/8)
static void
intaudio_filter(float v)
{
    float vin = v - 0.5f;
    float vout;
#define NZEROS 6
#define NPOLES 6
#define GAIN   9.508895986e+02f

    static float xv[NZEROS+1], yv[NPOLES+1];
    static int first_time = 1;
    if (first_time) {
	int i;
	for(i=0; i<NZEROS+1; i++)
	    xv[i] = yv[i] = 0.0f;
	first_time = 0;
    }

    xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3]; xv[3] = xv[4]; xv[4] = xv[5]; xv[5] = xv[6];
    xv[6] = vin / GAIN;
    yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3]; yv[3] = yv[4]; yv[4] = yv[5]; yv[5] = yv[6];
    yv[6] =   (xv[0] + xv[6]) + 6 * (xv[1] + xv[5]) + 15 * (xv[2] + xv[4])
		 + 20 * xv[3]
		 + ( -0.0433569884f * yv[0]) + (  0.3911172306f * yv[1])
		 + ( -1.5172788447f * yv[2]) + (  3.2597642798f * yv[3])
		 + ( -4.1360809983f * yv[4]) + (  2.9785299261f * yv[5]);
    vout = yv[6];

    intaudio_HP_filter(vout);
}
#endif
#if 0
// 8th order chebechev lowpass filter, 9KHz (of 44.1KHz)
void
intaudio_filter(float v)
{
    float vin = v - 0.5f;
    float vout;

#define NZEROS 8
#define NPOLES 8
#define GAIN   2.023751107e+03f

    static float xv[NZEROS+1], yv[NPOLES+1];
    static int first_time = 1;
    if (first_time) {
	int i;
	for(i=0; i<NZEROS+1; i++)
	    xv[i] = yv[i] = 0.0f;
	first_time = 0;
    }

    xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3]; xv[3] = xv[4]; xv[4] = xv[5]; xv[5] = xv[6]; xv[6] = xv[7]; xv[7] = xv[8];
    xv[8] = vin / GAIN;
    yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3]; yv[3] = yv[4]; yv[4] = yv[5]; yv[5] = yv[6]; yv[6] = yv[7]; yv[7] = yv[8];
    yv[8] =   (xv[0] + xv[8]) + 8 * (xv[1] + xv[7]) + 28 * (xv[2] + xv[6])
		 + 56 * (xv[3] + xv[5]) + 70 * xv[4]
		 + ( -0.1867934074f * yv[0]) + (  1.1332247493f * yv[1])
		 + ( -3.4847805116f * yv[2]) + (  6.9536631017f * yv[3])
		 + ( -9.7983517882f * yv[4]) + ( 10.0022344750f * yv[5])
		 + ( -7.3241701277f * yv[6]) + (  3.5784757403f * yv[7]);
    vout = yv[8];

    intaudio_HP_filter(vout);
}
#endif
#if 0
// 6th order butterworth lowpass filter, 4410.5Hz (88.2KHz/20)
/* Digital filter designed by mkfilter/mkshape/gencode   A.J. Fisher
   Command line: /www/usr/fisher/helpers/mkfilter -Bu -Lp -o 6 -a 5.0000000000e-02 0.0000000000e+00 -l */
static void
intaudio_filter(float v)
{
    float vin = v - 0.5f;
    float vout;
#define NZEROS 6
#define NPOLES 6
#define GAIN   (float)1.165969038e+05

    static float xv[NZEROS+1], yv[NPOLES+1];
    static int first_time = 1;
    if (first_time) {
	int i;
	for(i=0; i<NZEROS+1; i++)
	    xv[i] = yv[i] = 0.0f;
	first_time = 0;
    }

    xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3]; xv[3] = xv[4]; xv[4] = xv[5]; xv[5] = xv[6];
    xv[6] = vin / GAIN;
    yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3]; yv[3] = yv[4]; yv[4] = yv[5]; yv[5] = yv[6];
    yv[6] =   (xv[0] + xv[6]) + 6 * (xv[1] + xv[5]) + 15 * (xv[2] + xv[4])
		 + 20 * xv[3]
		 + ( -0.2951724313f * yv[0]) + (  2.1290387500f * yv[1])
		 + ( -6.4411118810f * yv[2]) + ( 10.4690788930f * yv[3])
		 + ( -9.6495177287f * yv[4]) + (  4.7871354989f * yv[5]);
    vout = yv[6];

    intaudio_HP_filter(vout);
}
#endif
#if 0
// 4th order butterworth lowpass filter, 4410.5Hz (88.2KHz/20)
/* Digital filter designed by mkfilter/mkshape/gencode   A.J. Fisher
   Command line: /www/usr/fisher/helpers/mkfilter -Bu -Lp -o 4 -a 5.0000000000e-02 0.0000000000e+00 -l */
static void
intaudio_filter(float v)
{
    float vin = v - 0.5f;
    float vout;

#define NZEROS 4
#define NPOLES 4
#define GAIN   2.400388646e+03f

    static float xv[NZEROS+1], yv[NPOLES+1];
    static int first_time = 1;
    if (first_time) {
	int i;
	for(i=0; i<NZEROS+1; i++)
	    xv[i] = yv[i] = 0.0f;
	first_time = 0;
    }

    xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3]; xv[3] = xv[4];
    xv[4] = vin / GAIN;
    yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3]; yv[3] = yv[4];
    yv[4] =   (xv[0] + xv[4]) + 4 * (xv[1] + xv[3]) + 6 * xv[2]
		 + ( -0.4382651423f * yv[0]) + (  2.1121553551f * yv[1])
		 + ( -3.8611943490f * yv[2]) + (  3.1806385489f * yv[3]);
    vout = yv[4];

    intaudio_HP_filter(vout);
}
#endif


// highpass filter to get rid of DC offset.
// vaguely models the DC blocking cap of the actual board.
// the input varies between -0.5 and +0.5, roughly.
static void
intaudio_HP_filter(float vin)
{
    float vout;

#define HP_NZEROS 2
#define HP_NPOLES 2
#define HP_GAIN   1.003026941e+00f
    static float xv[HP_NZEROS+1], yv[HP_NPOLES+1];
    static int first_time = 1;
    if (first_time) {
	int i;
	for(i=0; i<HP_NZEROS+1; i++)
	    xv[i] = yv[i] = 0.0f;
	first_time = 0;
    }

    xv[0] = xv[1]; xv[1] = xv[2];
    xv[2] = vin / HP_GAIN;
    yv[0] = yv[1]; yv[1] = yv[2];
    yv[2] =   (xv[0] + xv[2]) - 2 * xv[1]
		 + ( -0.9939734940f * yv[0]) + (  1.9939552796f * yv[1]);
    vout = yv[2];

    intaudio_downsample(vout);
}


// downsample to requested output rate, which must be a submultiple
// of SAMPRATE_AUDIO.  we can just toss out samples because we should
// be bandlimited enough that it doesn't matter.
// the input varies between -0.5 and +0.5, roughly.
static void
intaudio_downsample(float v)
{
    static int phase = 0;

    phase++;
    if (phase >= samprate_ratio) {
	phase = 0;
	mixer(v);
    }
}


// Mixer: mix various sound effects together at one point.  the inten
// samples are the driver for timing; for every one inten sample, we
// pull advance the sample pointers for the other active sounds.
// the input varies between -0.5 and +0.5, roughly.
static void
mixer(float inten_sample)
{
    float vout;

    vout = inten_sample
	 * audiostate.inten_logvol * 2.0f;	// crank up the volume!

    // output rate is 2x the sfx samples
#if 1
    vout += upsample_sfx();
#else
    {	// just duplicate samples
	static int toggle = 0;
	static float sfx = 0.0f;
	toggle = !toggle;
	if (toggle)
	    sfx = next_sfx_sample();
	vout += sfx;
    }
#endif

    UI_CreateAudioSample(vout + 0.5f);	// goes out in the range 0.0 to 1.0
}


// return the next sound effects sample.  the output rate is always 44100
// but the samples are stored at 22050.  simply duplicating samples is
// not exactly right.  the method chosen here is to effectively interleave
// zero samples with the 22050 Hz samples and then apply a lowpass filter.
static float
upsample_sfx(void)
{
    // http://www.dsptutor.freeuk.com/remez/RemezFIRFilterDesign.html
    // Parks-McClellan FIR Filter Design
    // 
    // Filter type: Low pass
    // Passband: 0 - 0.29478
    // Order: 31
    // Passband ripple: 0.1 dB
    // Transition band: 0.088
    // Stopband attenuation: 60.0 dB
    // nb: coefficients are given as doubles, but we're using floats.

    static float x[32];	// history of inputs
    static int xi = 0;	// index to circular x[] buffer
    static int phase = 1;	// we produce two outputs for each one input
    float vout;

    ASSERT(SAMPRATE_AUDIO == 44100);

    phase = !phase;		//


    // explanation of zero stuffing:
    //
    // orig seq: 1, 2, 3, 4, 5
    // new  seq: 1, 0, 2, 0, 3, 0, 4, 0, 5
    //
    // coef seq: c0,c1,c2,c4,c4,c5,c6,c7
    // out[n]   = c0*1 + c1*0 + c2*2 + c3*0 + c4*3 + c5*0 + c6*4 + c7*0
    // out[n+1] = c0*0 + c1*1 + c2*0 + c3*2 + c4*0 + c5*3 + c6*0 + c7*4
    //   or collapsing out the known zeros:
    // out[n]   = c0*1 +        c2*2 +        c4*3 +        c6*4       
    // out[n+1] =        c1*1 +        c3*2 +        c5*3 +        c7*4

    // note that lower xi indices (ignoring wrapping) are
    switch (phase) {
	case 0:
	    xi = (xi+1) & 15;	// wrap every 16
	    x[xi] = x[xi+16] = next_sfx_sample();
	    vout =
		// from most recent sample ...
		  (x[xi- 0+16] *  9.318547061649759E-5f)	// c[0]
		+ (x[xi- 1+16] * -7.879650240536664E-5f)	// c[2]
		+ (x[xi- 2+16] * -0.008319358993221289f)	// c[4]
		+ (x[xi- 3+16] *  0.014673392114194652f)	// c[6]
		+ (x[xi- 4+16] * -7.729689358275862E-4f)	// c[8]
		+ (x[xi- 5+16] * -0.0395692636904585f)		// c[10]
		+ (x[xi- 6+16] *  0.07288108952030582f)		// c[12]
		+ (x[xi- 7+16] * -0.0017892817203430443f)	// c[14]
		+ (x[xi- 8+16] *  0.5512619081067116f)		// c[16]
		+ (x[xi- 9+16] * -0.10457406256352055f)		// c[18]
		+ (x[xi-10+16] * -0.0013986781082109276f)	// c[20]
		+ (x[xi-11+16] *  0.03154507106307213f)		// c[22]
		+ (x[xi-12+16] * -0.018497508455466726f)	// c[24]
		+ (x[xi-13+16] * -1.5708164921279035E-4f)	// c[26]
		+ (x[xi-14+16] *  0.006085075514158386f)	// c[28]
		+ (x[xi-15+16] * -0.004621433681521645f);	// c[30]
		// ... to oldest sample
	    break;
	case 1:
	    vout =
		// from most recent sample ...
		  (x[xi- 0+16] * -0.004621433681521645f)	// c[1]
		+ (x[xi- 1+16] *  0.006085075514158386f)	// c[3]
		+ (x[xi- 2+16] * -1.5708164921279035E-4f)	// c[5]
		+ (x[xi- 3+16] * -0.018497508455466726f)	// c[7]
		+ (x[xi- 4+16] *  0.03154507106307213f)		// c[9]
		+ (x[xi- 5+16] * -0.0013986781082109276f)	// c[11]
		+ (x[xi- 6+16] * -0.10457406256352055f)		// c[13]
		+ (x[xi- 7+16] *  0.5512619081067116f)		// c[15]
		+ (x[xi- 8+16] * -0.0017892817203430443f)	// c[17]
		+ (x[xi- 9+16] *  0.07288108952030582f)		// c[19]
		+ (x[xi-10+16] * -0.0395692636904585f)		// c[21]
		+ (x[xi-11+16] * -7.729689358275862E-4f)	// c[23]
		+ (x[xi-12+16] *  0.014673392114194652f)	// c[25]
		+ (x[xi-13+16] * -0.008319358993221289f)	// c[27]
		+ (x[xi-14+16] * -7.879650240536664E-5f)	// c[29]
		+ (x[xi-15+16] *  9.318547061649759E-5f);	// c[31]
		// ... to oldest sample
	    break;
	default:
	    ASSERT(0);
	    break;
    }
    
    return 2.0f * vout;	// 2.0f compensates for zero stuffing
}


// this produces samples at 22.050 KHz.
// it is up to the caller to interpolate if necessary.
static float
next_sfx_sample(void)
{
    int fan_sfx, disk_sfx;	// fan and disk sound effects contribution
    float vout;

    // fan noise contribution

    fan_sfx = 0;
    if (audiostate.fan1_wav != NULL) {
	switch (audiostate.fan) {
	    case FAN_OFF:
		break;
	    case FAN_START:
		fan_sfx = audiostate.fan1_wav[audiostate.fan_phase++] - 128;
		if (audiostate.fan_phase >= audiostate.fan1_samples) {
		    audiostate.fan_phase = 0;
		    audiostate.fan       = FAN_ON;
		}
		break;
	    case FAN_ON:
		fan_sfx = audiostate.fan2_wav[audiostate.fan_phase++] - 128;
		if (audiostate.fan_phase >= audiostate.fan2_samples)
		    audiostate.fan_phase = 0;	// loop
		break;
	    default:
		ASSERT(0);
		break;
	}
    }


    // add up disk noise contributions

    disk_sfx = 0;

    if (audiostate.step_wav != NULL) {
	if (audiostate.step_phase >= 0) {
	    disk_sfx += audiostate.step_wav[audiostate.step_phase++] - 128;
	    if (audiostate.step_phase >= audiostate.step_samples)
		audiostate.step_phase = -1;	// one-shot
	}
    }

    switch (audiostate.disk) {
	case DISK_OFF:
	    break;
	case DISK_START:
	    if (audiostate.disk1_wav != NULL) {
		disk_sfx += audiostate.disk1_wav[audiostate.disk_phase++] - 128;
		if (audiostate.disk_phase >= audiostate.disk1_samples) {
		    audiostate.disk = DISK_ON;
		    audiostate.disk_phase = 0;
		}
	    }
	    break;
	case DISK_ON:
	    if (audiostate.disk2_wav != NULL) {
		disk_sfx += audiostate.disk2_wav[audiostate.disk_phase++] - 128;
		if (audiostate.disk_phase >= audiostate.disk2_samples) {
		    audiostate.disk = DISK_ON;	// loop
		    audiostate.disk_phase = 0;
		}
	    }
	    break;
    }

    switch (audiostate.door) {
	case DOOR_NONE:
	    break;
	case DOOR_OPEN:
	    if (audiostate.door1_wav != NULL) {
		disk_sfx += audiostate.door1_wav[audiostate.door_phase++] - 128;
		if (audiostate.door_phase >= audiostate.door1_samples)
		    audiostate.door = DOOR_NONE;
	    }
	    break;
	case DOOR_CLOSE:
	    if (audiostate.door2_wav != NULL) {
		disk_sfx += audiostate.door2_wav[audiostate.door_phase++] - 128;
		if (audiostate.door_phase >= audiostate.door2_samples)
		    audiostate.door = DOOR_NONE;
	    }
	    break;
    }

    // convert to output format
    vout = 0.0f;
    if (fan_sfx != 0)
	vout += (float)fan_sfx/255.0f * audiostate.fan_logvol;
    if (disk_sfx != 0)
	vout += (float)disk_sfx/255.0f * audiostate.disk_logvol;

    return vout;
}


// ======================================================================
//      Read a WAV file
// ======================================================================

// subdirectory from which WAV files are read
#define SUBDIR "resource\\"

static void
get_wav(char *filename, uint8 **wav, int *num_samples)
{
    FILE *fd;		// file descriptor
    uint8 *buf;		// wav buffer pointer
    int n_remain;	// expected # of samples
    int n_read;		// # of samples actually read
    char *subname;	// with subdirectory prepended
    char *fullname;	// full filename

    // default
    *wav = NULL;
    *num_samples = 0;

    // convert to a fully qualified path name
    subname = (char*)malloc(strlen(filename) + strlen(SUBDIR) + 1);
    if (subname == NULL) {
	UI_Alert("Error: trouble allocating memory");
	return;
    }
    strcpy(subname, SUBDIR);
    strcat(subname, filename);
    fullname = HostMakeAbsolutePath(subname);
    if (fullname == NULL) {
	UI_Alert("Error: trouble allocating memory");
	return;
    }

    fd = fopen(fullname, "rb");
    if (fd == NULL) {
	UI_Alert("Error: couldn't open wave sample file '%s'", fullname);
	free(fullname);
	return;
    }

    if (!WavHeaderOK(fd, &n_remain)) {
	UI_Alert("Error: wave sample file '%s' isn't the right format", fullname);
	fclose(fd);
	free(fullname);
	return;
    }

    buf = (uint8*)malloc(n_remain * sizeof(uint8));
    if (buf == NULL) {
	UI_Alert("Error: not enough memory to hold sample '%s'", fullname);
	fclose(fd);
	free(fullname);
	return;
    }

    // read in the expected number of samples, or EOF, whichever comes first
    n_read = fread(buf, 1, n_remain, fd);
    fclose(fd);
    free(fullname);

    // return what we actually were able to read
    *wav         = buf;
    *num_samples = n_read;
}


// return 1 if the file header conforms
static int
WavHeaderOK(FILE *fd, int *samples)
{
    RIFF_t        RiffHdr;
    FormatChunk_t FormatHdr;
    DataChunk_t   DataHdr;

    if (fread(&RiffHdr, sizeof(RiffHdr), 1, fd) != 1)
	return 0; // error accessing input file

    if ((RiffHdr.groupID  != RiffID) ||
	(RiffHdr.riffType != WaveID))
	return 0; // input file not a WAV file

    if (fread(&FormatHdr, sizeof(FormatHdr), 1, fd) != 1)
	return 0; // error accessing input file

    if ((FormatHdr.chunkID       != FmtID) ||
        (FormatHdr.chunkSize     != sizeof(FormatHdr)-8) ||
        (FormatHdr.BitsPerSample != 8) ||
	(FormatHdr.Channels      != 1) ||
	(FormatHdr.Frequency     != 22050))
	return 0; // not the expected format

    if (fread(&DataHdr, sizeof(DataHdr), 1, fd) != 1)
	return 0; // error accessing input file

    if (DataHdr.chunkID != DataID)
	return 0; // not the expected format

    // must all be one big chunk for this to work
    *samples = DataHdr.chunkSize / 
	       ((FormatHdr.BitsPerSample/8) * FormatHdr.Channels);

    return 1;	// OK
}
