// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000, 2001

// this code supplies a very simple scheduler that operates in
// simulated processor time.
//
// a fixed number of timers are supported; timers are one-shot,
// not oscillators.
//
// the timer is only a 32 bit integer, but only 31 bits should be
// used as things might get squirrely near 2^32.  At the nominal
// 14.318 MHz "base" frequency, this allows setting a timer event
// for about 2.5 minutes into the future.  This should be enough
// for our needs.  This could all be modified to use __int64 for
// the counters if it becomes an issue.
//
// a routine desiring later notification at some specific time
// calls TimerCreate(ticks, *fcn, arg1, arg2), which causes
// 'fcn' to be called with parameters arg1 and arg2 after simulating
// 'ticks' clock cycles.  then event is removed from the queue.
//
// The scheduler uses a "base" tick for all scheduling.  In the Sol,
// there is a 14.318 MHz oscillater that gets divided by 5, 6, or 7
// to create the CPU clock.  Because the user might change the scale
// at any point, we don't want old timers to get affected.  Thus we
// always schedule in base units (which are equivalent to wall time)
// and then as each CPU instruction is executed, scale that tick count
// by the ratio of 14.318MHz/CPU_frequency.

// FIXME: this needs to be rewritten.  It started out simply enough,
//        but it has gotten ugly with time.

#include <stdlib.h>
#include <limits.h>
#include "solace.h"
#include "solace_intf.h"	// used for UI_Alert() only


#define NUM_TIMERS 20	// number of active timers allowed

#define TEST_TIMER 0	// if 1, set up a few timers at time 0


static struct {

    int active_timers;		// # of active timers

    int updating;		// semaphore that we are updating

    // rather than updating all N counters every instruction,
    // instead we find the one which will expire first, and
    // transfer that into countdown and startcnt.  as simulated
    // time ticks by, we decrement countdown.
    //
    // when countdown goes <=0, all timers have (startcnt-countdown)
    // ticks removed from their own counter.
    uint32 countdown;
    uint32 startcnt;

    // any timer that has its own counter go <=0 will then
    // cause a callback to the supplied function using the
    // supplied parameters.
    struct {
	uint32    ctr;		// counter until expiration
	timercb_t callback;	// callback function
	uint32    arg1;		// arbitrary value
	uint32    arg2;		// arbitrary value
    } timer[NUM_TIMERS];

} timers;


// create an absolute tick timer.
// we update a uint32 version for speed, then roll that into a uint64 version
// either when rollover out of the uint32 accumulator is a danger, or when
// somebody want to get an absolute timer value.
timer_t timer64;
uint32  timer32;
static void TimerRollover(uint32 arg1, uint32 arg2);


// set up some events and check that they trigger properly
#if TEST_TIMER
#include <stdio.h>	// for printf
static int g_testtime;
static void
TimerTestCB(uint32 arg1, uint32 arg2)
{
    printf("got callback for timer %d after %d clocks\n", arg1, g_testtime);
    if (arg1 == 2)
	(void)TimerCreate(4, TimerTestCB, 5, 0);
}

static void
TimerTest(void)
{
    int t1, t2, t3, t4;

    g_testtime = 0;

    t1 = TimerCreate(30, TimerTestCB, 1, 0);
    t2 = TimerCreate(10, TimerTestCB, 2, 0);
    t3 = TimerCreate(50, TimerTestCB, 3, 0);
    t4 = TimerCreate(10, TimerTestCB, 4, 0);
}
#endif


// intialize Timer data structures.  Called once per simulation
void
TimerInit(void)
{
    int i;

    timers.active_timers = 0;
    timers.countdown     = 0;
    timers.updating      = 0;

    for(i=0; i<NUM_TIMERS; i++) {
	timers.timer[i].ctr = 0;	// inactive timer
    }

    timer64 = 0;
    timer32 = 0;
    // to make sure our 32b counters don't ever wrap,
    // this callback makes sure the 32b things propagate
    // into 64b things once in a while.
    (void)TimerCreate((1<<30), TimerRollover, 0, 0);

#if TEST_TIMER
    TimerTest();
#endif
}


// return the absolute 14.3M tick count
timer_t
TimeAbs(void)
{
    return (timer64 + timer32);
}

// compute time difference from earlier time until now
timer_t
TimeDiff(timer_t oldtime)
{
    return (timer64 + timer32 - oldtime);
}

// compute time difference, saturated to int32 range
uint32
TimeDiffSat(timer_t oldtime)
{
    timer_t diff = (timer64 + timer32 - oldtime);

    if (diff <= INT_MAX)
	return (int)diff;
    else
	return INT_MAX;
}


// roll over the 32b tick counter into the 64b version.
static void
TimerRollover(uint32 arg1, uint32 arg2)
{
    timer64 += timer32;
    timer32 = 0;

    arg1 = arg1; arg2 = arg2;	// keep lint happy

    // kick it off again
    (void)TimerCreate((1<<30), TimerRollover, 0, 0);
}


// creates a timer.  returns an int, which can be passed back to
// TimerKill to get rid of a timer.  "ticks" is in base units.
int
TimerCreate(uint32 ticks, timercb_t fcn, uint32 arg1, uint32 arg2)
{
    int tmr;

    // scan through all slots looking for an empty one
    int found = 0;
    for(tmr=0; tmr<NUM_TIMERS; tmr++) {
	if (timers.timer[tmr].ctr == 0) {
	    timers.timer[tmr].ctr      = ticks;
	    timers.timer[tmr].callback = fcn;
	    timers.timer[tmr].arg1     = arg1;
	    timers.timer[tmr].arg2     = arg2;
	    timers.active_timers++;
	    found = 1;
	    break;
	}
    }
    if (!found) {
	UI_Alert("Error: ran out of simulated timers");
	exit(-1);
    }

    if (timers.updating) {
	// an expiring timer scheduled a new timer, so at the
	// end of the TimerCredit, the startcnt and countdown
	// get recomputed, and we don't need to tweak them here.
	;
    } else if (timers.active_timers == 1) {
	// this is the only timer
	timers.startcnt = timers.countdown = ticks;
    } else if (ticks < timers.countdown) {
	// the new timer expires before current target
	uint32 diff = (timers.countdown - ticks);
	timers.startcnt  -= diff;
	timers.countdown -= diff;	// == ticks
    } else {
	// the timer state might be that we already have credits lined up
	// for other timers.  rather than doling out the elapsed time to
	// each timer (which may cause some timer to expire, which may cause
	// a callback to try and create another timer, causing reentrancy
	// issues), instead we just add the current credit to the new timer.
	// that is, if N cycles have gone by since the last time we
	// evaluated who to trigger next, we pretend that the new event
	// which is to trigger in M cycles was actually scheduled for M+N
	// ticks N cycles ago.
	int elapsed = (timers.startcnt - timers.countdown);
	timers.timer[tmr].ctr += elapsed;
    }

    return tmr;
}


// return the number of base ticks left before a timer runs out
// we return 0 if the timer is dead already.
// UNUSED to date
int
TimerRemaining(int n)
{
    int uncredited = (timers.startcnt - timers.countdown);
    int remains;

    if (timers.timer[n].ctr == 0)
	return 0;	// dead timer

    ASSERT(uncredited >= 0);

    remains = timers.timer[n].ctr - uncredited;
    if (remains < 0)
	remains = 0;

    return remains;
}


// kill a timer.  the timer number passed to this function
// is the one returned by the TimerCreate function.
// if we happen to be killing the timer nearest to completion,
// we don't bother messing with updating timers.countdown.
// instead, we just let that countdown expire, nothing will
// be triggered, and a new countdown will be established then.
void
TimerKill(int n)
{
    if (timers.timer[n].ctr == 0) {
	UI_Alert("Error: killing non-existent simulated timer");
	exit(-1);
    }

    timers.timer[n].ctr = 0;
    timers.active_timers--;
}


// transfer accumulated timer deficit to each active timer.
// n is added to the already elapsed time.
// this shouldn't need to be called very frequently.
static void
TimerCredit(uint32 n)
{
    uint32 elapsed;
    int i, flag;

    if ((timers.active_timers == 0) || (n == 0))
	return;	// no timers or no time

    elapsed = n + (timers.startcnt - timers.countdown);
    timers.updating = 1;

    // scan each old timer
    for(i=0; i<NUM_TIMERS; i++) {
	if (timers.timer[i].ctr != 0) {
	    int overshoot = (elapsed - timers.timer[i].ctr);
	    if (overshoot >= 0) {
		// a timer has expired
		timercb_t fcn = timers.timer[i].callback;
		uint32 arg1   = timers.timer[i].arg1;
		uint32 arg2   = timers.timer[i].arg2;
		// destroy timer
		timers.timer[i].ctr = 0;
		timers.active_timers--;
		// call the callback
		fcn(arg1,arg2);
		// ideally, we'd also pass the number of base ticks
		// of overshoot so that a periodic timer could make
		// sure the new period for the next cycle accounted
		// for this small (?) difference.
	    } else {
		// a timer hasn't expired
		timers.timer[i].ctr -= elapsed;
	    }
	}
    }


    // determine the new target time
    flag = 0;
    timers.countdown = 0;
    for(i=0; i<NUM_TIMERS; i++) {
	if (timers.timer[i].ctr != 0) {
	    if (!flag || (timers.timer[i].ctr < timers.countdown)) {
		timers.countdown = timers.timer[i].ctr;
		flag = 1;
	    }
	}
    }
    timers.startcnt = timers.countdown;

    timers.updating = 0;	// done updating
}


// each CPU clock corresponds to n base clocks.  Because all scheduling
// is done in base ticks, we need to scale CPU ticks by some factor to
// get base ticks.

#define MAXTICK 100
static int tickscale[MAXTICK];
static int tscale;

// this macro doesn't work if 't' isn't assignable
#define NORMALIZE_TICKS(t) \
	(((t) < MAXTICK) ? tickscale[(t)] : ((t)*tscale))

// this function establishes the scale factor between base frequency
// and CPU frequency.
void
TimerTickScale(int n)
{
    int i;
    tscale = n;
    for(i=0; i<MAXTICK; i++)
	tickscale[i] = tscale*i;
}


// let 'n' cpu cycles of simulated time go past
void
TimerTick(uint32 n)
{
    uint32 base_n = NORMALIZE_TICKS(n);

#if TEST_TIMER
    g_testtime += base_n;
    if (g_testtime < 100)
	printf("inc %d, time=%d\n", base_n, g_testtime);
#endif

    timer32 += base_n;

    if (timers.countdown > base_n) {

	// this should be the most common case
	timers.countdown -= base_n;

    } else if (timers.countdown > 0) {
// FIXME: timers.countdown is unsigned, so this is really only true
//        if contdown!=0.  what if countdown gets stuck between
//        0 and base_n?  do we keep calling TimerCredit() pointlessly?
//        the comment where countdown is defined indicates at one time
//        it was signed.

	// it is likely that one or more timers has expired.
	// it isn't guaranteed since it is possible that the timer
	// which was nearest to completion was recently TimerKill'd.
	TimerCredit(base_n);

    } // else no active timers
}

