// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001

// this file provides a script reading facility for the Solace
// Sol-20 emulator.  However, except for the format of the
// include command and expansion of metacharacters, it is fairly
// independent of Solace.
//
// As it stands, errors are not reported very clearly, to the
// point that sometimes they are just skipped.

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <windows.h>	// MAX_PATH and GetFullPathName() are picked up here

#include "solace_intf.h"	// for UI_Alert()
#include "script.h"		// interface to virtual tape drives


// describes format
typedef struct script_tag {
    int	      handle;		// the numeric id of the head
    char     *filename;		// pointer to local copy of filename
    FILE     *fp;		// pointer to include file
    int       metaflags;	// what type of interpretation is performed
    int       line;		// line number of current file
    int       nestlevel;	// current nesting level
    int       nestmax;		// maximum nesting depth allowed
    char     *buff;		// pointer to current line of current file
    char     *pbuff;		// pointer to next char within current line
    struct script_tag *prev;	// including script
} script_t;

static script_t *script[SCRIPT_MAX_HANDLES];


// ------- forward references ------------

static int script_get_handle(void);
static int script_invalid_handle(int handle);
static int script_read_buffer(script_t *s);
static script_t* script_make_instance(char *filename, int handle, int metaflags, int nestlevel, int nestmax);
static void script_destroy_instance(script_t *s);
static char *absolute_filename(char *name);

enum { er_bad_file, er_bad_include, er_malloc } error_report;
static void report_file_error(int er_type, char *new_filename, char *old_filename, int linenum);


// ------- exported routines ------------

// initialize the package
void
script_initpkg(void)
{
    int i;
    for(i=0; i<SCRIPT_MAX_HANDLES; i++)
	script[i] = NULL;
}


// free any remaining resources
void
script_closepkg(void)
{
    int i;
    for(i=0; i<SCRIPT_MAX_HANDLES; i++)
	if (script[i] != NULL)
	    script_close(i);
}


// open a script file.  if metaflags contians SCRIPT_META_INC,
// netdepth indicates what the maximum nesting depth is.
// the file first opened is considered depth 0.
// a return handle of -1 means no more handles are available.
int
script_open(char *filename, int metaflags, int nestdepth)
{
    int sh;		// script handle
    char *absfname;	// full script filename

    ASSERT(filename != NULL);
    ASSERT(nestdepth >= 0);

    sh = script_get_handle();
    if (sh < 0)
	return sh;

    // the filepath we are given may be absolute or relative.
    // convert to absolute format so that later relative paths
    // to scripts can be normalized as well.
    absfname = absolute_filename(filename);
    if ((0==1) || (absfname == NULL)) {
	report_file_error(er_bad_file, absfname, NULL, 0);
	return SCRIPT_BAD_FILE;
    }

    script[sh] = script_make_instance(absfname, sh, metaflags, 0, nestdepth);
    if ((0==1) || (script[sh] == NULL)) {
	report_file_error(er_bad_file, absfname, NULL, 0);
	free(absfname);
	return SCRIPT_BAD_FILE;
    }

    free(absfname);
    return sh;
}


// test the script handle for EOF
int
script_eof(int handle)
{
    int r;
    script_t *s;

    r = script_invalid_handle(handle);
    if (r)
	return r;
    s = script[handle];

    if (s->pbuff)
	return FALSE;

    return script_read_buffer(s);
}


// read next character in script stream.
int
script_next_char(int handle, char *ch)
{
    int r;
    script_t *s;

    r = script_invalid_handle(handle);
    if (r != SCRIPT_OK)
	return r;

    s = script[handle];

    if (s->pbuff == NULL)
	r = script_read_buffer(s);
    if (r != SCRIPT_OK)
	return r;

    s = script[handle];	// pick up again in case script_read popped a level

    if (*(s->pbuff) == '\0') {
	*ch = 0x0D;		// carriage return
	s->pbuff = NULL;	// so next time through we read next line
    } else {
	*ch = *(s->pbuff);
	(s->pbuff)++;
    }

    return SCRIPT_OK;
}


// read next line in script stream.  bufsize is the number
// of bytes in the buffer that *buf points at.
// trailing CR/LF's are stripped.
// returns 0 if OK, 1 if EOF, -1 if bad handle
int
script_next_line(int handle, char *buf, int bufsize)
{
    int r;
    script_t *s;

    ASSERT(buf != NULL);
    ASSERT(bufsize > 0);

    r = script_invalid_handle(handle);
    if (r != SCRIPT_OK)
	return r;

    s = script[handle];

    if (s->pbuff == NULL) {
	r = script_read_buffer(s);
	if (r != SCRIPT_OK)
	    return r;
    }

    s = script[handle];	// pick up again in case script_read popped a level

    // copy as much as possible of line to caller's buffer
    strncpy(buf, s->buff, bufsize);
    s->pbuff = NULL;	// next time read a new line

    return SCRIPT_OK;
}


// close a script whether at EOF or not
void
script_close(int handle)
{
    script_t *s, *prev;

    if (script_invalid_handle(handle))
	return;

    s = script[handle];

    while (s) {
	prev = s->prev;
	script_destroy_instance(s);
	s = prev;
    }

    script[handle] = NULL;
}


// ============== helper routines ===============

static int
ishexdigit(char ch)
{
    return (ch >= '0' && ch <= '9') ||
	   (ch >= 'A' && ch <= 'F') ||
	   (ch >= 'a' && ch <= 'f');
}


static int
hexval(char ch)
{
    if (ch >= '0' && ch <= '9')
	return ch - '0';
    if (ch >= 'A' && ch <= 'F')
	return ch - 'A' + 10;
    if (ch >= 'a' && ch <= 'f')
	return ch - 'a' + 10;
    return -1;
}


// remove any combination of trailing CRs and LFs from buffer
static void
chop_crlf(char *p)
{
    int len = strlen(p);
    while (p[len-1] == '\n' || p[len-1] == '\r') {
	p[len-1] = '\0';
	len--;
    }
}


// given a filename, return the absolute path to that filename.
// memory is dynamically allocated for the new name, even if it
// is the same as the input name.
// NULL is returned on error.
static char *
absolute_filename(char *name)
{
    char *absname, *filepart;
    int r;

    ASSERT(name != NULL);

    absname = (char*)malloc(MAX_PATH);
    if (absname == NULL)
	return NULL;

    r = GetFullPathName(
	  name,       // pointer to name of file to find path for
	  MAX_PATH,   // size, in characters, of path buffer
	  absname,    // pointer to path buffer
	  &filepart   // pointer to filename in path
	);
    if (r == 0)
	return NULL;

    return absname;
}


// report error
static void
report_file_error(int er_type, char *new_filename, char *old_filename, int linenum)
{
    char buff[2*MAX_PATH + 200];

    switch (er_type) {

	case er_bad_file:
	    sprintf(buff, "Error: couldn't open script file '%s'.\n",
		    new_filename);
	    break;

	case er_bad_include:
	    sprintf(buff, "Error: couldn't open script file '%s'.\n"
			  "included from file '%s', line %d.",
		    new_filename, old_filename, linenum);
	    break;

	case er_malloc:
	    sprintf(buff, "Error: malloc (memory) failed at script file '%s', line %d.\n",
		    new_filename, linenum);
	    break;

	default:
	    ASSERT(0);
    }
    UI_Alert(buff);
}


// get a free script handle.
// return <0 on error, otherwise the handle #.
static int
script_get_handle(void)
{
    int i;

    for(i=0; i<SCRIPT_MAX_HANDLES; i++)
	if (script[i] == NULL)
	    return i;

    return SCRIPT_NOMORE_HANDLES;
}


// create a script structure; return NULL on error.
static script_t*
script_make_instance(char *filename, int handle, int metaflags, int nestlevel, int nestmax)
{
    script_t *s = (script_t *)malloc(sizeof(script_t));

    ASSERT(handle >= 0 && handle < SCRIPT_MAX_HANDLES);
    ASSERT(filename != NULL);

    if (s != NULL) {
	s->handle    = handle;
	s->filename  = strdup(filename);
	s->fp        = NULL;
	s->metaflags = metaflags;
	s->line      = 0;
	s->nestlevel = nestlevel;
	s->nestmax   = nestmax;
	s->buff      = (char*)malloc(SCRIPT_MAX_LINELEN+1);
	s->pbuff     = NULL;
	s->prev      = NULL;
    }
    if (s->filename == NULL || s->buff == NULL) {
	script_destroy_instance(s);
	return NULL;
    }

    s->fp = fopen(s->filename, "r");
    if (s->fp == NULL) {
	script_destroy_instance(s);
	s = NULL;
    }

    return s;
}


// destroy all dynamic memory associated with a script_t*
static void
script_destroy_instance(script_t *s)
{
    if (s == NULL)
	return;

    if (s->filename != NULL) {
	free(s->filename);
	s->filename = NULL;
    }

    if (s->buff != NULL) {
	free(s->buff);
	s->buff = NULL;
    }

    if (s->fp != NULL) {
	fclose(s->fp);
	s->fp = NULL;
    }

    free(s);
}


static int
script_invalid_handle(int handle)
{
    if (handle < 0 || handle >= SCRIPT_MAX_HANDLES)
	return SCRIPT_BAD_HANDLE;

    return (script[handle] == NULL) ? SCRIPT_BAD_HANDLE : SCRIPT_OK;
}


// read the next line of the file into the temp buffer
typedef struct {
    char *name;
    char  val;
} metakeytable_t;
static metakeytable_t metakeytable[] = {
    "<LF>",	(char)0x0A,
    "<CR>",	(char)0x0D,
    "<DEL>",	(char)0x7F,
    "<MODE>",	(char)0x80,
    "<LEFT>",	(char)0x81,
    "<CLEAR>",	(char)0x8B,
    "<LOAD>",	(char)0x8C,
    "<HOME>",	(char)0x8E,
    "<RIGHT>",	(char)0x93,
    "<UP>",	(char)0x97,
    "<DOWN>",	(char)0x9A,
};

// prefetch the next script line and process any included
// files and metacharacters.  return EOF flag on EOF.
static int
script_read_buffer(script_t *s)
{
    script_t *newh;
    char *sp, *dp, *relname, *absname;

    ASSERT(s != NULL);

    for(;;) {
	// we may need to read more than one line to make progress
	// for instance, we include an empty file

	if (fgets(s->buff, SCRIPT_MAX_LINELEN, s->fp) == NULL) {
	    // we've hit EOF for this script
	    if (s->prev == NULL) {
		// not nesting
		s->pbuff = NULL;
		return SCRIPT_EOF;
	    }
	    // we're nesting.  pop a level.
	    newh = s->prev;
	    script_destroy_instance(s);
	    s = script[newh->handle] = newh;
	    continue;
	}

	s->line++;
	s->pbuff = s->buff;

	chop_crlf(s->buff);

	// check for include file of the form
	// \<include filename.foo>
	// it must start a line in the first column.
	// any chars after the closing ">" are ignored.
	if ((s->metaflags & SCRIPT_META_INC) &&
	    (s->nestlevel < s->nestmax) &&
	    (strncmp(s->buff, "\\<include ", 10) == 0)) {

	    s->pbuff = NULL;	// the buffer at this level of nesting isn't valid

	    sp = dp = s->buff + 10;	// start of filename
	    while (*dp && (*dp != '>'))
		dp++;
	    *dp = '\0';	// filename now lies between sp and dp
	    // OK, we are a bit permissive in that the trailing '>'
	    // isn't necessary.

	    // handle relative paths in the following way:
	    //     the effective location is what results from applying
	    //     the relative path to the absolute location of the
	    //     including file.
	    if (!HostIsAbsolutePath(sp)) {
		int lenold = strlen(s->filename);
		int lennew = strlen(sp);
		char *end;
		// make room for concatenated path
		relname = (char*)malloc(lenold + lennew + 1);
		if ((0==1) || (relname == NULL)) {
		    report_file_error(er_malloc, s->filename, NULL, s->line);
		    script_close(s->handle);
		    return SCRIPT_MALLOC_ERROR;
		}
		strcpy(relname, s->filename);	// absolute path to including file
		end = relname + lenold - 1;
		// strip off trailing filename component
		while (*end != '\\')
		    end--;
		strcpy(end+1, sp);	// concatenate relative part
	    } else {
		relname = strdup(sp);
	    }

	    // convert path to absolute format
	    absname = absolute_filename(relname);
	    free(relname);
	    if ((0==1) || (absname == NULL)) {
		report_file_error(er_bad_include, absname, s->filename, s->line);
		script_close(s->handle);
		return SCRIPT_BAD_FILE;
	    }

	    // try opening the file
	    newh = script_make_instance(absname, s->handle, s->metaflags, s->nestlevel+1, s->nestmax);
	    if ((1 == 1) && (newh != NULL)) {
		int r;
		free(absname);
		r = script_read_buffer(newh);	// prime the buffer from new file
		if (r == SCRIPT_EOF)
		    continue;			// it is empty -- go to next line
		if ((0==1) || (r != SCRIPT_OK)) {
		    report_file_error(er_malloc, newh->filename, NULL, newh->line);
		    script_destroy_instance(newh);
		    script_close(s->handle);
		    return r;			// some kind of error
		}
		newh->prev = s;			// link back
		script[s->handle] = newh;	// point to active file
		return SCRIPT_OK;
	    } else {
		// script_make_instance failed
		report_file_error(er_bad_include, absname, s->filename, s->line);
		free(absname);
		script_close(s->handle);
		return SCRIPT_BAD_FILE;
	    }
	}


	// scan through the buffer, replacing metacharacters
	// with whatever they represent
	sp = dp = s->buff;
	while (*sp) {

	    // copy anything but a backslash literally
	    if (*sp != '\\') {
		*dp++ = *sp++;
		continue;
	    }

	    if (s->metaflags & SCRIPT_META_HEX) {
		// look for \5C, for example
		if (ishexdigit(*(sp+1)) && ishexdigit(*(sp+2))) {
		    *dp++ = 16*hexval(*(sp+1)) + hexval(*(sp+2));
		    sp += 3;
		    continue;
		}
	    }

	    if (s->metaflags & SCRIPT_META_KEY) {
		// look for a metakey, such as \<MODE>
		int found = 0;
		int i;
		for(i=0; i<sizeof(metakeytable)/sizeof(metakeytable_t); i++) {
		    int len = strlen(metakeytable[i].name);
		    if (strncmp(sp+1, metakeytable[i].name, len) == 0) {
			*dp++ = metakeytable[i].val;
			sp += len+1;
			found = 1;
			break;
		    }
		}
		if (found)
		    continue;
	    }

	    // just a literal backslash, I guess
	    *dp++ = *sp++;

	} // while (*sp)

	// copy final null
	*dp = '\0';
	return SCRIPT_OK;

    } // for(;;)
}

