// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000, 2001

// This file contains stuff localized to the core emulator.
// The interface part should not look at any of the internal
// organs which get exposed here.

#ifndef _SOLACE_H_
#define _SOLACE_H_


#include "solace_types.h"
#include "z80.h"
#include "vtape.h"


// z80 state
extern Z80 Z80Regs;


// for use by Z80 emulator
typedef byte rdz80fp(word);
typedef void wrz80fp(word, byte);
typedef word exec8080fp(Z80 *);


// simstate holds things related to running the emulator that aren't
// part of the physical Sol system.  bookkeeping type things live here.
typedef struct {

    enum {
	RESET,		// CPU is in reset
	HALTED,		// emulator is not stepping 8080
	RUNNING,	// 8080 is free running until breakpoint
	STEP,		// step over N 8080 instructions, then -> HALTED
	STEPRUN,	// run until temp breakpoint, then -> STEP
	STEPI,		// step N 8080 instructions, then -> HALTED
    } runstate;
    int stepcount;
    int firststep;
    int inactivate_timer;	// how many slices before disabling dbg window

    exec8080fp *exec8080_func;	// normal or debugging routine

    rdz80fp *diz80_func;	// byte RdZ80(word Addr)
    rdz80fp *opz80_func;	// byte RdZ80(word Addr)
    rdz80fp *rdz80_func;	// byte RdZ80(word Addr)
    wrz80fp *wrz80_func;	// void WrZ80(word Addr, byte Value)
    rdz80fp *inz80_func;	// byte InZ80(word Port)
    wrz80fp *outz80_func;	// void OutZ80(word Port, byte Value)

    int32  timeslice_ms;	// how often we resync
    int32  tick_freq;		// # timer ticks per second
    int32  sol_ticks_per_ms;	// # of 8080 clocks per millisecond
    int32  sol_ticks_per_slice;	// # of 8080 clocks per timeslice
    speed_t up_speed;		// processor speed
    int    throttle;		// 1=slow down 8080 emulation for audio

    int prog_detect;		// # timeslices since PC was outside SOLOS

    int hscrolltimer;		// timer handle for port 0xFE
    int hscanadv;		// timer handle for port 0xFE

    int hkb_stat;		// timer handle for keyboard strobe timer
    int kb_script;		// kb processing suspended
    int kb_sh;			// handle to script handler
    int kb_suppress;		// suppress script input
    int kb_supptimer;		// suppress script input timer

    // virtual tape state
    int vt_realtime;		// boolean: 0=ultrafast, 1=real time
    int vt_wrtimer[2];		// write timer handle
    int vt_rdtimer[2];		// read timer handle

} simstate_t;


// allow memory ranges to be handled by specific modules.
// allow I/O ranges to be handled by specific modules.
typedef byte read_sub_t(word Addr);
typedef void write_sub_t(word Addr, byte Value);

void MapRdRange(int low_addr, int high_addr, read_sub_t fcn);
void MapWrRange(int low_addr, int high_addr, write_sub_t fcn);

void MapInRange(int low_addr, int high_addr, read_sub_t fcn);
void MapOutRange(int low_addr, int high_addr, write_sub_t fcn);

// contains a map per byte of whether there is a special
extern uint32 brkpt_map[2048];	// a bit indicating if anything special is going on
extern byte brkpt_iomap[256];	// is anything special is going on

// there are a number of breakpoints we support
typedef struct {
    break_t btype;	// type of breakpoint
    int     enabled;	// is breakpoint enabled
    int     sernum;	// breakpoint serial #
    int     addr;	// address to break on
    int     data;	// 8 or 16b data value to trigger on
    int     mask;	// 8 or 16b data mask to trigger on
    int     triggered;	// this breakpoint was detected
    int     dlo;	// rd16/wr16 low  datum (-1 means not touched)
    int     dhi;	// rd16/wr16 high datum (-1 means not touched)
} breakpoint_t;

#define NUM_BREAKPOINTS 32	// # of possible breakpoints
extern breakpoint_t breakpoints[NUM_BREAKPOINTS];
extern int breaknum;		// # of defined breakpoints
extern int breaknxt;		// breakpoint serial number


// built-in ROM
extern byte solos_rom[];


// ============== exported by solace.c ================

// really should move this to solace.c and make it private,
// but for now, winmain uses scrollbase stuff frequently for blitting.

typedef struct {

    // OUT 0FEH state:
    int scrolltimer;	// out port 0xFE tweaked within last 250 ms
    int scrollshade;	// how many beginning rows to blank
    int scrollbase;	// which line # is the first displayed
    int scanadv;	// horizontal sync (more or less)

    // this semaphore and register represent the keyboard status and
    // input data latch.  it is filled by winmain and drained by emulator.
    int kb_stat;	// 0=no key, 1=key
    int kb_key;

    byte dipsw1;	// display dip swith
    byte dipsw2;	// sense switches
    byte dipsw3;	// serial port baud rate
    byte dipsw4;	// serial port control bits

    // virtual tape state
    int vt_TTBE;		// tape TX buffer empty
    int vt_writebyte;		// if !TTBE, this is data to write
    int vt_rxready;		// boolean: 1=we already have sample
    int vt_readbyte;		// if rxready, what is data
    int vt_framing;		// if rxready, 1=framing error
    int vt_overrun;		// if rxready, 1=overrun error
    int vt_baud;		// 300 or 1200
    struct {
	int playstate;		// current tape mode
	int motor_on;		// boolean: 1=tape motor is enabled
	int motor_force;	// force motor enable
	vtape_t *tape;		// tape units 0 and 1
    } vt[2];

} sysstate_t;


// this holds state that has a physical correlate in the Sol system
extern sysstate_t sysstate;

// this holds state that is emulator bookkeeping information
extern simstate_t simstate;	// exported because debugger.c needs it

// a breakpoint changed
void bkptchange_coresim(void);


// ============== exported by scheduler.c ================

typedef void (*timercb_t)(uint32 arg1, uint32 arg2);
void TimerInit(void);
int  TimerCreate(uint32 ticks, timercb_t fcn, uint32 arg1, uint32 arg2);
int  TimerRemaining(int n);	// in base ticks, not CPU ticks
void TimerKill(int n);
void TimerTickScale(int n);	// specify # of base ticks per CPU clock
void TimerTick(uint32 n);

// scale a floating point time in microseconds to be an
// argument appropriate for the TimerCreate() function.
#define TIMER_US(f) ((int)(14.318180*(f)+0.5))
#define TIMER_MS(f) ((int)(14318.180*(f)+0.5))

timer_t TimeAbs(void);
timer_t TimeDiff(timer_t oldtime);
uint32  TimeDiffSat(timer_t oldtime);


// ============== exported by Z80.c ================

// z80 state
extern Z80 Z80Regs;


// ============== exported by Debug.c ================

int DAsm(char *S, word A, int extended);
int OpLen(word A);


// ============== exported by debugger.c ================

void breakpoint_init(void);
void breakpoint_genmap(void);
void breakpoint_temp_add(word addr);
void breakpoint_temp_kill(void);
int  breakpoint_test(break_t btype, word addr, word data, int *num);
int  breakpoint_verify(int *num);
int  any_breakpoints(break_t btype);


// ============== exported by fileio.c ================

void read_bin_file(char *filename, int report);
int  create_hexdump(char *filename, int start, int end, byte *memory);
int  create_ihexdump(char *filename, int start, int end, byte *memory);


// ============== exported by audio.c ================

void intaudio_init(void);
void intaudio_samplerate(int rate);
void intaudio_event(int inten);
void intaudio_advance_time(void);


#endif // ifdef _SOLACE_H_
