// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001

// This include file contains the interface between the core Sol
// emulator and the system-dependent interface.  This includes both
// what the core emulator is export and what it expect to interface
// to for an interface (exports and imports, roughly).
//
// No Win32-specific stuff should live in this interface.  In theory,
// this interface is all that is need to build a front-end for some
// other GUI (or even a text-mode interface).

#ifndef _SOLACE_INTF_H_
#define _SOLACE_INTF_H_


#include "solace_types.h"
#include "vtape.h"


// ============== exported by core emulator ================

// Abstract the 8080 register set
typedef enum {
	CPU_REG_A, CPU_REG_F,
	CPU_REG_B, CPU_REG_C,
	CPU_REG_D, CPU_REG_E,
	CPU_REG_H, CPU_REG_L,
	CPU_REG_AF,
	CPU_REG_BC,
	CPU_REG_DE,
	CPU_REG_HL,
	CPU_REG_PC,
	CPU_REG_SP
} cpureg_t;

// read a register or register pair
int CPU_RegRead(cpureg_t reg);

// trigger an interrupt, and specify a reason
void CPU_Break(brkpt_t cause);

// this is the memory image.  parts of it might be ghosted
// by other functions.  this is exposed in this interface
// to allow rapid access by display functions.
extern byte memimage[65536];

// read a byte at the specified 16b address.
// any memory-mapped devices are visible.
byte Sys_ReadMappedByte(word Addr);

// write a byte to the specified 16b address.
// any memory-mapped devices are visible.
void Sys_WriteMappedByte(word Addr, byte Value);

// character generators
extern uint8 charset_6574[];
extern uint8 charset_6575[];
extern uint8 charset_dzonkha[];
extern uint8 charset_devanagri[];

// initialize system state
void Sys_Init(void);	  // state unaffected by reset; called once
void Sys_Reset(int warm);  // state affected by reset
void Sys_LoadDefaultROM(void);

// register keyboard activity
void Sys_Keypress(byte data);

// open a keyboard script file
int Sys_Script(char *filename);

// run the emulator core until the user wants to quit
void Sys_DoTimeSlicing(void);

// slow down emulated 8080 (keeps synchronization with audio rate)
enum { Throttle_NORMAL=0, Throttle_FASTER=1, Throttle_SLOWER=2 };
void Sys_ThrottleScheduler(int throttle);

// the windows interface notifies the core simulator of certain events
// with the debugger
typedef enum {
    DEBUG_KEY = 1,	// halt CPU and start debugger
    RUN_KEY,		// run if halted; halt if running
    FORCERUN_KEY,	// run
    STEP_KEY,		// step over N instructions
    STEPI_KEY,		// step into N instructions
    CLOSE_WIN		// close the debugger and run
} sysnotify_t;
void Sys_DbgNotify(sysnotify_t event, int param);

// set and get various properties
void Sys_SetDipSwitch(int swtch, byte value);
byte Sys_GetDipswitch(int swtch);

// uP speed
void    Sys_Set8080Speed(speed_t clkrate);
speed_t Sys_Get8080Speed(void);

// specify the rate to produce samples from emulator
void Sys_SetAudioSamplerate(int rate);

// machine/program state
int Sys_IsProgramRunning(void);
int Sys_MachineIsReset(void);

// get/set state of indicated virtual tape unit
enum {	TPROP_POSITION	   = VTPROP_POSITION,
	TPROP_IGNOREBAUD   = VTPROP_IGNOREBAUD,
	TPROP_WRITEPROTECT = VTPROP_WRITEPROTECT,
	TPROP_DIRTY	   = VTPROP_DIRTY,
	TPROP_BAUDRATE	   = VTPROP_BAUDRATE,
	TPROP_LABEL   	   = VTPROP_LABEL,
	TPROP_REALTIME=7,
	TPROP_PLAYSTATE=8,
	TPROP_FORCEMOTOR=9 };
void Sys_GetTapeProp(int unit, int prop, int *param);
void Sys_SetTapeProp(int unit, int prop, int param);

enum {VTF_NewTape, VTF_LoadTape, VTF_SaveTape, VTF_EjectTape};
int Sys_TapeFile(int unit, int action, char *filename);

// playstate
enum { EMPTY, STOP, PLAY, FWD, REW, RECORD };


// ============== exported by audio.c ================

// accessor values for sound effects properties
enum {	SFXP_ENABLE=1,
	SFXP_VOL_INTEN=2,
	SFXP_VOL_FAN=3,
	SFXP_VOL_DISK=4
    };

void Sys_AudioSet(int prop, int value);
void Sys_AudioGet(int prop, int *value);


// accessor values for sound effects events
enum {	SFXE_FAN_ON=1,
	SFXE_FAN_OFF,

	SFXE_DISK_STEP,

	SFXE_MOTOR_ON,
	SFXE_MOTOR_OFF,

	SFXE_DISK_DOOR_OPEN,
	SFXE_DISK_DOOR_CLOSE,
    };

// trigger some sound effect
void Sys_AudioEvent(int event);


// ============== exported by Debug.c ================

int DAsm(char *S, word A, int extended);
int OpLen(word A);


// ============== exported by debugger.c ================

int  check_brk(word addr, break_t type);
void dbg_interp(char *str);


// ============== exported by GUI ================

// tell UI that a byte of the display memory changed
void UI_UpdateDisplay(int offset, byte data);

// redraw the contents of the screen
void UI_InvalidateScreen(void);

// change the display character generator
int UI_OutFFTickle(void);

// update the screen after the scrollshade register changes
void UI_UpdateScrollBase(int scrollshade, int scrollbase);

// process one windows message, if one is pending
int UI_HandleMessages(int *done);

// poll the keystate to see if we are still forcing reset
int UI_KBResetState(void);

// from vdisk to gui -- notify gui of disk drive state changes
enum {
    SVD_NOTIFY_EMPTY,		// disk drive is empty
    SVD_NOTIFY_OCCUPIED,	// disk drive is occupied
    SVD_NOTIFY_DISKOFF,		// disk motor is off
    SVD_NOTIFY_DISKON,		// disk motor is on
    SVD_NOTIFY_CONTROLLER	// major state change in controller
};
void UI_DiskNotify(int unit, int state);

// send a passive message to the user for some period of time
void UI_TimedNote(char *msg, int duration);

// send a message about CPU load
void UI_NoteSpeed(char *msg);

// send an error/warning to the user that requires an action to close
void UI_Alert(char *fmt, ...);

// like UI_Alert, but has more buttons and returns status
// about what to do in this situation
enum { DBGFLAG_REPORT=0, DBGFLAG_DEBUG=1, DBGFLAG_IGNORE=2 };
int  UI_DbgAlert(char *fmt, ...);

// ask if something is OK to do or if the request should be cancelled.
// returns TRUE if it is OK to do, FALSE if cancel is requested.
int UI_Confirm(char *fmt, ...);

// emit an audio sample
void UI_CreateAudioSample(float v);


// --- debugger interface functions ---

// add a message to the end of the debugger log window.
// it can contain embedded newlines.
void UI_DbgWinLog(char *str, int bold);

// is debugger window active?
int UI_DbgWinActive(void);

typedef enum {
    DW_Activate,
    DW_ActivateStep,
    DW_Inactivate,
    DW_Close,
    DW_Update
} DbgWinMsg_t;

void UI_DbgWin(DbgWinMsg_t msg, int arg);

// args for UI_DbgWin (used only by Update)
#define UPDATE_REG	0x0001	// registers subwindow
#define UPDATE_MEM	0x0002	// memory subwindow
#define UPDATE_DASM	0x0004	// dasm subwindow
#define UPDATE_LOG	0x0008	// command log subwindow
#define UPDATE_CMD	0x0010	// command entry subwindow
#define HOME_DASM	0x0100	// recenter dasm around pc
#define HOME_MEM	0x0200	// recenter mem around sp

#define UPDATE_REFRESH (UPDATE_REG | UPDATE_MEM | UPDATE_DASM | UPDATE_LOG | UPDATE_CMD)
#define UPDATE_ALL     (UPDATE_REFRESH | HOME_DASM | HOME_MEM)


// -------- not UI issue, but host specific anyway --------

// --- filename functions ---
int   HostIsAbsolutePath(const char *name);
char *HostExtractPath(const char *name);
char *HostMakeAbsolutePath(const char *name);
char *HostUniqueFilename(const char *dir, const char *prefix);

// --- system timer functions ---
int    HostQueryTimebase(void);
int32  HostGetRealTimeMS(void);

#endif // ifdef _SOLACE_INTF_H_
