// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2005

// Common virtual disk access functions.

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "vdisk_svdlib.h"
#include "solace_intf.h"	// required for Host* functions

// information that appears in the first 4K of the file
typedef struct {

    char format[64];	// "magic" file format string
    uint16 version;	// subtype of format
    uint16 writeprot;	// 1=read only disk
    uint16 density;	// single or double density
    uint16 sides;	// sides per disk
    uint16 tracks;	// tracks per side
    uint16 sectors;	// sectors per track
    uint8  pad[1024 - 64 - 6*sizeof(uint16)];
    char label[SVD_LABEL_SIZE];	// diskette label (at offset 1024)

} svd_header_t;


// return a descriptive string to decipher numeric error code
char *
vdisk_errstring(int err)
{
    switch (err) {
	case SVD_OK:		return "everything is OK";
	case SVD_FILEEXISTS:	return "file already exists on disk";
	case SVD_NOFILE:	return "requested file doesn't exist";
	case SVD_BADFILE:	return "bad file format";
	case SVD_OPENERROR:	return "couldn't open file";
	case SVD_ACCESSERROR:	return "couldn't read or write to file";
	case SVD_BADFORMAT:	return "the format of this disk is wrong";
	case SVD_INTERROR:	return "internal program error";
	default:
	    break;
    }
    return "impossible error code";
}


// given a filename, get information about a disk.
// returns SVD_OK if it worked, something else if it didn't.
int
svdlib_read_disk_info( char *filename,		// in
                       int  *filetype,		// the rest are out...
                       int  *sides,
                       int  *tracks,
                       int  *sectors,
                       int  *density,
                       int  *writeprotect,
                       char *label,
                       int  *maxsectorsize)
{
    svd_t svd;
    int stat;

    ASSERT(filename != NULL);

    stat = svdlib_read_file_header(filename, &svd);
    if (stat != SVD_OK)
	return stat;

    *filetype      = SVD_FORMAT_SVN;
    *sides         = svd.sides;
    *tracks        = svd.tracks;
    *sectors       = svd.sectors;
    *density       = svd.density;
    *writeprotect  = svd.writeprot;
    *maxsectorsize = (svd.density == 1) ? SVD_SNG_SECSIZE : SVD_DBL_SECSIZE;
    strcpy(label, svd.label);

    return SVD_OK;
}


// write a disk header to the specified file.
// returns SVD_OK if it worked, something else if it didn't.
// should this be changed so that only the writeprotect
// and the label can be modified?
int
svdlib_write_disk_info( char *filename,
                        int   sides,
                        int   tracks,
                        int   sectors,
                        int   density,
                        int   writeprotect,
                        char *label)
{
    svd_t svd;

    ASSERT(filename != NULL);
    ASSERT(strlen(filename) < sizeof(svd.filename));

    strncpy(svd.filename, filename, sizeof(svd.filename));
    svd.format    = SVD_FORMAT_SVN;
    svd.sides     = sides;
    svd.tracks    = tracks;
    svd.sectors   = sectors;
    svd.density   = density;
    svd.writeprot = writeprotect;
    strncpy(svd.label, label, sizeof(svd.label));

    return svdlib_write_file_header(&svd);
}


// given a filename, read the header into an svd struct
int
svdlib_read_file_header(const char *filename, svd_t *svd)
{
    svd_header_t hdr_block;
    int stat;
    FILE *fd;

    assert(sizeof(svd_header_t) == SVD_HDR_SIZE);

    fd = fopen(filename, "rb");
    if (fd == NULL)
	return SVD_OPENERROR;

    // read header block
    stat = fread((char*)&hdr_block, SVD_HDR_SIZE, 1, fd);
    fclose(fd);
    if (stat != 1)
	return SVD_ACCESSERROR;

    // validate and copy each parameter from the header to the svd_t

    if ( (strcmp(hdr_block.format, SVN_MAGIC_STR) == 0) &&
         (hdr_block.version == 1) ) {
	svd->format = SVD_FORMAT_SVN;
    } else if ( (strcmp(hdr_block.format, SVH_MAGIC_STR) == 0) &&
		(hdr_block.version == 1) ) {
	svd->format = SVD_FORMAT_SVH;
    } else
	return SVD_BADFORMAT;

    svd->writeprot = hdr_block.writeprot;
    svd->density   = hdr_block.density;
    svd->sides     = hdr_block.sides;
    svd->tracks    = hdr_block.tracks;
    svd->sectors   = hdr_block.sectors;
    memset(svd->label, 0, SVD_LABEL_SIZE);		// makes gdb "print svd" more sane
    hdr_block.label[SVD_LABEL_SIZE-1] = '\0';	// just in case
    strcpy(svd->label, hdr_block.label);

    if (hdr_block.writeprot != 0 && hdr_block.writeprot != 1)
	return SVD_BADFORMAT;

    switch (svd->format) {
	case SVD_FORMAT_SVN:
	    if ( (hdr_block.density != 1 && hdr_block.density != 2)
	      || (hdr_block.sides   != 1 && hdr_block.sides   != 2)
	      || (hdr_block.tracks < SVD_MINTRACKS)
	      || (hdr_block.tracks > SVD_MAXTRACKS)
	      || (hdr_block.sectors != NS_SECTORS_PER_TRACK) )
		return SVD_BADFORMAT;
	    break;
	case SVD_FORMAT_SVH:
	    if ( (hdr_block.density   != 1)
	      || (hdr_block.sides     != 1)
	      || (hdr_block.tracks    != SVH_TRACKS_PER_DISK)
	      || (hdr_block.sectors   != SVH_SECTORS_PER_TRACK) )
		return SVD_BADFORMAT;
	    break;
	default:
	    assert(0);
    }

    // save filename so we can open it again later
    strcpy(svd->filename, filename);

    return 0;
}


// modify the header block on the virtual disk without otherwise
// affecting the rest of the virtual disk.  the file must already
// exist as it won't create the file.
int
svdlib_write_file_header(svd_t *svd)
{
    svd_header_t hdr_block;
    int stat, i;
    FILE *fd;

    assert(sizeof(svd_header_t) == SVD_HDR_SIZE);

    assert(svd->filename != NULL);
    assert(strlen(svd->filename) < SVD_FILENAME_SIZE);
    assert(svd->writeprot == 0 || svd->writeprot == 1);
    assert(svd->label != NULL);
    assert(strlen(svd->label) <= SVD_LABEL_SIZE);

    switch (svd->format) {
	case SVD_FORMAT_SVN:
	    assert(svd->sides == 1 || svd->sides == 2);
	    assert(svd->tracks >= SVD_MINTRACKS && svd->tracks <= SVD_MAXTRACKS);
	    assert(svd->density == 1 || svd->density == 2);
	    strcpy(hdr_block.format, SVN_MAGIC_STR);
	    break;
	case SVD_FORMAT_SVH:
	    assert(svd->sides == 1);
	    assert(svd->tracks == SVH_TRACKS_PER_DISK);
	    assert(svd->density == 1);
	    strcpy(hdr_block.format, SVH_MAGIC_STR);
	    break;
	default:
	    assert(0);
    }

    hdr_block.version   = 1;
    hdr_block.writeprot = svd->writeprot;
    hdr_block.density   = svd->density;
    hdr_block.sides     = svd->sides;
    hdr_block.tracks    = svd->tracks;
    hdr_block.sectors   = svd->sectors;
    strcpy(hdr_block.label, svd->label);

    // pad fields with 0x00 bytes
    for(i=strlen(hdr_block.format); i<sizeof(hdr_block.format); i++)
	hdr_block.format[i] = '\0';
    for(i=0; i<sizeof(hdr_block.pad); i++)
	hdr_block.pad[i] = '\0';
    for(i=strlen(hdr_block.label); i<sizeof(hdr_block.label); i++)
	hdr_block.label[i] = '\0';


    fd = fopen(svd->filename, "rb+");	// open for update
    if (fd == NULL)
	return SVD_OPENERROR;

    // write header block
    stat = fwrite((char*)&hdr_block, SVD_HDR_SIZE, 1, fd);
    fclose(fd);
    if (stat != 1)
	return SVD_ACCESSERROR;

    return SVD_OK;
}


// safely as possible, replace the old file with the new one:
//   (1) rename the old file to a temporary name
//   (2) rename the new file with that of the old one
//   (3) delete the old file that now has a temporary name
// both names should be fully qualified; they don't have to be in the
// same directory.
// return SVD_OK if everything came off without a hitch, or an error
// code if things aren't OK.
int
svdlib_replace_file(const char *oldfile, const char *newfile)
{
    char *oldpath = NULL,	// path to oldfile
	 *newpath = NULL,	// path to newfile
         *tempname = NULL;	// name of temporary file
    int rv;

    // create or erase the old file
    {
	char *opath = HostExtractPath(oldfile);
	if (opath == NULL)
	    return SVD_INTERROR;
	oldpath = alloca(strlen(opath) + 1);
	if (oldpath == NULL) {
	    free(opath);
	    return SVD_INTERROR;
	}
	strcpy(oldpath, opath);
	free(opath);
    }

    {
	char *npath = HostExtractPath(newfile);
	if (npath == NULL)
	    return SVD_INTERROR;
	newpath = alloca(strlen(npath) + 1);
	if (newpath == NULL) {
	    free(npath);
	    return SVD_INTERROR;
	}
	strcpy(newpath, npath);
	free(npath);
    }

    // --- step 1: rename oldfile to some temporary name

    {
	char *tname = HostUniqueFilename(oldpath, "svn");
	if (tname == NULL)
	    return SVD_INTERROR;
	tempname = alloca(strlen(tname)+1);
	if (tempname == NULL) {
	    free(tname);
	    return SVD_INTERROR;
	}
	strcpy(tempname, tname);
	free(tname);
	// HostUniqueFilename creates a file that we don't want
	rv = unlink(tempname);
	if (rv != 0)
	    return SVD_ACCESSERROR;
    }

    rv = rename(oldfile, tempname);
    if (rv != 0)
	return SVD_ACCESSERROR;

    // --- step 2: rename the newfile to match the oldfile

    rv = rename(newfile, oldfile);
    if (rv != 0)
	return SVD_ACCESSERROR;

    // --- step 3: delete the old file
    rv = unlink(tempname);

    return (rv == 0) ? SVD_OK : SVD_ACCESSERROR;
}
