// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2005

/*
   This file adds support to Solace for emulating a P.T. Helios floppy
   disk system.  This "driver" can be installed and uninstalled
   dynamically for the convenience of the user.
 */

// FIXME: some places I use "unsigned char", others "byte" -- be consistent

#include <stdio.h>
#include <stdarg.h>	// for varargs
#include <string.h>
#include <io.h>		// for _open() and friends
#include <fcntl.h>	// for _open() and friends
#include <sys\stat.h>	// for _open() and friends
#include <malloc.h>
#include <memory.h>	// for memcpy()

#include "solace.h"
#include "solace_intf.h"
#include "vdisk_svh.h"


// --------------- local macros -----------------

#define ENSURE_TIMER_DEAD(thnd) \
    if ((thnd) >= 0) \
	TimerKill((thnd)); \
    (thnd) = -1;

// there are eight drives in the system
#define MAX_DRIVES (8)
#define FOREACH_DRIVE(i) for((i)=0; (i)<MAX_DRIVES; (i)++)

// ===================================================================
//                       Emulation Layer
// This layer overlays timing information onto the logic layer and
// handles the disk controller emulation.
//
//    - plug in/remove controller
//      (must reset system when this happens)
//
//    - adds simple indexing to four SVD objects so that windows code
//      only deals with the index, not the actual struct representing
//      the disk.
//
//    - adds register interface and timing
//         + register mapping
//         + a sense of time via timers
//
//    - GUI interface:
//           to gui: disk empty/occupied, disk select & motor on/off
//         from gui: set/clear debugging mode (applies to all disks)
//         from gui: empty disk drive
//         from gui: load (new) disk
//         from gui: format (new) disk
//         from gui: get header info (label, write protect, disk format info)
//         from gui: set header info (label, write protect, disk format info)
//
// ===================================================================

/* ===================================================================
		      SW Interface

The controller uses only eight addresses in I/O space and has no
memory footprint.  It occupies I/O ports 0xF0 through 0xF7.

                                         Direction
        I/O Addr  Function               wrt CPU
        --------  ----------------       ---------
          0xF0    Status                 Input
          0xF1    Transfer Command       Output
          0xF2    Unused                 --
          0xF3    Transfer Length, high  Output
          0xF4    Transfer Length, low   Output
          0xF5    Transfer Addr, high    Output
          0xF6    Transfer Addr, low     Output
          0xF7    Drive Command          Output

The ms four bits of "transfer length high" are ignored.

To set up a DMA operation, all the parameters need to be set up and as
a last step, the Transfer Command (0xF1) is given.

The Status register has some sticky bits that latch results from the
most recent DMA operation.  The state persists until the next time that
the "Transfer Address, low byte" is written.

Port 0xF0 (Status) bit assignments:

                             Active
        Bit  Signal Name     State   Significance
        ---  --------------  ------  ------------------
         0   TC              high    transfer complete
         1   SREADY          high    ready
         2   ABORT           high    error
         3   CRC ERROR       high    bad checksum on read
         4   CRC CHECKED     high    check complete
         5   !DISK_READY     low     ready
         6   !SEEK_COMPLETE  low     done
         7   !INDEX          low     index hole present

FIXME: say that the DMA transfer length is 800, but the block encodes
       896 bytes.  when do the different complete signals finish?
       perhaps SREADY goes active first and CRC CHECKED later.  hmm
       but the controller will be preoccuplied until CRC CHECKED is
       done.  when is (CRC_CHECKED != SREADY)?

Some of these bits need more explanation.

    TC is set by the DMA logic after all the requested bytes
    have been transferred, or a ABORT or other error occurs.

    SREADY goes inactive when a transfer command has been issued
    and goes active again only after the command has oompleted.
    
    ABORT occurs when something goes wrong in the requested
    DMA transfer.  There are five possible reasons for this
    to occur: (1) power on; (2) sync error, which occurs when
    an expected SYNC pattern isn't found before the next
    sector begins; (3) fifo over/under run, which can happen
    in real hardware if the S-100 bus doesn't yield to the
    controller's request for bus mastership in time;
    (4) over index, which occurs when a DMA transfer reads
    a data block and sees the sector mark for sector 0, which
    is against the rules -- it could happen, for example if
    bad software requested 500 bytes to be written to sector
    15, requring two sectors (15 and 0) to hold it all;
    (5) missed, which occurs when a pending data transfer
    command hasn't started by the time "crossover" occurs --
    this could happen, for example, if the software doesn't
    issue the read request quickly enough after the header
    read completes.

    CRC ERROR occurs on either a header read or data read
    operation when the checksum saved with the data don't
    agree.  However, this emulation doesn't model the CRC
    bytes because in theory, there should be no way to
    generate a bad block of data (other than if was imported
    with a bad checksum from a real disk).

    CRC COMPLETE ...

    DISK READY active indicates that a diskette is in the selected
    drive and that the motor is up to speed.

    SEEK COMPLETE goes active some time after the requested head
    stepping is done or the RESTORE has found track 0.

    INDEX goes active for 4 ms or so when the disk detects the
    index hole, which appears in the middle of the last of the
    32 hard sectors, which will appear to to the emulation as
    being 3/4 the way through the last of 16 sectors.

Port 0xF1 (Transfer Command) bit assignments:

                             Active
        Bit  Signal Name     State   Significance
        ---  --------------  ------  ------------------
         0   !ERASE          low     erase track
         1   R/!W            high    read
                             low     write
         2   !TR_DATA        high    header
                             low     data
         3   ???             ???     ???   FIXME
         4   unused
         5   unused
         6   unused
         7   unused

The ERASE command needs some explanation.  When this bit is set, the
formatter writes a header and data block to each of the 16 sectors on
the track.  The header will consist of 13 0xFF bytes, and the data will
consist of a single 0xFF byte.  Sectors keep getting formatted until
the signal is removed.  Software should enable this bit and wait until
the INDEX hole is seen twice before removing ERASE to make sure the
entire track is wiped clean.

Software writes a 0xFF to this register to cancel the last command.

Port 0xF7 (Drive Command) bit assignments:

                             Active
        Bit  Signal Name     State   Significance
        ---  --------------  ------  ------------------
         0   !STEP           low     step to next track
         1   !INWARD         low     away from track 0
                             high    toward track 0
         2   !DRIVE_SEL_1    low     select unit 2, 3, 6, or 7
                             high    select unit 0, 1, 4, or 5
         3   !DRIVE_SEL_2    low     select unit 4, 5, 6, or 7
                             high    select unit 0, 1, 2, or 3
         4   !RESTORE        low     send head to track 0
         5   !LOAD_HEAD_0    low     load head of unit 0, 2, 4, or 6
         6   !LOAD_HEAD_1    low     load head of unit 1, 3, 5, or 7
         7   !DISK_SEL_1     low     select unit 1, 3, 5, or 7
                             high    select unit 0, 2, 4, or 6

Another way to figure out which drive is selected is these equations:

    selected unit = { !r[3], !r[2], !r[7] };  // verilog

    selected unit = ((~r >> 1) & 6)   // C
                  | ((~r >> 7) & 1);

   =================================================================== */

// various disk timing parameters
// The disk rotates at 360 RPM,
//     therefore, it is 6 revolutions per second, or 166.7 ms/revolution.
// There are 16 sectors per disk,
//     therefore, it is 10.4167 ms per sector.
// FIXME: the timing model for track stepping will need an overhaul
#define MOTOR_TURN_OFF_DELAY 8000 // in milliseconds
#define MOTOR_TURN_ON_DELAY  1000 // in milliseconds
#define TRACK_STEP_TIME         4 // in milliseconds
#define REVOLUTION_TIME       167 // in milliseconds
#define SECTOR_TIME            10 // in milliseconds

// FIXME: other interesting timing
// p 7-21 of the Helios II manual says that the head load solenoid has
//        a one second one-shot.  The one-shot is reset any time there
//        is a falling edge on the appropriate head load signal, and
//        is cleared after on second of inactivity.
// p 7-25 says that DMA proceeds by grabbing the S-100 bus for 20 uS
//        out of each 390 uS or so, during which time 12 bytes are
//        transferred.

// this contains the state of the controller board.
// because this is statically allocated, there can be only one instance
// of this disk controller in the system, which is a real limitation of
// the Helios system.
static struct {

    // fictitious state:

    int installed;	// indicates if we're installed or not
    int base_addr;	// either 0xF0 (99.99% of the time), or 0xE0 (unusual)
    int drives;		// number of drives on controller (1-4)
    int debug_mode;	// flag

// FIXME: should this be per drive (or per drive pair?)
    int thnd_motor_off;		// timer handle for motor turn-off delay
    int at_speed_flag;		// flag: motor takes time to get up to speed
    int thnd_motor_at_speed;	// timer handle

// FIXME: unnecessary -- part of disk drive state?
    int thnd_sector_inc;	// timer handle: increment sector counter

    // real state:

    // writable (from 8080) state:
    int ctl_cmd_erase;	// 0xF1: erase track
    int ctl_cmd_read;	// 0xF1: 1=read, 0=write
    int ctl_cmd_header;	// 0xF1: 1=transfer header, 0=transfer data

    int ctl_xfer_len;	// 0xF3-0xF4: number of bytes to transfer
    int ctl_xfer_addr;	// 0xF5-0xF6: address to transfer to/from

    int ctl_step;	// 0xF7: step head to next track
    int ctl_step_up;	// 0xF7: direction of step (1=to higher track)
    int ctl_head_0;	// 0xF7: load head 0 of currently selected drive
    int ctl_head_1;	// 0xF7: load head 1 of currently selected drive
    int ctl_restore;  	// 0xF7: seek to track 0
    int ctl_drive_sel;	// 0xF7: which drive is addressed: 0 to 7

    // readonly (from 8080) state:
    int stat_tc;		// transfer complete (successfully or not)
    int stat_sready;		// ready to accept new command
    int stat_abort;		// something has gone wrong
    int stat_crc_error;		// CRC error on header or data block
    int stat_crc_checked;	// ...
    int stat_disk_ready;	// drive is selected and motor up to speed
    int stat_seek_complete;	// after stepping is done or RESTORE found track 0
    int stat_index;		// active 4ms or so during middle of last sector

} ctrlr_state;

// this maintains state for each of the eight possible disk drives
// attached to the controller board.
static struct {

    int occupied;	// 1=occupied, 0=empty
    int on_track;	// flag: head is not between tracks
    int thnd_on_track;	// timer handle for on_track
    int track;		// track counter
    int thnd_sector;	// timer handle for sector mark
    int sector;		// which sector is the disk physically on?
    int sec_time_us;	// sector rotation time, in uS

    svd_t disk;		// if occupied, disk header information

    byte *image;	// disk image, dynamically allocated

} dsk_state[MAX_DRIVES];


// --- forward references ---

void svh_board_reset(void);

static byte handle_IO_In_F0(word Addr);
static void handle_IO_Out_F1(word Addr, byte Value);
static void handle_IO_Out_F3(word Addr, byte Value);
static void handle_IO_Out_F4(word Addr, byte Value);
static void handle_IO_Out_F5(word Addr, byte Value);
static void handle_IO_Out_F6(word Addr, byte Value);
static void handle_IO_Out_F7(word Addr, byte Value);

static void tcb_motor_off(uint32 arg1, uint32 arg2);
static void tcb_motor_at_speed(uint32 arg1, uint32 arg2);
static void tcb_on_track(uint32 arg1, uint32 arg2);
static void tcb_sector(uint32 arg1, uint32 arg2);
static void tcb_seq(uint32 arg1, uint32 arg2);
static void tcb_sec_inc(uint32 arg1, uint32 arg2);

static void hel_new_sector(void);

static void complain(char *fmt, ...);
static void dbgcomplain(int *dbgflag, char *fmt, ...);

// if we have debugging messages on, don't complain too frequently
static int g_complained_recently;
static int g_thnd_complained;


// called once at power up
void
svh_emu_init(void)
{
    int i;

    g_complained_recently = 0;
    g_thnd_complained = -1;

    ctrlr_state.installed     = FALSE;
    ctrlr_state.debug_mode    = FALSE;
    ctrlr_state.drives        = 2;

    ctrlr_state.seqstate            = seq_none;
    ctrlr_state.thnd_seq            = -1;
    ctrlr_state.thnd_motor_off      = -1;
    ctrlr_state.thnd_motor_at_speed = -1;
    ctrlr_state.thnd_sector_inc     = -1;

    ctrlr_state.base_addr = 0xF0;	// default

    FOREACH_DRIVE(i) {
	dsk_state[i].occupied      = FALSE;
	dsk_state[i].on_track      = TRUE;
	dsk_state[i].track         = 0;
	dsk_state[i].thnd_on_track = -1;
	dsk_state[i].thnd_sector   = -1;
	dsk_state[i].sector        = 0;
	// just to keep things interesting, make the disk
	// drives rotate at slightly different speeds
	dsk_state[i].sec_time_us = 1000*SECTOR_TIME + 100*(i-2);

	dsk_state[i].secinfo = NULL;
	dsk_state[i].image   = NULL;

	// UI_DiskNotify(i, SVD_NOTIFY_EMPTY);
    }

    cursec.valid = FALSE;

#if 0
    // for initial debugging -- create a blank virtual disk
    {
	int rv;

	ctrlr_state.debug_mode = TRUE;

#define DISK_NAME "C:\\Jim\\sol\\solace\\boot.svh"
	rv = svh_format(DISK_NAME,
		0,	// 1=write protect
		"blank Helios disk");
	if (rv != SVD_OK)
	    complain("Error creating blank disk; code=%d", rv);

	// put the disk in drive 1 (A)
	svh_insert_disk(1, DISK_NAME);
    }
#endif
}


// install the card in the system
int
svh_install_driver(int base_address)
{
    ASSERT(base_address == 0xF0 || base_address == 0xE0);

    ASSERT(!ctrlr_state.installed);

    // occupy our place in memory
    MapInRange (base_address+0, base_address+0, handle_IO_In_F0);
    MapOutRange(base_address+1, base_address+1, handle_IO_Out_F1);
    MapOutRange(base_address+3, base_address+3, handle_IO_Out_F3);
    MapOutRange(base_address+4, base_address+4, handle_IO_Out_F4);
    MapOutRange(base_address+5, base_address+5, handle_IO_Out_F5);
    MapOutRange(base_address+6, base_address+6, handle_IO_Out_F6);
    MapOutRange(base_address+7, base_address+7, handle_IO_Out_F7);

    ctrlr_state.base_addr = base_address;
    ctrlr_state.installed = TRUE;

    svh_board_reset();

    return SVD_OK;
}


// this gives the driver a chance to clean up, such as flushing
// any buffers to disk.  returning 0 means OK, 1 means request denied.
int
svh_request_uninstall_driver(void)
{
    // FIXME: anything to do?
    //        complain if current command isn't done?
    return 0;
}


// pull the card from the system
int
svh_uninstall_driver(void)
{
    int i;

    ASSERT(ctrlr_state.installed);

    // return address range back to simple memory
    MapInRange (ctrlr_state.base_addr+0, ctrlr_state.base_addr+0, NULL);
    MapOutRange(ctrlr_state.base_addr+1, ctrlr_state.base_addr+1, NULL);
    MapOutRange(ctrlr_state.base_addr+3, ctrlr_state.base_addr+3, NULL);
    MapOutRange(ctrlr_state.base_addr+4, ctrlr_state.base_addr+4, NULL);
    MapOutRange(ctrlr_state.base_addr+5, ctrlr_state.base_addr+5, NULL);
    MapOutRange(ctrlr_state.base_addr+6, ctrlr_state.base_addr+6, NULL);
    MapOutRange(ctrlr_state.base_addr+7, ctrlr_state.base_addr+7, NULL);

    ctrlr_state.installed = FALSE;

    // kill any possible outstanding timers
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_seq);
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_motor_off);
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_motor_at_speed);
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_sector_inc);
    FOREACH_DRIVE(i) {
	ENSURE_TIMER_DEAD(dsk_state[i].thnd_on_track);
	ENSURE_TIMER_DEAD(dsk_state[i].thnd_sector);
    }

    return SVD_OK;
}


// must be called on reset
void
svh_board_reset(void)
{
    if (!ctrlr_state.installed)
	return;

    // fake state:
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_seq);
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_motor_off);
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_motor_at_speed);
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_sector_inc);

    ctrlr_state.at_speed_flag = 0;

    ctrlr_state.index_flag     = FALSE;
    ctrlr_state.sector_counter = 0;

    ctrlr_state.ctl_cmd_erase  = 0;
    ctrlr_state.ctl_cmd_read   = 1;	// seems more passive, but it shouldn't matter
    ctrlr_state.ctl_cmd_header = 0;

    ctrlr_state.ctl_xfer_len   = 0;
    ctrlr_state.ctl_xfer_addr  = 0;

    ctrlr_state.ctl_step       = 0;
    ctrlr_state.ctl_step_up    = 0;
    ctrlr_state.ctl_head_0     = 0;
    ctrlr_state.ctl_head_1     = 0;
    ctrlr_state.ctl_restore    = 0;
    ctrlr_state.ctl_drive_sel  = 0;

    // the drive motor is turned off too, which has other side effects
    tcb_motor_off(0, 0);

    // if nothing else, we start generating fake sector pulses
    hel_new_sector();
}


// --- file local routines ---

// read controller status
static byte
handle_IO_In_F0(word Addr)
{
}

// write controller transfer command
static void
handle_IO_Out_F1(word Addr, byte Value)
{
}

// write transfer length, high byte
static void
handle_IO_Out_F3(word Addr, byte Value)
{
    if (!ctrlr_state.stat_sready) {
	// FIXME: complain
    }
    ctrlr_state.ctl_xfer_len = (ctrlr_state.ctl_xfer_len & 0x00FF)
			     | (Value << 8);
}

// write transfer length, low byte
static void
handle_IO_Out_F4(word Addr, byte Value)
{
    if (!ctrlr_state.stat_sready) {
	// FIXME: complain
    }
    ctrlr_state.ctl_xfer_len = (ctrlr_state.ctl_xfer_len & 0xFF00)
			     | (Value);
}

// write transfer address, high byte
static void
handle_IO_Out_F5(word Addr, byte Value)
{
    if (!ctrlr_state.stat_sready) {
	// FIXME: complain
    }
    ctrlr_state.ctl_xfer_addr = (ctrlr_state.ctl_xfer_addr & 0x00FF)
			      | (Value << 8);
}

// write transfer address, low byte
static void
handle_IO_Out_F6(word Addr, byte Value)
{
    if (!ctrlr_state.stat_sready) {
	// FIXME: complain
    }
    ctrlr_state.ctl_xfer_addr = (ctrlr_state.ctl_xfer_addr & 0xFF00)
			      | (Value);

    // writing to this register also clears the status sticky bits
    // FIXME: is this all?
    ctrlr_state.stat_abort       = 0;
    ctrlr_state.stat_crc_error   = 0;
    ctrlr_state.stat_crc_checked = 0;
}

// write drive command
static void
handle_IO_Out_F7(word Addr, byte Value)
{
    int new_step     = ((~Value >> 0) & 1);
    int new_step_up  = ((~Value >> 1) & 1);
    int new_drivesel = ((~Value >> 1) & 6) | ((~Value >> 7) & 1);
    int new_restore  = ((~Value >> 4) & 1);
    int new_head_0   = ((~Value >> 5) & 1);
    int new_head_1   = ((~Value >> 6) & 1);

    static int dbgflag1=0, dbgflag2=0, dbgflag3=0, dbgflag4=0, dbgflag5=0,
	       dbgflag6=0, dbgflag7=0, dbgflag8=0, dbgflag9=0;

    if (!ctrlr_state.stat_sready) {
	// FIXME: complain
    }

    if ((ctrlr_state.ctl_restore || new_restore) && new_step) {
	// FIXME: complain about stepping while restore in progress
    }

    if (new_restore) {
	// FIXME: restore is a slow operation
	dsk_state[new_drivesel].track = 0;
    FIXME: what other state is affected?  is stat_sready changed?  etc
    SEEK COMPLETE goes active some time after the requested head
    stepping is done or the RESTORE has found track 0.
    }

    if (!ctrlr_state.ctl_step && new_step) {

	// step the head to the next track
	const int maxtrack = 77;

	if (ctrlr_state.ctl_step_up != new_step_up) {
	    dbgcomplain(&dbgflag5, "disk direction changed at same time step command issued");
	    ctrlr_state.ctl_dp = new_dp;
	}
	if (ctrlr_state.ctl_drivesel != new_drivesel)
	    dbgcomplain(&dbgflag6, "disk drive select changed at same time step command issued");

	if (new_step_up) {
	    // step in (away from track 0)
	    dsk_state[new_drivesel].track++;
	    if (dsk_state[new_drivesel].track > maxtrack-1) {
		dsk_state[new_drivesel].track = maxtrack-1;
		dbgcomplain(&dbgflag9, "stepped off end of the disk: track=%d", maxtrack);
	    }
	} else {
	    // step out (towards track 0)
	    // it is legitimate to overstep in this direction
	    if (dsk_state[new_drivesel].track > 0)
		dsk_state[new_drivesel].track--;
	}

#if 0
	// set a step timer for debugging purposes
	ENSURE_TIMER_DEAD(dsk_state[new_drivesel].thnd_on_track);
	dsk_state[new_drivesel].on_track = 0;
	dsk_state[new_drivesel].thnd_on_track =
	    TimerCreate(TIMER_MS(TRACK_STEP_TIME), tcb_on_track, new_drivesel, 0);
#endif
    } // track step requested


    // leave state set
    if (ctrlr_state.ctl_drivesel != new_drivesel) {
	int old = ctrlr_state.ctl_drivesel;
	ctrlr_state.ctl_drivesel = new_drivesel;
	// UI_DiskNotify(old, SVD_NOTIFY_DISKOFF);
	// UI_DiskNotify(new_drivesel, SVD_NOTIFY_DISKON);
    }

    ctrlr_state.ctl_step    = new_step;
    ctrlr_state.ctl_dir_up  = new_step_up;
    ctrlr_state.ctl_head_0  = new_head_0;
    ctrlr_state.ctl_head_1  = new_head_1;
    ctrlr_state.ctl_restore = new_restore;
}


/* -------------------------------------------------------------------
     The following note was lifted more or less verbatim from
     NorthStar document "Micro-disk system MDS-A-D Double Density"

     Case 1: Write byte of data (typically E900H)

      +----+----+----+----+----+----+----+----+
      |                 Data                  |
      +----+----+----+----+----+----+----+----+

     Write a byte of data to the disk.  Wait if the write shift
     register is not empty.  The low order 8 bits specify the
     byte to be written.
   ------------------------------------------------------------------- */
static byte
wdata_rdhandler(word Addr)
{
    const int floatbus = 0xFF;
    const int off = Addr & 0xFF;
    const int wrdata = off;
    const int maxbytes = (ctrlr_state.ctl_density) ? SVD_DBL_SECSIZE
						   : SVD_SNG_SECSIZE;
    static int dbgflag1=0, dbgflag2=0;

    if ( (ctrlr_state.seqstate != seq_write) ||
         !ctrlr_state.write_flag) {
	dbgcomplain(&dbgflag1, "Attempting to write data to disk at the wrong time.");
	return floatbus;
    }

    if (cursec.valid) {
	// FIXME: include timing effects
	if (cursec.off > maxbytes) {
	    dbgcomplain(&dbgflag2, "Error: writing off end of sector on disk %d, side %d, track %d, sector %d",
		    cursec.drive, cursec.side, cursec.track, cursec.sector);
	    cursec.off--;
	}
	cursec.data[cursec.off++] = wrdata;	// write to sector
	cursec.dirty = TRUE;
    }

    return floatbus;
}


/* -------------------------------------------------------------------
     The following note was lifted more or less verbatim from
     NorthStar document "Micro-disk system MDS-A-D Double Density"

     Case 2: Controller Orders (typically EA00H)

      +----+----+----+----+----+----+----+----+
      | DD | SS | DP | ST |        DS         |
      +----+----+----+----+----+----+----+----+

     Load 8-bit order register from low order 8 address bits.

     DD  Controls density on write.
         DD=1 for double density.
         DD=0 for single density.

     SS  specifies the side of a double-sided diskette.  The
         bottom side (and only side of a single-sided diskette)
         is selecte when SS=0.  The second (top) side is
         selected when SS=1.

     DP  has shared use.  During stepping operations, DP=0
         specifies a step out and DP=1 specifies a step in.
         During write operations, write precompensation is
         invoked if and only if DP=1.

     ST  controls the level of the head step signal to the disk
         drives.

     DS  is the drive select field, encoded as follows

            0=no drive selected
            1=drive 1 selected
            2=drive 2 selected
            4=drive 3 selected
            8=drive 4 selected
   ------------------------------------------------------------------- */
static byte
orders_rdhandler(word Addr)
{
    int off = Addr & 0xFF;
    int new_density = (off >> 7) & 1;
    int new_side    = (off >> 6) & 1;
    int new_dp      = (off >> 5) & 1;
    int new_step    = (off >> 4) & 1;
    int new_drivesel;
    static int dbgflag1=0, dbgflag2=0, dbgflag3=0, dbgflag4=0, dbgflag5=0,
	       dbgflag6=0, dbgflag7=0, dbgflag8=0, dbgflag9=0;

    // normally no orders should be given in the middle of writing
    // a sector.  flush out any data already written just in case.
    flush_cur_sector();
    cursec.valid = FALSE;

    if (off & 0x01) {
	new_drivesel = 1;
    } else if (off & 0x02) {
	new_drivesel = 2;
    } else if (off & 0x04) {
	new_drivesel = 3;
    } else if (off & 0x08) {
	new_drivesel = 4;
    } else
	new_drivesel = 0;

    if (ctrlr_state.debug_mode &&
	((off & (off-1) & 0x0F) != 0x0))
	dbgcomplain(&dbgflag1, "More than one drive select bit on: 0x%x", off & 0xF);

    if (ctrlr_state.seqstate == seq_write) {
	if (ctrlr_state.ctl_density != new_density)
	    dbgcomplain(&dbgflag2, "changing density in the middle of writing a sector");
	if (ctrlr_state.ctl_dp != new_dp)
	    dbgcomplain(&dbgflag3, "changing write precompensation in the middle of writing a sector");
	if (ctrlr_state.ctl_side != new_side)
	    dbgcomplain(&dbgflag4, "changing side select in the middle of writing a sector");
    }

    if (!ctrlr_state.ctl_step && new_step) {
	// step the head to the next track

	if ( (ctrlr_state.ctl_dp != new_dp) &&
	    (!ctrlr_state.write_flag)) {
	    dbgcomplain(&dbgflag5, "disk direction changed at same time step command issued");
	    // it turns out that my CBIOS actually does this and it
	    // seems to work.  therefore the direction bit used must
	    // be the new one.
	    ctrlr_state.ctl_dp = new_dp;
	}
	if (ctrlr_state.ctl_drivesel != new_drivesel)
	    dbgcomplain(&dbgflag6, "disk drive select changed at same time step command issued");

	if (new_drivesel == 0) {
	    dbgcomplain(&dbgflag7, "stepping without selecting a disk drive");
	} else {
	    // step the head
	    int maxtrack = dsk_state[new_drivesel].disk.tracks;
	    ASSERT(maxtrack < 96);  // just make sure it is reasonable

	    if (!dsk_state[new_drivesel].on_track)
		dbgcomplain(&dbgflag8, "disk head is stepping too fast");

	    if (!ctrlr_state.ctl_dp) {
		// step out (towards track 0)
		if (dsk_state[new_drivesel].track > 0)
		    dsk_state[new_drivesel].track--;
	    } else {
		// step in (away from track 0)
		dsk_state[new_drivesel].track++;
		if (dsk_state[new_drivesel].track > maxtrack-1) {
		    dsk_state[new_drivesel].track = maxtrack-1;
		    dbgcomplain(&dbgflag9, "stepped off end of the disk: track=%d", maxtrack);
		}
	    }

	    // set a step timer for debugging purposes
	    ENSURE_TIMER_DEAD(dsk_state[new_drivesel].thnd_on_track);
	    dsk_state[new_drivesel].on_track = 0;
	    dsk_state[new_drivesel].thnd_on_track =
		TimerCreate(TIMER_MS(TRACK_STEP_TIME), tcb_on_track, new_drivesel, 0);

	} // new_drivesel != 0

    } // track step requested


    // leave state set
    if (ctrlr_state.ctl_drivesel != new_drivesel) {
	int old = ctrlr_state.ctl_drivesel;
	ctrlr_state.ctl_drivesel = new_drivesel;
	// UI_DiskNotify(old, SVD_NOTIFY_DISKOFF);
	// UI_DiskNotify(new_drivesel, SVD_NOTIFY_DISKON);
    }
    ctrlr_state.ctl_step    = new_step;
    ctrlr_state.ctl_dp      = new_dp;
    ctrlr_state.ctl_side    = new_side;
    ctrlr_state.ctl_density = new_density;


    // any random value will do as it isn't supposed to return
    // anything of importance.  we'll assume it is undriven and
    // that pullups on the bus cause a 0xFF to be read.
    return 0xFF;
}


/* -------------------------------------------------------------------
     The following note was lifted more or less verbatim from
     NorthStar document "Micro-disk system MDS-A-D Double Density"

     Case 3: Controller Command (typically EB00H)

      +----+----+----+----+----+----+----+----+
      |        DM         |        CC         |
      +----+----+----+----+----+----+----+----+

     Perform a disk controller command.  The commands are
     specified by the 8 low order address bits.

     DM   The DM field controls what gets multiplexed onto the
          DI bus during the command.

             1=A-status
             2=B-status
             3=C-status
             4=Read data (may enter wait state)

     CC   Command code.

             0=no operation
             1=reset sector flag
             2=disarm interrupt
             3=arm interrupt
             4=set body (diagnostic)
             5=turn on drive motors
             6=begin write
             7=reset controller, deselect drives, stop motors

     DISK CONTROLLER STATUS BYTES

     There are three status bytes that can be read on the Data Input
     Bus.

         A-Status
              +----+----+----+----+----+----+----+----+
              | SF | IX | DD | MO | WI | RE | SP | BD |
              +----+----+----+----+----+----+----+----+

     SF   Sector Flag: set when sector hole detected, reset by
          software.

     IX   Index Detect: true if index hole detected during previous
          sector.

     DD   Double Density Indicator: true if data being read is encoded
          in double density.

     MO   Motor On: true while motor(s) are on.

     WI   Window: true during 96-microsecond window at beginning of
          sector.

     RE   Read Enable: true while phase-locked loop is enabled.

     BD   Body: set when sync character(s) is detected.

     SP   Spare: reserved for future use.


         B-Status
              +----+----+----+----+----+----+----+----+
              | SF | IX | DD | MO | WR | SP | WP | T0 |
              +----+----+----+----+----+----+----+----+

     SF, IX, DD, MO, SP: same as A-Status

     WR   Write: true during valid write operation.

     WP   Write Protect: true while the diskette installed in the
          selected drive is write protected.

     T0   Track 0: true if selected drive is at sector 0.
            [ NOTE: I think this is in error: it is true if the
                    head for the selected drive is at track 0,
                    not sector 0.  This isn't based on any
                    experiment, just the name and reason.  ]

         C-Status
              +----+----+----+----+----+----+----+----+
              | SF | IX | DD | MO |        SC         |
              +----+----+----+----+----+----+----+----+

     SF, IX, DD, MO: same as A-Status

     SC   Sector Counter: indicates the current sector position.
   ------------------------------------------------------------------- */
static byte
command_rdhandler(word Addr)
{
    int off = Addr & 0xFF;
    int cc = (off >> 0) & 7;	// bits [2:0] are a command
    int dm = (off >> 4) & 7;	// bits [6:4] select status read back
    int drv = ctrlr_state.ctl_drivesel;
    int top4bits, retval, writeprot_flag, track0_flag;
    int readdata;
    static int dbgflag1=0, dbgflag2=0;

    // note: if a command has a side effect, that effect will be
    //       visible to any status returned by that read.
    switch (cc) {

	case 0:	// no op
	    break;

	case 1:	// reset sector flag
	    ctrlr_state.sector_flag = FALSE;
	    break;

	case 2:	// disarm interrupt
	    ctrlr_state.intarmed_flag = FALSE;
	    // FIXME: (if Solace ever supports interrupts)
	    // if sector_flag is true at this time, we remove interrupt
	    // request.  emulating this is essential if interrupts are
	    // used, as the sector_flag may be set a long time.
	    break;

	case 3:	// arm interrupt
	    ctrlr_state.intarmed_flag = TRUE;
	    // FIXME: (if Solace ever supports interrupts)
	    // if sector_flag is true at this time, assert interrupt request.
	    // emulating this seems less important since there is a race
	    // between this flag getting set and the sector hole appearing.
	    // it is better to just detect the conjunction when the sector
	    // flag is set.
	    break;

	case 4:	// set body
	    ctrlr_state.body_flag = TRUE;
	    break;

	case 5:	// turn on drive motors
	    if (!ctrlr_state.motor_flag) {
		// for each inserted disk, start up its sector timer
		int i;
		FOREACH_DRIVE(i) {
		    if (dsk_state[i].occupied) {
			ASSERT(dsk_state[i].thnd_sector < 0);
			dsk_state[i].thnd_sector =
			    TimerCreate(TIMER_US(dsk_state[i].sec_time_us), tcb_sector, i, 0);
		    }
		}
		if (ctrlr_state.motor_flag == FALSE) {
		    ctrlr_state.motor_flag = TRUE;
		    // UI_DiskNotify(ctrlr_state.ctl_drivesel, SVD_NOTIFY_DISKON);
		}
	    }
	    ENSURE_TIMER_DEAD(ctrlr_state.thnd_motor_off);
	    ctrlr_state.thnd_motor_off = TimerCreate(TIMER_MS(MOTOR_TURN_OFF_DELAY), tcb_motor_off, 0, 0);
	    if (!ctrlr_state.at_speed_flag) {
		ENSURE_TIMER_DEAD(ctrlr_state.thnd_motor_at_speed);
		ctrlr_state.thnd_motor_at_speed = TimerCreate(TIMER_MS(MOTOR_TURN_ON_DELAY), tcb_motor_at_speed, 0, 0);
	    }
	    break;

	case 6:	// begin write
	    // this command is ignored unless we are in the 96 us window
	    // after a sector hole
	    if (ctrlr_state.window_flag)
		ctrlr_state.write_flag = TRUE;
	    break;

	case 7:	// reset controller, deselect drives, stop motors
	    svh_board_reset();
	    break;
    }


    // the readout mux is weird.  the top four bits are chosen
    // based on bit [6] of the address (=1 means data, =0 means status).
    // the bottom four bits are simply based on [5:4] of the address.

    if (dm & 4) {	// do byte read

	// in real life there are no side effects to reading the data,
	// but how can this be?  what keeps us from double reading the
	// same byte?  "read proceed" is ultimately what determines the
	// stall signal (!PRDY) whenever the read data register is
	// accessed.  read proceed is active only when all 8 bits of a
	// byte are in their proper alignment in the shift register.
	// as soon as another bit comes in off the floppy disk, the PRDY
	// signal is de-asserted again.  thus, if the read_data register
	// is accessed too quickly, we can indeed get the same byte twice.
	// what is the bit time?  "RefBitClk is nominally 250 KHz (page 6),
	// this gets divided by 2 by LS74 3D (page 5), so it would appear
	// that BITCK is 125 KHz, or 8 uS.  Apparently this is single
	// density, and it matches well with the back-of-the-envelope
	// bit clock estimate earlier.
	//
	// At a 2 MHz CPU frequency, the bit clock is 16 CPU clocks
	// single density, 8 CPU clocks double density.  Assuming only
	// the fastest possible instructions (4 clocks per), this means
	// we can't do two reads any faster than 4 instructions.  Because
	// we must compute a checksum and do other overhead, there is no
	// danger of that.  Phew, it works.

	if ((ctrlr_state.seqstate != seq_body) || !cursec.valid) {
	    readdata = 0xCA;	// gibberish (nothing but ca-ca)
	} else {
	    // FIXME: include timing effects
#if 0
if (cursec.off == 50)
 printf("Reading disk=%d, side=%d, track=%d, sector=%d, density=%d\n",
	cursec.drive, cursec.side, cursec.track, cursec.sector, cursec.density);
#endif
	    readdata = cursec.data[cursec.off++];
	    if (cursec.off > cursec.num_bytes) {
		dbgcomplain(&dbgflag1, "Error: reading off end of sector on disk %d, side %d, track %d, sector %d",
			cursec.drive, cursec.side, cursec.track, cursec.sector);
		cursec.off--;
	    }
	}
	top4bits = readdata & 0xF0;

    } else {
	// A, B, and C status share these common bits
	top4bits = (ctrlr_state.sector_flag  << 7)
		 | (ctrlr_state.index_flag   << 6)
		 | (ctrlr_state.density_flag << 5)
		 | (ctrlr_state.motor_flag   << 4);
    }

    switch (dm&3) {
	case 1:	// A-status
	    retval = top4bits
		   | (ctrlr_state.window_flag << 3)
		   | (ctrlr_state.readen_flag << 2)
		   | (ctrlr_state.head_flag   << 1)
		   | (ctrlr_state.body_flag   << 0);
	    break;

	case 2: // B-status
	    if (drv == 0) {
		track0_flag    = FALSE;	// no drive selected, so line undriven
		writeprot_flag = FALSE;	// ditto
	    } else {
		track0_flag = (dsk_state[drv].track == 0);
		if (dsk_state[drv].occupied)
		    writeprot_flag = dsk_state[drv].disk.writeprot;
		else
		    writeprot_flag = FALSE;
	    }
	    retval = top4bits
		   | (ctrlr_state.write_flag << 3)
		   | (0                      << 2)	// spare
    // on NS schematic, it is hooked up to "SPARE/RDY on p4.
    // this in turn comes from an inverted version of pin 6 of the
    // floppy controller cable.  however, some systems use it as
    // DS3 (disk select 3).  I've also seen it called "EDENSEL"; not
    // sure what that means.
		   | (writeprot_flag         << 1)
		   | (track0_flag            << 0);
	    break;

	case 3: // C-status
	    retval = top4bits
		   | ctrlr_state.sector_counter;
	    break;

	case 0: // read disk data (commonly dm==4 here)
	    if (~dm & 4)
		dbgcomplain(&dbgflag2,
			    "reading bad command/data hybrid, PC=%04X",
			    CPU_RegRead(CPU_REG_PC));
	    retval = top4bits | (readdata & 0xF);
	    break;
    }

    return retval;
}


// this routine is activated after the disk has not had a "motor-set" command
// for N seconds.
static void
tcb_motor_off(uint32 arg1, uint32 arg2)
{
    int i;
    int old_drive = ctrlr_state.ctl_drivesel;

    ctrlr_state.thnd_motor_off  = -1;		// disabled
    ctrlr_state.motor_flag      = FALSE;	// turn off motor
    ctrlr_state.at_speed_flag   = FALSE;

    // on the NS schematic, page 1, the order register is
    // cleared when the motor stops
    ctrlr_state.ctl_drivesel = 0;
    ctrlr_state.ctl_step     = 0;
    ctrlr_state.ctl_dp       = 0;
    ctrlr_state.ctl_side     = 0;
    ctrlr_state.ctl_density  = 0;

    ENSURE_TIMER_DEAD(ctrlr_state.thnd_motor_at_speed);

    FOREACH_DRIVE(i) {
	ENSURE_TIMER_DEAD(dsk_state[i].thnd_sector);
    }

    // UI_DiskNotify(old_drive, SVD_NOTIFY_DISKOFF);
}


// disk motor is up to speed
// this isn't per-disk because all disk motors are energized
// together, independently of the disk select.
static void
tcb_motor_at_speed(uint32 arg1, uint32 arg2)
{
    ctrlr_state.thnd_motor_at_speed = -1;
    ctrlr_state.at_speed_flag       = TRUE;
}


// disk head has settled on new track
static void
tcb_on_track(uint32 arg1, uint32 arg2)
{
    ASSERT(arg1 >= 1 && arg1 <= 4);

    dsk_state[arg1].on_track      =  1;
    dsk_state[arg1].thnd_on_track = -1;
}


/*
new_sector active forces a counter reload to step (1) below:

    1)  =  0000_0000_0100_0000 :     0 uS, WINDOW->1, RE->0, HE->0, BODY->0
    2)  =  0000_0001_0000_0000 :    96 uS, WINDOW->0 (cc7tc)
    3)  =  0000_0100_0000_0000 :   480 uS, RE->1 if any data rx'd
    4)  =  0000_1000_0000_0000 :   992 uS, RE->1 no matter what
    5)  =  0000_??01_0000_0000 :  +256 uS after RE->1, HE->1
    6) >=  1000_0000_0000_0000 : 15872 uS, new_sector asserted if hole
    7)  = 10000_0000_0000_0000 : 32736 uS, new_sector is forced active

when new_sector is asserted, the sector counter is incremented.
the sector counter is cleared to 0 when index is asserted, which
lasts for one 2 MHz cycle on the trailing edge of HOLE* when the
index is detected (hold & counter_msb_is_0).

WINDOW = 96 interval after SECTOR hole has been seen.
         (from start of HOLE)
         FIGURE OUT HOW LONG HOLE LASTS IF IT MATTERS.

RE = read enable, goes active roughly 480 uS after start of sector,
     or 384 uS after end of WINDOW.  The point of this is to start
     looking for the sync byte after the PLL has had the chance to
     track the data pulses for a while.  This prevents spurious data
     pulses at the start of the sector where we might accidentally
     recognize them as the sync character.  384 uS is 6 byte times
     in SD, 12 byte times in DD.

there are TWO sync bytes in DD?  How does one get swallowed?

HE = hunt enable, it means the read circuit is a few bytes out of
     the !WINDOW state and is presumably locked, and is looking for
     the 0xFB sync pattern.

BODY = goes active after sync byte has been read during read
       operation.  normally this is 17 (single density) or 34
       bytes (double density) after end of WINDOW.

Note that sector count is cleared when INDEX is seen (which is nominally
10 ms into sector 9), and it is incremented about 16 mS into the sector
(which is 20 mS long).  Also, there is no hardware interlock to prevent
reading the value just as it is changing.  Thus it is important to read
it only at a safe time, such as during WINDOW active.

*/

// this is called when a new sector hole is detected
static void
hel_new_sector(void)
{
    // when the sector hole is detected, a number of flags get set/cleared
    // this also happens when it has been too long since the last time
    // we saw a sector hole.
    ctrlr_state.sector_flag  = TRUE;

    // wait for WINDOW interval
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_seq);
    ctrlr_state.seqstate = seq_window;
    ctrlr_state.thnd_seq =
	TimerCreate(TIMER_US(T_DATA_WINDOW), tcb_seq, 0, 0);

    // bump the sector counter about 16 ms into the sector
    ENSURE_TIMER_DEAD(ctrlr_state.thnd_sector_inc);
    ctrlr_state.thnd_sector_inc =
	TimerCreate(TIMER_US(T_INC_SECTOR), tcb_sec_inc, 0, 0);
}


// increment the sector counter as appropriate
static void
tcb_sec_inc(uint32 arg1, uint32 arg2)
{
    if (!ctrlr_state.index_flag) {
	// index flag does async clear of sector counter
	ctrlr_state.sector_counter =
	    (ctrlr_state.sector_counter + 1) & 0xF;
    }

    ctrlr_state.thnd_sector_inc = -1;
}


// handle sequencer notifications
static void
tcb_seq(uint32 arg1, uint32 arg2)
{
    int sync_found, i, time, skipped;
    static int dbgflag1=0, dbgflag2=0, dbgflag3=0, dbgflag4=0;
    int bad_preamble;

    ctrlr_state.thnd_seq = -1;	// make sure it is dead

    switch (ctrlr_state.seqstate) {

	case seq_window:
	    // end of WINDOW (first 96 uS of sector)
	    ASSERT(ctrlr_state.window_flag == TRUE);
	    ctrlr_state.window_flag = FALSE;	// end of initial window
	    ctrlr_state.index_flag  = FALSE;	// cleared when WINDOW falls
	    // see if the write_flag has been set during the WINDOW interval
	    if (ctrlr_state.write_flag) {
		if (!cursec.valid)
		    dbgcomplain(&dbgflag1, "attempting to write to invalid drive %d", cursec.drive);
		// ignore whatever preread claims the density is: we set it
		// note: cursec uses 1/2 for density, while ctl_density is 0/1
		if (cursec.density != ctrlr_state.ctl_density+1)
		    cursec.num_bytes = 0;	// don't just overwrite -- wipe sector
		cursec.density   = ctrlr_state.ctl_density + 1;
		cursec.byte_time = (ctrlr_state.ctl_density == 1) ? T_SD_BYTE
								  : T_DD_BYTE;
		// set state to begin writing at start of temp sector buffer
		// write an initial 00 byte per spec NS spec
		cursec.data[cursec.off] = 0x00;
		cursec.off++;
		ctrlr_state.seqstate = seq_write;
		ctrlr_state.thnd_seq = TimerCreate(TIMER_US(cursec.byte_time), tcb_seq, 0, 0);
	    } else {
		// now wait a few bytes for the preamble to get PLL in phase
		ctrlr_state.seqstate = seq_read;
		// FIXME: in theory, this time depends if any data is found
		//        or not if nothing else, if no drive is selected or
		//        the drive is unoccupied, then pick slower time.
		ctrlr_state.thnd_seq =
		    TimerCreate(TIMER_US(T_DATA_READ-T_DATA_WINDOW), tcb_seq, 0, 0);
	    }
	    break;

	case seq_read:
	    // the PLL is presumably in sync by now.

	    // according to p.34 of the n* doc, software shouldn't
	    // rely upon the DD bit until RE has gone true.
	    ctrlr_state.density_flag = (cursec.density == 2);
	    ctrlr_state.readen_flag  = TRUE;

	    // wait 256 uS to begin header mode (hunt enable)
	    ctrlr_state.seqstate = seq_head;
	    ctrlr_state.thnd_seq = TimerCreate(TIMER_US(T_SYNC_HUNT-T_DATA_READ), tcb_seq, 0, 0);
	    break;

	case seq_head:
	    // wait for sync -- how long depends on data in sector
	    ctrlr_state.head_flag = TRUE;	// looking for sync byte(s)
	    sync_found = FALSE;

	    if (ctrlr_state.ctl_drivesel > 0 &&
		dsk_state[ctrlr_state.ctl_drivesel].occupied &&
		cursec.valid) {

		bad_preamble = 0;
		for(i=0; i<20*cursec.density; i++) {
		    skipped = i;
		    cursec.off++;
		    if (cursec.data[i] == 0x00)
			continue;
		    if (cursec.data[i] != SYNCBYTE) {
			// wasn't a valid preamble byte
//FIXME: if there is a problem, we end up complaining each revolution
//       instead of only when the sector is read.  is this a problem?:
			if (!bad_preamble) {
			    dbgcomplain(&dbgflag2, "Error: invalid sync byte in preamble on disk %d, side %d, track %d, sector %d",
				cursec.drive, cursec.side, cursec.track, cursec.sector);
			    bad_preamble = 1;
			}
			continue;
		    }
		    if (cursec.density == 2) {
			// double density has two sync bytes
			i++;
			cursec.off++;
			if (cursec.data[i] != SYNCBYTE)
			    break;
		    }
		    sync_found = TRUE;
		    break;
		}
		if (!sync_found) {
		    dbgcomplain(&dbgflag3, "Error: no sync byte on disk %d, side %d, track %d, sector %d",
			    cursec.drive, cursec.side, cursec.track, cursec.sector);
		} else if (skipped < 14*cursec.density) {
		    dbgcomplain(&dbgflag4, "Error: insufficient preamble on disk %d, side %d, track %d, sector %d",
			    cursec.drive, cursec.side, cursec.track, cursec.sector);
		    skipped = 14*cursec.density;
		    // otherwise real data appears before BODY can be set
		}
		time = skipped*cursec.byte_time;
	    } else {
		// no disk data, so assume we waited max time looking for sync
		time = 20*T_SD_BYTE;	// 20 bytes SD (40 bytes DD)
	    }
	    // we for "skipped" bytes after WINDOW ended, but since we've
	    // already waited some period of time since WINDOW ended,
	    // subtract it out.
	    time -= (T_SYNC_HUNT-T_DATA_WINDOW);
	    ASSERT(time > 0);

	    ctrlr_state.seqstate = seq_body;
	    // the time waiting from here ignores time "already served" since end of WINDOW
	    ctrlr_state.thnd_seq = TimerCreate(TIMER_US(time), tcb_seq, 0, 0);
	    break;

	case seq_body:
	    // we are reading the user data portion of the sector
	    ctrlr_state.body_flag = TRUE;
	    // we don't schedule any further events and wait for
	    // a call to hel_new_sector() to get things going again
	    break;

	case seq_write:
	    // we are write the data portion of the sector
	    // we don't schedule any further events and wait for
	    // a call to hel_new_sector() to get things going again
	    break;

	case seq_none: // here just to catch errors
	default:
	    ASSERT(0);
	    break;
    }
}


// this routine handles advancing the sector counter for each of the
// physical disks.  note that this is potentially different from the
// sector counter on the controller.  in effect, this is just feeding the
// sector hole pulses to the controller for the currently selected disk.
//
// for most sectors, the sequence is:
//    time 0:                    start of sector
//    time 20 ms (sec_time_us):  start new sector
//
// for the last sector of the disk, the timing is a bit different
// because the index hole appears in the middle of the sector:
//    time 0:                     start of sector
//    time 10 ms (sec_time_us/2): set IX flag, clear sector counter
//    time 20 ms (sec_time_us):   start new sector
//
// arg1 contains the disk drive number.
static void
tcb_sector(uint32 arg1, uint32 arg2)
{
    int drive    = arg1;
    int do_index = arg2;

    int sectors_per_track = dsk_state[drive].disk.sectors;
    int selected = (ctrlr_state.ctl_drivesel == drive);
    int next_time;

    ASSERT(drive >= 1 && drive <= 4);

    dsk_state[drive].thnd_sector = -1;	// make sure timer is dead

    if (!dsk_state[drive].occupied)	// someone removed the disk?
	return;

    next_time = dsk_state[drive].sec_time_us;	// sector rotation time

    if (do_index) {

	// we are in the middle of sector (N-1).
	// perform the index hole side effects. now it is only
	// half a sector time until the next sector pulse.
	if (selected) {
	    // perform index hole side effects
	    ctrlr_state.index_flag     = TRUE;
	    ctrlr_state.sector_counter = 0;
	}

	// schedule next event
	dsk_state[drive].thnd_sector =
	    TimerCreate(TIMER_US(next_time>>1), tcb_sector, drive, 0);

    } else {

	// we are at the boundary between two sectors
	// advance to next sector on disk
	dsk_state[drive].sector = (dsk_state[drive].sector + 1)
				% (dsk_state[drive].disk.sectors);

	// trigger new sector sequencer
	if (selected)
	    hel_new_sector();

	// schedule next event
	if (dsk_state[drive].sector == sectors_per_track-1) {
	    // schedule half a sector to generate the index hole
	    dsk_state[drive].thnd_sector =
		TimerCreate(TIMER_US(next_time>>1), tcb_sector, drive, 1);
	} else {
	    // full sector
	    dsk_state[drive].thnd_sector =
		TimerCreate(TIMER_US(next_time), tcb_sector, drive, 0);
	}
    }
}


// put a disk into a drive.  this amounts to reading the disk header
// block, then caching the contents of the entire virtual disk in memory.
// this memory is all dynamically allocated.
//
// this function trusts the caller not to put the same disk into two
// drives at the same time.
int
svh_insert_disk(int drive, char *filename)
{
    svd_t *svd;
    int stat;
    int num_blocks, maxsectorsize;
    int offset, blknum, side, track, sector;

    ASSERT(drive >= 1 && drive <= ctrlr_state.drives);

    if (dsk_state[drive].occupied) {
	complain("Take out the old disk from drive %d first!", drive);
	return SVD_ACCESSERROR;
    }


    // ENSURE_TIMER_DEAD(dsk_state[drive].thnd_sector);

    stat = svd_read_file_header(filename, &dsk_state[drive].disk);
    if (stat != SVD_OK)
	return stat;
    svd = &(dsk_state[drive].disk);

    // cache the disk image
    num_blocks = svd->sides * svd->tracks * svd->sectors;
    maxsectorsize = SVH_SECSIZE;
    dsk_state[drive].image = (byte*)malloc(num_blocks*maxsectorsize);
    ASSERT(dsk_state[drive].image != NULL);
    dsk_state[drive].secinfo = (sector_info_t*)malloc(num_blocks*sizeof(sector_info_t));
    ASSERT(dsk_state[drive].secinfo != NULL);

    // read in each sector
    offset = 0;
    blknum = 0;

    for(side=0; side<svd->sides; side++) {
	for(track=0; track<svd->tracks; track++) {

	    int done;
	    // this operation takes a little while.
	    // let windows drain messages while we're running.
	    // this is a bit dangerous, since if the machine is
	    // really slow, it might be possible for the user to
	    // kick off some operation that we're not ready for.
	    (void)UI_HandleMessages(&done);	// done is ignored

	    for(sector=0; sector<svd->sectors; sector++) {

		stat = svh_read_raw_sector(svd, side, track, sector,
					&dsk_state[drive].secinfo[blknum].density,
					&dsk_state[drive].secinfo[blknum].bytes,
					&dsk_state[drive].image[offset]);
		if (stat != SVD_OK) {
		    free(dsk_state[drive].image);
		    dsk_state[drive].image = NULL;
		    free(dsk_state[drive].secinfo);
		    dsk_state[drive].secinfo = NULL;
		    // UI_DiskNotify(drive, SVD_NOTIFY_EMPTY);
		    return stat;	// FAILED
		}

		dsk_state[drive].secinfo[blknum].offset = offset;
		offset += maxsectorsize;
		blknum++;
	    }
	}
    }

    dsk_state[drive].occupied = TRUE;
    dsk_state[drive].sector   = 0;	// it's got to be something
    // start the disk spinning
    if (ctrlr_state.motor_flag) {
	dsk_state[drive].thnd_sector =
	    TimerCreate(TIMER_US(dsk_state[drive].sec_time_us), tcb_sector, drive, 0);
    }
    // UI_DiskNotify(drive, SVD_NOTIFY_OCCUPIED);

    return SVD_OK;
}


// remove a disk from a drive
void
svh_remove_disk(int drive)
{
    ASSERT(drive >=1 && drive <= 4);

    if (!dsk_state[drive].occupied) {
	complain("attempting to empty out an empty drive, %d", drive);
	return;
    }

    if (cursec.valid && (cursec.drive == drive))
	flush_cur_sector();

    dsk_state[drive].occupied = FALSE;
    ENSURE_TIMER_DEAD(dsk_state[drive].thnd_sector);
    // UI_DiskNotify(drive, SVD_NOTIFY_EMPTY);

    // free resources associated with cached disk image
    ASSERT(dsk_state[drive].image != NULL);
    free(dsk_state[drive].image);
    dsk_state[drive].image = NULL;

    ASSERT(dsk_state[drive].secinfo != NULL);
    free(dsk_state[drive].secinfo);
    dsk_state[drive].secinfo = NULL;
}


#if 0
// ====================================================================
// GUI interface abstraction layer
// ====================================================================
// FIXME:
//     Should I split this into just a "vdisk.c" part
//     with the logical layer, and then the vdisk_ns.c
//     for the controller level interface?  For instance,
//     the logical layer should be more or less the same
//     no matter whose controller you use (if the controllers
//     are disk compatible) whereas the _ns.c one contains
//     the driver-level software/hardware interface.

// return a property from the virtual controller state as *value.
// return 0 on success, non-zero on failure.
int
GetDiskControllerProp(int prop, int *value)
{
    switch (prop) {
	case VDCPROP_INSTALLED:
	    *value = ctrlr_state.installed;
	    break;
	case VDCPROP_BASEADDR:
	    *value = ctrlr_state.base_addr;
	    break;
	case VDCPROP_DRIVES:
	    *value = ctrlr_state.drives;
	    break;
	case VDCPROP_DEBUGMODE:
	    *value = ctrlr_state.debug_mode;
	    break;
	default:
	    return VDCPROP_BADPROP;	// illegal request
    }

    return VDCPROP_OK;
}


// set a property from the virtual controller state.
// return 0 on success, non-zero on failure.
int
SetDiskControllerProp(int prop, int value)
{
    int i, stat;

    switch (prop) {

	case VDCPROP_INSTALLED:
	    ASSERT(value == 0 || value == 1);
	    if (value == ctrlr_state.installed)
		return VDCPROP_OK;	// unchanged
	    if (ctrlr_state.installed) {
		// remove the card at the current address
		stat = svd_request_uninstall_driver();
		if (stat != 0)
		    return VDCPROP_NOTNOW;	// not a good time
		stat = svd_uninstall_driver();
		ASSERT(stat == SVD_OK);
		// now reinstall the card at the new address
	    } else {
		// install the card
		stat = svd_install_driver(ctrlr_state.base_addr);
		ASSERT(stat == SVD_OK);
	    }
	    // UI_DiskNotify(0, SVD_NOTIFY_CONTROLLER);
	    return VDCPROP_OK;

	case VDCPROP_BASEADDR:
	    if (value == ctrlr_state.base_addr)
		return VDCPROP_OK;	// unchanged
	    if (ctrlr_state.installed) {
		// first we must remove the card at the current address
		stat = svd_request_uninstall_driver();
		if (stat != 0)
		    return VDCPROP_NOTNOW;	// not a good time
		stat = svd_uninstall_driver();
		ASSERT(stat == SVD_OK);
		// now reinstall the card at the new address
		stat = svd_install_driver(value);
		ASSERT(stat == SVD_OK);
	    }
	    ctrlr_state.base_addr = value;
	    break;

	case VDCPROP_DRIVES:
	    ASSERT(value >= 1 && value <= 4);
	    if (value < ctrlr_state.drives) {
		// removing drives -- check if they are in use
		int inuse = 0;
		for(i=value+1; i<=ctrlr_state.drives; i++)
		    inuse |= (dsk_state[i].occupied);
		if (inuse) {
		    int confirm = UI_Confirm(
			"Some of the drives that you are removing contain disks.\n"
			"Do you want to reduce the number of drives anyway?");
		    if (!confirm)
			return VDCPROP_NOTNOW;
		}
		// removing drives -- check if they are in use
		for(i=value+1; i<=ctrlr_state.drives; i++) {
		    if (dsk_state[i].occupied) {
			svh_remove_disk(i);
		    }
		}
	    }
	    ctrlr_state.drives = value;
	    // UI_DiskNotify(0, SVD_NOTIFY_CONTROLLER);
	    break;

	case VDCPROP_DEBUGMODE:
	    ASSERT(value == 0 || value == 1);
	    ctrlr_state.debug_mode = value;
	    break;

	default:
	    ASSERT(0);
	    return VDCPROP_BADPROP;	// illegal request
    }

    return VDCPROP_OK;
}


// returns 0 if the disk is not in use in another drive, and
// returns N if the disk is in use in drive N.
// the filenames must already be absolute and normalized.
// otherwise, we can't tell they both reference the same file.
int
DiskInUse(char *filename)
{
    int rv = 0;
    int i;

    ASSERT(filename != NULL);

    FOREACH_DRIVE(i) {
	if (dsk_state[i].occupied &&
	    (strcmp(dsk_state[i].disk.filename, filename) == 0)) {
	    rv = i;
	    break;
	}
    }

    return rv;
}


// get some state about the disk in question.
// returns zero if OK, non-zero on error.
int
GetDiskProp(int unit, int prop, int *value)
{
    if (unit < 1 || unit > 4)
	return VDPROP_BADDRIVE;

    if ((prop != VDPROP_EMPTY) && (prop != VDPROP_SELECTED) &&
	!dsk_state[unit].occupied)
	return VDPROP_NODISK;

    switch (prop) {
	case VDPROP_EMPTY:	// read-only; drive state
	    *value = !dsk_state[unit].occupied;
	    break;
	case VDPROP_SELECTED:	// read-only; drive state
	    *value = (ctrlr_state.ctl_drivesel == unit);
	    break;
	case VDPROP_SIDES:	// read-only; disk state
	    *value = dsk_state[unit].disk.sides;
	    break;
	case VDPROP_TRACKS:	// read-only; disk state
	    *value = dsk_state[unit].disk.tracks;
	    break;
	case VDPROP_SECTORS:	// read-only; disk state
	    *value = dsk_state[unit].disk.sectors;
	    break;
	case VDPROP_DENSITY:	// read-only; disk state
	    *value = dsk_state[unit].disk.density;	// (1 or 2)
	    break;
	case VDPROP_READONLY:	// disk state (write protect tab)
	    *value = dsk_state[unit].disk.writeprot;
	    break;
	case VDPROP_LABEL:	// disk state -- cast char* to int
	    *value = (int)(dsk_state[unit].disk.label);
	    break;
	case VDPROP_FILENAME:	// disk state -- cast char* to int
	    if (!dsk_state[unit].occupied)
		return VDPROP_NODISK;
	    *value = (int)(dsk_state[unit].disk.filename);
	    break;
	default:
	    return VDPROP_BADPROP;
    }

    return VDPROP_OK;
}


// set some state for the disk in question.
// returns zero if OK, non-zero on error.
int
SetDiskProp(int unit, int prop, int  value)
{
    if (unit < 1 || unit > 4)
	return VDPROP_BADDRIVE;
    if (!dsk_state[unit].occupied)
	return VDPROP_NODISK;

    switch (prop) {
	case VDPROP_EMPTY:	// read-only; drive state
	case VDPROP_SELECTED:	// read-only; drive state
	case VDPROP_SIDES:	// read-only; disk state
	case VDPROP_TRACKS:	// read-only; disk state
	case VDPROP_SECTORS:	// read-only; disk state
	case VDPROP_DENSITY:	// read-only; disk state
	case VDPROP_FILENAME:	// read-only; disk state
	    ASSERT(0);
	    return VDPROP_CANTSET;
	case VDPROP_READONLY:	// disk state (write protect tab)
	    ASSERT(value == 0 || value == 1);
	    dsk_state[unit].disk.writeprot = value;
	    svd_write_file_header(&dsk_state[unit].disk); // write back header
	    break;
	case VDPROP_LABEL:	// disk state -- cast char* to int*
	    ASSERT((void*)value != NULL);
	    strncpy(dsk_state[unit].disk.label, (char*)value, SVD_LABEL_SIZE);
	    svd_write_file_header(&dsk_state[unit].disk); // write back header
	    break;
	default:
	    return VDPROP_BADPROP;
    }

    return VDPROP_OK;
}

#endif

#if 0
/* -------------------------------------------------------------------
   logical layer external interface

   these routines provide a "cleaner" external interface as they don't
   mess with the svd_t structure at all.  however, they are less efficient
   since each call results in fetching the header block of the file
   before the main work is done.
   ------------------------------------------------------------------- */

// given a filename, get information about a disk.
// returns SVD_OK if it worked, something else if it didn't.
int
svh_read_disk_info( char *filename,		// in
		    int  *filetype,		// the rest are out...
		    int  *sides,
		    int  *tracks,
		    int  *sectors,
		    int  *density,
		    int  *writeprotect,
		    char *label,
		    int  *maxsectorsize)
{
    svd_t svd;
    int stat;

    ASSERT(filename != NULL);

    stat = svd_read_file_header(filename, &svd);
    if (stat != SVD_OK)
	return stat;

    *filetype      = VDFORMAT_NORTHSTAR;
    *sides         = svd.sides;
    *tracks        = svd.tracks;
    *sectors       = svd.sectors;
    *density       = svd.density;
    *writeprotect  = svd.writeprot;
    *maxsectorsize = (svd.density == 1) ? SVD_SNG_SECSIZE : SVD_DBL_SECSIZE;
    strcpy(label, svd.label);

    return SVD_OK;
}


// write a disk header to the specified file.
// returns SVD_OK if it worked, something else if it didn't.
// should this be changed so that only the writeprotect
// and the label can be modified?
int
svh_write_disk_info( char *filename,
		     int   sides,
		     int   tracks,
		     int   sectors,
		     int   density,
		     int   writeprotect,
		     char *label)
{
    svd_t svd;
    int stat, unit;

    ASSERT(filename != NULL);
    ASSERT(strlen(filename) < sizeof(svd.filename));

    strncpy(svd.filename, filename, sizeof(svd.filename));
    svd.sides     = sides;
    svd.tracks    = tracks;
    svd.sectors   = sectors;
    svd.density   = density;
    svd.writeprot = writeprotect;
    strncpy(svd.label, label, sizeof(svd.label));

    stat = svd_write_file_header(&svd);

    // check if the disk is in use -- if it is,
    // update our cached copy
    unit = DiskInUse(filename);
    if (unit != 0) {
	dsk_state[unit].disk.writeprot = writeprotect;
	strncpy(dsk_state[unit].disk.label, label, sizeof(svd.label));
    }

    return stat;
}
#endif


#if 0
// install boot tracks on an already formatted disk.
// this works only for double density images (512 bytes/sector).
// this routine computes and appends the sector checksum.
#define SECSIZE (512)
int
svd_make_bootable(char *filename, byte *image, int tracks)
{
    int side, track, sector;
    int stat;
    svd_t svd;

    ASSERT(filename != NULL);

    stat = svd_read_file_header(filename, &svd);
    if (stat != SVD_OK)
	return stat;

    if (svd.sectors != 10)
	return SVD_BADFORMAT;

    side = 0;

    for(track=0; track<tracks; track++) {
	for(sector=0; sector<10; sector++) {
	    byte *rawdata = &image[SECSIZE*(track*(svd.sectors)+sector)];
	    byte secbuf[SECSIZE+1];
	    int off, checksum = 0x00;
	    for(off=0; off<SECSIZE; off++) {
		secbuf[off] = rawdata[off];
		checksum = (rawdata[off] ^ checksum);
		checksum = 0xFF & ((checksum<<1) + (checksum>>7));
	    }
	    secbuf[SECSIZE] = checksum;
	    stat = svd_write_sector(&svd, side, track, sector, 2,
					SECSIZE+1, secbuf);
	    if (stat != SVD_OK)
		return stat;
	}
    }

    return SVD_OK;
}
#endif


#if 0
// debugging aid -- dump contents of virtual disk file
int
svh_dump_disk(char *diskfilename, char *dumpfilename)
{
    FILE *fp;
    int side, track, sector;
    int stat, off, density, bytes;
    byte rawbuff[SVD_DBL_SECSIZE];
    char *p;
    svd_t svd;

    ASSERT(diskfilename != NULL);
    ASSERT(dumpfilename != NULL);

    stat = svd_read_file_header(diskfilename, &svd);
    if (stat != SVD_OK)
	return stat;

    fp = fopen(dumpfilename, "w");
    if (fp == NULL)
	return SVD_BADFILE;

    // dump header
    fprintf(fp, "//        format: Solace Virtual Northstar Disk 1.0\n");
    fprintf(fp, "//         sides: %d\n", svd.sides);
    fprintf(fp, "//   tracks/side: %d\n", svd.tracks);
    fprintf(fp, "// sectors/track: %d\n", svd.sectors);
    fprintf(fp, "//  disk density: %d\n", svd.density);
    fprintf(fp, "// write protect: %d\n", svd.writeprot);
    p = svd.label;
    while (*p != 0x00) {
	fprintf(fp, "// label: ");
	while (*p && (*p != '\r') && (*p != '\n')) {
	    fprintf(fp, "%c", *p);
	    p++;
	}
	fprintf(fp, "\n");
	while ((*p == '\r') || (*p == '\n'))
	    p++;
    }
    fprintf(fp, "//\n");


    for(side=0; side<svd.sides; side++) {
	for(track=0; track<svd.tracks; track++) {
	    for(sector=0; sector<svd.sectors; sector++) {
		fprintf(fp, "// Side=%d, Track=%d, Sector=%d\n", side, track, sector);
		stat = svh_read_raw_sector(&svd, side, track, sector,
					      &density, &bytes, rawbuff);
		if (stat != SVD_OK) {
		    fprintf(fp, "Error reading sector\n\n");
		    continue;
		}
		fprintf(fp, "// density=%d, #bytes=%d\n", density, bytes);
		
		for(off=0; off<bytes; off++) {
		    if ((off%16 == 0) && (off != 0))
			fprintf(fp, "\n");
		    fprintf(fp, "%02X ", rawbuff[off]);
		}
		fprintf(fp, "\n");
	    }
	}
    }

    fclose(fp);

    return SVD_OK;
}
#endif


/* -------------------------------------------------------------------
 * report error conditions, but only if we haven't squawked recently
 */

// complain only about errors that are the result of software
// driver problems, not user error.  also, only complain if
// there hasn't recently been an error just to guard against
// perpetual error reports.
//
// the dbgflag parameter is used to keep track of which errors
// should be ignored or not.

static void
dbgcomplain(int *dbgflag, char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    if (ctrlr_state.debug_mode && (*dbgflag != DBGFLAG_IGNORE) &&
	!g_complained_recently) {

	char buf[1000];
	vsprintf(buf, fmt, args);
	*dbgflag = UI_DbgAlert(buf);
	g_complained_recently = 1;
	g_thnd_complained =
	    TimerCreate(TIMER_MS(1000), tcb_complain, 0, 0);

	// if the user wants to investigate further
	if (*dbgflag == DBGFLAG_DEBUG)
	    CPU_Break(BRKPT_DEBUG);
    }

    va_end(args);
}


// always complain
static void
complain(char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    UI_Alert(fmt, args);
    va_end(args);
}
