// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2005

// Functions for manipulating the virtual northstar disk images independent
// of any specific emulator needs.  These functions are useful for the
// emulator but also utility programs.

#ifndef _VDISK_SVNLIB_H_
#define _VDISK_SVNLIB_H_

#include "vdisk_svdlib.h"

// ===================================================================
// VIRTUAL NORTHSTAR DISK FORMAT:
//
// Each disk consists of a header and a fixed number of blocks.
// The header occupies the first 4 KB of the file and is stored
// as CR/LF-terminated ASCII strings, with any remainder of the 4 KB
// padded with 0x00 bytes.
//
// This header block has this fixed sequence:
//
//   struct {
//     char format[64];     "SVD:Solace Virtual Disk, Northstar\00\00..."
//     uint16 version;      [=1]
//     uint16 density;      [=1|2]
//     uint16 writeprotect; [=0|1]
//     uint16 sides;        [=1|2]
//     uint16 tracks;       [=35-96]
//     uint16 sectors;      [=10]
//     unsigned char pad[1024 - 64 - 6*2];
//     char label[3*1024];
//   }
//
// The format string is a null-terminated string that identifies
// what the rest of the header block might contain.  Any unused
// portion of the string should be set to 0x00.
//
// All the uint16 fields are stored little-endian, meaning that the
// least-significant byte of the 16b quantity is stored earlier in
// the file than the most-significant byte.
//
// Density indicates if the entire disk is single or double density.
// This field is used to minimize the file size if all sectors are
// known to be single-density.  It is OK to write nothing but single-
// density sectors to a double-density disk, or any mix, of course.
//
// If writeprotect is 1, the disk shouldn't be written to; if it is 0,
// it is OK for the driver to write data to the disk.
//
// Sides indicates the number of sides that are writable on the disk.
// Single sided virtual disks take up less space on the real disk.
//
// Tracks is the number of tracks per side; it is 35 for the Northstar
// controller, but room has been left in the header for generality.
//
// Sectors is the number of sectors per track; it is fixed at 10 for
// the Northstar controller, but room has been left in the header
// for generality.
//
// Pad should be filled with 0x00 bytes to round out the first 1KB
// of the header block.
//
// The label lines are a single null-terminated string, although
// there can be any number of lines, which are CR/LF terminated.
//
// The rest of the file is accessed randomly as fixed size blocks.
// These blocks are saved as a raw binary format.
//
// The number of blocks is (# sides)*(# tracks)*(10).
//
// The size of each block depends on the density.  Nominally there
// are 256 bytes/sector in single density, and 512 bytes/sector in
// double density.  However, the actual number (unformatted) is
// dependent only on the rotational speed and the bit transfer rate.
// At the nominal 300 RPM, each rotation takes 200 ms.  At a transfer
// rate of 250 Kbps, this works out to 6250 bytes/track double density,
// and 3125 bytes/track single density.  In real life not all these
// bytes can be used, as the disk speed might be higher or lower,
// and each sector requires preamble and sync bytes.
//      single density (density==1): 256 +  56 = 312 bytes/block
//      double density (density==2): 512 + 112 = 624 bytes/block
//
// Although the standard Northstar format only uses a few of these
// "extra" bytes, for preamble and checksum, in theory they can be
// used and Solace should work for any such scheme too.
//
// The file that contains the virtual disk image adds yet another
// four bytes of overhead per sector.  Only one bit of this is
// currently used, which is for indicating if the sector is
// recorded in single or double density.
// ===================================================================

// create a blank virtual disk of the specified type.
// returns SVD_OK if successful, otherwise some error SVD_* indication.
// it does not check if file already exists.  the file is not actually
// written until svnlib_flush() is called using the returned svd_t pointer.
int svnlib_create_disk(
        const char *filename,
        int sides, int tracks, int sectors,
        int density, int writeprotect, char *label, char fillchar,
        svd_t *svd
    );

// this function returns a handle to a virtual disk file in svd*.
// the function returns SVD_OK on success, otherwise some SVD_* error.
// if the parameter "cached" is non-zero, the entire disk image is read
// into memory and subsequent modifications (via the read/write sector
// routines) are done on the in-memory image; the accumulated changes
// are finally commited to memory on svnlib_flush().  if "cached" is
// zero, svd just caches the disk metadata and subsequent sector read/
// write operations happen immediately; svnlib_flush should still be
// called after all changes are done.
int svnlib_open(const char *filename, int cached, svd_t *svd);

// this function is called on an svd_t* handle after changes have been
// made to a disk image.  the function returns SVD_OK on success, otherwise
// some other SVD_* error status is returned.  
int svnlib_flush(svd_t *svd);

// read a sector: preamble, data, and any checksum are returned
// SVD_* status is returned (SVD_OK on success).
int svnlib_read_raw_sector(svd_t *svd, int side, int track, int sector,
                           int *density, int *bytes, unsigned char *secbuf);

// write a sector: preamble, data, and any checksum must be supplied
// SVD_* status is returned (SVD_OK on success).
int svnlib_write_raw_sector(svd_t *svd, int side, int track, int sector,
                            int density, int bytes, unsigned char *secbuf);

// read a sector, skipping the preamble; checksum is returned, as that
// is part of software-visible state.
// SVD_* status is returned (SVD_OK on success).
int svnlib_read_sector(svd_t *svd, int side, int track, int sector,
                       int *density, int *bytes, unsigned char *secbuf);

// write a sector, provided only the "payload" data
// SVD_* status is returned (SVD_OK on success).
int svnlib_write_sector(svd_t *svd, int side, int track, int sector,
                        int density, int bytes, unsigned char *secbuf);

#endif // ifdef _VDISK_SVNLIB_H_
