// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000, 2001, 2002

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>	// for varargs
#include <string.h>
#include <ctype.h>	// character type: isalpha()
#include <io.h>		// required for open_osfhandle() function
#include <fcntl.h>	// needed for stdio redirection hack

#include <windows.h>	// main windows includes
#include <windowsx.h>	// message cracker macros
#include <direct.h>	// needed for _getcwd() etc
#include <mmsystem.h>	// needed for timeGetTime() and related funcs
#include <commctrl.h>	// needed for status bar

#include "solace_intf.h" // core emulator definitions
#include "wingui.h"	// GUI-specific definitions
#include "vdisk_svn.h"	// northstar virtual disk definitions
#include "wintape.h"	// interface to virtual tape drives
#include "hexfile.h"	// ENT and HEX file I/O
#include "srcdbg.h"	// needed for OvlParse()
#include "resource.h"

#define BLINK_RATE 3.0	// in Hz (actually, blink period is half of this)


// ------ globals ------

// state relevant to the windows environment
winstate_t winstate;


// -------- forward references --------

static void RegisterWindowClass(HINSTANCE hInstance, char *appname);
static void CreateMainWindow(HINSTANCE hInstance, char *appname);

static LRESULT CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM);
static LRESULT CALLBACK CreditsDlgProc(HWND, UINT, WPARAM, LPARAM);
static LRESULT CALLBACK KeysDlgProc(HWND, UINT, WPARAM, LPARAM);
static LRESULT CALLBACK DbgAlertDlgProc(HWND, UINT, WPARAM, LPARAM);

static void BuildStatusBar(HWND hwnd);
static void InitWinstate(void);
static void InitOpenDlg(void);
static void CreateConsole(void);

int hexfile_read_overlay(char *filename, int report, byte *memory);


int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
	PSTR szCmdLine, int iCmdShow)
{
    static char szAppName[] = "Solace";

    if (0)
	CreateConsole();	// create console window for printf messages

    InitWinstate();		// winstate.foo = blah
    InitScreen();		// stuff for drawing the emulated screen
    InitOpts();			// options dialog

    RegisterWindowClass(hInstance, szAppName);	// register main window class
    CreateMainWindow(hInstance, szAppName);	// create main window

    InitDebugger();			// one-time initializations
    InitOpenDlg();			// set up "Open" dialog params
	// InitOpenDlg must happen before Sys_Init() so winstate.basedir is valid
	// when get_wav() in audio.c calls Host* functions.
    Sys_Init();				// set some emulator state
    WinTapeInit(hInstance, winstate.hWnd); // stuff related to virtual tape
    WinAudioInit();			// initialize audio output

    LoadOptions();			// load .ini file to override defaults

    WinResetSim(FALSE);			// cold reset the simulator
    SetDispColors();			// set foreground/background colors
    CreateCharMap();			// create font map

    Sys_DbgNotify(FORCERUN_KEY, 0);

    WinUpdateToolBar();

    ShowWindow(winstate.hWnd, iCmdShow);	// not visible until this point
    UpdateWindow(winstate.hWnd);		// cause redraw of everything

    Sys_DoTimeSlicing();		// run emulation

    return winstate.exitcode;
}


static void
RegisterWindowClass(HINSTANCE hInstance, char *appname)
{
    WNDCLASSEX  wndclass;

    wndclass.cbSize        = sizeof(wndclass);
    wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
    wndclass.lpfnWndProc   = WndProc;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = 0;
    wndclass.hInstance     = hInstance;
    wndclass.hIcon         = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON1));
    wndclass.hCursor       = NULL; //LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground = NULL;
    wndclass.lpszMenuName  = MAKEINTRESOURCE(IDR_MENU_MAIN);
    wndclass.lpszClassName = appname;
    wndclass.hIconSm       = NULL;

    RegisterClassEx(&wndclass);
    winstate.hInst = hInstance;
}


static void
CreateMainWindow(HINSTANCE hInstance, char *appname)
{
    DWORD	winstyle;

    // open up main window
#if 1
    // typical windows window
    winstyle = WS_OVERLAPPEDWINDOW;
#else
    // don't allow resizing of the window
    winstyle = WS_OVERLAPPED | WS_CAPTION;
#endif

    winstate.hWnd = CreateWindow(
			appname,	// window class name
			"Solace",	// window caption
			winstyle,	// window style
			CW_USEDEFAULT,	// initial x position
			0,		// initial y position
			CW_USEDEFAULT,	// initial x size
			0,		// initial y size
			NULL,		// parent window handle
			NULL,		// window menu handle
			hInstance,	// program instance handle
			NULL);		// creation parameters

    ASSERT(winstate.hWnd != NULL);

    winstate.hdcWin = GetDC(winstate.hWnd);
}


static void
InitWinstate(void)
{
    winstate.hWnd         = NULL;	// WinResetSim looks at it before it is set
    winstate.hdcWin       = NULL;	// = GetDC(winstate.hWnd);

    winstate.hToolbar     = NULL;
    winstate.toolbar_h    = 0;	// determined on WM_CREATE

    winstate.hStatus      = NULL;
    winstate.status_h     = 0;	// determined on WM_CREATE

    winstate.hOpts        = NULL;

    winstate.exitcode     = 0;
    winstate.hNullCursor  = 0;
    winstate.hArrowCursor = 0;

    winstate.hidemouse    = 0;
    winstate.show_stats   = 0;
    winstate.holdmsg      = 0;

    winstate.client_w     = 0;	// determined on WM_SIZE
    winstate.client_h     = 0;
}


// return the size of the emulated Sol screen.
// this doesn't include any windows window dressing.
void
GetSolScreenSize(HWND hwnd, int *pixels, int *rows)
{
    int window_w, window_h;
    RECT rcwin;

#if 1
    GetClientRect(hwnd, &rcwin);
    window_w = rcwin.right;
    window_h = rcwin.bottom
	     - winstate.toolbar_h	// height of toolbar
	     - winstate.status_h;	// height of status bar
#else
    int menuheight = GetSystemMetrics(SM_CYMENU);	// in pixels
    int cx_border  = GetSystemMetrics(SM_CXSIZEFRAME);	// in pixels
    int cy_border  = GetSystemMetrics(SM_CYSIZEFRAME);	// in pixels

    // get the entire app frame size
    GetWindowRect(hwnd, &rcwin);

    window_w = rcwin.right - rcwin.left
	     - 2*cx_border;		// left and right window borders

    window_h = rcwin.bottom - rcwin.top
	     - menuheight		// FIXME: titlebar (fudged)
	     - menuheight		// height of menu bar
	     - winstate.toolbar_h	// height of toolbar
	     - winstate.status_h	// height of status bar
	     - 2*cy_border;		// left and right window borders
#endif

    *pixels = window_w;
    *rows   = window_h;
}


// resize the main window such that the emulated screen is the specified size.
// don't move origin, though.
// AdjustWindowRect() might be a better way to do this, but this works for me.
void
SetSolScreenSize(HWND hwnd, int pixels, int rows)
{
    int menuheight = GetSystemMetrics(SM_CYMENU);	// in pixels
    int cx_border  = GetSystemMetrics(SM_CXSIZEFRAME);	// in pixels
    int cy_border  = GetSystemMetrics(SM_CYSIZEFRAME);	// in pixels

    int window_w, window_h;
    RECT rcwin;

    window_w = pixels			// the Sol display canvas
	     + 2*cx_border;		// left and right window borders

    window_h = menuheight		// FIXME: titlebar (fudged)
	     + menuheight		// height of menu bar
	     + winstate.toolbar_h	// height of toolbar
	     + rows			// the Sol display canvas
	     + winstate.status_h	// height of status bar
	     + 2*cy_border;		// top and bottom window borders

    // preserve location, but change size
    GetWindowRect(hwnd, &rcwin);
    (void)MoveWindow(hwnd, rcwin.left, rcwin.top, window_w, window_h, FALSE);
}


// reset the emulated computer
void
WinResetSim(int warm_reset)
{
    int use_builtin_rom = 0;

    Sys_Reset(warm_reset);		// init Sol state
    if (!warm_reset) {
	if (strlen(winstate.ofn[FILE_ROM].lpstrFile) > 0) {
	    int stat = hexfile_read_overlay(winstate.ofn[FILE_ROM].lpstrFile, 0, memimage);	// get personality module ROM
	    if (stat != HFILE_OK) {
		UI_Alert("The personality ROM image couldn't be loaded.\n"
			 "Using the built-in standard SOLOS ROM.");
		use_builtin_rom = 1;
	    }
	} else {
	    // silently use the internal ROM image
	    use_builtin_rom = 1;
	}
    }

    if (use_builtin_rom) {
	char *fname = "ROMs\\solos_cpm.prn";
	char *absname = HostMakeAbsolutePath(fname);
	Sys_LoadDefaultROM();
	// attempt to load the source code overlay
	(void)OvlParse(absname, 1);	// 1=yes, autoloaded
	OvlHitInit();			// rebuild overlay index table
    }

    UI_InvalidateScreen();
}


// if any messages are waiting, dispatch one and return TRUE.
// the *done flag is is set to 1 if we see a WM_QUIT message.
int
UI_HandleMessages(int *done)
{
    MSG msg;

    // update cursor blink state
    UpdateCursorBlinkState();

    if (!PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
	return FALSE;

    if (winstate.hOpts == NULL || !PropSheet_IsDialogMessage(winstate.hOpts, &msg)) {
	TranslateMessage(&msg);
	DispatchMessage(&msg);
    }

    if (msg.message == WM_QUIT) {
	winstate.exitcode = msg.wParam;
	*done = TRUE;
    }

    return TRUE;
}


// -------------- message handlers ------------------

static BOOL
OnCreateMain(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    int menuheight = GetSystemMetrics(SM_CYMENU);	// in pixels
    int cx_border  = GetSystemMetrics(SM_CXSIZEFRAME);	// in pixels
    int cy_border  = GetSystemMetrics(SM_CYSIZEFRAME);	// in pixels
    int screenfudge = 6;

    // we use two different mouse cursors at different times
    winstate.hNullCursor  = LoadCursor(NULL, IDC_ICON);
    winstate.hArrowCursor = LoadCursor(NULL, IDC_ARROW);

    // build controls then resize window to show entire screen.
    // this resizing may get overridden later by solace.ini processing.
    InitCommonControls();
    BuildToolBar(hwnd);
    BuildStatusBar(hwnd);

    SetSolScreenSize(hwnd,
			SolScreenWidth()  + screenfudge,
			SolScreenHeight() + screenfudge);

    // get messages provide for cursor blinking
    SetTimer(hwnd, 1, (int)(1000/BLINK_RATE), NULL);

    return TRUE;
}


// handle window resize events
static void
OnSizeMain(HWND hwnd, UINT state, int cx, int cy)
{
    winstate.client_w = cx;	// width of client area
    winstate.client_h = cy;	// height of client area

    // if the window is bigger than the display,
    // center the display in the window
    winstate.xoff = (cx - SolScreenWidth())/2;
    winstate.yoff = (cy - winstate.toolbar_h - winstate.status_h - SolScreenHeight())/2;
    if (winstate.xoff < 0) winstate.xoff = 0;
    if (winstate.yoff < 0) winstate.yoff = 0;

    // the sol display starts below the toolbar
    winstate.yoff += winstate.toolbar_h;

    // wndclass has CS_HREDRAW | CS_VREDRAW, so we don't need to:
    //    InvalidateRect(winstate.hWnd, NULL, TRUE);

    // let the toolbar know the size has changed
    SendMessage(winstate.hToolbar, TB_AUTOSIZE, (WPARAM)0, (LPARAM)0);

    // let the status bar know the size has changed
    SendMessage(winstate.hStatus, WM_SIZE, (WPARAM)0, (LPARAM)0);
}


static void
OnCommandMain(HWND hwnd, int id, HWND hctl, UINT codenotify)
{
    char buf[MAX_PATH];

    switch (id) {

    // --- File submenu ---

    case IDM_FILE_LOADROM:
	if (FileOpenDlg(hwnd, FILE_ROM)) {
	    // always reset after changing the ROM
	    WinResetSim(FALSE);
	}
	return;

    case IDM_FILE_LOADPROGRAM:
	if (Sys_IsProgramRunning()) {
	    int confirm = UI_Confirm(
		"A program is already running.  Are you sure\n" \
		"you want to load another? (maybe reset first)");
	    if (confirm != IDOK)
		return;
	}
	if (FileOpenDlg(hwnd, FILE_PROGRAM)) {
	    (void)hexfile_read_overlay(winstate.ofn[FILE_PROGRAM].lpstrFile, 1, memimage);
	    // just in case the program loaded into screen memory,
	    // refresh the display (one program cutely has its
	    // ascii instructions loaded directly to the screen)
	    UI_InvalidateScreen();
	}
	return;


    case IDM_FILE_DISKFACTORY:
	CreateDiskPropsDialog(FALSE, 0);
	return;

    case IDM_FILE_SCRIPT:
	if (FileOpenDlg(hwnd, FILE_SCRIPT))
	    (void)Sys_Script(winstate.ofn[FILE_SCRIPT].lpstrFile);
	return;

    case IDM_FILE_SNAPSHOT:
	CreateSnapshot();
	return;

    case IDM_FILE_EXIT:
	SendMessage(hwnd, WM_CLOSE, 0, 0L);
	return;


    // --- Processor submenu ---

    case IDM_PROC_COLDBOOT:
	WinResetSim(FALSE);	// cold reset
	return;

    case IDM_PROC_WARMBOOT:
	WinResetSim(TRUE);	// warm reset
	return;


    // --- Windows submenu ---

    case IDM_WIN_TAPE1:
	CreateTapeWindow(1);
	break;

    case IDM_WIN_TAPE2:
	CreateTapeWindow(2);
	break;

    case IDM_WIN_DEBUG:
	if (UI_DbgWinActive())
	    SendMessage(winstate.hDbgWnd, WM_CLOSE, 0, 0);
	else
	    Sys_DbgNotify(RUN_KEY, 0);
	break;

    case IDM_WIN_OPTIONS:
	CreateOptsDialog();
	break;


    // --- Options submenu ---

    case IDM_OPTIONS_DIP1:
    case IDM_OPTIONS_DIP2:
    case IDM_OPTIONS_DIP3:
    case IDM_OPTIONS_DIP4:
    case IDM_OPTIONS_CPU:
    case IDM_OPTIONS_DISPLAY:
    case IDM_OPTIONS_TAPES:
    case IDM_OPTIONS_NSDISK:
    case IDM_OPTIONS_CHARSET:
    case IDM_OPTIONS_SOUNDS:
	ForceOption(id);
	break;

    case IDM_OPTIONS_SAVE:
	SaveOptions();
	return;


    // --- HELP submenu ---

    case IDM_HELP_CREDITS:
	(void)DialogBox(winstate.hInst, MAKEINTRESOURCE(IDD_HELP_CREDITS),
			hwnd, (DLGPROC)CreditsDlgProc);
	return;

    case IDM_HELP_KEYS:
	(void)DialogBox(winstate.hInst, MAKEINTRESOURCE(IDD_HELP_KEYS),
			hwnd, (DLGPROC)KeysDlgProc);
	return;

    case IDM_HELP_OPTIONS:
	strcpy(buf, winstate.basedir);
	strcat(buf, "\\html");
	(void)ShellExecute(NULL, "open", "solace_options.html", NULL, buf, 0);
	return;

    case IDM_HELP_DEBUG:
	strcpy(buf, winstate.basedir);
	strcat(buf, "\\html");
	(void)ShellExecute(NULL, "open", "solace_dbg.html", NULL, buf, 0);
	return;

    case IDM_HELP_TAPES:
	strcpy(buf, winstate.basedir);
	strcat(buf, "\\html");
	(void)ShellExecute(NULL, "open", "solace_tape.html", NULL, buf, 0);
	return;

    case IDM_HELP_NSDISK:
	strcpy(buf, winstate.basedir);
	strcat(buf, "\\html");
	(void)ShellExecute(NULL, "open", "solace_vdiskns.html", NULL, buf, 0);
	return;

    case IDM_HELP_CPM:
	strcpy(buf, winstate.basedir);
	strcat(buf, "\\html");
	(void)ShellExecute(NULL, "open", "solace_cpm.html", NULL, buf, 0);
	return;


    // --- toolbar buttons ---

    case ID_TBBUT_TAPE1:
    case ID_TBBUT_TAPE2:
    case ID_TBBUT_DISKA:
    case ID_TBBUT_DISKB:
    case ID_TBBUT_DISKC:
    case ID_TBBUT_DISKD:
    case ID_TBBUT_PROP:
    case ID_TBBUT_DEBUG:
	ToolbarMessage(id);
	return;

    default:
	break;
    }

    // we aren't handling it, pass it through to default handler
    FORWARD_WM_COMMAND(hwnd, id, hctl, codenotify, DefWindowProc);
}


// FIXME: I can't find anything that says what the return value needs
//        to be, and whether or not the message needs to be forwarded
//        if we don't handle it.
static LRESULT
OnNotifyMain(HWND hwnd, int id, NMHDR *nmhdr)
{
    switch (nmhdr->code) {

	case TTN_NEEDTEXT:
	    // get tooltip text
	    {
		LPTOOLTIPTEXT lpttt = (LPTOOLTIPTEXT)nmhdr;
		lpttt->lpszText = ToolbarToolTipStr(lpttt->hdr.idFrom);
		return TRUE;
	    }
	    break;

	case NM_RCLICK:
	    // right mouse button clicked
	    if (id == ToolbarID) {
		LPNMMOUSE lpnmmouse = (LPNMMOUSE)nmhdr;
		ToolbarToolRClick(lpnmmouse->dwItemSpec);
		return TRUE;
	    }
	    break;

	default:
	    break;
    }

#if 0
    // we aren't handling it, pass it through to default handler
// compiler complains about not returning a value.
    FORWARD_WM_NOTIFY(hwnd, id, nmhdr, DefWindowProc);
#else
    return FALSE;
#endif
}


// potentially hide mouse cursor when it is over the Sol screen
static void
OnMouseMoveMain(HWND hwnd, int xClient, int yClient, UINT keyflags)
{
    if (winstate.hidemouse)
	SetCursor(winstate.hNullCursor);
    else
	SetCursor(winstate.hArrowCursor);
}


// update the checkmarks, etc, of the popup menus
static void
OnInitMenuPopupMain(HWND hwnd, HMENU hmenu, UINT item, BOOL sysmenu)
{
    if (!sysmenu) {
	HMENU hMenu = GetMenu(winstate.hWnd);
	HMENU hSubmenu;

	hSubmenu = GetSubMenu(hMenu, 2);	// Windows menu
	CheckMenuItem(hSubmenu, IDM_WIN_TAPE1,   (TapeWindowVisible(1))    ? MFS_CHECKED : MFS_UNCHECKED);
	CheckMenuItem(hSubmenu, IDM_WIN_TAPE2,   (TapeWindowVisible(2))    ? MFS_CHECKED : MFS_UNCHECKED);
	CheckMenuItem(hSubmenu, IDM_WIN_DEBUG,   (DebuggerWindowVisible()) ? MFS_CHECKED : MFS_UNCHECKED);
	CheckMenuItem(hSubmenu, IDM_WIN_OPTIONS, (OptionsWindowVisible())  ? MFS_CHECKED : MFS_UNCHECKED);
    }
}


static void
OnMenuSelectMain(HWND hwnd, HMENU hmenu, int item, HMENU hpopup, UINT flags)
{
    if ((hmenu == NULL) && (flags == 0xFFFFFFFF)) {
	// the menu has been dismissed
	StatusBarMsg(0, "");
	return;
    }

    if (!(flags & (MF_SYSMENU | MF_POPUP))) {
	// offer help only for non-system, non-popup menus;
	// that is, our own submenu items
	char buf[80];
	if (item == IDM_FILE_SNAPSHOT)
	    sprintf(buf, "Save an image of the screen in snap%04d.bmp", NextSnapshot());
	else
	    (void)LoadString(winstate.hInst, item, buf, sizeof(buf));
	StatusBarMsg(0, buf);
    }
}


// a request has been made to shut down.  we might honor it, or not.
static void
OnCloseMain(HWND hwnd)
{
    if (WinTapeClose())
	return;		// cancelled

    // OK, go ahead and shut down
    DestroyWindow(winstate.hWnd);
}


// terminate the application, including killing children.
// WM_DESTROY kills proper child winows (in this case, the Options window)
// but because the tape windows and the debugger window aren't children,
// we must explicitly clobber them.
static void
OnDestroyMain(HWND hwnd)
{
    WinTapeDestroy();	// includes killing kids
    WinAudioClose();
    svd_flush_disks();	// flush any dirty disks

    DestroyDebugger();

    if (winstate.hOpts != NULL)
	DestroyWindow(winstate.hOpts);

    KillTimer(hwnd, 1);
    PostQuitMessage(0);
}


// main window handler
static LRESULT CALLBACK
WndProc(HWND hWnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {

	HANDLE_MSG(hWnd, WM_CREATE,        OnCreateMain);
	HANDLE_MSG(hWnd, WM_SIZE,          OnSizeMain);
	HANDLE_MSG(hWnd, WM_PAINT,         OnPaintMain);
	HANDLE_MSG(hWnd, WM_MOUSEMOVE,     OnMouseMoveMain);
	HANDLE_MSG(hWnd, WM_KEYDOWN,       OnKeyDownMain);
	HANDLE_MSG(hWnd, WM_INITMENUPOPUP, OnInitMenuPopupMain);
	HANDLE_MSG(hWnd, WM_MENUSELECT,    OnMenuSelectMain);
	HANDLE_MSG(hWnd, WM_COMMAND,       OnCommandMain);
	HANDLE_MSG(hWnd, WM_NOTIFY,        OnNotifyMain);
	HANDLE_MSG(hWnd, WM_TIMER,         OnTimerMain);
	HANDLE_MSG(hWnd, WM_CLOSE,         OnCloseMain);
	HANDLE_MSG(hWnd, WM_DESTROY,       OnDestroyMain);
	HANDLE_MSG(hWnd, WM_SETFOCUS,      OnSetFocusMain);
	HANDLE_MSG(hWnd, WM_KILLFOCUS,     OnKillFocusMain);

	// we can't use a standard cracker because it doesn't give us
	// the transition state we need to defeat auto repeat
	case WM_CHAR:
	    return My_Handle_WM_CHAR(wParam, lParam);
    }

    return DefWindowProc(hWnd, iMsg, wParam, lParam);
}


// -------------- dialog box procedures ------------------

static LRESULT CALLBACK
CreditsDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    char *webpage = "http://www.thebattles.net/sol20/sol.html";

    switch (iMsg) {

	case WM_INITDIALOG:
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD(wParam)) {
		case IDOK:
		    EndDialog(hDlg, LOWORD(wParam));
		    return TRUE;
		case IDCANCEL:
		    EndDialog(hDlg, LOWORD(wParam));
		    return TRUE;
		case IDC_BUTTON_SOLPAGE:
		    (void)ShellExecute(NULL, "open", webpage, NULL, NULL, SW_SHOWNORMAL);
		    return TRUE;
		default:
		    break;
	    }
	    break;

	default:
	    break;

    } // switch (iMsg)

    return FALSE;	// message not handled
}


static LRESULT CALLBACK
KeysDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {

	case WM_INITDIALOG:
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD(wParam)) {
		case IDOK:
		    EndDialog(hDlg, LOWORD(wParam));
		    return TRUE;
		case IDCANCEL:
		    EndDialog(hDlg, LOWORD(wParam));
		    return TRUE;
		default:
		    break;
	    }
	    break;

	default:
	    break;

    } // switch (iMsg)

    return FALSE;	// message not handled
}


// we maintain state for search out ROMs and for searching out programs.
// it is possible that they exist in two different places.
static void
InitOpenDlg(void)
{
    char *fullpath[FILE_NUMTYPES], *initdir[FILE_NUMTYPES];
    int i;

    (void)_getcwd(winstate.basedir, MAX_PATH);

    for(i=0; i<FILE_NUMTYPES; i++) {

	fullpath[i]  = (char*)malloc(MAX_PATH);
	initdir[i]   = (char*)malloc(MAX_PATH);
	ASSERT(fullpath[i] != NULL);
	ASSERT(initdir[i]  != NULL);

	memset(&winstate.ofn[i], 0x00, sizeof(OPENFILENAME));
	winstate.ofn[i].lStructSize = sizeof(OPENFILENAME);
	winstate.ofn[i].nFilterIndex    = 1;
	winstate.ofn[i].lpstrFile       = fullpath[i];
	winstate.ofn[i].nMaxFile        = MAX_PATH;
	winstate.ofn[i].lpstrInitialDir = initdir[i];
	winstate.ofn[i].Flags           = OFN_HIDEREADONLY | OFN_FILEMUSTEXIST;
	switch (i) {
	    case FILE_PROGRAM:
		    winstate.ofn[i].lpstrFilter =
					"ENT/HEX Files (*.ent,*.hex)\0*.ENT;*.HEX\0"
					"ENT Files (*.ent)\0*.ENT\0"
					"HEX Files (*.hex)\0*.HEX\0"
					"All Files (*.*)\0*.*\0\0";
		    winstate.ofn[i].lpstrDefExt = "ent";
		    // set default search directory to where we launched from
		    strcpy(initdir[i], winstate.basedir);
		    strcat(initdir[i], "\\binaries");
		break;
	    case FILE_ROM:
		    winstate.ofn[i].lpstrFilter =
					"ROM/HEX Files (*.rom,*.hex)\0*.ROM;*.HEX\0"
					"ROM Files (*.rom)\0*.ROM\0"
					"HEX Files (*.hex)\0*.HEX\0"
					"All Files (*.*)\0*.*\0\0";
		    winstate.ofn[i].lpstrDefExt = "rom";
		    // set default search directory to where we launched from
		    strcpy(initdir[i], winstate.basedir);
		    strcat(initdir[i], "\\ROMs");
		break;
	    case FILE_VDISK:
		    winstate.ofn[i].lpstrFilter =
					"Northstar Files (*.svn)\0*.SVN\0"
					"Helios Files (*.svh)\0*.SVH\0"
					"All Files (*.*)\0*.*\0\0";
		    winstate.ofn[i].lpstrDefExt = "svn";
		    // set default search directory to where we launched from
		    strcpy(initdir[i], winstate.basedir);
		    strcat(initdir[i], "\\binaries");
		break;
	    case FILE_SCRIPT:
		    winstate.ofn[i].lpstrFilter =
					"SCR Files (*.scr)\0*.SCR\0"
					"TXT Files (*.txt)\0*.TXT\0"
					"ENT Files (*.ent)\0*.ENT\0"
					"All Files (*.*)\0*.*\0\0";
		    winstate.ofn[i].lpstrDefExt = "scr";
		    // set default search directory to where we launched from
		    strcpy(initdir[i], winstate.basedir);
		    strcat(initdir[i], "\\scripts");
		break;
	    case FILE_WAVOUT:
		    winstate.ofn[i].Flags = OFN_HIDEREADONLY;	// override default -- writable
		    winstate.ofn[i].lpstrFilter =
					"WAV Files (*.wav)\0*.WAV\0"
					"All Files (*.*)\0*.*\0\0";
		    winstate.ofn[i].lpstrDefExt = "wav";
		    // set default search directory to where we launched from
		    strcpy(initdir[i], winstate.basedir);
		break;
	    default:
		ASSERT(0);
	}

	// default filename to open
	fullpath[i][0] = '\0';
    }

#if 0
  for(i=0; i<FILE_NUMTYPES; i++) {
     printf("init: ofn[%d].lpstrFile       = %s\n", i, winstate.ofn[i].lpstrFile);
     printf("init: ofn[%d].lpstrInitialDir = %s\n", i, winstate.ofn[i].lpstrInitialDir);
  }
#endif
}


BOOL
FileOpenDlg(HWND hWnd, int filetype)
{
    BOOL flag;
    int idx = filetype;
    char *prevfile = strdup(winstate.ofn[idx].lpstrFile);

    winstate.ofn[idx].hwndOwner = hWnd;
    winstate.ofn[idx].lpstrFile[0] = '\0';

    flag = GetOpenFileName(&winstate.ofn[idx]);

    // if we succeed in opening a file, we want to remember what
    // directory to start the search in next time.
    if (flag) {
	char *newpath = HostExtractPath(winstate.ofn[idx].lpstrFile);
	strcpy((char*)winstate.ofn[idx].lpstrInitialDir, newpath);
	free(newpath);
    } else {
	// restore it to what it was in case user hit CANCEL button
	if (prevfile != NULL)
	    strcpy(winstate.ofn[idx].lpstrFile, prevfile);
    }

    free(prevfile);
    return flag;
}


// -------------- manipulate status bar ------------------

#define StatusID (200)		// arbitrary (?) unique constant

// build a status bar, located at the bottom of the window
static void
BuildStatusBar(HWND hwnd)
{
    static int iStatusWidths[] = {300, -1};
    RECT rStatus;

    // create a basic status bar
    winstate.hStatus = CreateStatusWindow(WS_CHILD | WS_VISIBLE,
				    (LPCTSTR)NULL, hwnd, StatusID);

    // tell the status bar it has two message areas
    SendMessage(winstate.hStatus, SB_SIMPLE, (WPARAM)FALSE, (LPARAM)0);
    SendMessage(winstate.hStatus, SB_SETPARTS, 2, (LPARAM)iStatusWidths);

    // save its height for building entire client window geometry
    GetWindowRect(winstate.hStatus, &rStatus);
    winstate.status_h = rStatus.bottom - rStatus.top;
}


void
StatusBarMsg(int subwin, char *msg)
{
    // we report stats only when user wants them
    if (subwin == 1 && !winstate.show_stats)
	return;

    SendMessage(winstate.hStatus, SB_SETTEXT, subwin, (LPARAM)msg);
}


// put up a notification of some kind for a few time slices
void
UI_TimedNote(char *msg, int duration)
{
    StatusBarMsg(0, msg);
    // we use the cursor blink timer to know how long to keep message up
    winstate.holdmsg = (int)(duration*BLINK_RATE+0.8);
}

// send a message about CPU load
void
UI_NoteSpeed(char *msg)
{
    StatusBarMsg(1, msg);
}


// -------------- system timer functions ------------------

// figure out what type of timing capability the machine
// supports, and return the frequency of the time reference.
// we save the timer capabilities so HostCurTick() knows which
// facility to use.

int
HostQueryTimebase(void)
{
    LARGE_INTEGER HPTimerFreq;
    LARGE_INTEGER time;

    winstate.useHPTimer = QueryPerformanceFrequency(&HPTimerFreq);
#if 0
    winstate.useHPTimer = FALSE; // for testing purposes
#endif
    winstate.useLPTimer = FALSE;

    if (winstate.useHPTimer) {

	//printf("This machine has a high-performance timer of %I64d Hz\n", HPTimerFreq.QuadPart);
	winstate.fTickFreq = (double)HPTimerFreq.QuadPart;
	winstate.fTickPer  = 1.0 / winstate.fTickFreq;

	(void)QueryPerformanceCounter(&time);
	winstate.startTime = time;

    } else {

	MMRESULT mm;

	//printf("Using 1ms timer function\n");
	mm = timeBeginPeriod(1);	// 1 ms timer resolution
	winstate.useLPTimer = (mm == TIMERR_NOERROR);
#if 0
winstate.useLPTimer = FALSE;	// for testing
#endif
	mm = timeEndPeriod(1);		// must be paired with timeBeginPeriod()
	if (!winstate.useLPTimer) {
	    DWORD dwtime;

	    winstate.fTickFreq = 0.0;	// no useful timer
	    winstate.fTickPer  = 1.0;

	    (void)timeBeginPeriod(1);	// 1 ms timer resolution
	    dwtime = timeGetTime();
	    (void)timeEndPeriod(1);	// must be paired with timeBeginPeriod()
	    time.HighPart = 0;
	    time.LowPart = dwtime;
	    winstate.startTime = time;

	} else {
	    winstate.fTickFreq = 1000.0;	// 1 KHz timer resolution
	    winstate.fTickPer  = 1.0 / winstate.fTickFreq;
	}
    }

    return (winstate.fTickFreq == 0.0) ? 0 : 1000;
}


// return a 32b count of the number of milliseconds since the start of the simulation.
// this is enough to span 24 days of run time before wrapping negative.
int32
HostGetRealTimeMS(void)
{
    if (winstate.useHPTimer) {
	LARGE_INTEGER time;
	(void)QueryPerformanceCounter(&time);
	return (int32)((time.QuadPart - winstate.startTime.QuadPart) * winstate.fTickPer * 1000.0 + 0.5);
    } else if (winstate.useLPTimer) {
	DWORD dwtime;
	(void)timeBeginPeriod(1);	// 1 ms timer resolution
	dwtime = timeGetTime();
	(void)timeEndPeriod(1);		// must be paired with timeBeginPeriod()
	return (int32)(dwtime - winstate.startTime.LowPart);
    }

    return 0;
}


// -------------- utility functions ------------------


// create a console window so printf messages come out
static void
CreateConsole(void)
{
    int hCrt, i;
    FILE *hf;

    AllocConsole();
    hCrt = _open_osfhandle((long)GetStdHandle(STD_OUTPUT_HANDLE), _O_TEXT);
    hf = _fdopen( hCrt, "w" );
    *stdout = *hf;
    i = setvbuf( stdout, NULL, _IONBF, 0 );
}


// if something goes wrong and the only thing to do is report an
// error, this is the way to do it.
void
UI_Alert(char *fmt, ...)
{
    char buff[1000];
    va_list args;

    va_start(args, fmt);
    _vsnprintf(buff, sizeof(buff), fmt, args);
    va_end(args);

    (void)MessageBox(winstate.hWnd, buff, "Warning", MB_OK | MB_ICONEXCLAMATION);
}


// ask if something is OK to do or if the request should be cancelled.
// returns TRUE if it is OK to do, FALSE if cancel is requested.
int
UI_Confirm(char *fmt, ...)
{
    char buff[1000];
    va_list args;
    int confirm;

    va_start(args, fmt);
    _vsnprintf(buff, sizeof(buff), fmt, args);
    va_end(args);

    confirm = MessageBox(winstate.hWnd, buff, "Confirm", MB_OKCANCEL | MB_ICONEXCLAMATION);
    return (confirm == IDOK);
}


// this is like UI_Alert, except it has more options:
//    ignore it this time,
//    ignore it forever,
//    bring up the debugger
// error, this is the way to do it.
typedef struct {
    char *titlebar;	// string on titlebar, else generic msg if NULL
    char *explanation;	// mandatory
} dlgargs_t;

int
UI_DbgAlert(char *fmt, ...)
{
    char buff[1000];
    dlgargs_t dlgargs;
    va_list args;
    int resp;

    va_start(args, fmt);
    _vsnprintf(buff, sizeof(buff), fmt, args);
    va_end(args);

    dlgargs.titlebar    = "Warning: N* Disk Controller Driver Problem";
    dlgargs.explanation = buff;

    resp = DialogBoxParam(winstate.hInst, MAKEINTRESOURCE(IDD_DBGALERT),
			  winstate.hWnd, (DLGPROC)DbgAlertDlgProc, (LPARAM)&dlgargs);

    //(void)MessageBox(winstate.hWnd, buff, "Warning", MB_OK | MB_ICONEXCLAMATION);
    return resp;
}


// special dialog box for asking whether to ignore a problem or debug it
// the parameters are passed in via the dialog param, which is cast to
// a pointer of all the args
static LRESULT CALLBACK
DbgAlertDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    HWND hwndStatic, hwndIcon;
    HICON hIcon;
    dlgargs_t *dlgargs;

    switch (iMsg) {

	case WM_INITDIALOG:
	    dlgargs = (dlgargs_t*)lParam;

	    if (dlgargs->titlebar == NULL)
		SetWindowText(hDlg, "Warning");
	    else
		SetWindowText(hDlg, dlgargs->titlebar);

	    ASSERT(dlgargs->explanation != NULL);
	    hwndStatic = GetDlgItem(hDlg, IDC_DBGALERT_TEXT);
	    Static_SetText(hwndStatic, dlgargs->explanation);

	    hIcon = LoadIcon(NULL, IDI_EXCLAMATION);
	    hwndIcon = GetDlgItem(hDlg, IDC_DBGALERT_ICON);
	    Static_SetIcon(hwndIcon, hIcon);

	    return TRUE;

	case WM_COMMAND:
	    if (HIWORD(wParam) == BN_CLICKED) {
		switch (LOWORD(wParam)) {
		    case IDC_DBGALERT_IGNORE:
			EndDialog(hDlg, DBGFLAG_REPORT);
			return TRUE;
		    case IDC_DBGALERT_NEVER:
			EndDialog(hDlg, DBGFLAG_IGNORE);
			return TRUE;
		    case IDC_DBGALERT_DEBUG:
			EndDialog(hDlg, DBGFLAG_DEBUG);
			return TRUE;
		    default:
			break;
		}
	    }
	    break;

	default:
	    break;

    } // switch (iMsg)

    return FALSE;	// message not handled
}

// read the specified hex file, but if available, load the
// corresponding .prn file as an overlay.
int
hexfile_read_overlay(char *filename, int report, byte *memory)
{
    int rv = hexfile_read(filename, report, memory);
    if (rv == HFILE_OK) {

	int len = strlen(filename);
	char *modname;
	int repl = 0;
	int n;

	// see if the filename ends with a .??? suffix
	// replace trailing .??? with .prn if it exists
	modname = (char *)malloc(len + 4);
	strcpy(modname, filename);
	for(n=len-1; (n >= 0) && (n >= len-4) && !repl; n--) {
	    switch (modname[n]) {
		case '\\':
		case ':':
		    // oops
		    break;
		case '.':
		    strcpy(&modname[n], ".PRN");
		    repl = 1;	// replaced
		    break;
		default:
		    break;
	    }
	}
	if (!repl)
	    strcpy(&modname[len], ".PRN");

	// now try to load the overlay.
	// if the load fails, silently ignore it.
#if 1
	(void)OvlParse(modname, 1);	// 1=yes, autoloaded
	OvlHitInit();			// rebuild overlay index table
#else
	// dbg_interp("overlay %modname");		// interpret it
#endif

	free(modname);
    }

    return rv;
}


// -------------- filename utility functions ------------------

// return a pointer to an allocated string containing the path leading
// to the absolute file name passed in.  the caller must free the
// returned string at some point.
char *
HostExtractPath(const char *name)
{
    char *path, *p;
    int len;

    ASSERT(name != NULL);

    len = strlen(name);
    ASSERT(len > 0);

    path = strdup(name);
    ASSERT(path != NULL);

    // ...then erase the text after the last backslash.
    p = &path[len - 1];
    while (*p != '\\' && len > 1) {
	ASSERT(p > path); // at least one backslash
	p--;
	len--;
    }
    *p = '\0';	// wipe out tail

    return path;
}


// this function classifies the supplied filename as being either
// relative (returns 0) or absolute (returns 1).
// to be considered absolute, it must have one of these forms
//     [driveletter]:\path\to\file.c
//     \path\to\file.c
// otherwise it is considered relative.
int
HostIsAbsolutePath(const char *name)
{
    ASSERT(name != NULL);
    if (isalpha(name[0]) && (name[1] == ':') && (name[2] == '\\'))
	return 1;
    if (name[1] == '\\')
	return 1;
    return 0;
}

// if the filename given isn't an absolute path, make an absolute path
// assuming the name given is relative to the install directory.  either way,
// the string returned has been allocated and must be free'd by the caller.
// returns NULL on error.
char *
HostMakeAbsolutePath(const char *name)
{
    char *newname;

    if (name == NULL)
	return NULL;
    if (strlen(name) > MAX_PATH)
	return NULL;

    if (HostIsAbsolutePath(name)) {
	newname = strdup(name);
    } else {
	int newlen = strlen(winstate.basedir) + 1 + strlen(name) + 1;
	if (newlen > MAX_PATH)
	    return NULL;
	newname = (char *)malloc(newlen);
	strcpy(newname, winstate.basedir);
	strcat(newname, "\\");
	strcat(newname, name);
    }

    return newname;
}


// return a filename in the specified directory, dir, using a prefix name,
// prefix, if possible.  returns a malloc'd string if successful, NULL if not.
// an empty file is created with the given name.
char *
HostUniqueFilename(const char *dir, const char *prefix)
{
#if 0  // not used because under XP, at least, the dir was being ignored
    return tempnam(dir, prefix);
#else
    char *tempname = (char *)malloc(MAX_PATH);
    int rv;

    if (tempname == NULL)
	return NULL;

    rv = GetTempFileName(dir, prefix, 0, tempname);
    if (rv == 0) {
	free(tempname);
	return NULL;
    }

    return tempname;
#endif
}


// return a pointer to just the filename, ignoring any leading path.
char *
just_filename(char *fullname)
{
    char *p;
    int len;

    ASSERT(fullname != NULL);

    len = strlen(fullname);
    ASSERT(len > 0);

    // back up until a backslash or colon
    p = &fullname[len - 1];
    while (len > 0) {
	if (*p == '\\')
	    break;
	if (*p == ':')
	    break;
	p--;
	len--;
    }

    return (p+1);
}
