// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000, 2001

#include <stdlib.h>
#include <stdio.h>

#include <windows.h>	// main windows includes
#include <windowsx.h>	// message cracker macros

#include "solace_intf.h" // core emulator definitions
#include "wingui.h"	// GUI-specific definitions
#include "resource.h"


// FIXME: not used by this file, but it really belongs here
#define BLINK_RATE 3.0	// in Hz (actually, blink period is half of this)


// ------ globals ------

// state local to this file
typedef struct {

    int timerflag;	// gets set when it is time to change blink state
    int blinkmask;	// cursor blink gate

    int scrollshade;	// scroll blanking
    int scrollbase;	// scroll offset

    disp_color_t disp_color;	// pick a monitor color
    COLORREF fg, bg;		// corresponding fg and bg colors

    int snapcnt;	// snapshot count

    // ------ the following params are screen geometry information ------

    HBITMAP hCharmap;	// handle to font bitmap
    HDC  hdcCharMap;	// = CreateCompatibleDC(hdcWin);

    fontmode_t fontmode; // choose font
    byte *pMainFont;	// pointer to character generator ROM
    byte *pAltFont;	// pointer to character generator ROM

    int mapwidth;	// font table width,  in pixels
    int mapheight;	// font table height, in pixels
    int mapstride;	// font table width,  in pixels

    int cellwidth;	// # pixels per font character
    int cellheight;	// # pixels per font character

    int scrn_w;		// # pixels in display
    int scrn_h;		// # pixels in display
    int scrn_stride;	// # bytes per row of image

} screenstate_t;

static screenstate_t screenstate;


// -------- forward references --------

static void blit_row(int disprow, int startcol, int endcol);
static void update_inv_chars();


// --------------- access functions to screenstate ---------------

int SolScreenWidth(void)  { return screenstate.scrn_w; }
int SolScreenHeight(void) { return screenstate.scrn_h; }
int NextSnapshot(void)    { return screenstate.snapcnt; }

disp_color_t GetSolDispColor(void)       { return screenstate.disp_color; }
void SetSolDispColor(disp_color_t color) { screenstate.disp_color = color; }

fontmode_t GetSolFontMode(void)          { return screenstate.fontmode; }
void SetSolFontMode(fontmode_t fontmode) { screenstate.fontmode = fontmode; }


void
SetSolMainFont(chargen_t font)
{
    switch (font) {
	case CHARGEN_6574:
	    screenstate.pMainFont = charset_6574;
	    break;
	case CHARGEN_6575:
	    screenstate.pMainFont = charset_6575;
	    break;
	default:
	    ASSERT(0);
    }
}

chargen_t
GetSolMainFont(void)
{
    return (screenstate.pMainFont == charset_6574) ? CHARGEN_6574
						   : CHARGEN_6575;
}

void
SetSolAltFont(chargen_t font)
{
    switch (font) {
	case CHARGEN_DZONKHA:
	    screenstate.pAltFont = charset_dzonkha;
	    break;
	case CHARGEN_DEVANAGRI:
	    screenstate.pAltFont = charset_devanagri;
	    break;
	default:
	    ASSERT(0);
    }
}

chargen_t
GetSolAltFont(void)
{
    return (screenstate.pAltFont == charset_dzonkha) ? CHARGEN_DZONKHA
						     : CHARGEN_DEVANAGRI;
}

char *
SolMainFontName(void)
{ return (screenstate.pMainFont == charset_6574) ? "6574" : "6575"; }

char *
SolAltFontName(void)
{ return (screenstate.pAltFont == charset_dzonkha) ? "Dzonkha" : "Devanagri"; }


// called once at the beginning of time
void
InitScreen(void)
{
    screenstate.hdcCharMap   = NULL;	// = CreateCompatibleDC(hdcWin);
    screenstate.hCharmap     = NULL;

    screenstate.cellwidth    = 9;
    screenstate.cellheight   = 13*2;
    screenstate.mapwidth     = 32;	// the cell is narrower
    screenstate.mapheight    = 256*screenstate.cellheight;

    screenstate.scrn_w       = 64*screenstate.cellwidth;
    screenstate.scrn_h       = 16*screenstate.cellheight;
    screenstate.scrn_stride  = screenstate.scrn_w/8;

    screenstate.fontmode     = MAINONLY;
    screenstate.pMainFont    = charset_6574;
    screenstate.pAltFont     = charset_devanagri;

    screenstate.disp_color   = DISP_WHITE;

    screenstate.timerflag    = 0;
    screenstate.blinkmask    = 1;	// 1 = on

    screenstate.scrollbase   = 0;
    screenstate.scrollshade  = 0;

    screenstate.snapcnt      = 0;

}


// this is called any time the processor updates this register
// from chapter 8 of the sol manual:
//
//    U2 latches the starting row address from DIO0-3 and U13
//    latches the data on DIO4-7, with !PORT_OUT_FE being the
//    strobe.  Data on DIO4-7 specifies where the first line will
//    be displayed.  Thus, the number loaded into U1 is the
//    address of the first displayable scan line, and the number
//    loaded into U22 defines the character row (0 through 15).
//
// in other words, bits [7:4] specify that the first 'n' otherwise
// displayable lines are blank, and [3:0] specify the row number of
// the first actually displayed line.
//
// Note: SOLOS uses only [3:0], and bits [7:4] are always 0.
void
UI_UpdateScrollBase(int scrollshade, int scrollbase)
{
    if ((scrollbase  == screenstate.scrollbase) &&
        (scrollshade == screenstate.scrollshade))
	return;

    // changing scrollbase causes a roll.  Ideally, we would
    // detect where the screen wraps from 0xCFFF back to 0xCC00
    // and scroll the larger portion up or down; alas, we don't.
    // for now, just handle the most typical case.
    if ( (1==1) &&
         (scrollshade == screenstate.scrollshade) &&
         (scrollshade == 0) &&
         (winstate.yoff > winstate.toolbar_h) && // all screen rows are visible
	(((scrollbase-screenstate.scrollbase) & 0xF) == 1)) {

	RECT rc_scroll;	// area to scroll

	// in case screen is clipped on the bottom
	int bot = min(winstate.yoff + screenstate.scrn_h,
		      winstate.client_h - winstate.status_h);

	// top is shifted down by a row because of how rc_scroll
	// is used by ScrollWindowEx().  this rect describes the
	// source area; thus, scrolling by N pixels will cause
	// the N rows above the scroll area to get written to,
	// which is a waste of time and necessitates redrawing
	// that area too to put it back to normal.
	rc_scroll.left   = winstate.xoff;
	rc_scroll.right  = winstate.xoff + screenstate.scrn_w;
	rc_scroll.top    = winstate.yoff + screenstate.cellheight;
	rc_scroll.bottom = bot;

	screenstate.scrollbase  = scrollbase;
	screenstate.scrollshade = scrollshade;

	// scroll up one row
	ScrollWindowEx(winstate.hWnd,
			0, -(int)screenstate.cellheight,
			&rc_scroll, NULL, NULL, NULL,
			SW_INVALIDATE);	// don't scroll statusbar

    } else {
	// just redraw the whole window
	screenstate.scrollbase  = scrollbase;
	screenstate.scrollshade = scrollshade;
	UI_InvalidateScreen();
    }

}


// if the cursor toggle flag has been seen since the last
// time slice, redraw the appropriate parts of the screen.
// if the alternate font mode is active, there is no update
// because there are no inverse characters.
// also, the status message area gets blanked after a while.
void
UpdateCursorBlinkState(void)
{
    if (screenstate.timerflag) {

	screenstate.timerflag = 0;

	screenstate.blinkmask = !screenstate.blinkmask;
	if ((Sys_GetDipswitch(1) & SW1_BLINKCUR) &&
	    (screenstate.fontmode != ALTNOW))
		update_inv_chars();

	// after displaying a timed note for some period, erase it
// FIXME: this should really be disentangled from the blink timer
	if (winstate.holdmsg > 0) {
	    winstate.holdmsg--;
	    if (winstate.holdmsg == 0)
		StatusBarMsg(0, "");
	}
    }
}


void
OnPaintMain(HWND hwnd)
{
    PAINTSTRUCT	ps;
    HDC         hdc    = BeginPaint(hwnd, &ps);
    HBRUSH      hBrush = CreateSolidBrush(screenstate.bg);
    HPEN        hPen   = CreatePen(PS_SOLID, 1, screenstate.bg);
    HBRUSH      hOldBrush;
    HPEN        hOldPen;
    RECT        rc, todraw;

    // clear the overscan around the display
    // set the brush to the background color for filling
    hOldBrush = SelectObject(hdc, hBrush);
    hOldPen   = SelectObject(hdc, hPen);

    //printf("OnPaint: (%d,%d,%d,%d)\n", ps.rcPaint.left, ps.rcPaint.top, ps.rcPaint.right, ps.rcPaint.bottom);

    // see if any border needs to be painted
    if ((ps.rcPaint.left   <  winstate.xoff) ||
        (ps.rcPaint.top    <  winstate.yoff) ||
        (ps.rcPaint.right  >= winstate.xoff + 64*screenstate.cellwidth) ||
        (ps.rcPaint.bottom >= winstate.yoff + 16*screenstate.cellheight)) {
    //printf("erasing some of the border\n");

	// top strip
	rc.left   = 0;
	rc.top    = winstate.toolbar_h;
	rc.right  = winstate.client_w;
	rc.bottom = winstate.yoff;
	if (IntersectRect(&todraw, &rc, &ps.rcPaint)) {
	    //printf("  erasing top strip: (%d,%d,%d,%d)\n", todraw.left, todraw.top, todraw.right, todraw.bottom);
	    Rectangle(hdc, todraw.left, todraw.top, todraw.right, todraw.bottom);
	}

	// bottom strip
	rc.left   = 0;
	rc.top    = winstate.yoff + screenstate.scrn_h;
	rc.right  = winstate.client_w;
	rc.bottom = winstate.client_h;
	if (IntersectRect(&todraw, &rc, &ps.rcPaint)) {
	    //printf("  erasing bottom strip: (%d,%d,%d,%d)\n", todraw.left, todraw.top, todraw.right, todraw.bottom);
	    Rectangle(hdc, todraw.left, todraw.top, todraw.right, todraw.bottom);
	}

	// left strip
	rc.left   = 0;
	rc.top    = winstate.yoff;
	rc.right  = winstate.xoff;
	rc.bottom = winstate.yoff + screenstate.scrn_h;
	if (IntersectRect(&todraw, &rc, &ps.rcPaint)) {
	    //printf("  erasing left strip: (%d,%d,%d,%d)\n", todraw.left, todraw.top, todraw.right, todraw.bottom);
	    Rectangle(hdc, todraw.left, todraw.top, todraw.right, todraw.bottom);
	}

	// right strip
	rc.left   = winstate.xoff + screenstate.scrn_w;
	rc.top    = winstate.yoff;
	rc.right  = winstate.client_w;
	rc.bottom = winstate.yoff + screenstate.scrn_h;
	if (IntersectRect(&todraw, &rc, &ps.rcPaint)) {
	    //printf("  erasing right strip: (%d,%d,%d,%d)\n", todraw.left, todraw.top, todraw.right, todraw.bottom);
	    Rectangle(hdc, todraw.left, todraw.top, todraw.right, todraw.bottom);
	}
    }

    // update all the chars of the display
    if (Sys_MachineIsReset()) {
	rc.left   = winstate.xoff;
	rc.top    = winstate.yoff;
	rc.right  = winstate.client_w;
	rc.bottom = winstate.yoff + screenstate.scrn_h;
	if (IntersectRect(&todraw, &rc, &ps.rcPaint))
	    Rectangle(hdc, todraw.left, todraw.top, todraw.right, todraw.bottom);
    } else {
	rc.left   = winstate.xoff;
	rc.top    = winstate.yoff;
	rc.right  = winstate.xoff + screenstate.scrn_w;
	rc.bottom = winstate.yoff + screenstate.scrn_h;
	if (IntersectRect(&todraw, &rc, &ps.rcPaint)) {
	    int topscrrow = (todraw.top      - winstate.yoff) / screenstate.cellheight;
	    int botscrrow = (todraw.bottom-1 - winstate.yoff) / screenstate.cellheight;
	    int lftscrcol = (todraw.left     - winstate.xoff) / screenstate.cellwidth;
	    int rgtscrcol = (todraw.right-1  - winstate.xoff) / screenstate.cellwidth;
	    int disprow;
//printf("  drawing chars: (%d,%d,%d,%d)\n", lftscrcol, topscrrow, rgtscrcol, botscrrow);
	    for(disprow = topscrrow; disprow <= botscrrow; disprow++)
		blit_row(disprow, lftscrcol, rgtscrcol);
	}
    }

    // clean up
    SelectObject(hdc, hOldBrush);
    SelectObject(hdc, hOldPen);
    DeleteObject(hBrush);
    DeleteObject(hPen);
    EndPaint(hwnd, &ps);
}


// blink timer interrupt
void
OnTimerMain(HWND hwnd, UINT id)
{
    screenstate.timerflag = 1;
}


// cause redraw of entire Sol screen
void
UI_InvalidateScreen(void)
{
    if (winstate.hWnd) {
	RECT rc;
	rc.left   = 0;
	rc.top    = winstate.toolbar_h;
	rc.right  = winstate.client_w;
	rc.bottom = winstate.client_h - winstate.status_h;
	// don't invalidate status bar
	InvalidateRect(winstate.hWnd, &rc, FALSE);
    }
}


// the Sol character generator uses a fixed character cell grid of
// 8 bits wide by 11 rows high.  The screen is 64 chars wide by 24
// rows, so the pixel grid on the Sol is 512x264.  We don't want
// to duplicate this exactly because the pixels on our display are
// approximately square, while those on the Sol aren't.  The typical
// TV screen is 4:3.  We simply double each row in our font -- it
// ends up being too tall, but it is better than the squat look.
//
// we create a 1bpp bitmap, 16 chars wide and 8 chars tall.  Sol
// defines only 128 chars, with the top bit specifying inverse chars.

void
CreateCharMap(void)
{
    byte *charmap;
    int    mapbytes;
    int    chr, r;

    // release existing objects/handles
    if (screenstate.hCharmap != NULL) {
	DeleteObject(screenstate.hCharmap);
	screenstate.hCharmap = NULL;
    }
    if (screenstate.hdcCharMap != NULL) {
	DeleteDC(screenstate.hdcCharMap);
	screenstate.hdcCharMap = NULL;
    }

    // this is the actual size of each character on the display
    screenstate.cellwidth  = 9;
    screenstate.cellheight = 13*2;

    // this is the size of each character in the font table
    screenstate.mapwidth   = 32;	// the cell is narrower
    screenstate.mapheight  = 256*screenstate.cellheight;
    screenstate.mapstride  = screenstate.mapwidth/8;

    // allocate space for the expanded bitmap
    mapbytes = screenstate.mapheight * screenstate.mapstride;
    charmap = (byte*)malloc(mapbytes);
    if (charmap == NULL) {
	printf("died allocating charmap\n");
	exit(-1);
    }
    memset(charmap,            0xFF, mapbytes/2);	// first 128 are normal
    if (screenstate.fontmode == ALTNOW)
	memset(charmap+mapbytes/2, 0xFF, mapbytes/2);	// in this mode no inverse
    else
	memset(charmap+mapbytes/2, 0x00, mapbytes/2);	// second 128 are inverse

    // pre-expand the charset into the charmap.
    // also, 0s are foreground, 1's are background, so we must invert.
    // each char begins on a 16 bit boundary.
    for(chr=0; chr<256; chr++) {

	int invert = (screenstate.fontmode != ALTNOW) && (chr >= 128);

	byte *CurFont;
	if ((screenstate.fontmode == ALTNOW) && chr < 128)
	    CurFont = screenstate.pAltFont;
	else
	    CurFont = screenstate.pMainFont;

	for(r=0; r<13; r++) {
	    byte pix = CurFont[13*(chr&0x7F) + r];
	    charmap[screenstate.mapstride*(26*chr + 2*r + 0)] =
	    charmap[screenstate.mapstride*(26*chr + 2*r + 1)] =
						(invert) ? pix : ~pix;
	}
    }

    // cache new DC and charmap
    // create windows data structure for the character map
    screenstate.hCharmap = CreateBitmap(screenstate.mapwidth,
					screenstate.mapheight,
					1, 1,
					charmap);
    screenstate.hdcCharMap = CreateCompatibleDC(winstate.hdcWin);
    SelectObject(screenstate.hdcCharMap, screenstate.hCharmap);

    free(charmap);

    UI_InvalidateScreen();
}


// every time OUT 0FFH is performed, this function
// is called from the core emulator.  it returns
// '1' if we are in DH mode, and 0 otherwise.
int
UI_OutFFTickle(void)
{
    switch (screenstate.fontmode) {
	case MAINNOW:
	    screenstate.fontmode = ALTNOW;
	    CreateCharMap();
	    return 1;
	case ALTNOW:
	    screenstate.fontmode = MAINNOW;
	    CreateCharMap();
	    return 1;
	default:
	    ASSERT(0);
    }
    return 0;
}


// set screen fg/bg colors
void
SetDispColors(void)
{
    COLORREF fg,bg,tmp;

    if (winstate.hWnd) {

	HDC hdc = GetDC(winstate.hWnd);
	int iBitsPixel = GetDeviceCaps(hdc, BITSPIXEL);

	switch (screenstate.disp_color) {
	    case DISP_WHITE: fg = RGB(0xFF,0xFF,0xFF); break;
	    case DISP_GREEN: fg = RGB(0x10,0xF0,0x10); break;
	    case DISP_AMBER: fg = RGB(0xD0,0xD0,0x00); break;
	}
	bg = RGB(0x00,0x00,0x00);
	if (!(Sys_GetDipswitch(1) & SW1_POLARITY)) {
	    tmp = fg; fg = bg; bg = tmp;
	}

	// if we have a palettized display, we must pick
	// the nearest color otherwise it might get dithered
	if (iBitsPixel <= 8) {
	    fg = GetNearestColor(hdc, fg);
	    bg = GetNearestColor(hdc, bg);
	}

	SetTextColor(hdc, fg);
	SetBkColor  (hdc, bg);
	SetBkMode(hdc, OPAQUE); // this is the default anyway

	// cache these
	screenstate.fg = fg;	// used by snapshot function
	screenstate.bg = bg;	// used by snapshot function

	UI_InvalidateScreen();
    }
}


// draw one specific character on the display given the (row,col)
// of the memory image, before display, so scrollbase is removed
// to figure out where it shows up in the display.
void
UI_UpdateDisplay(int offset, byte data)
{
    int inverted = (data & 0x80);
    int dip1 = Sys_GetDipswitch(1);
    int left, upper, height;
    int row, col;
    int disprow;

    ASSERT(offset >=0 && offset < 1024);

    row = (offset >> 6);
    col = (offset & 0x3F);

    // the display is relative to the scroll register
    disprow = (row - screenstate.scrollbase + screenstate.scrollshade) & 0xF;

    left   = winstate.xoff +     col*screenstate.cellwidth;
    upper  = winstate.yoff + disprow*screenstate.cellheight;
    height = screenstate.cellheight;

    if (upper + height > winstate.client_h - winstate.status_h) {
	height -= (upper + height) - (winstate.client_h - winstate.status_h);
	if (height < 0)
	    return;
    }

    if (dip1 & SW1_BLANKCTL) {
	// control characters turn into blanks
	if ((data & 0x7F) < 0x20)
	    data = inverted | 0x20;	// space, preserve polarity
    }

    if ( inverted &&
	! (dip1 & SW1_SOLIDCUR) &&
	!((dip1 & SW1_BLINKCUR) && screenstate.blinkmask)) {
	    data &= 0x7F;
    }

    // if we are above scrollshade, blank it
    if (disprow < screenstate.scrollshade)
	data = 0x20;

    BitBlt(winstate.hdcWin,		// dest DC
	   left, upper,			// upper left corner of dest
	   screenstate.cellwidth,	// width
	   height,			// height
	   screenstate.hdcCharMap,	// source DC
	   0,				// left ...
	   data*screenstate.cellheight,	// ... upper
	   SRCCOPY);			// ROP
}


// draw a span of characters from startcol to endcol, inclusively,
// on the specified row of the screen.  The row is specified in terms
// of the display row, so that before we access the character memory,
// we must take scrollshade into account.
static void
blit_row(int disprow, int startcol, int endcol)
{
    int dip1      = Sys_GetDipswitch(1);
    int blank_ctl =  (dip1 & SW1_BLANKCTL);
    int obey_inv  =  (dip1 & SW1_SOLIDCUR)
		 || ((dip1 & SW1_BLINKCUR) && screenstate.blinkmask);
    int memrow = (disprow + screenstate.scrollbase - screenstate.scrollshade) & 0xF;
    int col, left, top, inverted, data;
    byte *dp;

    ASSERT(disprow  >=0  &&  disprow  <= 15);
    ASSERT(startcol >=0  &&  startcol <= 63);
    ASSERT(endcol   >=0  &&  endcol   <= 63);

    // the display is relative to the scroll register
    dp   = &memimage[0xCC00 + (memrow<<6) + startcol];

    left = winstate.xoff + screenstate.cellwidth * startcol;
    top  = winstate.yoff + screenstate.cellheight * disprow;

    for(col = startcol; col <= endcol; col++) {

	data     = *dp++;
	inverted = (data & 0x80);
	data    &= 0x7F;

	if (blank_ctl && (data < 0x20))
	    data = 0x20;	// control characters turn into blanks

	if (obey_inv)
	    data |= inverted;

	// if we are above scrollshade, blank it
	if (disprow < screenstate.scrollshade)
	    data = 0x20;

	BitBlt(winstate.hdcWin,			// dest DC
	       left, top,			// upper left corner of dest
	       screenstate.cellwidth,		// width
	       screenstate.cellheight,		// height
	       screenstate.hdcCharMap,		// source DC
	       0,				// left ...
	       data*screenstate.cellheight,	// ... upper
	       SRCCOPY);			// ROP

	left += screenstate.cellwidth;

    } // for(col)
}


// scan the screen: if the MSB is set, then redraw that character.
// this is used for blinking cursors.
static void
update_inv_chars(void)
{
    int row,offset;
    byte data;

    for(row=0; row < 1024; row += 64) {
	for(offset=row; offset < row+64; offset++) {
	    data = memimage[0xCC00 + offset];
	    if (data >= 0x80)
		UI_UpdateDisplay(offset, data);
	}
    }
}


// scan the display buffer and return and return a bitmap that
// corresponds to the image it generates.  a pointer to the bitmap to
// write to is passed in.  it must be of the correct size; if the
// pointer is NULL, a map is allocated of the correct size.  in either
// case, the pointer to the map is returned.   note that the
// screenstate.scrollbase value must be taken into account.
static byte *
CreateScrnImage(byte *imgmap)
{
    // 9b field masks for 8 possible alignments
    //                 off =   0      1      2      3      4      5      6      7
    static byte maskL[8] = { 0xFF, 0x7F, 0x3F, 0x1F, 0x0F, 0x07, 0x03, 0x01 };
    static byte maskR[8] = { 0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE, 0xFF };
    DWORD dstride_1 = (screenstate.scrn_stride >> 2);	// one display row
    DWORD dstride_2 = (screenstate.scrn_stride >> 1);	// two display rows
    int   blank_controls = Sys_GetDipswitch(1) &  SW1_BLANKCTL;
    int   any_cursor     = Sys_GetDipswitch(1) & (SW1_SOLIDCUR | SW1_BLINKCUR);
    int   row, col, imgbytes;

    imgbytes = screenstate.scrn_h * screenstate.scrn_stride;

    if (imgmap == NULL) {
	// allocate space for the display bitmap
	imgmap = (byte*)malloc(imgbytes);
	if (imgmap == NULL) {
	    printf("died allocating imgmap in creat_scrnimg\n");
	    exit(-1);
	}
    }


    // initialize the array to the background color
    memset(imgmap, 0xFF, imgbytes);


    for(row=0; row<16; row++) {

	int r = row * 13;	// font table cell height
	int blank_shade = (row < screenstate.scrollshade);
	int effective_row = (row + screenstate.scrollbase - screenstate.scrollshade) & 0xF;
	byte *pData = &memimage[0xCC00 + 64*effective_row];

	for(col=0; col<64; col++, pData++) {

	    int c   = col * 9;	// font table cell width, although only b[6:0] are used
	    int off = (c & 31);	// 32 possible character alignments in DWORD

	    byte data    = *pData;
	    int inverted = (data & 0x80);

	    byte *bp, *cp, *ce, t;
	    DWORD pix;

	    if ((inverted && !any_cursor) ||
		(screenstate.fontmode == ALTNOW))
		inverted = 0;

	    data &= 0x7F;

	    if ( blank_shade ||
		(blank_controls && (data < 0x20)) )
		// control characters turn into blanks
		data = 0x20;

#if 0
	    // speedup -- however, it makes the time very
	    // display dependent
	    if (data == 0x20 && !inverted)
		continue;
#endif

	    bp = imgmap + r*screenstate.scrn_stride*2 + (c>>3);
	    if ((screenstate.fontmode == ALTNOW) && (data < 128))
		cp = &screenstate.pAltFont[13*data];
	    else
		cp = &screenstate.pMainFont[13*data];

	    ce = cp + 13;	// end of char


	    if (off < 24 && (1==1)) {

		// 9b char map doesn't span 32b boundary
		DWORD *dp = (DWORD *)((DWORD)bp & ~3);
		DWORD mmask = ~ (0x1FF << (23-off));
		DWORD nmask = (inverted) ? 0x00000000 : 0xFFFFFFFF;

		off = 24-off;	// later loop invarient

		while (cp<ce) {

		    // get current row of font, align to bit position
		    // possible invert fg/bg, and merge background mask
		    pix = ((*cp << off) ^ nmask) | mmask;

		    // swizzle pixels due to the fact that the format
		    // MS chose for monochrome bitmaps was not the best
		    // for little endian machines.  this is because
		    // bits 7 and 8 of a word are non-contiguous in the
		    // image.  why did they do that?
#if 0
		    pix = ((pix & 0x000000FF) << 24)
		        | ((pix & 0x0000FF00) <<  8)
		        | ((pix & 0x00FF0000) >>  8)
		        | ((pix & 0xFF000000) >> 24);
#else
		    __asm mov eax, pix
		    __asm bswap eax
		    __asm mov pix, eax
#endif
		    // merge pixels into imagemap
		    *(dp + dstride_1) = *dp = (*dp & pix);

		    dp += dstride_2;	// advance two display rows
		    cp++;		// advance one character row

		} // while(cp < ce)

	    } else {

		// the 9b character row straddles a 32b boundary
		// (col * 9) & 7 == ((col * 8) + col) mod 8 == (col & 7)
		off = (col & 7);	// 8 possible character alignments

		while(cp<ce) {

		    // get current row of font
		    pix = *cp << 8;	// leftmost bit is bit 15
		    if (!inverted)
			pix = ~pix;
		    else
			pix = pix | 0xFFFF007F;

		    // left half of 9b
#if 0
		    // this is required if we are updating against a
		    // not-necessarily blank background
		    t = ((*bp) & ~maskL[off]) | ((pix >> (8+off)) & maskL[off]);
#else
		    t = *bp & (byte)(pix >> (8+off));
#endif
		    *(bp + screenstate.scrn_stride) = *bp = t;

		    // right half of 9b
		    bp++;
#if 0
		    t = ((*bp) & ~maskR[off]) | ((pix >> off) & maskR[off]);
#else
		    t = *bp & (byte)(pix >> off);
#endif
		    *(bp + screenstate.scrn_stride) = *bp = t;

		    bp += 2*screenstate.scrn_stride - 1;
		    cp++;

		} // while (cp < ce)
	    }

	} // for col
    } // for row

    return imgmap;
}


static RGBQUAD
COLORREF_TO_RGBQUAD(COLORREF color)
{
    RGBQUAD v;

    v.rgbBlue     = (BYTE)((color >> 16) & 0xFF);
    v.rgbGreen    = (BYTE)((color >>  8) & 0xFF);
    v.rgbRed      = (BYTE)((color >>  0) & 0xFF);
    v.rgbReserved = (BYTE)0x00;

    return v;
}


// copy screen to a file.  on each successive call, a new
// filename is generate by simply incrementing a counter.
void
CreateSnapshot(void)
{
    BITMAPFILEHEADER bmfh;
    BITMAPINFOHEADER bmih;
    RGBQUAD          aColors[2];

    byte *imgmap, *ip, *ep;
    char  filename[MAX_PATH];
    int   imgbytes, r;
    FILE *fp;

    // allocate space for the display bitmap
    imgbytes = screenstate.scrn_h * screenstate.scrn_stride;
    imgmap = (byte*)malloc(imgbytes);
//{
//    DWORD t1, t2, t3, t4;
//    t1 = CurTick();
//    t2 = CurTick();
    (void)CreateScrnImage(imgmap);
//    t3 = CurTick();
//    t4 = (t3-t2) - (t2-t1);
//    printf("time to generate image: %d ticks, or %f uS\n", t4, (t4*1000000.0f)/winstate.tick_freq);
//}
// it consistently took 7.1 ms on my machine, a P55C-200

    // write out a file
    bmfh.bfType      = 'MB';	// "BM" (bitmap)
    bmfh.bfSize      = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER)
		     + 2*sizeof(RGBQUAD) + imgbytes;
    bmfh.bfReserved1 = 0;
    bmfh.bfReserved2 = 0;
    bmfh.bfOffBits   = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER)
		     + 2*sizeof(RGBQUAD);

    bmih.biSize          = sizeof(BITMAPINFOHEADER);
    bmih.biWidth         =  screenstate.scrn_w;	// in pixels
    bmih.biHeight        =  screenstate.scrn_h;// in pixels, top down
    bmih.biPlanes        = 1;
    bmih.biBitCount      = 1;			// bpp
    bmih.biCompression   = BI_RGB;		// no compression
    bmih.biSizeImage     = 0;			// windows figures it out
    bmih.biXPelsPerMeter = 0;
    bmih.biYPelsPerMeter = 0;
    bmih.biClrUsed       = 0;			// 0 = all
    bmih.biClrImportant  = 0;			// 0 = all

    // the BitBlt command uses 1 as bg and 0 as fg, so we swap
    // the expected mapping to match BitBlt
    aColors[0] = COLORREF_TO_RGBQUAD(screenstate.fg);
    aColors[1] = COLORREF_TO_RGBQUAD(screenstate.bg);

    sprintf(filename,"%s\\snap%04d.bmp", winstate.basedir, screenstate.snapcnt);
    printf("dumping image to file '%s'\n", filename);
    screenstate.snapcnt++;

    fp = fopen(filename, "wb");		// write binary mode
    if (fp == NULL) {
	printf("failed trying to create snapshot file '%s'\n", filename);
	return;
    }

    // write headers
    (void)fwrite(&bmfh,    sizeof(bmfh),    1, fp);
    (void)fwrite(&bmih,    sizeof(bmih),    1, fp);
    (void)fwrite(&aColors, sizeof(RGBQUAD), 2, fp);

    // write bitmap -- 1bpp dib can apparently be only bottom-up
    // this code assumes fwrite() always writes the full amount.
    ip = imgmap;
    ep = imgmap + imgbytes;
    for(r=screenstate.scrn_h-1; r>=0; r--) {
	(void)fwrite(imgmap + screenstate.scrn_stride*r,
		     sizeof(byte), screenstate.scrn_stride, fp);
    }

    // we're done -- clean up
    fclose(fp);
    free(imgmap);
}

