// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <windows.h>
#include <commctrl.h>	// needed for toolbar

#include "solace_intf.h"
#include "wingui.h"
#include "wintape.h"	// interface to virtual tape drives
#include "vdisk_svn.h"	// interface to virtual disk drives
#include "resource.h"

// if 1, have more lengthy tooltips
#define VERBOSE_TOOLTIPS 1


// -------------- build the toolbar ------------------

// offset within imagelist of different images
enum { TBICON_TAPE=0,
       TBICON_DISKOFF=2,    TBICON_DISKON=3,
       TBICON_DISKOFF_MT=4, TBICON_DISKON_MT=5,
       TBICON_PROP=6,
       TBICON_DEBUG=7 };


// build a status bar, located at the bottom of the window
void
BuildToolBar(HWND hwnd)
{
    DWORD winstyle = WS_CHILD | WS_CLIPCHILDREN | WS_CLIPSIBLINGS
		   | WS_VISIBLE | CCS_TOP
		   | TBSTYLE_TOOLTIPS;
    RECT rc;

    const char *tbstrings = "Tape 1\0"
			    "Tape 2\0"
			    "Disk A\0"
			    "Disk B\0"
			    "Disk C\0"
			    "Disk D\0"
			    "Props\0"
			    "Debug\0"
			    "Reset\0"
			    "Help\0"
			    "\0";

    TBBUTTON tbuts[] = {
	TBICON_TAPE,       ID_TBBUT_TAPE1, TBSTATE_ENABLED, TBSTYLE_CHECK,  0, 0, 0, 0,
	TBICON_TAPE,       ID_TBBUT_TAPE2, TBSTATE_ENABLED, TBSTYLE_CHECK,  0, 0, 0, 1,
	0,                 0,              TBSTATE_ENABLED, TBSTYLE_SEP,    0, 0, 0, 0,
	TBICON_DISKOFF_MT, ID_TBBUT_DISKA, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, 2,
	TBICON_DISKOFF_MT, ID_TBBUT_DISKB, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, 3,
	TBICON_DISKOFF_MT, ID_TBBUT_DISKC, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, 4,
	TBICON_DISKOFF_MT, ID_TBBUT_DISKD, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, 5,
	0,                 0,              TBSTATE_ENABLED, TBSTYLE_SEP,    0, 0, 0, 0,
	TBICON_PROP,       ID_TBBUT_PROP,  TBSTATE_ENABLED, TBSTYLE_CHECK,  0, 0, 0, 6,
	TBICON_DEBUG,      ID_TBBUT_DEBUG, TBSTATE_ENABLED, TBSTYLE_CHECK,  0, 0, 0, 7,
	};

    winstate.hToolbar =
	CreateToolbarEx(
	    hwnd,		// parent window
	    winstyle,
	    ToolbarID,		// identifies the toolbar
	    8,			// number of images in list
	    winstate.hInst,	// application instance
	    IDR_TOOLBAR,
	    tbuts,		// array of button descriptions
	    sizeof(tbuts)/sizeof(TBBUTTON),	// # button instances
	     0,  0,		// button w,h (let windows figure it out)
	    20, 20,		// image w,h
	    sizeof(TBBUTTON)
	);

#if 0
    // add explanatory text under each toolbar button
    SendMessage(winstate.hToolbar, TB_ADDSTRING, (WPARAM)0, (LPARAM)tbstrings);
#endif

    // tell the toolbar to recompute its size
    SendMessage(winstate.hToolbar, TB_AUTOSIZE, (WPARAM)0, (LPARAM)0);

    // get toolbar height
    SendMessage(winstate.hToolbar, TB_GETITEMRECT, (WPARAM)0, (LPARAM)&rc);
    winstate.toolbar_h = rc.bottom - rc.top
		       + 6;	// magic fudge; but Petzold p 622 recommends 5
				// using 5 causes one row of flashing on win95 B though
}


// update the toolbar on the main window
void
WinUpdateToolBar(void)
{
    int i, stat, enabled, drives;
    int fState;


    // indicate if tape 1 or tape 2 window is open

    for(i=0; i<2; i++) {
	int ID[] = { ID_TBBUT_TAPE1, ID_TBBUT_TAPE2 };
	fState = (TapeWindowVisible(i+1)) ? (TBSTATE_ENABLED | TBSTATE_CHECKED)
					  : (TBSTATE_ENABLED);
	SendMessage(winstate.hToolbar, TB_SETSTATE,
		    (WPARAM)(ID[i]), (LPARAM) MAKELONG(fState, 0));
    }


    // hide any buttons for non-existent disk drives

    stat = GetDiskControllerProp(VDCPROP_INSTALLED, &enabled);
    ASSERT(stat == VDCPROP_OK);
    stat = GetDiskControllerProp(VDCPROP_DRIVES, &drives);
    ASSERT(stat == VDCPROP_OK);

    for(i=0; i<4; i++) {

	int ID[] = { ID_TBBUT_DISKA, ID_TBBUT_DISKB, ID_TBBUT_DISKC, ID_TBBUT_DISKD };

	fState = (enabled && (drives > i)) ? TBSTATE_ENABLED
					   : TBSTATE_HIDDEN;
	SendMessage(winstate.hToolbar, TB_SETSTATE,
		    (WPARAM)(ID[i]), (LPARAM) MAKELONG(fState, 0));

	if (fState == TBSTATE_ENABLED) {
	    int empty, on, iBitmap;
	    static int imgID[] = { TBICON_DISKOFF,    TBICON_DISKON,
				   TBICON_DISKOFF_MT, TBICON_DISKON_MT };
	    stat = GetDiskProp(i+1, VDPROP_EMPTY, &empty);
	    ASSERT(stat == VDPROP_OK);
	    stat = GetDiskProp(i+1, VDPROP_SELECTED, &on);
	    ASSERT(stat == VDPROP_OK);
	    iBitmap = imgID[2*empty + on];
	    SendMessage(winstate.hToolbar, TB_CHANGEBITMAP,
			(WPARAM)(ID[i]), (LPARAM) MAKELONG(iBitmap, 0));
	}
    }


    // show state of options dialog

    fState = (OptionsWindowVisible()) ? (TBSTATE_ENABLED | TBSTATE_CHECKED)
				      : (TBSTATE_ENABLED);
    SendMessage(winstate.hToolbar, TB_SETSTATE,
		(WPARAM)(ID_TBBUT_PROP), (LPARAM) MAKELONG(fState, 0));


    // show state of debugger

    fState = (DebuggerWindowVisible()) ? (TBSTATE_ENABLED | TBSTATE_CHECKED)
				       : (TBSTATE_ENABLED);
    SendMessage(winstate.hToolbar, TB_SETSTATE,
		(WPARAM)(ID_TBBUT_DEBUG), (LPARAM) MAKELONG(fState, 0));
}


void
ToolbarMessage(int id)
{
    char *diskname = winstate.ofn[FILE_VDISK].lpstrFile;
    int tapeunit = 2;
    int diskunit = 4;
    int stat, diskempty, drive;

    switch (id) {

	case ID_TBBUT_TAPE1: tapeunit--;
	case ID_TBBUT_TAPE2:
	    CreateTapeWindow(tapeunit);
	    break;

	case ID_TBBUT_DISKA: diskunit--;
	case ID_TBBUT_DISKB: diskunit--;
	case ID_TBBUT_DISKC: diskunit--;
	case ID_TBBUT_DISKD:
	    stat = GetDiskProp(diskunit, VDPROP_EMPTY, &diskempty);
	    ASSERT(stat == VDPROP_OK);
	    if (!diskempty) {
		svd_remove_disk(diskunit);
	    } else {
		if (!FileOpenDlg(winstate.hWnd, FILE_VDISK))
		    return;
		drive = DiskInUse(diskname);
		if (drive != 0) {
		    UI_Alert("That disk is already in use in drive %c", 'A'+drive-1);
		    return;
		}
		stat = svd_insert_disk(diskunit, diskname);
		if (stat != SVD_OK)
		    UI_Alert("Error openining disk '%s', drive %d, %s",
				diskname, diskunit, vdisk_errstring(stat));
	    }
	    break;

	case ID_TBBUT_PROP:
	    CreateOptsDialog();
	    break;

	case ID_TBBUT_DEBUG:
#if 1
	    if (UI_DbgWinActive())
		SendMessage(winstate.hDbgWnd, WM_CLOSE, 0, 0);
	    else
#endif
		Sys_DbgNotify(RUN_KEY, 0);
	    break;

	default:
	    ASSERT(0);
    }
}


// return a statically allocated string describing button for tooltip

static char ttbuff[200];	// holds tooltip message string

char *
ToolbarToolTipStr(int id)
{
    int tapeunit = 2;
    int diskunit = 4;
#if VERBOSE_TOOLTIPS
    int stat;
    char *filename;
#endif

    switch (id) {

	case ID_TBBUT_TAPE1: tapeunit--;
	case ID_TBBUT_TAPE2:
#if VERBOSE_TOOLTIPS
	    filename = GetTapeFilename(tapeunit & 1);
	    if (strlen(filename) == 0)
		sprintf(ttbuff, "Tape %d: empty", tapeunit);
	    else
		sprintf(ttbuff, "Tape %d: %s", tapeunit, just_filename(filename));
#else
	    sprintf(ttbuff, "Tape %d", tapeunit);
#endif
	    return ttbuff;

	case ID_TBBUT_DISKA: diskunit--;
	case ID_TBBUT_DISKB: diskunit--;
	case ID_TBBUT_DISKC: diskunit--;
	case ID_TBBUT_DISKD:
#if VERBOSE_TOOLTIPS
	    stat = GetDiskProp(diskunit, VDPROP_FILENAME, (int*)(&filename));
	    if (stat == VDPROP_NODISK) {
		sprintf(ttbuff, "Drive %c: empty\nClick to load",
			    'A'-1 + diskunit);
	    } else if (stat == VDPROP_OK) {
		sprintf(ttbuff, "Drive %c: %s\nClick to eject",
			    'A'-1 + diskunit, just_filename(filename));
	    } else
		ASSERT(0);
#else
	    sprintf(ttbuff, "Drive %c", 'A'-1 + diskunit);
#endif
	    return ttbuff;

	case ID_TBBUT_PROP:
	    return "Solace Options";

	case ID_TBBUT_DEBUG:
#if 0
	    // test really long string
	    return "xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xxxxxx xx";
#else
	    return "8080 Debugger";
#endif

	default:
	    ASSERT(0);
    }

    return "";
}


// something on the toolbar had a right click
void
ToolbarToolRClick(int id)
{
    int tapeunit = 2;
    int diskunit = 4;

    switch (id) {

	case ID_TBBUT_TAPE1: tapeunit--;
	case ID_TBBUT_TAPE2:
	    break;

	case ID_TBBUT_DISKA: diskunit--;
	case ID_TBBUT_DISKB: diskunit--;
	case ID_TBBUT_DISKC: diskunit--;
	case ID_TBBUT_DISKD:
	    // pop open the inspector dialog on the disk in this drive
	    CreateDiskPropsDialog(TRUE, diskunit);
	    break;

	case ID_TBBUT_PROP:
	    break;

	case ID_TBBUT_DEBUG:
	    break;

	default:
	    break;
    }
}


// this is called from the disk drive emulator to tell us of
// changes in that affect the toolbar appearance
void
UI_DiskNotify(int unit, int state)
{
    // FIXME: this is the lazy way.  if it flashes,
    //        incrementally update the toolbar icons
    WinUpdateToolBar();
}

