// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000, 2001, 2002

// Create a modeless dialog box with either a tabbed or listbox-controlled
// interface (compile-time selectable), one page for each option group.

#include <stdlib.h>
#include <stdio.h>
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>

#include "solace_intf.h"
#include "wingui.h"
#include "wintape.h"	// interface to virtual tape drives
#include "resource.h"
#include "vdisk_svn.h"

//#include "extensions.h"

#define Trackbar_SetPos(hwnd, moveit, pos) \
                (void)SendMessage((hwnd), TBM_SETPOS, (WPARAM)(BOOL)(moveit), \
                                (LPARAM)(pos))

#define Trackbar_SetRange(hwnd, redraw, lo, hi) \
                (void)SendMessage((hwnd), TBM_SETRANGE, \
                                  (WPARAM)(BOOL)(redraw), \
                                  MAKELONG((WORD)(lo), (WORD)(hi)))


// define to be 0 to get a listbox-controlled options dialog.
// define to be 1 to get a tabbed property-sheet options dialog.
#define TABBED_PROPSHEET 0


#define NUMTABS 10

static struct {

    PROPSHEETHEADER pshdr;		// prop sheet header structure
    PROPSHEETPAGE   pspage[NUMTABS];	// prop sheet page structures
    HWND            hwndPage[NUMTABS];	// handle to page

    BOOL            valid_position;	// is dialog position set?
    int             orig_x, orig_y;	// window origin

#if !TABBED_PROPSHEET
    // if we are using the alternative interface:
    int		curdlg;		// which dialog is currently shown
    HWND	hCurDlg;	// handle of current sub-dialog
    int		subdlg_offx;	// x-offset for subdialog
#endif

} optstate;


// ========================================================================
// The following code handle setting up and managing the container
// dialog.
// ========================================================================

static void
Refresh_dips(HWND hwndDlg, int byte)
{
    static int ids[] = { IDC_DIP_1, IDC_DIP_2, IDC_DIP_3, IDC_DIP_4,
                         IDC_DIP_5, IDC_DIP_6, IDC_DIP_7, IDC_DIP_8 };
    int i;
    for(i=0; i<8; i++)
	CheckDlgButton(hwndDlg, ids[i], (byte>>i) & 1);
}


static int
MapSW2Int(WPARAM wParam)
{
    int n;
    switch (wParam) {
	case IDC_DIP_1: n=0; break;
	case IDC_DIP_2: n=1; break;
	case IDC_DIP_3: n=2; break;
	case IDC_DIP_4: n=3; break;
	case IDC_DIP_5: n=4; break;
	case IDC_DIP_6: n=5; break;
	case IDC_DIP_7: n=6; break;
	case IDC_DIP_8: n=7; break;
    }
    return n;
}


// the Sol hardware design is cheesy in certain areas.  For instance,
// there are user accessible dip switches whereby some outputs get
// shorted together and fight, or worse, as in this case, an output
// gets shorted to ground, potentially damaging a chip.
static void
SW1_Error(HWND hWnd)
{
    (void)MessageBox(hWnd,
	    "You can't have both SW1-5 and SW1-6 on at the same time.\n" \
	    "You've blown the output driver of U88, pin 6.\n" \
	    "Time to break out the soldering iron!",
	    "Bozo Box",
	    MB_OK | MB_ICONEXCLAMATION);
}


// this function is called on creation/destruction of prop sheet
// it is used to associate a switch number with a given window.
//
// this is pretty useless ... hwnd is null, and ppsp doesn't
// provide a way to recover the window handle for the sheet
// it is associated with, so instead we resort to maintaining
// it in a global array; first one here deletes them all.

static UINT CALLBACK
DipSheetProc(HWND hwnd, UINT uMsg, LPPROPSHEETPAGE ppsp)
{
    int i;

    switch (uMsg) {

	case PSPCB_CREATE:
	    return TRUE;

	case PSPCB_RELEASE:
	    for(i=0; i<4; i++)
		if (optstate.hwndPage[i] != NULL) {
		    RemoveProp(optstate.hwndPage[i], "switchnum");
		    optstate.hwndPage[i] = NULL;
		}
	    return 0;
    }

    return 0;
}


static void
SaveOptPosition(void)
{
    if (winstate.hOpts != NULL) {
	RECT rc;
	GetWindowRect(winstate.hOpts, &rc);
	optstate.valid_position = TRUE;
	optstate.orig_x = rc.left;
	optstate.orig_y = rc.top;
    }
}


// some common code handling if using TABBED interface
static BOOL
Handle_WM_NOTIFY(LPNMHDR pnmh)
{
#if TABBED_PROPSHEET
    switch (pnmh->code) {

	case PSN_SETACTIVE:
	    return TRUE;

	case PSN_QUERYCANCEL:
	case PSN_APPLY:
	    // closing dialog
	    SaveOptPosition();
	    ShowWindow(winstate.hOpts, FALSE);
	    WinUpdateToolBar();
	    return TRUE;

	default:
	    break;
    }
#endif
    return FALSE;	// didn't handle the message
}


static BOOL
handle_sw1(HWND hwndDlg, byte dip, int n)
{
    byte newdip = (byte)(dip ^ (1<<n));

    switch (n) {

	case 0:
	    Sys_SetDipSwitch(1, newdip);
	    WinResetSim(TRUE);
	    return TRUE;

	case 1:
	    Sys_SetDipSwitch(1, newdip);
	    return TRUE;

	case 2:
	    Sys_SetDipSwitch(1, newdip);
	    UI_InvalidateScreen();
	    return TRUE;

	case 3:
	    Sys_SetDipSwitch(1, newdip);
	    SetDispColors();
	    return TRUE;

	case 4:
	    if (  (dip & SW1_SOLIDCUR) &&
		 !(dip & SW1_BLINKCUR)) {
		SW1_Error(hwndDlg);
		Refresh_dips(hwndDlg, dip);
	    } else {
		Sys_SetDipSwitch(1, newdip);
		UI_InvalidateScreen();
	    }
	    return TRUE;

	case 5:
	    if ( !(dip & SW1_SOLIDCUR) &&
		  (dip & SW1_BLINKCUR)) {
		SW1_Error(hwndDlg);
		Refresh_dips(hwndDlg, dip);
	    } else {
		Sys_SetDipSwitch(1, newdip);
		UI_InvalidateScreen();
	    }
	    return TRUE;

	default:
	    break;

    } // WM_COMMAND switch

    return FALSE;	// shouldn't ever happen
}


static BOOL
handle_sw2(HWND hwndDlg, byte dip, int n)
{
    Sys_SetDipSwitch(2, (byte)(dip ^ (1<<n)));
    return TRUE;
}


static BOOL
handle_sw3(HWND hwndDlg, byte dip, int n)
{
    byte newdip = dip ^ (1<<n);
    if ((newdip & (newdip-1)) != 0) {
	(void)MessageBox(hwndDlg,
		"Only one of SW3-1 to SW3-8 can be on at the same time.\n" \
		"You've herniated the output driver of U86.\n" \
		"Time to break out the soldering iron!",
		"Bozo Box",
		MB_OK | MB_ICONEXCLAMATION);
	Refresh_dips(hwndDlg, dip);
    } else
	Sys_SetDipSwitch(3, newdip);
    return TRUE;
}


static BOOL
handle_sw4(HWND hwndDlg, byte dip, int n)
{
    Sys_SetDipSwitch(4, (byte)(dip ^ (1<<n)));
    return TRUE;
}


static BOOL CALLBACK
SWDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    int switchnum;
    byte dip;
    int n;

    switch (msg) {

	case WM_INITDIALOG:
	    // associate the switch number with the window handle
#if TABBED_PROPSHEET
	    n = (int)(((PROPSHEETPAGE*)lParam)->lParam);
#else
	    n = (int)lParam;
#endif
	    SetProp(hwndDlg, "switchnum", (HANDLE)n);
	    optstate.hwndPage[n] = hwndDlg;
	    dip = Sys_GetDipswitch(n);
	    Refresh_dips(hwndDlg, dip);
	    return TRUE;

	case WM_COMMAND:
	    // retrieve the switch number associated with the window handle
	    switchnum = (int)GetProp(hwndDlg, "switchnum");
	    ASSERT(switchnum > 0);
	    dip = Sys_GetDipswitch(switchnum);
	    n = MapSW2Int(wParam);
	    switch (switchnum) {
		case 1: return handle_sw1(hwndDlg, dip, n);
		case 2: return handle_sw2(hwndDlg, dip, n);
		case 3: return handle_sw3(hwndDlg, dip, n);
		case 4: return handle_sw4(hwndDlg, dip, n);
	    }
	    return FALSE;


	case WM_NOTIFY:
	    return Handle_WM_NOTIFY((LPNMHDR)lParam);

	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


static void
set_fontmode_display(HWND hwndDlg)
{
    HWND hMainFont = GetDlgItem(hwndDlg, IDC_CGEN_BASEFONT);
    HWND hAltFont  = GetDlgItem(hwndDlg, IDC_CGEN_ALTFONT);
    HWND hFontMode = GetDlgItem(hwndDlg, IDC_CGEN_DH);
    HWND hText     = GetDlgItem(hwndDlg, IDC_STATIC_ALT);
    BOOL b = (GetSolFontMode() != MAINONLY);

    ComboBox_SetCurSel(hMainFont, (GetSolMainFont() == CHARGEN_6575));
    ComboBox_SetCurSel(hAltFont,  (GetSolAltFont()  == CHARGEN_DZONKHA));

    Button_SetCheck(hFontMode, b ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(hAltFont, b);
    EnableWindow(hText,    b);
}


static BOOL CALLBACK
CharsetDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HWND hMainFont = GetDlgItem(hwndDlg, IDC_CGEN_BASEFONT);
    HWND hAltFont  = GetDlgItem(hwndDlg, IDC_CGEN_ALTFONT);
    HWND hFontMode = GetDlgItem(hwndDlg, IDC_CGEN_DH);
    int n;

    switch (msg) {

	case WM_INITDIALOG:
	    ComboBox_InsertString(hMainFont, 0, (LPARAM)"6574");
	    ComboBox_InsertString(hMainFont, 1, (LPARAM)"6575");
	    ComboBox_InsertString(hAltFont,  0, (LPARAM)"Devanagri");
	    ComboBox_InsertString(hAltFont,  1, (LPARAM)"Dzonkha");
	    set_fontmode_display(hwndDlg);
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD(wParam)) {

		case IDC_CGEN_DH:
		    if (HIWORD(wParam) == BN_CLICKED) {
			int bst = Button_GetState(hFontMode) & 0x0003;
			SetSolFontMode((bst == BST_CHECKED) ? MAINNOW : MAINONLY);
			CreateCharMap();
			set_fontmode_display(hwndDlg);
			return TRUE;
		    }
		    return FALSE;

		case IDC_CGEN_BASEFONT:
		    switch (HIWORD(wParam)) {
			case CBN_SELENDOK:
			    n = ComboBox_GetCurSel(hMainFont);
			    SetSolMainFont((n) ? CHARGEN_6575
					       : CHARGEN_6574);
			    set_fontmode_display(hwndDlg);
			    CreateCharMap();
			    return TRUE;
			default:
			    break;
		    }
		    return FALSE;

		case IDC_CGEN_ALTFONT:
		    switch (HIWORD(wParam)) {
			case CBN_SELENDOK:
			    n = ComboBox_GetCurSel(hAltFont);
			    SetSolAltFont((n) ? CHARGEN_DZONKHA
					      : CHARGEN_DEVANAGRI);
			    set_fontmode_display(hwndDlg);
			    CreateCharMap();
			    return TRUE;
			default:
			    break;
		    }
		    return FALSE;

	    }
	    return FALSE;	// shouldn't ever happen


	case WM_NOTIFY:
	    return Handle_WM_NOTIFY((LPNMHDR)lParam);


	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


// pick up all the state and update the dialog to match
static void
refresh_audio_display(HWND hwndDlg)
{
    HWND hRate = GetDlgItem(hwndDlg, IDC_STATIC_AUDIO_RATE);
    HWND hBits = GetDlgItem(hwndDlg, IDC_STATIC_AUDIO_BITS);
    HWND hBut, hTB;
    int audioen, checked, pos, capture;

    hTB = GetDlgItem(hwndDlg, IDC_AUDIO_VOL_INTEN);
    Sys_AudioGet(SFXP_VOL_INTEN, &pos);
    Trackbar_SetPos(hTB, TRUE, pos);

    hTB = GetDlgItem(hwndDlg, IDC_AUDIO_VOL_FAN);
    Sys_AudioGet(SFXP_VOL_FAN, &pos);
    Trackbar_SetPos(hTB, TRUE, pos);

    hTB = GetDlgItem(hwndDlg, IDC_AUDIO_VOL_DISK);
    Sys_AudioGet(SFXP_VOL_DISK, &pos);
    Trackbar_SetPos(hTB, TRUE, pos);

    hBut = GetDlgItem(hwndDlg, IDC_AUDIO_EN);
    Sys_AudioGet(SFXP_ENABLE, &checked);
    Button_SetCheck(hBut, (checked) ? BST_CHECKED : BST_UNCHECKED);
    audioen = checked;

    if (!audioen) {
	SetWindowText(hRate, "");
	SetWindowText(hBits, "");
    } else {
	int rate, bits;
	WinAudioGet(WA_PROP_SAMPLERATE, &rate);
	WinAudioGet(WA_PROP_BITSPERSAMPLE, &bits);
	SetWindowText(hRate, (rate == 44100) ? "44 KHz" : "22 KHz");
	SetWindowText(hBits, (bits == 16) ? "16 bits" : "8 bits");
    }

    hBut = GetDlgItem(hwndDlg, IDC_BUT_AUDIO_START);
    EnableWindow(hBut, audioen);
    WinAudioGet(WA_PROP_CAPTURING, &capture);
    SetWindowText(hBut, (capture) ? "Stop" : "Capture...");
}


static BOOL CALLBACK
AudioDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {

	case WM_INITDIALOG:
	    {
		HWND hTB;
		hTB = GetDlgItem(hwndDlg, IDC_AUDIO_VOL_INTEN);
		Trackbar_SetRange(hTB, (BOOL)TRUE, (WORD)0, (WORD)100);
		hTB = GetDlgItem(hwndDlg, IDC_AUDIO_VOL_FAN);
		Trackbar_SetRange(hTB, (BOOL)TRUE, (WORD)0, (WORD)100);
		hTB = GetDlgItem(hwndDlg, IDC_AUDIO_VOL_DISK);
		Trackbar_SetRange(hTB, (BOOL)TRUE, (WORD)0, (WORD)100);
	    }
	    refresh_audio_display(hwndDlg);
	    return TRUE;

	case WM_COMMAND:

	    if (HIWORD(wParam) == BN_CLICKED) {
		HWND hBut = GetDlgItem(hwndDlg, LOWORD(wParam));
		int checked = ((Button_GetState(hBut) & 0x0003) == BST_CHECKED);
		int capture;
		switch (LOWORD(wParam)) {

		    case IDC_AUDIO_EN:
			if (checked) {
			    // see if we can get an audio channel
			    int rv = WinAudioOpen();
			    if (rv != WA_STAT_OK) {
				Button_SetCheck(hBut, BST_UNCHECKED);
				checked = FALSE;
			    }
			} else {
			    // close the open audio channel
			    WinAudioGet(WA_PROP_CAPTURING, &capture);
			    if (capture)
				SaveWaveClose();
			    WinAudioClose();
			}
			Sys_AudioSet(SFXP_ENABLE, checked);
			refresh_audio_display(hwndDlg);
			return TRUE;

		    case IDC_BUT_AUDIO_START:
			WinAudioGet(WA_PROP_CAPTURING, &capture);
			if (capture) {
			    SaveWaveClose();
			} else {
			    // pop up a file dialog to get a file name
			    if (FileOpenDlg(hwndDlg, FILE_WAVOUT)) {
				int ok;
				// remove it if it exists
				(void)DeleteFile((LPCTSTR)(winstate.ofn[FILE_WAVOUT].lpstrFile));
				// open it
				ok = SaveWaveOpen(winstate.ofn[FILE_WAVOUT].lpstrFile);
			    }
			}
			refresh_audio_display(hwndDlg);
			return TRUE;

		    default:
			return TRUE;	// not handled
		}
	    }
	    return FALSE;	// shouldn't ever happen


	case WM_HSCROLL:
	    {
		HWND hTB = (HWND)lParam;
		static const DWORD tbarid[] = { IDC_AUDIO_VOL_INTEN, IDC_AUDIO_VOL_FAN, IDC_AUDIO_VOL_DISK };
		static const int   propid[] = { SFXP_VOL_INTEN,      SFXP_VOL_FAN,      SFXP_VOL_DISK };
		int code = LOWORD(wParam);
		int curpos, newpos;
		int i;
		for(i=0; i<sizeof(tbarid)/sizeof(DWORD); i++) {
		    HWND hTB_ctl = GetDlgItem(hwndDlg, tbarid[i]);
		    if (hTB == hTB_ctl) {
			Sys_AudioGet(propid[i], &curpos);
			switch (code) {
			    case TB_TOP:      newpos =   0; break;
			    case TB_BOTTOM:   newpos = 100; break;
			    case TB_PAGEUP:   newpos = max(  0, curpos - 10); break;
			    case TB_PAGEDOWN: newpos = min(100, curpos + 10); break;
			    case TB_LINEUP:   newpos = max(  0, curpos -  1); break;
			    case TB_LINEDOWN: newpos = min(100, curpos +  1); break;
			    case TB_THUMBPOSITION:
			    case TB_THUMBTRACK:
				newpos = HIWORD(wParam);
				break;
			    default:
				return FALSE;	// not handled
			}
			Sys_AudioSet(propid[i], newpos);
			Trackbar_SetPos(hTB, TRUE, newpos);
			return TRUE;
		    } // if
		} // for
	    }
	    return TRUE;
	    

	case WM_NOTIFY:
	    return Handle_WM_NOTIFY((LPNMHDR)lParam);


	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


static void
set_svtopts_display(HWND hwndDlg)
{
    HWND hReal = GetDlgItem(hwndDlg, IDC_SVT_SPEEDREAL);
    HWND hFast = GetDlgItem(hwndDlg, IDC_SVT_SPEEDFAST);
    HWND hBaud = GetDlgItem(hwndDlg, IDC_SVT_IGNOREBAUD);
    HWND hMot  = GetDlgItem(hwndDlg, IDC_SVT_MOTORBUT);
    int realtime, ignorebaud, motorbut;

    Sys_GetTapeProp(1, TPROP_REALTIME,   &realtime);
    Sys_GetTapeProp(1, TPROP_IGNOREBAUD, &ignorebaud);
    GetSVTOpt(1,   SVTOpt_MOTENB,    &motorbut);

    Button_SetCheck(hReal,  realtime   ? BST_CHECKED : BST_UNCHECKED);
    Button_SetCheck(hFast, !realtime   ? BST_CHECKED : BST_UNCHECKED);
    Button_SetCheck(hBaud,  ignorebaud ? BST_CHECKED : BST_UNCHECKED);
    Button_SetCheck(hMot,   motorbut   ? BST_CHECKED : BST_UNCHECKED);
}


static BOOL CALLBACK
SVTDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    int bst;

    switch (msg) {

	case WM_INITDIALOG:
	    set_svtopts_display(hwndDlg);
	    return TRUE;

	case WM_COMMAND:
	    if (HIWORD(wParam) == BN_CLICKED) {
		switch (LOWORD(wParam)) {

		    case IDC_SVT_SPEEDREAL:
			Sys_SetTapeProp(0, TPROP_REALTIME, 1);
			return TRUE;

		    case IDC_SVT_SPEEDFAST:
			Sys_SetTapeProp(0, TPROP_REALTIME, 0);
			return TRUE;

		    case IDC_SVT_IGNOREBAUD:
			bst = Button_GetState(GetDlgItem(hwndDlg, IDC_SVT_IGNOREBAUD)) & 0x0003;
			Sys_SetTapeProp(0, TPROP_IGNOREBAUD, bst);
			Sys_SetTapeProp(1, TPROP_IGNOREBAUD, bst);
			return TRUE;

		    case IDC_SVT_MOTORBUT:
			bst = Button_GetState(GetDlgItem(hwndDlg, IDC_SVT_MOTORBUT)) & 0x0003;
			SetSVTOpt(0, SVTOpt_MOTENB, bst);
			SetSVTOpt(1, SVTOpt_MOTENB, bst);
			return TRUE;

		    default:
			break;
		}
	    }
	    return FALSE;


	case WM_NOTIFY:
	    return Handle_WM_NOTIFY((LPNMHDR)lParam);


	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


static BOOL CALLBACK
CPUDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    speed_t cpu_speed;
    BOOL regulated;
    HWND hButton;

    switch (msg) {

	case WM_INITDIALOG:
	    cpu_speed = Sys_Get8080Speed();
	    regulated = winstate.useHPTimer || winstate.useLPTimer;

	    hButton = GetDlgItem(hwndDlg, IDC_CPU_2_04MHZ);
	    Button_SetCheck(hButton, (cpu_speed == MHZ_2_04) ? BST_CHECKED : BST_UNCHECKED);
	    EnableWindow(hButton, regulated);

	    hButton = GetDlgItem(hwndDlg, IDC_CPU_2_38MHZ);
	    Button_SetCheck(hButton, (cpu_speed == MHZ_2_38) ? BST_CHECKED : BST_UNCHECKED);
	    EnableWindow(hButton, regulated);

	    hButton = GetDlgItem(hwndDlg, IDC_CPU_2_86MHZ);
	    EnableWindow(hButton, regulated);
	    Button_SetCheck(hButton, (cpu_speed == MHZ_2_86) ? BST_CHECKED : BST_UNCHECKED);

	    hButton = GetDlgItem(hwndDlg, IDC_CPU_999MHZ);
	    Button_SetCheck(hButton, (cpu_speed == MHZ_unregulated) ? BST_CHECKED : BST_UNCHECKED);
	    EnableWindow(hButton, regulated);

	    hButton = GetDlgItem(hwndDlg, IDC_CPU_STATS);
	    Button_SetCheck(hButton, (winstate.show_stats) ? BST_CHECKED : BST_UNCHECKED);
	    EnableWindow(hButton, regulated);
	    return TRUE;

	case WM_COMMAND:

	    if (HIWORD(wParam) == BN_CLICKED) {
		switch (LOWORD(wParam)) {

		    case IDC_CPU_2_04MHZ:
			Sys_Set8080Speed(MHZ_2_04);
			return TRUE;

		    case IDC_CPU_2_38MHZ:
			Sys_Set8080Speed(MHZ_2_38);
			return TRUE;

		    case IDC_CPU_2_86MHZ:
			Sys_Set8080Speed(MHZ_2_86);
			return TRUE;

		    case IDC_CPU_999MHZ:
			Sys_Set8080Speed(MHZ_unregulated);
			return TRUE;

		    case IDC_CPU_STATS:
			if (winstate.show_stats)
			    UI_NoteSpeed("");
			winstate.show_stats = !winstate.show_stats;
			return TRUE;

		    default:
			break;
		}
	    }
	    return FALSE;


	case WM_NOTIFY:
	    return Handle_WM_NOTIFY((LPNMHDR)lParam);


	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


static BOOL CALLBACK
DisplayDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HWND hButton;

    switch (msg) {

	case WM_INITDIALOG:
	    {
		disp_color_t color = GetSolDispColor();
		hButton = GetDlgItem(hwndDlg, IDC_DISP_WHITE);
		Button_SetCheck(hButton, (color == DISP_WHITE) ? BST_CHECKED : BST_UNCHECKED);
		hButton = GetDlgItem(hwndDlg, IDC_DISP_GREEN);
		Button_SetCheck(hButton, (color== DISP_GREEN)  ? BST_CHECKED : BST_UNCHECKED);
		hButton = GetDlgItem(hwndDlg, IDC_DISP_AMBER);
		Button_SetCheck(hButton, (color== DISP_AMBER)  ? BST_CHECKED : BST_UNCHECKED);

		hButton = GetDlgItem(hwndDlg, IDC_DISP_HIDEMOUSE);
		Button_SetCheck(hButton, (winstate.hidemouse) ? BST_CHECKED : BST_UNCHECKED);
		return TRUE;
	    }


	case WM_COMMAND:

	    if (HIWORD(wParam) == BN_CLICKED) {
		switch (LOWORD(wParam)) {

		    case IDC_DISP_WHITE:
			SetSolDispColor(DISP_WHITE);
			SetDispColors();
			return TRUE;

		    case IDC_DISP_GREEN:
			SetSolDispColor(DISP_GREEN);
			SetDispColors();
			return TRUE;

		    case IDC_DISP_AMBER:
			SetSolDispColor(DISP_AMBER);
			SetDispColors();
			return TRUE;

		    case IDC_DISP_HIDEMOUSE:
			winstate.hidemouse = !winstate.hidemouse;
			// don't need to do anything else.  as soon as
			// the user moves the mouse, the cursor will
			// disappear on the WM_MOUSEMOVE msg.
			return TRUE;

		    default:
			break;
		}
	    }
	    return FALSE;


	case WM_NOTIFY:
	    return Handle_WM_NOTIFY((LPNMHDR)lParam);


	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


// these are the choices for the base address of the NS disk controller.
// 0xE800 is the default one.
static int ns_base_addrs[] = { 0xE000, 0xE400, 0xE800, 0xEC00 };

static void
UpdateNStarDlgCtls(HWND hwndDlg)
{
    HWND hPresent  = GetDlgItem(hwndDlg, IDC_NSCTRLR_PRESENT);
    HWND hDrives1  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D1);
    HWND hDrives2  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D2);
    HWND hDrives3  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D3);
    HWND hDrives4  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D4);
    HWND hBaseAddr = GetDlgItem(hwndDlg, IDC_NSCTRLR_BASEADDR);
    HWND hDebug    = GetDlgItem(hwndDlg, IDC_NSCTRLR_DEBUG);
    HWND hStatic1  = GetDlgItem(hwndDlg, IDC_NSCTRLR_STATIC1);
    HWND hStatic2  = GetDlgItem(hwndDlg, IDC_NSCTRLR_STATIC2);
    int i, stat, v, enabled;

    stat = GetDiskControllerProp(VDCPROP_INSTALLED, &enabled);
    ASSERT(stat == VDCPROP_OK);
    Button_SetCheck(hPresent, (enabled) ? BST_CHECKED : BST_UNCHECKED);

    stat = GetDiskControllerProp(VDCPROP_BASEADDR, &v);
    ASSERT(stat == VDCPROP_OK);
    for(i=0; i<sizeof(ns_base_addrs)/sizeof(int); i++) {
	if (ns_base_addrs[i] == v)
	    ComboBox_SetCurSel(hBaseAddr, i);
    }

    stat = GetDiskControllerProp(VDCPROP_DRIVES, &v);
    ASSERT(stat == VDCPROP_OK);
    Button_SetCheck(hDrives1, (v==1) ? BST_CHECKED : BST_UNCHECKED);
    Button_SetCheck(hDrives2, (v==2) ? BST_CHECKED : BST_UNCHECKED);
    Button_SetCheck(hDrives3, (v==3) ? BST_CHECKED : BST_UNCHECKED);
    Button_SetCheck(hDrives4, (v==4) ? BST_CHECKED : BST_UNCHECKED);

    stat = GetDiskControllerProp(VDCPROP_DEBUGMODE, &v);
    ASSERT(stat == VDCPROP_OK);
    Button_SetCheck(hDebug, (v) ? BST_CHECKED : BST_UNCHECKED);

    // if the controller isn't installed, don't allow modifying
    // the properties of the non-existant controller
    EnableWindow(hDrives1,  enabled);
    EnableWindow(hDrives2,  enabled);
    EnableWindow(hDrives3,  enabled);
    EnableWindow(hDrives4,  enabled);
    EnableWindow(hBaseAddr, enabled);
    EnableWindow(hDebug,    enabled);
    EnableWindow(hStatic1,  enabled);
    EnableWindow(hStatic2,  enabled);
}


static BOOL CALLBACK
NStarDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HWND hPresent  = GetDlgItem(hwndDlg, IDC_NSCTRLR_PRESENT);
    HWND hDrives1  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D1);
    HWND hDrives2  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D2);
    HWND hDrives3  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D3);
    HWND hDrives4  = GetDlgItem(hwndDlg, IDC_NSCTRLR_D4);
    HWND hBaseAddr = GetDlgItem(hwndDlg, IDC_NSCTRLR_BASEADDR);
    HWND hDebug    = GetDlgItem(hwndDlg, IDC_NSCTRLR_DEBUG);
    char buff[20];
    int i, stat;

    switch (msg) {

	case WM_INITDIALOG:
	    for(i=0; i<sizeof(ns_base_addrs)/sizeof(int); i++) {
		int this_addr = ns_base_addrs[i];
		sprintf(buff, "0x%04X", this_addr);
		ComboBox_InsertString(hBaseAddr, i, (LPARAM)buff);
	    }
	    UpdateNStarDlgCtls(hwndDlg);
	    return TRUE;

	case WM_COMMAND:

	    switch (LOWORD(wParam)) {

		case IDC_NSCTRLR_PRESENT:
		    if (HIWORD(wParam) == BN_CLICKED) {
			i = (Button_GetCheck(hPresent) == BST_CHECKED);
			stat = SetDiskControllerProp(VDCPROP_INSTALLED, i);
			if (stat != VDCPROP_OK) {
			    UI_Alert("Error: didn't work");
			}
			UpdateNStarDlgCtls(hwndDlg);
			return TRUE;
		    }
		    return FALSE;

		case IDC_NSCTRLR_D1:
		case IDC_NSCTRLR_D2:
		case IDC_NSCTRLR_D3:
		case IDC_NSCTRLR_D4:
		    if (HIWORD(wParam) == BN_CLICKED) {
			i = 1*(Button_GetCheck(hDrives1) == BST_CHECKED)
			  + 2*(Button_GetCheck(hDrives2) == BST_CHECKED)
			  + 3*(Button_GetCheck(hDrives3) == BST_CHECKED)
			  + 4*(Button_GetCheck(hDrives4) == BST_CHECKED);
			stat = SetDiskControllerProp(VDCPROP_DRIVES, i);
			if (stat != VDCPROP_OK)
			    UpdateNStarDlgCtls(hwndDlg);
		    }
		    return FALSE;

		case IDC_NSCTRLR_BASEADDR:
		    if (HIWORD(wParam) == CBN_SELENDOK) {
			i = ComboBox_GetCurSel(hBaseAddr);
			stat = SetDiskControllerProp(VDCPROP_BASEADDR, ns_base_addrs[i]);
			if (stat != VDCPROP_OK) {
			    UI_Alert("Error: didn't work");
			    UpdateNStarDlgCtls(hwndDlg);
			}
			return TRUE;
		    }
		    return FALSE;

		case IDC_NSCTRLR_DEBUG:
		    if (HIWORD(wParam) == BN_CLICKED) {
			i = (Button_GetCheck(hDebug) == BST_CHECKED);
			stat = SetDiskControllerProp(VDCPROP_DEBUGMODE, i);
			if (stat != VDCPROP_OK) {
			    UI_Alert("Error: didn't work");
			    UpdateNStarDlgCtls(hwndDlg);
			}
			return TRUE;
		    }
		    return FALSE;

		default:
		    break;
	    }
	    return FALSE;


	case WM_NOTIFY:
	    return Handle_WM_NOTIFY((LPNMHDR)lParam);


	default:
	    break;
    }

    return FALSE;	// didn't handle the message
}


// ========================================================================
// The following code handle setting up and managing the container
// dialog.
// ========================================================================

// called once at startup
void
InitOpts(void)
{
    int i;

    optstate.valid_position = FALSE;

    for(i=0; i<NUMTABS; i++)
	optstate.hwndPage[i] = NULL;
}


// create a list of all sub-dialog windows
// perhaps not: check out PSP_USETITLE in dwFlags param and the
//      pszTitle member.  mentioned on p675 of Petzold book.
static void
InitOptsDialogs(void)
{
    int i;

    ZeroMemory(&optstate.pshdr, sizeof(PROPSHEETHEADER));
    ZeroMemory(&optstate.pspage, NUMTABS*sizeof(PROPSHEETPAGE));

    optstate.pshdr.dwSize  = sizeof(PROPSHEETHEADER);
    optstate.pshdr.dwFlags = PSH_MODELESS	// modeless dialog
			   | PSH_PROPSHEETPAGE	// use psp structures
			   | PSH_NOAPPLYNOW;	// hide APPLY button
    optstate.pshdr.hInstance   = winstate.hInst;
    optstate.pshdr.pszCaption  = "Emulator Options";
    optstate.pshdr.nPages      = NUMTABS;
    optstate.pshdr.nStartPage  = 0;
    optstate.pshdr.ppsp        = optstate.pspage;
    optstate.pshdr.pfnCallback = NULL;	// only if PSH_USECALLBACK

    // set all fields to defaults -- may get overridden later
    for(i=0; i<NUMTABS; i++) {
	optstate.pspage[i].dwSize      = sizeof(PROPSHEETPAGE);
	optstate.pspage[i].dwFlags     = 0;
	optstate.pspage[i].dwFlags     = PSP_USETITLE;
	optstate.pspage[i].hInstance   = winstate.hInst;
	optstate.pspage[i].pfnCallback = NULL;
	optstate.pspage[i].lParam      = (LPARAM)(i+1);
    }

    for(i=0; i<4; i++) {
	static WORD template[] =
		{ IDD_DIPSW1, IDD_DIPSW2, IDD_DIPSW3, IDD_DIPSW4 };
	optstate.pspage[i].dwFlags    |= PSP_USECALLBACK;
	optstate.pspage[i].hInstance   = winstate.hInst;
	optstate.pspage[i].pszTemplate = MAKEINTRESOURCE(template[i]);
	optstate.pspage[i].pfnDlgProc  = SWDlgProc;
	optstate.pspage[i].pfnCallback = DipSheetProc;
    }

    optstate.pspage[0].pszTitle = "DIP 1";
    optstate.pspage[1].pszTitle = "DIP 2";
    optstate.pspage[2].pszTitle = "DIP 3";
    optstate.pspage[3].pszTitle = "DIP 4";

    optstate.pspage[4].pszTemplate = MAKEINTRESOURCE(IDD_CPU);
    optstate.pspage[4].pfnDlgProc  = CPUDlgProc;
    optstate.pspage[4].pszTitle    = "CPU";

    optstate.pspage[5].pszTemplate = MAKEINTRESOURCE(IDD_DISPLAY);
    optstate.pspage[5].pfnDlgProc  = DisplayDlgProc;
    optstate.pspage[5].pszTitle    = "Display";

    optstate.pspage[6].pszTemplate = MAKEINTRESOURCE(IDD_SVT);
    optstate.pspage[6].pfnDlgProc  = SVTDlgProc;
    optstate.pspage[6].pszTitle    = "Tape";

    optstate.pspage[7].pszTemplate = MAKEINTRESOURCE(IDD_NSTAR_OPTS);
    optstate.pspage[7].pfnDlgProc  = NStarDlgProc;
    optstate.pspage[7].pszTitle    = "NS Disk";

    optstate.pspage[8].pszTemplate = MAKEINTRESOURCE(IDD_CHARSET);
    optstate.pspage[8].pfnDlgProc  = CharsetDlgProc;
    optstate.pspage[8].pszTitle    = "Charset";

    optstate.pspage[9].pszTemplate = MAKEINTRESOURCE(IDD_AUDIOPROPS);
    optstate.pspage[9].pfnDlgProc  = AudioDlgProc;
    optstate.pspage[9].pszTitle    = "Audio";

#if !TABBED_PROPSHEET
    // if we are using the alternate dialog interface...
    optstate.curdlg  = -1;	// not in use
    optstate.hCurDlg = NULL;	// not in use
#endif
}


// return status of whether the options window is showing or not
int
OptionsWindowVisible(void)
{
    return IsWindow(winstate.hOpts) &&
           IsWindowVisible(winstate.hOpts);
}


// get the window size and placement.
// return 0 if it isn't valid, 1 if it is valid.
static int
OptionsWindowGetSize(RECT *rc, int *visible)
{
    if (!optstate.valid_position || (winstate.hOpts==NULL))
	return 0;

    GetWindowRect(winstate.hOpts, rc);
    *visible = OptionsWindowVisible();
    return 1;
}

// set the window size and placement.
static void
OptionsWindowSetSize(RECT *rc)
{
    optstate.valid_position = TRUE;
    optstate.orig_x = rc->left;
    optstate.orig_y = rc->top;

    if (winstate.hOpts)
	SetWindowPos(winstate.hOpts, NULL,
		     optstate.orig_x, optstate.orig_y, 0, 0, 
		     SWP_NOSIZE | SWP_NOZORDER | SWP_SHOWWINDOW);
}


#if TABBED_PROPSHEET

// if the options window is closed, open it.
// then auto-pick the selected option.
void
ForceOption(int n)
{
    int i = 0;

    if (!OptionsWindowVisible())
	CreateOptsDialog();

    switch (n) {
	case IDM_OPTIONS_DIP1:    i = 0; break;
	case IDM_OPTIONS_DIP2:    i = 1; break;
	case IDM_OPTIONS_DIP3:    i = 2; break;
	case IDM_OPTIONS_DIP4:    i = 3; break;
	case IDM_OPTIONS_CPU:     i = 4; break;
	case IDM_OPTIONS_DISPLAY: i = 5; break;
	case IDM_OPTIONS_TAPES:   i = 6; break;
	case IDM_OPTIONS_NSDISK:  i = 7; break;
	case IDM_OPTIONS_CHARSET: i = 8; break;
	case IDM_OPTIONS_SOUNDS:  i = 9; break;
	default: ASSERT(0);
    }

    PropSheet_SetCurSel(winstate.hOpts, (HPROPSHEETPAGE)0, i);
}


// Propery sheets have an "OK" and a "CANCEL" button.  We don't want to
// have a cancel button at all, but if we just remove the cancel button,
// the OK button is in the wrong place.  So we go throught the steps below.
static void
HideCancelButton(void)
{
    HWND hwndCancel = GetDlgItem(winstate.hOpts, IDCANCEL);
    HWND hwndOK     = GetDlgItem(winstate.hOpts, IDOK);
#if 0
// the following code works fine on Win95/98, but for some
// unknown reason causes a core dump on NT.  More specifically,
// *this* code doesn't; even if this routine isn't called,
// clicking on the OK button results in a core dump, while
// clicking on CANCEL doesn't, even though the two cases take
// the same code path in the solace user code.
    RECT rc;

    // find location of CANCEL button
    GetWindowRect(hwndCancel, &rc);
    ScreenToClient(winstate.hOpts, (LPPOINT)&rc.left);
    ScreenToClient(winstate.hOpts, (LPPOINT)&rc.right);

    // disable cancel button
    EnableWindow(hwndCancel, FALSE);
    ShowWindow(hwndCancel, FALSE);

    // change label on OK button
    SetWindowText(hwndOK, "Done");

    // move OK button to where Cancel was
    (void)MoveWindow(hwndOK,
	  rc.left, rc.top, rc.right-rc.left, rc.bottom-rc.top,
	  TRUE);
#else
// because clicking on DONE causes a core dump on NT 4 (s.p. 3)
// for me, here we take a different approach.  We disable the
// OK button and change the label on CANCEL to read DONE instead.
// it is actually less code, but more obscure.

    // disable the OK button
    EnableWindow(hwndOK, FALSE);
    ShowWindow(hwndOK, FALSE);

    // change label on the Cancel button
    SetWindowText(hwndCancel, "Done");
#endif
}


// if the options dialog hasn't been accessed before, create the
// dialog and store a handle to the dialog.
// if the options dialog already exists and it is visible, hide it.
// if the options dialog already exists and it is invisible, show it.
void
CreateOptsDialog(void)
{
    if (IsWindow(winstate.hOpts)) {
	// it exists already
	if (IsWindowVisible(winstate.hOpts))
	    ShowWindow(winstate.hOpts, FALSE);
	else {
	    ShowWindow(winstate.hOpts, TRUE);
	    SetFocus(winstate.hOpts);
	}
	WinUpdateToolBar();
	return;
    }

    // create the list of sub-dialogs
    InitOptsDialogs();

    // OK, create it!
    winstate.hOpts = (HWND)PropertySheet(&optstate.pshdr);

    HideCancelButton();

    WinUpdateToolBar();

    if (optstate.valid_position)
	SetWindowPos(winstate.hOpts, NULL,
		     optstate.orig_x, optstate.orig_y, 0, 0, 
		     SWP_NOSIZE | SWP_NOZORDER | SWP_SHOWWINDOW);
    else
	SaveOptPosition();

    SetFocus(winstate.hOpts);
}

#else // !TABBED_PROPSHEET

// set up dialog to have chosen sub-dialog displayed on the right
// side of the main dialog
static void
PickOption(HWND hDlg, int n)
{
    ASSERT(hDlg != NULL);
    ASSERT(n >=0 && n < NUMTABS);

    // update only if different
    if (n == optstate.curdlg)
	return;	// nothing to do
    optstate.curdlg = n;

    // destroy old sub-dialog, if it exists
    if (optstate.hCurDlg != NULL) {
	DestroyWindow(optstate.hCurDlg);
	optstate.hCurDlg = NULL;
    }

    // create new dialog
    optstate.hCurDlg = CreateDialogParam(
			    winstate.hInst,
			    optstate.pspage[n].pszTemplate,
			    hDlg,	// make it subordinate
			    optstate.pspage[n].pfnDlgProc,
			    (LPARAM)(n+1) // indicate which switch it is
			);
    ASSERT(optstate.hCurDlg != NULL);

    SetWindowPos(optstate.hCurDlg,
		NULL,	// ignored
		optstate.subdlg_offx,
		0,
		1,1,	// ignored
		SWP_NOSIZE | SWP_NOZORDER);

    ShowWindow(optstate.hCurDlg, TRUE);

    SetFocus(GetDlgItem(hDlg, IDC_LIST_CATEGORY));
}


// if the options window is closed, open it.
// then auto-pick the selected option.
void
ForceOption(int n)
{
    HWND hDlg, hLB;
    int i = 0;

    if (!OptionsWindowVisible())
	CreateOptsDialog();

    hDlg = winstate.hOpts;
    hLB  = GetDlgItem(hDlg, IDC_LIST_CATEGORY);

    switch (n) {
	case IDM_OPTIONS_DIP1:    i = 0; break;
	case IDM_OPTIONS_DIP2:    i = 1; break;
	case IDM_OPTIONS_DIP3:    i = 2; break;
	case IDM_OPTIONS_DIP4:    i = 3; break;
	case IDM_OPTIONS_CPU:     i = 4; break;
	case IDM_OPTIONS_DISPLAY: i = 5; break;
	case IDM_OPTIONS_TAPES:   i = 6; break;
	case IDM_OPTIONS_NSDISK:  i = 7; break;
	case IDM_OPTIONS_CHARSET: i = 8; break;
	case IDM_OPTIONS_SOUNDS:  i = 9; break;
	default: ASSERT(0);
    }

    ListBox_SetCurSel(hLB, i);
    PickOption(hDlg, i);
}


// handle messages to options window
static LRESULT CALLBACK
OptsDialog(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HWND hLB = GetDlgItem(hDlg, IDC_LIST_CATEGORY);
    RECT rcLB;

    switch (msg) {

	case WM_INITDIALOG:
	    // stuff category strings into listbox
	    ListBox_AddString(hLB, "DIP Switch 1");
	    ListBox_AddString(hLB, "DIP Switch 2");
	    ListBox_AddString(hLB, "DIP Switch 3");
	    ListBox_AddString(hLB, "DIP Switch 4");
	    ListBox_AddString(hLB, "8080 Speed");
	    ListBox_AddString(hLB, "Display");
	    ListBox_AddString(hLB, "Tape Players");
	    ListBox_AddString(hLB, "N* Disk Ctlr");
	    ListBox_AddString(hLB, "Character set");
	    ListBox_AddString(hLB, "Audio");

	    // find out how wide the listbox is
	    GetWindowRect(hLB, &rcLB);	// in screen coordinates
	    ScreenToClient(hDlg, (LPPOINT)&rcLB.left);
	    ScreenToClient(hDlg, (LPPOINT)&rcLB.right);

	    optstate.subdlg_offx = rcLB.right + 10;	// 10 is fudge

	    ListBox_SetCurSel(hLB, 0);
	    PickOption(hDlg, 0);
	    return TRUE;

	case WM_COMMAND:
	    if ((LOWORD(wParam) == IDC_LIST_CATEGORY) &&
		(HIWORD(wParam) == LBN_SELCHANGE)) {

		int i = SendMessage(hLB, LB_GETCURSEL, 0, 0);
		PickOption(hDlg, i);

	    } else if (HIWORD(wParam) == BN_CLICKED) {

		int i = SendMessage(hLB, LB_GETCURSEL, 0, 0);

		switch (LOWORD(wParam)) {
		    char url[MAX_PATH];
		    char *tag;
		    case IDC_OPTION_HELP:
			switch (i) {
			    case 0: tag = "DIP1";    break;
			    case 1: tag = "DIP2";    break;
			    case 2: tag = "DIP3";    break;
			    case 3: tag = "DIP4";    break;
			    case 4: tag = "CPU";     break;
			    case 5: tag = "Display"; break;
			    case 6: tag = "Tape";    break;
			    case 7: tag = "NSDisk";  break;
			    case 8: tag = "Chargen"; break;
			    case 9: tag = "Audio";   break;
			    default: ASSERT(0);
			}
#if 0
    // FIXME:
    // this doesn't work because the call to ShellExecute()
    // craps out because it can't find the file named "...#tag".
    // do I have to go tromping through the registry to figure
    // out which browser is preferred and then explicitly launch
    // it with the full URL as an argument?  bleh.
			sprintf(url, "%s\\html\\solace_options.html#%s",
				winstate.basedir, tag);
#else
			sprintf(url, "%s\\html\\solace_options.html",
				winstate.basedir);
#endif
			(void)ShellExecute(NULL, "open", url, NULL, "c:\\", SW_SHOW);
			return TRUE;
		    default:
			break;
		}

	    }
	    break;

	case WM_CLOSE:
	    ShowWindow(hDlg, FALSE); // we don't really close -- we hide
	    SaveOptPosition();	// save it for later
	    WinUpdateToolBar();
	    break;

	default:
	    break;
    }

    return FALSE;
}


// if the options dialog hasn't been accessed before, create the
// dialog and store a handle to the dialog.
// if the options dialog already exists and it is visible, hide it.
// if the options dialog already exists and it is invisible, show it.
void
CreateOptsDialog(void)
{
    HWND hWnd;

    if (IsWindow(winstate.hOpts)) {
	// it exists already
	if (IsWindowVisible(winstate.hOpts))
	    ShowWindow(winstate.hOpts, FALSE);
	else {
	    ShowWindow(winstate.hOpts, TRUE);
	    SetFocus(winstate.hOpts);
	}
	WinUpdateToolBar();
	return;
    }

    // create the list of sub-dialogs
    InitOptsDialogs();

    // open up debugging container window
    hWnd = CreateDialog(
		winstate.hInst,			// program instance handle
		MAKEINTRESOURCE(IDD_OPTIONS),	// dialog template
		NULL,				// parent window handle
		OptsDialog			// dialog procedure
	    );

    ASSERT(hWnd != NULL);
    winstate.hOpts = hWnd;

    if (optstate.valid_position)
	SetWindowPos(winstate.hOpts, NULL,
		     optstate.orig_x, optstate.orig_y, 0, 0, 
		     SWP_NOSIZE | SWP_NOZORDER | SWP_SHOWWINDOW);
    else
	SaveOptPosition();

    ShowWindow(hWnd, TRUE);

    WinUpdateToolBar();

    SetFocus(winstate.hOpts);
}
#endif // TABBED_PROPSHEET


// ========================================================================
// Save and Load options from the .ini file
// ========================================================================

// .ini file name (sans directory)
const char *g_ini_file = "\\solace.ini";


// scan solace.ini for the options.
void
LoadOptions(void)
{
    char ini_file[MAX_PATH];
    char buf[80];
    char ini_version[20];
    int i, visible, w, h;
    RECT rc;
    UINT arg;
    DWORD fstat;

    strcpy(ini_file, winstate.basedir);
    strcat(ini_file, g_ini_file);

    fstat = GetFileAttributes(ini_file);
    if (fstat == 0xFFFFFFFF)
	return;


    // get ini file format
    GetPrivateProfileString("fileformat", "version", "", ini_version, sizeof(ini_version), ini_file);


    GetPrivateProfileString("cpu", "speed", "2.04MHz", buf, sizeof(buf), ini_file);
         if (!strcmp(buf, "999MHz"))  Sys_Set8080Speed(MHZ_unregulated);
    else if (!strcmp(buf, "2.86MHz")) Sys_Set8080Speed(MHZ_2_86);
    else if (!strcmp(buf, "2.38MHz")) Sys_Set8080Speed(MHZ_2_38);
    else                              Sys_Set8080Speed(MHZ_2_04);


    GetPrivateProfileString("display", "color", "white", buf, sizeof(buf), ini_file);
         if (!strcmp(buf, "amber")) SetSolDispColor(DISP_AMBER);
    else if (!strcmp(buf, "green")) SetSolDispColor(DISP_GREEN);
    else                            SetSolDispColor(DISP_WHITE);

    winstate.hidemouse  = GetPrivateProfileInt("display", "hidemouse", 0, ini_file);
    winstate.show_stats = GetPrivateProfileInt("display", "showstats", 0, ini_file);


    // state for four dip switches
    for(i=1; i<5; i++) {
	char buf2[10], buf3[10];
	sprintf(buf2, "sw%d", i);
	sprintf(buf3, "0x%02X", Sys_GetDipswitch(i));
	GetPrivateProfileString("dipswitch", buf2, buf3,  buf, sizeof(buf), ini_file);
	sscanf(buf, "0x%02X", &arg);
	Sys_SetDipSwitch(i, (byte)arg);
    }


    // state for font selection method and current font
    GetPrivateProfileString("charset", "mode", "normal", buf, sizeof(buf), ini_file);
    if (!strcmp(buf, "dual"))
	SetSolFontMode(MAINNOW);	// allow toggle between main and alt
    else
	SetSolFontMode(MAINONLY);	// only main

    GetPrivateProfileString("charset", "mainfont", "6574", buf, sizeof(buf), ini_file);
    if (strcmp(buf, "6575")) SetSolMainFont(CHARGEN_6574);
			else SetSolMainFont(CHARGEN_6575);
    GetPrivateProfileString("charset", "altfont",  "Devanagri", buf, sizeof(buf), ini_file);
    if (strcmp(buf, "Dzonkha")) SetSolAltFont(CHARGEN_DZONKHA);
			   else SetSolAltFont(CHARGEN_DEVANAGRI);


    // restore main window size and placement
    if (strlen(ini_version) > 0) {

	int top, left, w, h;

	GetPrivateProfileString("windows", "main", "", buf, sizeof(buf), ini_file);
	if (sscanf(buf, "org=%d,%d, size=%d,%d", &left, &top, &w, &h) == 4) {
	    // move the window without resizing
	    SetWindowPos(winstate.hWnd, HWND_TOP,
			 left, top, 0, 0,
			 SWP_NOSIZE | SWP_NOACTIVATE);
	    // resize the window to give sol screen given dimensions
	    SetSolScreenSize(winstate.hWnd, w, h);
	}

    } else {

	int top, left, bottom, right;

	// restore main window size and placement
	GetPrivateProfileString("windows", "main", "", buf, sizeof(buf), ini_file);
	if (sscanf(buf, "%d,%d,%d,%d", &top, &left, &bottom, &right) == 4) {
	    // dirty, dirty: add in size of toolbar since that is the new element vs pre-3.0
	    bottom += winstate.toolbar_h;
	    MoveWindow(winstate.hWnd, left, top, right-left, bottom-top, FALSE);
	}
    }


    // restore debugger window size and placement
    GetPrivateProfileString("windows", "debug", "", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d:org=%d,%d, size=%d,%d", &visible, &rc.left, &rc.top, &w, &h) == 5) {
	rc.right  = rc.left + w;
	rc.bottom = rc.top  + h;
	DebuggerWindowSetSize(&rc);
	if (visible)
	    UI_DbgWin(DW_Activate, 0); // open debugger window
    }


    // restore options window placement
    GetPrivateProfileString("windows", "options", "", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d,%d", &rc.left, &rc.top) == 2) {
	rc.right  = rc.left + w;
	rc.bottom = rc.top  + h;
	OptionsWindowSetSize(&rc);
    }


    // ------ save default directories for ROMs, programs, and scripts ------

    GetPrivateProfileString("files", "ROMname", "", buf, sizeof(buf), ini_file);
    if (strlen(buf) > 0) {
	char *path = HostExtractPath(buf);
	strcpy((char*)winstate.ofn[FILE_ROM].lpstrFile, buf);
	strcpy((char*)winstate.ofn[FILE_ROM].lpstrInitialDir, path);
	free(path);
    }

    GetPrivateProfileString("files", "PROGdir", "", buf, sizeof(buf), ini_file);
    if (strlen(buf) > 0)
	strcpy((char*)winstate.ofn[FILE_PROGRAM].lpstrInitialDir, buf);

    GetPrivateProfileString("files", "VDISKdir", "", buf, sizeof(buf), ini_file);
    if (strlen(buf) > 0)
	strcpy((char*)winstate.ofn[FILE_VDISK].lpstrInitialDir, buf);

    GetPrivateProfileString("files", "SCRIPTdir", "", buf, sizeof(buf), ini_file);
    if (strlen(buf) > 0)
	strcpy((char*)winstate.ofn[FILE_SCRIPT].lpstrInitialDir, buf);

    GetPrivateProfileString("files", "WAVOUTdir", "", buf, sizeof(buf), ini_file);
    if (strlen(buf) > 0)
	strcpy((char*)winstate.ofn[FILE_WAVOUT].lpstrInitialDir, buf);


    // ------ get options for NS disk controller ------

    GetPrivateProfileString("nstar", "present", "", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d", &i) == 1)
	SetDiskControllerProp(VDCPROP_INSTALLED, i);

    GetPrivateProfileString("nstar", "baseaddr", "", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "0x%x", &i) == 1)
	SetDiskControllerProp(VDCPROP_BASEADDR, i);

    GetPrivateProfileString("nstar", "numdrives", "", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d", &i) == 1)
	SetDiskControllerProp(VDCPROP_DRIVES, i);

    GetPrivateProfileString("nstar", "debug", "", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d", &i) == 1)
	SetDiskControllerProp(VDCPROP_DEBUGMODE, i);


    // ------ get options for audio interface ------

    // set defaults first
    Sys_AudioSet(SFXP_ENABLE, 0);
    Sys_AudioSet(SFXP_VOL_INTEN, 0);
    Sys_AudioSet(SFXP_VOL_FAN, 0);
    Sys_AudioSet(SFXP_VOL_DISK, 0);

    GetPrivateProfileString("audio", "enabled", "1", buf, sizeof(buf), ini_file);
    if ((sscanf(buf, "%d", &i) == 1) && (i != 0)) {
	int rv = WinAudioOpen();
	if (rv == WA_STAT_OK)
	    Sys_AudioSet(SFXP_ENABLE, 1);	// turn it on
    }

    GetPrivateProfileString("audio", "volume_inten", "0", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d", &i) == 1)
	Sys_AudioSet(SFXP_VOL_INTEN, i);

    GetPrivateProfileString("audio", "volume_fan", "60", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d", &i) == 1)
	Sys_AudioSet(SFXP_VOL_FAN, i);

    GetPrivateProfileString("audio", "volume_disk", "60", buf, sizeof(buf), ini_file);
    if (sscanf(buf, "%d", &i) == 1)
	Sys_AudioSet(SFXP_VOL_DISK, i);


    // ------ get options for tape interface ------

    WinTapeLoadOptions(ini_file);
}


// save program options to solace.ini.
// the ini file is located wherever the current program is located.
void
SaveOptions(void)
{
    char ini_file[MAX_PATH];
    char buf[80];
    char *str;
    RECT rc;
    DWORD fstat;
    int i, w, h, visible;

    strcpy(ini_file, winstate.basedir);
    strcat(ini_file, g_ini_file);

    fstat = GetFileAttributes(ini_file);
    if (fstat == 0xFFFFFFFF) {
	int rslt;
	sprintf(buf, "Creating file '%s'", ini_file);
	rslt = MessageBox(winstate.hWnd, buf, "Confirm",
			    MB_OKCANCEL | MB_ICONEXCLAMATION | MB_TOPMOST);
	if (rslt != IDOK)
	    return;
    } else if (fstat & (FILE_ATTRIBUTE_READONLY | FILE_ATTRIBUTE_DIRECTORY)) {
	UI_Alert("Can't open file '%s': it is not writable", ini_file);
	return;
    }


    // just in case it changes monumentally in the future
    WritePrivateProfileString("fileformat", "version", "3.0", ini_file);


    switch (Sys_Get8080Speed()) {
	case MHZ_2_04: str = "2.04MHz"; break;
	case MHZ_2_38: str = "2.38MHz"; break;
	case MHZ_2_86: str = "2.86MHz"; break;
	      default: str = "999MHz";  break;
    }
    WritePrivateProfileString("cpu", "speed", str, ini_file);


    switch (GetSolDispColor()) {
	case DISP_WHITE: str = "white"; break;
	case DISP_GREEN: str = "green"; break;
	case DISP_AMBER: str = "amber"; break;
    }
    WritePrivateProfileString("display", "color", str, ini_file);

    str = (winstate.hidemouse) ? "1" : "0";
    WritePrivateProfileString("display", "hidemouse", str, ini_file);
    str = (winstate.show_stats) ? "1" : "0";
    WritePrivateProfileString("display", "showstats", str, ini_file);


    sprintf(buf, "0x%02X", Sys_GetDipswitch(1));
    WritePrivateProfileString("dipswitch", "sw1", buf, ini_file);
    sprintf(buf, "0x%02X", Sys_GetDipswitch(2));
    WritePrivateProfileString("dipswitch", "sw2", buf, ini_file);
    sprintf(buf, "0x%02X", Sys_GetDipswitch(3));
    WritePrivateProfileString("dipswitch", "sw3", buf, ini_file);
    sprintf(buf, "0x%02X", Sys_GetDipswitch(4));
    WritePrivateProfileString("dipswitch", "sw4", buf, ini_file);


    // it doesn't seem likely that the original hardware could
    // remember the state of the font chosen between power cycles,
    // so all we keep is whether we allow an alternate mode.
    str = (GetSolFontMode() == MAINONLY) ? "normal" : "dual";
    WritePrivateProfileString("charset", "mode", str, ini_file);
    str = SolMainFontName();
    WritePrivateProfileString("charset", "mainfont", str, ini_file);
    str = SolAltFontName();
    WritePrivateProfileString("charset", "altfont", str, ini_file);


    // save main window placement
    // we save the size not of the window, but of the sol screen.
    // this allows us to change the appearance of the frame yet keep
    // the same amount of the sol screen visible.
    GetWindowRect(winstate.hWnd, &rc);
    GetSolScreenSize(winstate.hWnd, &w, &h);
    sprintf(buf, "org=%d,%d, size=%d,%d", rc.left, rc.top, w, h);
    WritePrivateProfileString("windows", "main", buf, ini_file);

    // save debugger window placement
    if (DebuggerWindowGetSize(&rc, &visible)) {
	sprintf(buf, "%d:org=%d,%d, size=%d,%d", visible,
		rc.left, rc.top,				// x,y
		rc.right-rc.left, rc.bottom-rc.top);		// w,h
	WritePrivateProfileString("windows", "debug", buf, ini_file);
    }

    // save options window placement
    if (OptionsWindowGetSize(&rc, &visible)) {
	sprintf(buf, "%d,%d", rc.left, rc.top);		// x,y
	WritePrivateProfileString("windows", "options", buf, ini_file);
    }


    // save default directories for ROMs and programs

    if (strlen(winstate.ofn[FILE_ROM].lpstrFile) > 0)
	WritePrivateProfileString("files", "ROMname", winstate.ofn[FILE_ROM].lpstrFile, ini_file);

    if (strlen(winstate.ofn[FILE_PROGRAM].lpstrInitialDir) > 0)
	WritePrivateProfileString("files", "PROGdir", winstate.ofn[FILE_PROGRAM].lpstrInitialDir, ini_file);

    if (strlen(winstate.ofn[FILE_VDISK].lpstrInitialDir) > 0)
	WritePrivateProfileString("files", "VDISKdir", winstate.ofn[FILE_VDISK].lpstrInitialDir, ini_file);

    if (strlen(winstate.ofn[FILE_SCRIPT].lpstrInitialDir) > 0)
	WritePrivateProfileString("files", "SCRIPTdir", winstate.ofn[FILE_SCRIPT].lpstrInitialDir, ini_file);

    if (strlen(winstate.ofn[FILE_WAVOUT].lpstrInitialDir) > 0)
	WritePrivateProfileString("files", "WAVOUTdir", winstate.ofn[FILE_WAVOUT].lpstrInitialDir, ini_file);


    // ------ save options for NS disk controller ------

    GetDiskControllerProp(VDCPROP_INSTALLED, &i);
    sprintf(buf, "%d", i);
    WritePrivateProfileString("nstar", "present", buf, ini_file);

    GetDiskControllerProp(VDCPROP_BASEADDR, &i);
    sprintf(buf, "0x%04X", i);
    WritePrivateProfileString("nstar", "baseaddr", buf, ini_file);

    GetDiskControllerProp(VDCPROP_DRIVES, &i);
    sprintf(buf, "%d", i);
    WritePrivateProfileString("nstar", "numdrives", buf, ini_file);

    GetDiskControllerProp(VDCPROP_DEBUGMODE, &i);
    sprintf(buf, "%d", i);
    WritePrivateProfileString("nstar", "debug", buf, ini_file);


    // ------ save options for audio interface ------

    Sys_AudioGet(SFXP_ENABLE, &i);
    sprintf(buf, "%d", i);
    WritePrivateProfileString("audio", "enabled", buf, ini_file);

    Sys_AudioGet(SFXP_VOL_INTEN, &i);
    sprintf(buf, "%d", i);
    WritePrivateProfileString("audio", "volume_inten", buf, ini_file);

    Sys_AudioGet(SFXP_VOL_FAN, &i);
    sprintf(buf, "%d", i);
    WritePrivateProfileString("audio", "volume_fan", buf, ini_file);

    Sys_AudioGet(SFXP_VOL_DISK, &i);
    sprintf(buf, "%d", i);
    WritePrivateProfileString("audio", "volume_disk", buf, ini_file);


    // ------ save options for tape interface ------

    WinTapeSaveOptions(ini_file);
}

