// SCU.C
// Titan Hardware Routines

#include <string.h>
#include "global.h"
#include "mem.h"
#include "scu.h"

int Timer0;

// DMA Stuff
//bool dmaOn;
dmaStruct *dma;

void dmaTransfer(int);
void Interrupt(int);

void initHW(void)
{
	memset(SCU, 0, 0xd0);

	SCU_INT_MASK = 0xbfff;
	SCU_INT_STATUS = 0;

	//dmaOn = FALSE;
	dma = (dmaStruct*)(SCU);
	dma[0].add = dma[1].add = dma[2].add = 0x101;
}

// updateHW(): called to update hardware and interrupts
void updateHW(void)
{
}

// checkHW(): called when register is written to memory
// NOTE: don't forget to check for DMA triggers
void checkHW(uint32 addr)
{
	int tmp, scrap;

	switch (addr & 0xff) {
		case (0x10):
			if (DMACHECK(0)) dmaTransfer(0);
			break;
		case (0x30):
			if (DMACHECK(1)) dmaTransfer(1);
			break;
		case (0x50):
			if (DMACHECK(2)) dmaTransfer(2);
			break;

		case (0xa4): // Interrupt Register
			if (SCU_INT_MASK & SCU_INT_STATUS)
			{
				tmp = 13;
				scrap = (SCU_INT_MASK & SCU_INT_STATUS) & 0x2fff;
				while (SCU_INT_STATUS & 0x2fff) { if ((scrap >> tmp--) & 0x1) Interrupt(tmp+1); }
			}
			break;
		default:
			break;
	}
}

void Interrupt(int level)
{
	// disable interrupt when complete
	SCU_INT_STATUS &= ~(1 << level);
}

void dmaTransfer(int level)
{
	uint32 raddr, waddr;
	register uint32 count;
	uint8 rstep, wstep;
	const uint8 itab[8] = { 0, 2, 4, 8, 16, 32, 64, 128 };

	//dmaOn = TRUE;
	if (dma[level].mode & 0x1000000)
	{
		ConsoleMsg(MSG_NORM, "DMA %d Indirect Mode Intercepted", level);
	}
    else //Direct DMA Mode
	{
        raddr = dma[level].raddr & 0x7ffffff;
        waddr = dma[level].waddr & 0x7ffffff;
        count = dma[level].count;

		// is mem range in CS2?
		if ((waddr >= 0x5800000) && (waddr < 0x5900000)) {
		    rstep = dma[level].add & 0x100 ? 4 : 0;
			wstep = itab[dma[level].add & 0x2];
		}
		else {
	        wstep = itab[dma[level].add & 0x7];
			rstep = 4;
		}

		// inside A-Bus/B-Bus
		if ((waddr >= 0x5a00000) && (waddr < 0x5fe0000)) 
		{
			//if the read add value is 4, make the step equal to 1
			rstep = (rstep && rstep);

			while (count > 0) {
				memSetByte(waddr, memGetByte(raddr));
				waddr += wstep;
				raddr += rstep;
				count--;
			}
		}
		else { 	// outside of A-Bus/B-Bus
			while (count > 0) {
				memSetLong(waddr, memGetLong(raddr));
				waddr += 4;
				raddr += rstep;
				count -= 4;
			}
		}

		// update/save addresses
		if (dma[level].mode & 0x10000)
			dma[level].raddr = raddr;
		if (dma[level].mode & 0x100)
			dma[level].waddr = waddr;
	}

	//dmaOn = FALSE;
	//Interrupt(SH2_INT_DMA0 - level);
}

void updateMiscRegs(unsigned long addr)
{
	uint64 temp;

	switch (addr & 0xff)
	{
        case 0x04:
			if(SCU_DIVU_DVSR != 0)
			{
				(*(uint32*)&temp) = SCU_DIVU_DVDNT;
                SCU_DIVU_DVDNT  = (*(uint32*)&temp) / SCU_DIVU_DVSR;
                SCU_DIVU_DVDNTL = (*(uint32*)&temp);
                SCU_DIVU_DVDNTH = (*(uint32*)&temp) % SCU_DIVU_DVSR;
			}
			break;

		case 0x14:
			if(SCU_DIVU_DVSR != 0)
			{
                temp = ((uint64)SCU_DIVU_DVDNTH << 32) | SCU_DIVU_DVDNTL;
                SCU_DIVU_DVDNTH=(long)((sint64)temp % (long)SCU_DIVU_DVSR);
                SCU_DIVU_DVDNTL=(long)((sint64)temp / (long)SCU_DIVU_DVSR);
			}
			break;

		default: break;
	}
}