// SH2.C
// SH2 CPU Emulation Routines

#include <stdlib.h>
#include <string.h>
#include "global.h"
#include "opcodes.h"
#include "mem.h"
#include "sh2.h"

#define source		(uint8)((sh2->instr >> 4) & 0xf) //m
#define dest		(uint8)((sh2->instr >> 8) & 0xf) //n
#define offset1		(sh2->instr & 0xf)
#define offset2		(sh2->instr & 0xff)
#define offset3		(sh2->instr & 0xfff)
#define immediate	(signed char)(sh2->instr)

#define setTBit(x)		SR = (SR & 0xfffffffe) | x

#define setMBit(x)		SR = (SR & 0xfffffdff) | ((x&1) << 9)
#define setQBit(x)		SR = (SR & 0xfffffeff) | ((x&1) << 8)
#define setSBit(x)		SR = (SR & 0xfffffffd) | ((x&1) << 1)

#define DelayBranch(x)	PC += 2; \
						DelayWait = x; \
						doInst(sh2); \
						PC = DelayWait - 2;

tagSH2 master, slave;

i_desc opcode_list[] =
{
        { ZERO_F,	"clrt",						0xffff, 0x8,	0, sh2_CLRT},
        { ZERO_F,   "clrmac",					0xffff, 0x28,	0, sh2_CLRMAC},
        { ZERO_F,   "div0u",					0xffff, 0x19,	0, sh2_DIV0U},
        { ZERO_F,   "nop",						0xffff, 0x9,	0, sh2_NOP},
        { ZERO_F,   "rte",						0xffff, 0x2b,	0, sh2_RTE},
        { ZERO_F,   "rts",						0xffff, 0xb,	0, sh2_RTS},
        { ZERO_F,   "sett",						0xffff, 0x18,	0, sh2_SETT},
        { ZERO_F,   "sleep",					0xffff, 0x1b,	0, sh2_SLEEP},
        { N_F,      "cmp/pl r%d",				0xf0ff, 0x4015,	0, sh2_CMP_PL},
        { N_F,      "cmp/pz r%d",				0xf0ff, 0x4011,	0, sh2_CMP_PZ},
        { N_F,      "dt r%d",					0xf0ff, 0x4010,	0, sh2_DT},
        { N_F,      "movt r%d",					0xf0ff, 0x0029,	0, sh2_MOVT},
        { N_F,      "rotl r%d",					0xf0ff, 0x4004,	0, sh2_ROTL},
        { N_F,      "rotr r%d",					0xf0ff, 0x4005,	0, sh2_ROTR},
        { N_F,      "rotcl r%d",				0xf0ff, 0x4024,	0, sh2_ROTCL},
        { N_F,      "rotcr r%d",				0xf0ff, 0x4025,	0, sh2_ROTCR},
        { N_F,      "shal r%d",					0xf0ff, 0x4020,	0, sh2_SHL},
        { N_F,      "shar r%d",					0xf0ff, 0x4021,	0, sh2_SHAR},
        { N_F,      "shll r%d",					0xf0ff, 0x4000,	0, sh2_SHL},
        { N_F,      "shlr r%d",					0xf0ff, 0x4001,	0, sh2_SHLR},
        { N_F,      "shll2 r%d",				0xf0ff, 0x4008,	0, sh2_SHLL2},
        { N_F,      "shlr2 r%d",				0xf0ff, 0x4009,	0, sh2_SHLR2},
        { N_F,      "shll8 r%d",				0xf0ff, 0x4018,	0, sh2_SHLL8},
        { N_F,      "shlr8 r%d",				0xf0ff, 0x4019,	0, sh2_SHLR8},
        { N_F,      "shll16 r%d",				0xf0ff, 0x4028,	0, sh2_SHLL16},
        { N_F,      "shlr16 r%d",				0xf0ff, 0x4029,	0, sh2_SHLL16},
        { N_F,      "stc sr, r%d",				0xf0ff, 0x0002,	0, sh2_STC_SR},
        { N_F,      "stc gbr, r%d",				0xf0ff, 0x0012,	0, sh2_STC_GBR},
        { N_F,      "stc vbr, r%d",				0xf0ff, 0x0022,	0, sh2_STC_VBR},
        { N_F,      "sts mach, r%d",			0xf0ff, 0x000a,	0, sh2_STS_MACH},
        { N_F,      "sts macl, r%d",			0xf0ff, 0x001a,	0, sh2_STS_MACL},
        { N_F,      "sts pr, r%d",				0xf0ff, 0x002a,	0, sh2_STS_PR},
        { N_F,      "tas.b @r%d",				0xf0ff, 0x401b,	0, sh2_TAS_B},
        { N_F,      "stc.l sr, @-r%d",			0xf0ff, 0x4003,	0, sh2_STC_SR_DEC},
        { N_F,      "stc.l gbr, @-r%d",			0xf0ff, 0x4013,	0, sh2_STC_GBR_DEC},
        { N_F,      "stc.l vbr, @-r%d",			0xf0ff, 0x4023,	0, sh2_STC_VBR_DEC},
        { N_F,      "sts.l mach, @-r%d",		0xf0ff, 0x4002,	0, sh2_STS_MACH_DEC},
        { N_F,      "sts.l macl, @-r%d",		0xf0ff, 0x4012,	0, sh2_STS_MACL_DEC},
        { N_F,      "sts.l pr, @-r%d",			0xf0ff, 0x4022,	0, sh2_STS_PR_DEC},
        { M_F,      "ldc r%d, sr",				0xf0ff, 0x400e,	0, sh2_LDC_SR},
        { M_F,      "ldc r%d, gbr",				0xf0ff, 0x401e,	0, sh2_LDC_GBR},
        { M_F,      "ldc r%d, vbr",				0xf0ff, 0x402e,	0, sh2_LDC_VBR},
        { M_F,      "lds r%d, mach",			0xf0ff, 0x400a,	0, sh2_LDS_MACH},
        { M_F,      "lds r%d, macl",			0xf0ff, 0x401a,	0, sh2_LDS_MACL},
        { M_F,      "lds r%d, pr",				0xf0ff, 0x402a,	0, sh2_LDS_PR},
        { M_F,      "jmp @r%d",					0xf0ff, 0x402b,	0, sh2_JMP},
        { M_F,      "jsr @r%d",					0xf0ff, 0x400b,	0, sh2_JSR},
        { M_F,      "ldc.l @r%d+, sr",			0xf0ff, 0x4007,	0, sh2_LDC_SR_INC},
        { M_F,      "ldc.l @r%d+, gbr",			0xf0ff, 0x4017,	0, sh2_LDC_GBR_INC},
        { M_F,      "ldc.l @r%d+, vbr",			0xf0ff, 0x4027,	0, sh2_LDC_VBR_INC},
        { M_F,      "lds.l @r%d+, mach",		0xf0ff, 0x4006,	0, sh2_LDS_MACH_INC},
        { M_F,      "lds.l @r%d+, macl",		0xf0ff, 0x4016,	0, sh2_LDS_MACL_INC},
        { M_F,      "lds.l @r%d+, pr",			0xf0ff, 0x4026,	0, sh2_LDS_PR_INC},
        { M_F,      "braf r%d",					0xf0ff, 0x0023,	0, sh2_BRAF},
        { M_F,      "bsrf r%d",					0xf0ff, 0x0003,	0, sh2_BSRF},
        { NM_F,     "add r%d, r%d",				0xf00f, 0x300c,	0, sh2_ADD},
        { NM_F,     "addc r%d, r%d",			0xf00f, 0x300e,	0, sh2_ADDC},
        { NM_F,     "addv r%d, r%d",			0xf00f, 0x300f,	0, sh2_ADDV},
        { NM_F,     "and r%d, r%d",				0xf00f, 0x2009,	0, sh2_AND},
        { NM_F,     "cmp/eq r%d, r%d",			0xf00f, 0x3000,	0, sh2_CMP_EQ},
        { NM_F,     "cmp/hs r%d, r%d",			0xf00f, 0x3002,	0, sh2_CMP_HS},
        { NM_F,     "cmp/ge r%d, r%d",			0xf00f, 0x3003,	0, sh2_CMP_GE},
        { NM_F,     "cmp/hi r%d, r%d",			0xf00f, 0x3006,	0, sh2_CMP_HI},
        { NM_F,     "cmp/gt r%d, r%d",			0xf00f, 0x3007,	0, sh2_CMP_GT},
        { NM_F,     "cmp/str r%d, r%d",			0xf00f, 0x200c,	0, sh2_CMP_STR},
        { NM_F,     "div1 r%d, r%d",			0xf00f, 0x3004,	0, sh2_DIV1},
        { NM_F,     "div0s r%d, r%d",			0xf00f, 0x2007,	0, sh2_DIV0S},
        { NM_F,     "dmuls.l r%d, r%d",			0xf00f, 0x300d,	0, sh2_DMULS_L},
        { NM_F,     "dmulu.l r%d, r%d",			0xf00f, 0x3005,	0, sh2_DMULU_L},
        { NM_F,     "exts.b r%d, r%d",			0xf00f, 0x600e,	0, sh2_EXTS_B},
        { NM_F,     "exts.w r%d, r%d",			0xf00f, 0x600f,	0, sh2_EXTS_W},
        { NM_F,     "extu.b r%d, r%d",			0xf00f, 0x600c,	0, sh2_EXTU_B},
        { NM_F,     "extu.w r%d, r%d",			0xf00f, 0x600d,	0, sh2_EXTU_W},
        { NM_F,     "mov r%d, r%d",				0xf00f, 0x6003,	0, sh2_MOVR},
        { NM_F,     "mul.l r%d, r%d",			0xf00f, 0x0007,	0, sh2_MUL_L},
        { NM_F,     "muls.w r%d, r%d",			0xf00f, 0x200f,	0, sh2_MULS},
        { NM_F,     "mulu.w r%d, r%d",			0xf00f, 0x200e,	0, sh2_MULU},
        { NM_F,     "neg r%d, r%d",				0xf00f, 0x600b,	0, sh2_NEG},
        { NM_F,     "negc r%d, r%d",			0xf00f, 0x600a,	0, sh2_NEGC},
        { NM_F,     "not r%d, r%d",				0xf00f, 0x6007,	0, sh2_NOT},
        { NM_F,     "or r%d, r%d",				0xf00f, 0x200b,	0, sh2_OR},
        { NM_F,     "sub r%d, r%d",				0xf00f, 0x3008,	0, sh2_SUB},
        { NM_F,     "subc r%d, r%d",			0xf00f, 0x300a,	0, sh2_SUBC},
        { NM_F,     "subv r%d, r%d",			0xf00f, 0x300b,	0, sh2_SUBV},
        { NM_F,     "swap.b r%d, r%d",			0xf00f, 0x6008,	0, sh2_SWAP_B},
        { NM_F,     "swap.w r%d, r%d",			0xf00f, 0x6009,	0, sh2_SWAP_W},
        { NM_F,     "tst r%d, r%d",				0xf00f, 0x2008,	0, sh2_TST},
        { NM_F,     "xor r%d, r%d",				0xf00f, 0x200a,	0, sh2_XOR},
        { NM_F,     "xtrct r%d, r%d",			0xf00f, 0x200d,	0, sh2_XTRCT},
        { NM_F,     "mov.b r%d, @r%d",			0xf00f, 0x2000,	0, sh2_MOVB},
        { NM_F,     "mov.w r%d, @r%d",			0xf00f, 0x2001,	0, sh2_MOVW},
        { NM_F,     "mov.l r%d, @r%d",			0xf00f, 0x2002,	0, sh2_MOVL},
        { NM_F,     "mov.b @r%d, r%d",			0xf00f, 0x6000,	0, sh2_MOVB_MEM},
        { NM_F,     "mov.w @r%d, r%d",			0xf00f, 0x6001,	0, sh2_MOVW_MEM},
        { NM_F,     "mov.l @r%d, r%d",			0xf00f, 0x6002,	0, sh2_MOVL_MEM},
        { NM_F,     "mac.l @r%d+, @r%d+",		0xf00f, 0x000f,	0, sh2_MAC_L},
        { NM_F,     "mac.w @r%d+, @r%d+",		0xf00f, 0x400f,	0, sh2_MAC_W},
        { NM_F,     "mov.b @r%d+, r%d",			0xf00f, 0x6004,	0, sh2_MOVB_INC},
        { NM_F,     "mov.w @r%d+, r%d",			0xf00f, 0x6005,	0, sh2_MOVW_INC},
        { NM_F,     "mov.l @r%d+, r%d",			0xf00f, 0x6006,	0, sh2_MOVL_INC},
        { NM_F,     "mov.b r%d, @-r%d",			0xf00f, 0x2004,	0, sh2_MOVB_DEC},
        { NM_F,     "mov.w r%d, @-r%d",			0xf00f, 0x2005,	0, sh2_MOVW_DEC},
        { NM_F,     "mov.l r%d, @-r%d",			0xf00f, 0x2006,	0, sh2_MOVL_DEC},
        { NM_F,     "mov.b r%d, @(r0, r%d)",	0xf00f, 0x0004,	0, sh2_MOVB_R0},
        { NM_F,     "mov.w r%d, @(r0, r%d)",	0xf00f, 0x0005,	0, sh2_MOVW_R0},
        { NM_F,     "mov.l r%d, @(r0, r%d)",	0xf00f, 0x0006,	0, sh2_MOVL_R0},
        { NM_F,     "mov.b @(r0, r%d), r%d",	0xf00f, 0x000c,	0, sh2_MOVB_R0_MEM},
        { NM_F,     "mov.w @(r0, r%d), r%d",	0xf00f, 0x000d,	0, sh2_MOVW_R0_MEM},
        { NM_F,     "mov.l @(r0, r%d), r%d",	0xf00f, 0x000e,	0, sh2_MOVL_R0_MEM},
        { MD_F,     "mov.b @(0x%03X, r%d), r0",	0xff00, 0x8400,	0, sh2_MOVB_DISP_R0},
        { MD_F,     "mov.w @(0x%03X, r%d), r0", 0xff00, 0x8500,	0, sh2_MOVW_DISP_R0},
        { ND4_F,    "mov.b r0, @(0x%03X, r%d)", 0xff00, 0x8000,	0, sh2_MOVB_R0_DISP},
        { ND4_F,    "mov.w r0, @(0x%03X, r%d)", 0xff00, 0x8100,	0, sh2_MOVW_R0_DISP},
        { NMD_F,    "mov.l r%d, @(0x%03X, r%d)",0xf000, 0x1000,	0, sh2_MOVL_DISP_MEM},
		{ NMD_F,    "mov.l @(0x%03X, r%d), r%d",0xf000, 0x5000,	0, sh2_MOVL_MEM_DISP},
        { D_F,      "mov.b r0, @(0x%03X, gbr)",	0xff00, 0xc000,	1, sh2_MOVB_R0_GBR},
        { D_F,      "mov.w r0, @(0x%03X, gbr)",	0xff00, 0xc100,	2, sh2_MOVW_R0_GBR},
        { D_F,      "mov.l r0, @(0x%03X, gbr)",	0xff00, 0xc200,	4, sh2_MOVL_R0_GBR},
        { D_F,      "mov.b @(0x%03X, gbr), r0",	0xff00, 0xc400,	1, sh2_MOVB_GBR_R0},
        { D_F,      "mov.w @(0x%03X, gbr), r0",	0xff00, 0xc500,	2, sh2_MOVW_GBR_R0},
        { D_F,      "mov.l @(0x%03X, gbr), r0",	0xff00, 0xc600,	4, sh2_MOVL_GBR_R0},
        { D_F,      "mova @(0x%03X, pc), r0",	0xff00, 0xc700,	4, sh2_MOVA},
        { D_F,      "bf 0x%08X",				0xff00, 0x8b00,	4, sh2_BF},
        { D_F,      "bf/s 0x%08X",				0xff00, 0x8f00,	5, sh2_BF_S},
        { D_F,      "bt 0x%08X",				0xff00, 0x8900,	5, sh2_BT},
        { D_F,      "bt/s 0x%08X",				0xff00, 0x8d00,	5, sh2_BT_S},
        { D12_F,    "bra 0x%08X",				0xf000, 0xa000,	5, sh2_BRA},
        { D12_F,    "bsr 0x%08X",				0xf000, 0xb000,	0, sh2_BSR},
        { ND8_F,    "mov.w @(0x%03X, pc), r%d", 0xf000, 0x9000,	0, sh2_MOV_DISP_W},
        { ND8_F,    "mov.l @(0x%03X, pc), r%d",	0xf000, 0xd000,	2, sh2_MOV_DISP_L},
        { I_F,      "and.b #0x%02X, @(r0, gbr)",0xff00, 0xcd00,	4, sh2_AND_B},
        { I_F,      "or.b #0x%02X, @(r0, gbr)", 0xff00, 0xcf00,	0, sh2_OR_B},
        { I_F,      "tst.b #0x%02X, @(r0, gbr)",0xff00, 0xcc00,	0, sh2_TST_B},
        { I_F,      "xor.b #0x%02X, @(r0, gbr)",0xff00, 0xce00,	0, sh2_XOR_B},
        { I_F,      "and #0x%02X, r0",			0xff00, 0xc900,	0, sh2_ANDI},
        { I_F,      "cmp/eq #0x%02X, r0",		0xff00, 0x8800,	0, sh2_CMP_EQ_IMM},
        { I_F,      "or #0x%02X, r0",			0xff00, 0xcb00,	0, sh2_ORI},
        { I_F,      "tst #0x%02X, r0",			0xff00, 0xc800,	0, sh2_TST_R0},
        { I_F,      "xor #0x%02X, r0",			0xff00, 0xca00,	0, sh2_XORI},
        { I_F,      "trapa #0x%X",				0xff00, 0xc300,	0, sh2_TRAPA},
        { NI_F,     "add #0x%02X, r%d",			0xf000, 0x7000,	0, sh2_ADDI},
        { NI_F,     "mov #0x%02X, r%d",			0xf000, 0xe000,	0, sh2_MOVI},
        { 0,        NULL,						0,      0,		0, 0}
};

int opcodeIndex(unsigned short op)
{
	register int i=0;
	while (((op & opcode_list[i].mask) != opcode_list[i].bits) && opcode_list[i].mnem != 0) i++;
	return i;
}

void ResetCPU(tagSH2 *sh2)
{
	int i;

	memset((void*)sh2->GenReg, 0, sizeof(uint32)*16);
	memset((void*)sh2->CtrlReg, 0, sizeof(uint32)*3);
	memset((void*)sh2->SysReg, 0, sizeof(uint32)*4);

	// SMPC Stuff
	for (i=1; i<0x80*2; i+=2) 
		*(unsigned short*)&SMPC[i] = 0xff;

	SMPC[0x5f] = 0x06;

	sh2->CtrlReg[0] = 0xf;
	sh2->GenReg[15] = 0x06002000;
}

void sh2_ReturnFromFunction(tagSH2 *sh2)
{
	PC = PR;
}

uint32	DelayWait;

void doInst(tagSH2 *sh2)
{
	void (FASTCALL *opfunc)(tagSH2*);

	sh2->instr = memGetWord(sh2->SysReg[3]);
	if ((opfunc = opcode_list[opcodeIndex(sh2->instr)].func) == 0)
		ConsoleMsg(MSG_FATAL, "Unhandled Opcode (0x%4x) at 0x%8x", sh2->instr, PC);
	else
		opfunc(sh2);
	
	PC += 2;
}

void FASTCALL sh2_STC_SR(tagSH2 *sh2)
{
	sh2->GenReg[dest] = SR;
}

void FASTCALL sh2_STC_GBR(tagSH2 *sh2)
{
	sh2->GenReg[dest] = GBR;
}

void FASTCALL sh2_STC_VBR(tagSH2 *sh2)
{
	sh2->GenReg[dest] = VBR;
}

void FASTCALL sh2_BSRF(tagSH2 *sh2)
{
	uint32 temp;

	PR = PC + 4;
	temp = PC + sh2->GenReg[dest] + 4;
	DelayBranch(temp);
}

void FASTCALL sh2_BRAF(tagSH2 *sh2)
{
	uint32 temp;

	temp = PC + sh2->GenReg[dest] + 4;
	DelayBranch(temp)
}

void FASTCALL sh2_MOVB_R0(tagSH2 *sh2)
{
	memSetByte(sh2->GenReg[0]+sh2->GenReg[dest], (uint8)sh2->GenReg[source]);
}

void FASTCALL sh2_MOVW_R0(tagSH2 *sh2)
{
	memSetWord(sh2->GenReg[0]+sh2->GenReg[dest], (uint16)sh2->GenReg[source]);
}

void FASTCALL sh2_MOVL_R0(tagSH2 *sh2)
{
	memSetLong(sh2->GenReg[0]+sh2->GenReg[dest], sh2->GenReg[source]);
}

void FASTCALL sh2_MUL_L(tagSH2 *sh2)
{
	MACL = (uint32)(sh2->GenReg[source] * sh2->GenReg[dest]);
}

void FASTCALL sh2_CLRT(tagSH2 *sh2)
{
	SR &= 0xfffffffe;
}

void FASTCALL sh2_SETT(tagSH2 *sh2)
{
	setTBit(1);
}

void FASTCALL sh2_CLRMAC(tagSH2 *sh2)
{
	MACH = MACL = 0;
}

void FASTCALL sh2_NOP(tagSH2 *sh2)
{}

void FASTCALL sh2_DIV0U(tagSH2 *sh2)
{
	SR &= 0xfcfe;
}

void FASTCALL sh2_MOVT(tagSH2 *sh2)
{
	sh2->GenReg[dest] = SR & 1;
}

void FASTCALL sh2_STS_MACH(tagSH2 *sh2)
{
	sh2->GenReg[dest] = MACH;
}

void FASTCALL sh2_STS_MACL(tagSH2 *sh2)
{
	sh2->GenReg[dest] = MACL;
}

void FASTCALL sh2_STS_PR(tagSH2 *sh2)
{
	sh2->GenReg[dest] = PR;
}

void FASTCALL sh2_RTS(tagSH2 *sh2)
{
	//uint32 temp;

	//temp = PR;
	DelayBranch(PR);
}

void FASTCALL sh2_SLEEP(tagSH2 *sh2)
{
	ConsoleMsg(MSG_WARN, "SLEEP Opcode Interception");
}

void FASTCALL sh2_RTE(tagSH2 *sh2)
{
	uint32 temp;

	temp = memGetLong(sh2->GenReg[15]) + 4;
	sh2->GenReg[15] += 4;
	SR = memGetLong(sh2->GenReg[15]) & 0x3f3;
	sh2->GenReg[15] += 4;
	DelayBranch(temp);
}

void FASTCALL sh2_MOVB_R0_MEM(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed long)memGetByte(sh2->GenReg[0]+sh2->GenReg[source]);
}

void FASTCALL sh2_MOVW_R0_MEM(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed long)memGetWord(sh2->GenReg[0]+sh2->GenReg[source]);
}

void FASTCALL sh2_MOVL_R0_MEM(tagSH2 *sh2)
{
	sh2->GenReg[dest] = memGetLong(sh2->GenReg[0]+sh2->GenReg[source]);
}

void FASTCALL sh2_MAC_L(tagSH2 *sh2)
{
	uint64 mult;

	mult = (uint64)((signed long)memGetLong(sh2->GenReg[source])*(signed long)memGetLong(sh2->GenReg[dest]));
	sh2->GenReg[source] += 4;
	sh2->GenReg[dest] += 4;
	mult += (((uint64)MACH << 32) | (uint64)(MACL));
	MACL = (uint32)(mult & 0xffffffff);
	MACH = (uint32)((mult >> 32) & 0xffffffff);

	if (SR&1)
		MACH &= 0xffff;
}

void FASTCALL sh2_MOVL_DISP_MEM(tagSH2 *sh2)
{
	memSetLong(sh2->GenReg[dest]+(offset1<<2), sh2->GenReg[source]);
}

				
void FASTCALL sh2_MOVB(tagSH2 *sh2)
{
	memSetByte(sh2->GenReg[dest], (uint8)sh2->GenReg[source]);
}

void FASTCALL sh2_MOVW(tagSH2 *sh2)
{
	memSetWord(sh2->GenReg[dest], (uint16)sh2->GenReg[source]);
}

void FASTCALL sh2_MOVL(tagSH2 *sh2)
{
	memSetLong(sh2->GenReg[dest], sh2->GenReg[source]);
}

void FASTCALL sh2_MOVB_DEC(tagSH2 *sh2)
{
	memSetByte(--sh2->GenReg[dest], (uint8)sh2->GenReg[source]);
}

void FASTCALL sh2_MOVW_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 2;
	memSetWord(sh2->GenReg[dest], (uint16)sh2->GenReg[source]);
}
				
void FASTCALL sh2_MOVL_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 4;
	memSetLong(sh2->GenReg[dest], sh2->GenReg[source]);
}

void FASTCALL sh2_DIV0S(tagSH2 *sh2)
{
	if ((sh2->GenReg[source]&0x80000000)^(sh2->GenReg[dest]&0x80000000))
		setTBit(0);
	else
		setTBit(1);
}

void FASTCALL sh2_TST(tagSH2 *sh2)
{
	if ((sh2->GenReg[dest] & sh2->GenReg[source]) == 0)
		setTBit(1);
	else
		setTBit(0);
}

void FASTCALL sh2_AND(tagSH2 *sh2)
{
	sh2->GenReg[dest] &= sh2->GenReg[source];
}

void FASTCALL sh2_XOR(tagSH2 *sh2)
{
	sh2->GenReg[dest] ^= sh2->GenReg[source];
}

void FASTCALL sh2_OR(tagSH2 *sh2)
{
	sh2->GenReg[dest] |= sh2->GenReg[source];
}

void FASTCALL sh2_CMP_STR(tagSH2 *sh2)
{				
	uint32 temp;

	temp = sh2->GenReg[dest]^sh2->GenReg[source];
	if ((temp>>12)&0xff || (temp>>8)&0xff || (temp>>4)&0xff || temp&0xff)
		setTBit(1);
	else
		SR &= 0xfffffffe;
}

void FASTCALL sh2_XTRCT(tagSH2 *sh2)
{
	sh2->GenReg[dest] = ((sh2->GenReg[source]<<16)&0xffff0000) | ((sh2->GenReg[dest]>>16)&0xffff);
}

void FASTCALL sh2_MULU(tagSH2 *sh2)
{
	MACL=((uint32)(uint16)sh2->GenReg[dest]*(uint32)(uint16)sh2->GenReg[source]);
}


void FASTCALL sh2_MULS(tagSH2 *sh2)
{
	MACL=((signed long)(short)sh2->GenReg[dest]*(signed long)(short)sh2->GenReg[source]);
}

void FASTCALL sh2_CMP_EQ(tagSH2 *sh2) 
{
	if (sh2->GenReg[source] == sh2->GenReg[dest])
		setTBit(1);
	else
		SR &= 0xfffffffe;
}

void FASTCALL sh2_CMP_HS(tagSH2 *sh2) 
{
	if (sh2->GenReg[dest] >= sh2->GenReg[source])
		setTBit(1);
	else
		SR &= 0xfffffffe;
}

void FASTCALL sh2_CMP_GE(tagSH2 *sh2)
{
	if ((signed)sh2->GenReg[dest] >= (signed)sh2->GenReg[source])
		setTBit(1);
	else
		SR &= 0xfffffffe;
}

void FASTCALL sh2_DIV1(tagSH2 *sh2)
{
	uint32 temp1, temp2;

	temp1 = (0x80000000 & sh2->GenReg[dest]) != 0;
	sh2->GenReg[dest] = ((sh2->GenReg[dest]<<1) | (SR&1));

	switch((SR&0x100 != 0)) {
		case 0:
			switch((SR&0x200 != 0)) {
				case 0:
					temp2 = sh2->GenReg[dest];
					sh2->GenReg[dest] -= sh2->GenReg[source];
					setQBit((sh2->GenReg[dest] > temp2) ^ temp1);
					break;
				case 1:
					temp2 = sh2->GenReg[dest];
					sh2->GenReg[dest] += sh2->GenReg[source];
					setQBit((sh2->GenReg[dest] >= temp2) ^ temp1);
					break;
			}
			break;
		case 1:
			switch((SR&0x200 != 0)) {
				case 0:
					temp2 = sh2->GenReg[dest];
					sh2->GenReg[dest] += sh2->GenReg[source];
					setQBit((sh2->GenReg[dest] < temp2) ^ temp1);
					break;
				case 1:
					temp2 = sh2->GenReg[dest];
					sh2->GenReg[dest] -= sh2->GenReg[source];
					setQBit((sh2->GenReg[dest] <= temp2) ^ temp1);
					break;
			}
			break;
	}
	setTBit((SR&0x100)^(SR&0x200));
}

void FASTCALL sh2_DMULU_L(tagSH2 *sh2)
{
	uint64 mult;
	
	mult = ((uint64)sh2->GenReg[source] * (uint64)sh2->GenReg[dest]);
	MACL = (uint32)mult;
	MACH = (uint32)(mult >> 32);
}
				
void FASTCALL sh2_CMP_HI(tagSH2 *sh2)
{
	if (sh2->GenReg[dest] > sh2->GenReg[source])
		setTBit(1);
	else
		SR &= 0xfffffffe;
}
				
void FASTCALL sh2_CMP_GT(tagSH2 *sh2)
{
	if ((signed)sh2->GenReg[dest] > (signed)sh2->GenReg[source])
		setTBit(1);
	else
		SR &= 0xfffffffe;
}
				
void FASTCALL sh2_SUB(tagSH2 *sh2) {
	sh2->GenReg[dest] -= sh2->GenReg[source];
}

void FASTCALL sh2_SUBC(tagSH2 *sh2)
{
	uint32 temp;

	temp = sh2->GenReg[dest] - sh2->GenReg[source];
	if ((sh2->GenReg[dest] < temp) || (temp < temp - (SR&1)))
	{
		sh2->GenReg[dest] = temp - SR&1;
		setTBit(1);
	} else {
		sh2->GenReg[dest] = temp - SR&1;
		setTBit(0);
	}
}
				
void FASTCALL sh2_SUBV(tagSH2 *sh2) //**test
{
	uint32 temp;

	temp = sh2->GenReg[dest] - sh2->GenReg[source];
	if ((((signed long)sh2->GenReg[dest] < 0) ^ ((signed long)sh2->GenReg[source] < 0)) && ((signed)temp < 0))
		setTBit(1);
	else
		setTBit(0);
	sh2->GenReg[dest] = temp;
}
	
void FASTCALL sh2_ADD(tagSH2 *sh2) {
	sh2->GenReg[dest] += sh2->GenReg[source];
}

void FASTCALL sh2_DMULS_L(tagSH2 *sh2)
{
	uint64 mult;
	
	mult = ((sint64)sh2->GenReg[source] * (sint64)sh2->GenReg[dest]);
	MACL = (uint32)mult;
	MACH = (uint32)(mult >> 32);
}
				
void FASTCALL sh2_ADDC(tagSH2 *sh2) //**test
{
	uint32 temp;

	temp = (uint32)(sh2->GenReg[dest] + sh2->GenReg[source]);
	if ((temp+(SR&1) < temp) || (temp < sh2->GenReg[dest]))
	{
		sh2->GenReg[dest] = temp + (SR&1);
		setTBit(1);
	} else {
		sh2->GenReg[dest] = temp + (SR&1);
		SR &= 0xfffffffe;
	}
}
				
void FASTCALL sh2_ADDV(tagSH2 *sh2)
{
	uint32 temp;
	
	temp = sh2->GenReg[source] + sh2->GenReg[dest];
	if (!((((signed long)sh2->GenReg[source] < 0) ^ ((signed long)sh2->GenReg[dest] < 0))) && ((signed)temp < 0))
		setTBit(1);
	else
		setTBit(0);
	sh2->GenReg[dest] = temp;
}

void FASTCALL sh2_SHL(tagSH2 *sh2)
{
	setTBit((sh2->GenReg[dest] >> 31) & 1);
	sh2->GenReg[dest] <<= 1;
}
				
void FASTCALL sh2_SHLR(tagSH2 *sh2)
{
	setTBit(sh2->GenReg[dest] & 1);
	sh2->GenReg[dest] >>= 1;
}
				
void FASTCALL sh2_STS_MACH_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 4;
	memSetLong(sh2->GenReg[dest], MACH);
}
				
void FASTCALL sh2_STC_SR_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 4;
	memSetLong(sh2->GenReg[dest], SR);
}

void FASTCALL sh2_STC_GBR_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 4;
	memSetLong(sh2->GenReg[dest], GBR);
}

void FASTCALL sh2_STC_VBR_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 4;
	memSetLong(sh2->GenReg[dest], VBR);
}

				
void FASTCALL sh2_ROTL(tagSH2 *sh2)
{
	uint32 temp;

	temp = ((sh2->GenReg[dest] >> 31) & 1);
	sh2->GenReg[dest] = (sh2->GenReg[dest] << 1) | (SR&1);
	setTBit(temp);
}

void FASTCALL sh2_ROTR(tagSH2 *sh2)
{
	uint32 temp;

	temp = sh2->GenReg[dest] & 1;
	sh2->GenReg[dest] = (sh2->GenReg[dest] >> 1) | (uint32)((SR&1)<<31);
	setTBit(temp);
}
				
void FASTCALL sh2_LDS_MACH_INC(tagSH2 *sh2)
{
	MACH = memGetLong(sh2->GenReg[dest]);
	sh2->GenReg[dest] += 4;
}
				
void FASTCALL sh2_LDC_SR_INC(tagSH2 *sh2)
{
	SR = memGetLong(sh2->GenReg[dest]) & 0x3f3;
	sh2->GenReg[dest] += 4;
}
				
void FASTCALL sh2_SHLL2(tagSH2 *sh2)
{
	sh2->GenReg[dest] <<= 2;
}
				
void FASTCALL sh2_SHLR2(tagSH2 *sh2)
{
	sh2->GenReg[dest] >>= 2;
}
				
void FASTCALL sh2_LDS_MACH(tagSH2 *sh2)
{
	MACH = sh2->GenReg[dest];
}
				
void FASTCALL sh2_JSR(tagSH2 *sh2)
{
	uint32 temp;

	PR = PC + 4;
	temp = sh2->GenReg[dest];
	DelayBranch(temp);
}
				
void FASTCALL sh2_LDC_SR(tagSH2 *sh2)
{
	SR = sh2->GenReg[dest] & 0x3f3;
}
				
void FASTCALL sh2_MAC_W(tagSH2 *sh2)
{
	uint32 temp;
	uint64 mult;

	temp = ((signed long)memGetWord(sh2->GenReg[dest])*(signed long)memGetWord(sh2->GenReg[source]));
	sh2->GenReg[dest] += 4;
	sh2->GenReg[source] += 4;

	if (SR&2)
	{
		MACL = (uint32)((signed)MACL + (signed)temp);
		if (MACL > 0x80000000)
		{
			MACH = 1;
			MACL &= 0x8fffffff;
		}
	}
	else {
		mult = (((uint64)MACH<<32))+MACL+temp;
		MACL = (uint32)mult;
		MACH = (uint32)(mult>>32);
	}
}

void FASTCALL sh2_DT(tagSH2 *sh2)
{
	if (--sh2->GenReg[dest] == 0)
		setTBit(1);
	else
		SR &= 0xfffffffe;
}

void FASTCALL sh2_CMP_PZ(tagSH2 *sh2)
{
	if ((signed)sh2->GenReg[dest] >= 0)
		setTBit(1);
	else
		SR &= 0xfffffffe;
}
				
void FASTCALL sh2_STS_MACL_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 4;
	memSetLong(sh2->GenReg[dest], MACL);
}
				
void FASTCALL sh2_CMP_PL(tagSH2 *sh2)
{
	if ((signed)sh2->GenReg[dest] > 0)
		setTBit(1);
	else
		SR &= 0xfffffffe;
}
				
void FASTCALL sh2_LDS_MACL_INC(tagSH2 *sh2)
{
	MACL = memGetLong(sh2->GenReg[dest]);
	sh2->GenReg[dest] += 4;
}

void FASTCALL sh2_LDC_GBR_INC(tagSH2 *sh2)
{
	GBR = memGetLong(sh2->GenReg[dest]);
	sh2->GenReg[dest] += 4;
}
				
void FASTCALL sh2_SHLL8(tagSH2 *sh2)
{
	sh2->GenReg[dest] <<= 8;
}
				
void FASTCALL sh2_SHLR8(tagSH2 *sh2)
{
	sh2->GenReg[dest] >>= 8;
}
				
void FASTCALL sh2_LDS_MACL(tagSH2 *sh2)
{
	MACL = sh2->GenReg[dest];
}

void FASTCALL sh2_TAS_B(tagSH2 *sh2)
{
	uint32 temp;

	temp = (uint32)memGetByte(sh2->GenReg[dest]);
	if (temp == 0) setTBit(1);
	else setTBit(0);
	temp |= 0x80;
	memSetByte(sh2->GenReg[dest], (uint8)temp);
}

void FASTCALL sh2_LDC_GBR(tagSH2 *sh2)
{
	GBR = sh2->GenReg[dest];
}

void FASTCALL sh2_SHAR(tagSH2 *sh2)
{
	uint32 temp;

	temp = sh2->GenReg[dest] & 0x80000000;
	setTBit(sh2->GenReg[dest] & 1);
	sh2->GenReg[dest] = ((sh2->GenReg[dest] >> 1) & 0x7fffffff) | temp;
}
				
void FASTCALL sh2_STS_PR_DEC(tagSH2 *sh2)
{
	sh2->GenReg[dest] -= 4;
	memSetLong(sh2->GenReg[dest], PR);
}
			
void FASTCALL sh2_ROTCL(tagSH2 *sh2)
{
	uint32 temp;

	temp = (sh2->GenReg[dest] >> 31) & 1;
	sh2->GenReg[dest] = (sh2->GenReg[dest] << 1) | (SR&1);
	setTBit(temp);
}

void FASTCALL sh2_ROTCR(tagSH2 *sh2)
{
	uint32 temp;
	
	temp = sh2->GenReg[dest] & 1;
	sh2->GenReg[dest] = (sh2->GenReg[dest] >> 1) | (uint32)((SR&1) << 31);
	setTBit(temp);
}
				
void FASTCALL sh2_LDS_PR_INC(tagSH2 *sh2)
{
	PR = memGetLong(sh2->GenReg[dest]);
	sh2->GenReg[dest] += 4;
}
				
void FASTCALL sh2_LDC_VBR_INC(tagSH2 *sh2)
{
	VBR = memGetLong(sh2->GenReg[dest]);
	sh2->GenReg[dest] += 4;
}
				
void FASTCALL sh2_SHLL16(tagSH2 *sh2)
{
	sh2->GenReg[dest] <<= 16;
}
				
void FASTCALL sh2_SHLR16(tagSH2 *sh2)
{
	sh2->GenReg[dest] >>= 16;
}
				
void FASTCALL sh2_LDS_PR(tagSH2 *sh2)
{
	PR = sh2->GenReg[dest];
}
				
void FASTCALL sh2_JMP(tagSH2 *sh2)
{
	uint32 temp;

	temp = sh2->GenReg[dest];
	DelayBranch(temp);
}

void FASTCALL sh2_LDC_VBR(tagSH2 *sh2)
{
	VBR = sh2->GenReg[dest];
}

void FASTCALL sh2_MOVL_MEM_DISP(tagSH2 *sh2)
{
	//MOV.L @(disp,Rm),Rn
	sh2->GenReg[dest] = memGetLong(sh2->GenReg[source] + (offset1<<2));
}

void FASTCALL sh2_MOVB_MEM(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed char)memGetByte(sh2->GenReg[source]);
}

void FASTCALL sh2_MOVW_MEM(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed short)memGetWord(sh2->GenReg[source]);
}

void FASTCALL sh2_MOVL_MEM(tagSH2 *sh2)
{
	sh2->GenReg[dest] = memGetLong(sh2->GenReg[source]);
}

void FASTCALL sh2_MOVR(tagSH2 *sh2)
{
	sh2->GenReg[dest] = sh2->GenReg[source];
}
				
void FASTCALL sh2_MOVB_INC(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed char)memGetByte(sh2->GenReg[source]++);
}

void FASTCALL sh2_MOVW_INC(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed short)memGetWord(sh2->GenReg[source]);
	sh2->GenReg[source] += 2;
}
				
void FASTCALL sh2_MOVL_INC(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed long)memGetLong(sh2->GenReg[source]);
	sh2->GenReg[source] += 4;
}

void FASTCALL sh2_NOT(tagSH2 *sh2)
{
	sh2->GenReg[dest] = ~sh2->GenReg[source];
}
				
void FASTCALL sh2_SWAP_B(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (sh2->GenReg[dest] & 0xffff0000)|(uint16)(((uint16)sh2->GenReg[source]>>8)|((uint16)sh2->GenReg[source]<<8));
}

void FASTCALL sh2_SWAP_W(tagSH2 *sh2)
{
	sh2->GenReg[dest] = swapLong(sh2->GenReg[source]); //(sh2->GenReg[source] >> 16) | (sh2->GenReg[source] << 16);
}
				
void FASTCALL sh2_NEGC(tagSH2 *sh2)
{
	uint32 temp;

	temp = (signed long)(0-sh2->GenReg[source]);
	sh2->GenReg[dest] = (signed long)(temp) - (SR&1);
	if (0 < (signed)temp)
		setTBit(1);
	else
		SR &= 0xfffffffe;
	if (temp < sh2->GenReg[dest]) setTBit(1);
}
				
void FASTCALL sh2_NEG(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed long)(0-sh2->GenReg[source]);
}
				
void FASTCALL sh2_EXTU_B(tagSH2 *sh2)
{
	sh2->GenReg[dest] = sh2->GenReg[source] & 0xff;
}

void FASTCALL sh2_EXTU_W(tagSH2 *sh2)
{
	sh2->GenReg[dest] = sh2->GenReg[source] & 0xffff;
}
				
void FASTCALL sh2_EXTS_B(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed long)((char)sh2->GenReg[source]);
}

void FASTCALL sh2_EXTS_W(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed long)((short)sh2->GenReg[source]);
}

void FASTCALL sh2_ADDI(tagSH2 *sh2)
{
	sh2->GenReg[dest] += (signed long)immediate;
}

void FASTCALL sh2_MOVB_R0_DISP(tagSH2 *sh2)
{
	memSetByte(sh2->GenReg[source]+offset1, (uint8)sh2->GenReg[0]);
}

void FASTCALL sh2_MOVW_R0_DISP(tagSH2 *sh2)
{
	memSetWord(sh2->GenReg[source]+(offset1<<1), (uint16)sh2->GenReg[0]);
}
				
void FASTCALL sh2_MOVB_DISP_R0(tagSH2 *sh2)
{
	sh2->GenReg[0] = (signed long)memGetByte(sh2->GenReg[source]+offset1);
}

void FASTCALL sh2_MOVW_DISP_R0(tagSH2 *sh2)
{
	sh2->GenReg[0] = (signed long)memGetWord(sh2->GenReg[source]+(offset1<<1));
}

void FASTCALL sh2_CMP_EQ_IMM(tagSH2 *sh2)
{
	if ((signed)sh2->GenReg[0] == (signed long)immediate)
		setTBit(1);
	else
		SR &= 0xfffffffe;
}
				
void FASTCALL sh2_BT(tagSH2 *sh2)
{
	if (SR&1)
		PC += (signed char)((immediate << 1) + 4) - 2;
}
				
void FASTCALL sh2_BF(tagSH2 *sh2)
{
	if (!(SR&1))
		PC += (signed char)((immediate << 1) + 4) - 2;
}
				
void FASTCALL sh2_BT_S(tagSH2 *sh2)
{
	uint32 temp;

	if (SR&1)
	{
		temp = PC + (signed char)((immediate << 1) + 4);
		DelayBranch(temp);
	}
}
				
void FASTCALL sh2_BF_S(tagSH2 *sh2)
{
	uint32 temp;

	if (!(SR&1))
	{
		temp = PC + (signed char)((immediate << 1) + 4);
		DelayBranch(temp);
	}
}

void FASTCALL sh2_MOV_DISP_W(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (signed long)memGetWord(PC+(offset2<<1)+4);
}

void FASTCALL sh2_BRA(tagSH2 *sh2)
{
	uint32 temp;

	temp = (offset3 << 1);
	if (temp & 0x1000)
		temp |= 0xfffff000;
	temp = PC + (signed long)temp + 4;
	DelayBranch(temp);
}
				
void FASTCALL sh2_BSR(tagSH2 *sh2)
{
	uint32 temp;

	PR = PC + 4;
	temp = (signed short)(offset3 << 1);
	if (temp & 0x1000)
		temp |= 0xfffff000;
	temp = PC + (signed long)temp + 4;
	DelayBranch(temp);
}

void FASTCALL sh2_MOVB_R0_GBR(tagSH2 *sh2)
{
	memSetByte(offset2+GBR, (uint8)sh2->GenReg[0]);
}

void FASTCALL sh2_MOVW_R0_GBR(tagSH2 *sh2)
{
	memSetWord((offset2<<1)+GBR, (uint16)sh2->GenReg[0]);
}

void FASTCALL sh2_MOVL_R0_GBR(tagSH2 *sh2)
{
	memSetLong((offset2<<2)+GBR, sh2->GenReg[0]);
}

void FASTCALL sh2_TRAPA(tagSH2 *sh2)
{
	sh2->GenReg[15] -= 4;
	memSetLong(sh2->GenReg[15], SR);
	sh2->GenReg[15] -= 4;
	memSetLong(sh2->GenReg[15], PC-2);
	PC = memGetLong(VBR + ((uint32)immediate << 2)) + 4 - 2;
}

void FASTCALL sh2_MOVB_GBR_R0(tagSH2 *sh2)
{
	sh2->GenReg[0] = (signed long)memGetByte(offset2+GBR);
}
				
void FASTCALL sh2_MOVW_GBR_R0(tagSH2 *sh2)
{
	sh2->GenReg[0] = (signed long)memGetWord((offset2<<1)+GBR);
}

void FASTCALL sh2_MOVL_GBR_R0(tagSH2 *sh2)
{
	sh2->GenReg[0] = memGetLong((offset2<<2)+GBR);
}

void FASTCALL sh2_MOVA(tagSH2 *sh2)
{
	sh2->GenReg[0] = (PC&0xfffffffc)+(offset2<<2)+4;
}

void FASTCALL sh2_TST_R0(tagSH2 *sh2)
{
	if (((uint8)sh2->GenReg[0] & offset2) == 0)
		setTBit(1);
	else
		setTBit(0);
}

void FASTCALL sh2_ANDI(tagSH2 *sh2)
{
	sh2->GenReg[0] &= (uint32)immediate;
}

void FASTCALL sh2_XORI(tagSH2 *sh2)
{
	sh2->GenReg[0] ^= (uint32)immediate;
}
				
void FASTCALL sh2_ORI(tagSH2 *sh2)
{
	sh2->GenReg[0] |= (uint32)immediate;
}

void FASTCALL sh2_TST_B(tagSH2 *sh2)
{
	if (((uint32)memGetByte(GBR + sh2->GenReg[0]) & (uint32)immediate) == 0)
		setTBit(1);
	else
		setTBit(0);
}

void FASTCALL sh2_AND_B(tagSH2 *sh2)
{
	memSetByte(GBR + sh2->GenReg[0], (uint8)((uint32)memGetByte(GBR + sh2->GenReg[0]) & (uint32)immediate));
}

void FASTCALL sh2_XOR_B(tagSH2 *sh2)
{
	memSetByte(GBR + sh2->GenReg[0], (uint8)((uint32)memGetByte(GBR + sh2->GenReg[0]) ^ (uint32)immediate));
}

void FASTCALL sh2_OR_B(tagSH2 *sh2)
{
	memSetByte(GBR + sh2->GenReg[0], (uint8)((uint32)memGetByte(GBR + sh2->GenReg[0]) | (0x000000FF & (uint32)immediate)));
}

void FASTCALL sh2_MOV_DISP_L(tagSH2 *sh2)
{
	sh2->GenReg[dest] = memGetLong((PC&0xfffffffc)+(offset2<<2)+4);
}

void FASTCALL sh2_MOVI(tagSH2 *sh2)
{
	sh2->GenReg[dest] = (long)immediate;
}
