// i386.C
// x86 Specific Module for Titan

#if defined(__i386__) || defined(_M_IX86)

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <malloc.h>
#include "../../global.h"
#include "../../mem.h"
#include "../../dynamap.h"
#include "../../plugin.h"
#include "dynalib.h"

#include "../../scu.h"
#include "../../smpc.h"

short getOffset(Block*, uint32, uint32);
void opcodePass(x86op_desc*, uint16, uint8*);

#define addJumpTable(x,y,z)		jumps = realloc((void*)jumps, sizeof(jmpTag)*((++jmp)+1)); \
								memset((void*)&jumps[jmp], 0, sizeof(jmpTag)); \
								jumps[jmp-1].ptr = x; \
								jumps[jmp-1].src = y; \
								jumps[jmp-1].dest = z;

#define opdesc(op, y)	{x86_##op, &##op##_size, &##op##_src, &##op##_dest, &##op##_off1, &##op##_imm, &##op##_off3, y}
#define opNULL			{0,0,0,0,0,0,0,0}

char instrSize[NUMOFBLOCKS][MAXINSTRCNT];

x86op_desc asm_list[] =
{
	opdesc(CLRT,0),       opdesc(CLRMAC,0),      opdesc(DIV0U,0),    opdesc(NOP,0),
	opdesc(RTE,4),        opdesc(RTS,4),         opdesc(SETT,0),     opdesc(SLEEP,0),
	opdesc(CMP_PL,0),     opdesc(CMP_PZ,0),      opdesc(DT,0),       opdesc(MOVT,0),
    opdesc(ROTL,0),       opdesc(ROTR,0),	     opdesc(ROTCL,0),    opdesc(ROTCR,0),
	opdesc(SHL,0),        opdesc(SHAR,0),	     opdesc(SHL,0),      opdesc(SHLR,0), 
	opdesc(SHLL2,0),      opdesc(SHLR2,0),	     opdesc(SHLL8,0),    opdesc(SHLR8,0),
	opdesc(SHLL16,0),     opdesc(SHLR16,0),      opdesc(STC_SR,0),   opdesc(STC_GBR,0),
	opdesc(STC_VBR,0),    opdesc(STS_MACH,0),    opdesc(STS_MACL,0), opdesc(STS_PR,0),
	opNULL,
      //{ N_F,      "tas.b @r%d",          0xf0ff, 0x401b},
	opdesc(STC_SR_MEM,0), opdesc(STC_GBR_MEM,0), opdesc(STC_VBR_MEM,0),
	opdesc(STS_MACH_DEC,0), opdesc(STS_MACL_DEC,0), opdesc(STS_PR_DEC,0),
	opdesc(LDC_SR,0),     opdesc(LDC_GBR,0),     opdesc(LDC_VBR,0),  opdesc(LDS_MACH,0),
	opdesc(LDS_MACL,0),   opdesc(LDS_PR,0),      opdesc(JMP, 3),     opdesc(JSR, 4),
	opdesc(LDC_SR_INC,0), opdesc(LDC_SR_INC,0),  opdesc(LDC_VBR_INC,0),
	opdesc(LDS_MACH_INC,0), opdesc(LDS_MACL_INC,0), opdesc(LDS_PR_INC,0),
	opdesc(BRAF,4),       opdesc(BSRF,4),        opdesc(ADD,0),      opdesc(ADDC,0),
	opNULL,
        //{ NM_F,     "addv r%d, r%d",       0xf00f, 0x300f},
	opdesc(AND,0), opdesc(CMP_EQ,0), opdesc(CMP_HS,0), opdesc(CMP_GE,0),
	opdesc(CMP_HI,0), opdesc(CMP_GT,0), 
	opNULL, opNULL, opNULL, opNULL, opNULL,
      /*{ NM_F,     "cmp/str r%d, r%d",    0xf00f, 0x200c},
        { NM_F,     "div1 r%d, r%d",       0xf00f, 0x3004},
        { NM_F,     "div0s r%d, r%d",      0xf00f, 0x2007},
        { NM_F,     "dmuls.l r%d, r%d",    0xf00f, 0x300d},
        { NM_F,     "dmulu.l r%d, r%d",    0xf00f, 0x3005},*/
	opdesc(EXTS_B,0), opdesc(EXTS_W,0), opdesc(EXTU_B,0), opdesc(EXTU_W,0),
	opdesc(MOV_REG,0),
	opNULL, opNULL, opNULL,
      /*{ NM_F,     "mul.l r%d, r%d",      0xf00f, 0x0007},
        { NM_F,     "muls.w r%d, r%d",     0xf00f, 0x200f},
        { NM_F,     "mulu.w r%d, r%d",     0xf00f, 0x200e},*/
	opdesc(NEG,0), opdesc(NEGC,0), opdesc(NOT,0), opdesc(OR,0), opdesc(SUB,0),
	opdesc(SUBC,0), 
	opNULL,
      //{ NM_F,     "subv r%d, r%d",       0xf00f, 0x300b},
	opdesc(SWAP_B,0), opdesc(SWAP_W,0), opdesc(TST,0),
	opdesc(XOR,0), opdesc(XTRCT,0), opdesc(MOV_B_MEM,0), 
	opdesc(MOV_W_MEM,0), opdesc(MOV_L_MEM,0),
	opdesc(MOVB_MEM_REG,0), opdesc(MOVW_MEM_REG,0), opdesc(MOVL_MEM_REG,0),
	opNULL, opNULL,
      /*{ NM_F,     "mac.l @r%d+, @r%d+",  0xf00f, 0x000f},
        { NM_F,     "mac.w @r%d+, @r%d+",  0xf00f, 0x400f},*/
	opdesc(MOV_B_INC,0), opdesc(MOV_W_INC,0), opdesc(MOV_L_INC,0),
	opNULL, opNULL, opNULL,
      /*{ NM_F,     "mov.b r%d, @-r%d",    0xf00f, 0x2004},
        { NM_F,     "mov.w r%d, @-r%d",    0xf00f, 0x2005},
        { NM_F,     "mov.l r%d, @-r%d",    0xf00f, 0x2006},*/
	opdesc(MOVB_R0_N,0), opdesc(MOVW_R0_N,0), opdesc(MOVL_R0_N,0),
	opdesc(MOVB_M_R0,0), opdesc(MOVW_M_R0,0), opdesc(MOVL_M_R0,0),
	opNULL, opNULL, opNULL, opNULL,
      /*{ MD_F,     "mov.b @(0x%03X, r%d), r0", 0xff00, 0x8400},
        { MD_F,     "mov.w @(0x%03X, r%d), r0", 0xff00, 0x8500},
        { ND4_F,    "mov.b r0, @(0x%03X, r%d)", 0xff00, 0x8000},
        { ND4_F,    "mov.w r0, @(0x%03X, r%d)", 0xff00, 0x8100},*/
    opdesc(MOVL_REG_DISP,0),
	opNULL, opNULL, opNULL, opNULL, opNULL, opNULL, opNULL,
      /*{ NMD_F,    "mov.l @(0x%03X, r%d), r%d", 0xf000, 0x5000},
        { D_F,      "mov.b r0, @(0x%03X, gbr)", 0xff00, 0xc000},
        { D_F,      "mov.w r0, @(0x%03X, gbr)", 0xff00, 0xc100},
        { D_F,      "mov.l r0, @(0x%03X, gbr)", 0xff00, 0xc200},
        { D_F,      "mov.b @(0x%03X, gbr), r0", 0xff00, 0xc400},
        { D_F,      "mov.w @(0x%03X, gbr), r0", 0xff00, 0xc500},
        { D_F,      "mov.l @(0x%03X, gbr), r0", 0xff00, 0xc600},*/
	opdesc(MOVA,0), opdesc(BF,1), opdesc(BF,3), opdesc(BT,1), opdesc(BT,3),
	opdesc(BRA,2), opdesc(BSR,2), opdesc(MOVW_DISP,0), opdesc(MOVL_DISP,0),
	opdesc(AND_B,3), opdesc(OR_B,3), opdesc(TST_B,3), opdesc(XOR_B,3),
	opdesc(ANDI,0), opdesc(CMP_EQ_IMM,0), opdesc(ORI,0),
	opNULL,
      //{ I_F,      "tst #0x%02X, r0",     0xff00, 0xc800},
	opdesc(XORI,0),	opNULL,
      //{ I_F,      "trapa #0x%X",         0xff00, 0xc300},
    opdesc(ADDI,0), opdesc(MOVI,0), opNULL
};

// CPU Specific execution function
void dynaExecute(tagSH2 *sh2, Block *dCode)
{
	int i, temp = 0;

	if (!dynaCheckPC(&master, dCode))
		dynaMakeBlock(&master, dCode);
	else {
		// find location of first address
		i = ((PC - dCode[blockCount].b_addr)>>1)-1;
		while (i >= 0)
			temp += instrSize[blockCount][i--];

		temp += (uint32)(dCode[blockCount].code) + PROLOGSIZE;
	}

	((dynaFunc)((void*)(dCode[blockCount].code)))(sh2, temp);
	
	blockCount++;
	if (blockCount > NUMOFBLOCKS)
		blockCount = 0;
}

void xlatPage(Block *page)
{
	int i, j, jmp = 0, count = 0;
	uint16 op, temp;
	uint32 addr = page->b_addr;
	uint8 *ptr, *startptr;
	jmpTag *jumps;

	startptr = ptr = page->code;
	jumps = malloc(sizeof(jmpTag));
	memset(jumps, 0, sizeof(jmpTag));
	memset((void*)instrSize[blockCount], 0, sizeof(char)*MAXINSTRCNT);


	// add prolog and pass address of registers
	memcpy((void*)ptr, (void*)prologue, PROLOGSIZE);
	ptr += PROLOGSIZE;

	//First Pass -- Regular Opcodes and Incomplete Jump Ocodes
	while (ptr - startptr < SAFEPAGESIZE)
	{
		//translate the opcode and insert code
		op = memGetWord(addr);
		addr += 2;
		i = opcodeIndex(op);

		// stop code generation when data/text (unknown opcode) is found
		if (i == 0x8e) break;

		if (asm_list[i].func == 0) {
			ConsoleMsg(MSG_WARN, "Unimplemented Opcode (0x%4x) at 0x%8x", op, addr-2);
			break;
		}

		if (asm_list[i].delay <= 1) //Regular Opcode (No Delay Branch)
		{ 
			memcpy((void*)ptr, (void*)(asm_list[i].func), *(asm_list[i].size));
			memcpy((void*)(ptr + *(asm_list[i].size)), (void*)seperator, SEPERATORSIZE);
			instrSize[blockCount][count++] = *(asm_list[i].size) + SEPERATORSIZE;
			opcodePass(&asm_list[i], op, ptr); 
			//temp = i;

			ptr += *(asm_list[i].size) + SEPERATORSIZE;
		}
		else
		{ 
			memcpy((void*)ptr, (void*)(asm_list[i].func), *(asm_list[i].size));
			memcpy((void*)(ptr + *(asm_list[i].size)), (void*)seperator, SEPERATORSIZE);
			instrSize[blockCount][count++] = *(asm_list[i].size) + SEPERATORSIZE;
			opcodePass(&asm_list[i], op, ptr);
			ptr += *(asm_list[i].size) + SEPERATORSIZE;

			temp = memGetWord(addr);
			addr += 2;
			j = opcodeIndex(temp);

			//delay branch instruction
 			memcpy((void*)ptr, (void*)(asm_list[j].func), *(asm_list[j].size));
			memcpy((void*)(ptr + *(asm_list[j].size)), (void*)seperator, SEPERATORSIZE);
			instrSize[blockCount][count++] = *(asm_list[j].size) + SEPERATORSIZE + DELAYJUMPSIZE;
			opcodePass(&asm_list[j], temp, ptr);
			ptr += *(asm_list[j].size) + SEPERATORSIZE + DELAYJUMPSIZE;
		}

		// Delay Branch Address Codes
		if (asm_list[i].delay == 1) { //immediate w/o delay branch
			addJumpTable(ptr, addr-2, addr+((signed char)(op&0xff)<<1)+2);
			ptr += DELAYJUMPSIZE;
		}
		else if (asm_list[i].delay == 2) { //offset3
			temp = (op&0xfff) << 1;
			if (temp & 0x1000)
				temp |= 0xfffff000;
			addJumpTable(ptr-DELAYJUMPSIZE, addr-2, addr+((signed)(op&0xfff)<<1));
		}
		else if (asm_list[i].delay == 3) { //immediate
			addJumpTable(ptr-DELAYJUMPSIZE, addr-2, addr+((signed char)(op&0xff)<<1));
		}
		else if (asm_list[i].delay == 4) { //indeterminate
			//forces a page flip
			addJumpTable(ptr-DELAYJUMPSIZE, 0, 0);
		}
	}

	page->e_addr = addr-2;

	//Second Pass -- Complete jump opcodes
	for (i=0; i<jmp; i++)
	{
		if (temp = getOffset(page, jumps[i].src, jumps[i].dest)) {
 			memcpy((void*)(jumps[i].ptr), (void*)PageJump, DELAYJUMPSIZE);
			*(signed long*)((jumps[i].ptr)+DELAYJUMPSIZE-4) = (signed short)temp;
		}
		else
			memcpy((void*)jumps[i].ptr, (void*)PageFlip, DELAYJUMPSIZE);
	}

	free((void*)jumps);
	memcpy((void*)ptr, (void*)epilogue, EPILOGSIZE);
}

uint16 swapWord(uint16 temp)
{
	uint16 scrap;

#if defined(__GNUC__)
	__asm__ __volatile__ ("mov %ax, _temp");
	__asm__ __volatile__ ("xchg %ah, %al");
	__asm__ __volatile__ ("mov _scrap, %ax");
#else
	_asm {
		mov ax,temp	
		xchg ah,al
		mov scrap,ax
	}
#endif
	return scrap;
}

uint32 swapLong(uint32 temp)
{
	uint32 scrap;

#if defined(__GNUC__)
	__asm__ __volatile__ ("mov %eax, _temp");
	__asm__ __volatile__ ("bswap %eax");
	__asm__ __volatile__ ("mov _scrap, %eax");
#else
	_asm {
		mov eax,temp
		bswap eax
		mov scrap,eax
	}
#endif
	return scrap;
}

short getOffset(Block *tmp, uint32 src, uint32 dest)
{
	int i = 0;
	uint8 sign = 0;
	uint16 offset = 0;
	uint32 current, begin, end;

	if (src < tmp->b_addr || src > tmp->e_addr || dest < tmp->b_addr || dest > tmp->e_addr)
		return 0;

	if (src == dest)
		return 0;

	if (src > dest) {
		sign = 1;
		begin = dest;
		end = src;
	}
	else {
		begin = src;
		end = dest;
	}

	// find location of first address
	current = begin + 2;
	i = ((begin - tmp->b_addr)>>1) + 1;

	// include jumps opcode if jumping in reverse
	if (sign)
		offset += instrSize[blockCount][i-1];// + 2;

	while (current <= end) {
		offset += instrSize[blockCount][i++];
		current += 2;
	}

	//offset += 2;
	if (sign) offset = 0-offset;

	return offset;
}

void opcodePass(x86op_desc *op, uint16 opcode, uint8 *ptr)
{
	// multiply source and dest regions by 4 (size of register)

	if (*(op->src) != 0)
		*(ptr + *(op->src)) = (uint8)(((opcode >> 4) & 0xf)<<2);

	if (*(op->dest) != 0)
		*(ptr + *(op->dest)) = (uint8)(((opcode >> 8) & 0xf)<<2);

	if (*(op->off1) != 0)
		*(ptr + *(op->off1)) = (uint8)(opcode & 0xf);

	if (*(op->imm) != 0)
		*(ptr + *(op->imm)) = (uint8)(opcode & 0xff);

	if (*(op->off3) != 0)
		*(uint16*)(ptr + *(op->off3)) = (uint16)(opcode & 0xfff);
}

/* i386 Optimized Functions

void memSetByte(uint32 addr, uint8 data)
{
	uint32 temp;
	__asm {
		mov eax, addr //addr, eax
		shr eax, 001Ch  // addr >> 28
		and eax, 000fh  // (addr >> 28) & 0x4
		cmp eax, 0000h  // case (0x0)
		cmp eax, 0001h  // case (0x1)
		cmp eax, 0002h  // case (0x2)
		cmp eax, 0003h  // case (0x3)
		jne	L4
		call Process_C 
		jmp End         // break
	L4:	cmp eax, 0004h  // case (0x4)
		jne	L6
		jmp End			// break
	L6:	cmp eax, 0006h  // case (0x6)
		jne	LC
		jmp End			// break
	LC:	cmp eax, 000Ch  // case (0xc)
		jne LF
		jmp End			// break
	LF:	cmp eax, 000Fh  // case (0xf)
		jne Default			// I need to figure out how to handle a default
		jmp End
	}
	
	__asm {
Process_C:
		mov eax, addr  // temp = ((addr & 0x7ffffff)>>16);
		and addr, 0x7FFFFFF
		shr eax, 10h
		mov temp, eax
		cmp temp, 0x0 // if (temp>=0x0 && temp<=0xf)
		jl	LA1
		cmp temp, 0xF
		jg	LA1
		cmp temp, 0x10 // else if (temp == 0x10)
		je  LA2
		cmp temp, 0x18 // else if (temp == 0x18 || temp == 0x19)
		je	LA3
		cmp temp, 0x19
		je	LA3
		cmp temp, 0x20 // else if (temp >= 0x20 && temp <= 0x2f)
		jl  LA4
		cmp temp, 0x2F
		jg	LA4
		cmp temp, 0x200 // else if (temp >= 0x200 && temp <= 0x57f)
		jl	LA5
		cmp temp, 0x57F
		jg	LA5
		cmp temp, 0x580 // else if (temp >= 0x580 && temp <= 0x58f)
		jl	LA6
		cmp temp, 0x58F
		jg	LA6
		cmp temp, 0x5A0 // else if (temp >= 0x5a0 && temp <= 0x5b0)
		jl	LA7
		cmp temp, 0x5B0
		jg	LA7
		cmp temp, 0x5C0 // else if (temp >= 0x5c0 && temp <= 0x5cc)
		jl	LA8
		cmp temp, 0x5CC
		jl	LA8
		cmp temp, 0x5D0 // else if (temp == 0x5d0)
		je	LA9
		cmp temp, 0x5E0 // else if (temp >= 0x5e0 && temp <= 0x5e7)
		jl	LA10
		cmp temp, 0x5E7
		jg	LA10
		cmp temp, 0x5F0 // else if (temp == 0x5f0)
		je	LA11
		cmp temp, 0x5F8 // else if (temp == 0x5f8)
		je	LA12
		cmp temp, 0x5FE // else if (temp == 0x5fe)
		je	LA13
		cmp temp, 0x600 // else if (temp >= 0x600 && temp <= 0x60f)
		jl	LA14
		cmp temp, 0x60F
		jg	LA14
		cmp temp, 0x7FF // else if (temp == 0x7ff)
		je	LA15
		jmp End // break
	}
	__asm LA1: 
		IPL[addr & 0x7ffff] = data;
	__asm jmp End
	__asm LA2:
	{
		SMPC[addr & 0xff] = data;
		if ((addr & 0xff) == 0x1f)
		updateSMPC();
	}
	__asm jmp End
	__asm LA3:
		Backup[addr & 0x1ffff] = data;
	__asm jmp End
	__asm LA4:
		RAM[addr & 0x1fffff] = data;
	__asm jmp End
	__asm LA5:
		ABus[addr & 0x3fffff] = data;
	__asm jmp End
	__asm LA6:
		SCD_WRITE(addr, data);
	__asm jmp End
	__asm LA7:
		Sound[addr & 0x1fffff] = data;
	__asm jmp End
	__asm LA8:
		VDP1[addr & 0x3ffff] = data;
	__asm jmp End
	__asm LA9:
	{
		VDP1[(addr & 0x1f)+0x30000] = data;
		VDP_UPD_REGS(addr);
	}
	__asm jmp End
	__asm LA10:
		VDP2[addr & 0xfffff] = data;
	__asm jmp End
	__asm LA11:
		VDP2[(addr & 0x1fff)+0x80000] = data;
	__asm jmp End
	__asm LA12:
	{
		VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
		VDP_UPD_REGS(addr);
	}
	__asm jmp End
	__asm LA13:
	{
		SCU[addr & 0xff] = data;
		checkHW(addr);
	}
	__asm jmp End
	__asm LA14:
		RAM[(addr & 0x3fffff)+0x100000] = data;
	__asm jmp End
	__asm LA15:
	{
		Special[(addr & 0xff) + 0x1ff] = data;
		updateMiscRegs(addr);
	}
	__asm jmp End
	__asm Default:
		ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	__asm End:

}

void memSetWord(uint32 addr, uint16 data)
{
	uint32 temp;
	__asm {
		mov eax, addr
		shr eax, 001Ch  // addr >> 28
		and eax, 000fh  // (addr >> 28) & 0xf
		cmp eax, 0000h  // case (0x0)
		cmp eax, 0001h  // case (0x1)
		cmp eax, 0002h  // case (0x2)
		cmp eax, 0003h  // case (0x3)
		jne	L4
		call Process_C 
		jmp End         // break
	L4:	cmp eax, 0004h  // case (0x4)
		jne	L6
		jmp End			// break
	L6:	cmp eax, 0006h  // case (0x6)
		jne	LC
		jmp End			// break
	LC:	cmp eax, 000Ch  // case (0xc)
		jne LF
		jmp End			// break
	LF:	cmp eax, 000Fh  // case (0xf)
		jne Default			// I need to figure out how to handle a default
		jmp End
	}
	
	__asm {
Process_C:
		mov eax, addr  // temp = ((addr & 0x7ffffff)>>16);
		and addr, 0x7FFFFFF
		shr eax, 10h
		mov temp, eax
		cmp temp, 0x0 // if (temp>=0x0 && temp<=0xf)
		jl	LA1
		cmp temp, 0xF
		jg	LA1
		cmp temp, 0x10 // else if (temp == 0x10)
		je  LA2
		cmp temp, 0x18 // else if (temp == 0x18 || temp == 0x19)
		je	LA3
		cmp temp, 0x19
		je	LA3
		cmp temp, 0x20 // else if (temp >= 0x20 && temp <= 0x2f)
		jl  LA4
		cmp temp, 0x2F
		jg	LA4
		cmp temp, 0x200 // else if (temp >= 0x200 && temp <= 0x57f)
		jl	LA5
		cmp temp, 0x57F
		jg	LA5
		cmp temp, 0x580 // else if (temp >= 0x580 && temp <= 0x58f)
		jl	LA6
		cmp temp, 0x58F
		jg	LA6
		cmp temp, 0x5A0 // else if (temp >= 0x5a0 && temp <= 0x5b0)
		jl	LA7
		cmp temp, 0x5B0
		jg	LA7
		cmp temp, 0x5C0 // else if (temp >= 0x5c0 && temp <= 0x5cc)
		jl	LA8
		cmp temp, 0x5CC
		jl	LA8
		cmp temp, 0x5D0 // else if (temp == 0x5d0)
		je	LA9
		cmp temp, 0x5E0 // else if (temp >= 0x5e0 && temp <= 0x5e7)
		jl	LA10
		cmp temp, 0x5E7
		jg	LA10
		cmp temp, 0x5F0 // else if (temp == 0x5f0)
		je	LA11
		cmp temp, 0x5F8 // else if (temp == 0x5f8)
		je	LA12
		cmp temp, 0x5FE // else if (temp == 0x5fe)
		je	LA13
		cmp temp, 0x600 // else if (temp >= 0x600 && temp <= 0x60f)
		jl	LA14
		cmp temp, 0x60F
		jg	LA14
		cmp temp, 0x7FF // else if (temp == 0x7ff)
		je	LA15
		jmp End // break
	}
	__asm LA1: 
		*(uint16*)IPL[addr & 0x7ffff] = data;
	__asm jmp End
	__asm LA2:
	{
		*(uint16*)SMPC[addr & 0xff] = data;
		if ((addr & 0xff) == 0x1f)
		updateSMPC();
	}
	__asm jmp End
	__asm LA3:
		*(uint16*)Backup[addr & 0x1ffff] = data;
	__asm jmp End
	__asm LA4:
		*(uint16*)RAM[addr & 0x1fffff] = data;
	__asm jmp End
	__asm LA5:
		*(uint16*)ABus[addr & 0x3fffff] = data;
	__asm jmp End
	__asm LA6:
		SCD_WRITE(addr, data);
	__asm jmp End
	__asm LA7:
		*(uint16*)Sound[addr & 0x1fffff] = data;
	__asm jmp End
	__asm LA8:
		*(uint16*)VDP1[addr & 0x3ffff] = data;
	__asm jmp End
	__asm LA9:
	{
		*(uint16*)VDP1[(addr & 0x1f)+0x30000] = data;
		VDP_UPD_REGS(addr);
	}
	__asm jmp End
	__asm LA10:
		*(uint16*)VDP2[addr & 0xfffff] = data;
	__asm jmp End
	__asm LA11:
		*(uint16*)VDP2[(addr & 0x1fff)+0x80000] = data;
	__asm jmp End
	__asm LA12:
	{
		*(uint16*)VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
		VDP_UPD_REGS(addr);
	}
	__asm jmp End
	__asm LA13:
	{
		*(uint16*)SCU[addr & 0xff] = data;
		checkHW(addr);
	}
	__asm jmp End
	__asm LA14:
		*(uint16*)RAM[(addr & 0x3fffff)+0x100000] = data;
	__asm jmp End
	__asm LA15:
	{
		*(uint16*)Special[(addr & 0xff) + 0x1ff] = data;
		updateMiscRegs(addr);
	}
	__asm jmp End
	__asm Default:
		ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	__asm End:

}

void memSetLong(uint32 addr, uint32 data)
{
	uint32 temp;
	__asm {
		mov eax, addr
		shr eax, 001Ch  // addr >> 28
		and eax, 000fh  // (addr >> 28) & 0xf
		cmp eax, 0000h  // case (0x0)
		cmp eax, 0001h  // case (0x1)
		cmp eax, 0002h  // case (0x2)
		cmp eax, 0003h  // case (0x3)
		jne	L4
		call Process_C 
		jmp End         // break
	L4:	cmp eax, 0004h  // case (0x4)
		jne	L6
		jmp End			// break
	L6:	cmp eax, 0006h  // case (0x6)
		jne	LC
		jmp End			// break
	LC:	cmp eax, 000Ch  // case (0xc)
		jne LF
		jmp End			// break
	LF:	cmp eax, 000Fh  // case (0xf)
		jne Default			// I need to figure out how to handle a default
		jmp End
	}
	
	__asm {
Process_C:
		mov eax, addr  // temp = ((addr & 0x7ffffff)>>16);
		and addr, 0x7FFFFFF
		shr eax, 10h
		mov temp, eax
		cmp temp, 0x0 // if (temp>=0x0 && temp<=0xf)
		jl	LA1
		cmp temp, 0xF
		jg	LA1
		cmp temp, 0x10 // else if (temp == 0x10)
		je  LA2
		cmp temp, 0x18 // else if (temp == 0x18 || temp == 0x19)
		je	LA3
		cmp temp, 0x19
		je	LA3
		cmp temp, 0x20 // else if (temp >= 0x20 && temp <= 0x2f)
		jl  LA4
		cmp temp, 0x2F
		jg	LA4
		cmp temp, 0x200 // else if (temp >= 0x200 && temp <= 0x57f)
		jl	LA5
		cmp temp, 0x57F
		jg	LA5
		cmp temp, 0x580 // else if (temp >= 0x580 && temp <= 0x58f)
		jl	LA6
		cmp temp, 0x58F
		jg	LA6
		cmp temp, 0x5A0 // else if (temp >= 0x5a0 && temp <= 0x5b0)
		jl	LA7
		cmp temp, 0x5B0
		jg	LA7
		cmp temp, 0x5C0 // else if (temp >= 0x5c0 && temp <= 0x5cc)
		jl	LA8
		cmp temp, 0x5CC
		jl	LA8
		cmp temp, 0x5D0 // else if (temp == 0x5d0)
		je	LA9
		cmp temp, 0x5E0 // else if (temp >= 0x5e0 && temp <= 0x5e7)
		jl	LA10
		cmp temp, 0x5E7
		jg	LA10
		cmp temp, 0x5F0 // else if (temp == 0x5f0)
		je	LA11
		cmp temp, 0x5F8 // else if (temp == 0x5f8)
		je	LA12
		cmp temp, 0x5FE // else if (temp == 0x5fe)
		je	LA13
		cmp temp, 0x600 // else if (temp >= 0x600 && temp <= 0x60f)
		jl	LA14
		cmp temp, 0x60F
		jg	LA14
		cmp temp, 0x7FF // else if (temp == 0x7ff)
		je	LA15
		jmp End // break
	}
	__asm LA1: 
		*(uint32*)IPL[addr & 0x7ffff] = data;
	__asm jmp End
	__asm LA2:
	{
		*(uint32*)SMPC[addr & 0xff] = data;
		if ((addr & 0xff) == 0x1f)
		updateSMPC();
	}
	__asm jmp End
	__asm LA3:
		*(uint32*)Backup[addr & 0x1ffff] = data;
	__asm jmp End
	__asm LA4:
		*(uint32*)RAM[addr & 0x1fffff] = data;
	__asm jmp End
	__asm LA5:
		*(uint32*)ABus[addr & 0x3fffff] = data;
	__asm jmp End
	__asm LA6:
		SCD_WRITE(addr, data);
	__asm jmp End
	__asm LA7:
		*(uint32*)Sound[addr & 0x1fffff] = data;
	__asm jmp End
	__asm LA8:
		*(uint32*)VDP1[addr & 0x3ffff] = data;
	__asm jmp End
	__asm LA9:
	{
		*(uint32*)VDP1[(addr & 0x1f)+0x30000] = data;
		VDP_UPD_REGS(addr);
	}
	__asm jmp End
	__asm LA10:
		*(uint32*)VDP2[addr & 0xfffff] = data;
	__asm jmp End
	__asm LA11:
		*(uint32*)VDP2[(addr & 0x1fff)+0x80000] = data;
	__asm jmp End
	__asm LA12:
	{
		*(uint32*)VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
		VDP_UPD_REGS(addr);
	}
	__asm jmp End
	__asm LA13:
	{
		*(uint32*)SCU[addr & 0xff] = data;
		checkHW(addr);
	}
	__asm jmp End
	__asm LA14:
		*(uint32*)RAM[(addr & 0x3fffff)+0x100000] = data;
	__asm jmp End
	__asm LA15:
	{
		*(uint32*)Special[(addr & 0xff) + 0x1ff] = data;
		updateMiscRegs(addr);
	}
	__asm jmp End
	__asm Default:
		ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	__asm End:

}*/

void memSetByte(uint32 addr, uint8 data)
{
	uint32 temp;
	dynaLock();

	switch ((addr >> 28) & 0xf)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp>=0x0 && temp<=0xf)
				IPL[addr & 0x7ffff] = data;
			else if (temp == 0x10)
			{
				SMPC[addr & 0xff] = data;
				if ((addr & 0xff) == 0x1f)
					updateSMPC();
			}
			else if (temp == 0x18 || temp == 0x19)
				Backup[addr & 0x1ffff] = data;
			else if (temp >= 0x20 && temp <= 0x2f)
				RAM[addr & 0x1fffff] = data;
			else if (temp >= 0x200 && temp <= 0x57f)
				ABus[addr & 0x3fffff] = data;
			else if (temp >= 0x580 && temp <= 0x58f)
				SCD_WRITE(addr, data);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				Sound[addr & 0x1fffff] = data;
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				VDP1[addr & 0x3ffff] = data;
			else if (temp == 0x5d0)
			{
				VDP1[(addr & 0x1f)+0x30000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				VDP2[addr & 0xfffff] = data;
			else if (temp == 0x5f0)
				VDP2[(addr & 0x1fff)+0x80000] = data;
			else if (temp == 0x5f8)
			{
				VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp == 0x5fe)
			{
				SCU[addr & 0xff] = data;
				checkHW(addr);
			}
			else if (temp >= 0x600 && temp <= 0x60f)
				RAM[(addr & 0x3fffff)+0x100000] = data;
			else if (temp == 0x7ff)
			{
				Special[(addr & 0xff) + 0x1ff] = data;
				updateMiscRegs(addr);
			}
			break;
		case (0x4):
		case (0x6): // (0x60000000)
		case (0xc):
			//unknown memory area 
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				casLatency = (uint32)data;
			else
				Special[addr & 0x1ff] = data;
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	}
	dynaFree();
}

void memSetWord(uint32 addr, uint16 data)
{
	uint32 temp;
	dynaLock();

	data = swapWord(data);
	switch ((addr & 0xf0000000)>>28)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				*(uint16*)&IPL[addr & 0x7ffff] = data;
			else if (temp == 0x10)
			{
				*(uint16*)&SMPC[addr & 0xff] = data;
				if ((addr & 0xff) == 0x1f)
					updateSMPC();
			}
			else if (temp == 0x18 || temp == 0x19)
				*(uint16*)&Backup[addr & 0x1ffff] = data;
			else if (temp >= 0x20 && temp <= 0x2f)
				*(uint16*)&RAM[addr & 0x1fffff] = data;
			else if (temp >= 0x200 && temp <= 0x57f)
				*(uint16*)&ABus[addr & 0x3fffff] = data;
			else if (temp >= 0x580 && temp <= 0x58f)
				SCD_WRITE(addr, data);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				*(uint16*)&Sound[addr & 0x1fffff] = data;
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				*(uint16*)&VDP1[addr & 0x3ffff] = data;
			else if (temp == 0x5d0)
			{
				*(uint16*)&VDP1[(addr & 0x1f)+0x30000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				*(uint16*)&VDP2[addr & 0xfffff] = data;
			else if (temp == 0x5f0)
				*(uint16*)&VDP2[(addr & 0x1fff)+0x80000] = data;
			else if (temp == 0x5f8)
			{
				*(uint16*)&VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp == 0x5fe)
			{
				*(uint16*)&SCU[addr & 0xff] = swapWord(data);
				checkHW(addr);
			}
			else if (temp >= 0x600 && temp <= 0x60f)
				*(uint16*)&RAM[(addr & 0x3fffff)+0x100000] = data;
			else if (temp == 0x7ff)
			{
				*(uint16*)&Special[(addr & 0xff) + 0x1ff] = data;
				updateMiscRegs(addr);
			}
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			//unknown memory area 
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				casLatency = (uint32)data;
			else
				*(uint16*)&Special[addr & 0x1ff] = data;
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	}
	dynaFree();
}

void memSetLong(uint32 addr, uint32 data)
{
	uint32 temp;
	dynaLock();

	data = swapLong(data);
	switch ((addr & 0xf0000000)>>28)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				*(uint32*)&IPL[addr & 0x7ffff] = data;
			else if (temp == 0x10)
			{
				*(uint32*)&SMPC[addr & 0xff] = data;
				// just incase COMREG is referenced like this
				if ((addr & 0xff) == 0x1f)
					updateSMPC();
			}
			else if (temp == 0x18 || temp == 0x19)
				*(uint32*)&Backup[addr & 0x1ffff] = data;
			else if (temp >= 0x20 && temp <= 0x2f)
				*(uint32*)&RAM[addr & 0x1fffff] = data;
			else if (temp >= 0x200 && temp <= 0x57f)
				*(uint32*)&ABus[addr & 0x3fffff] = data;
			else if (temp >= 0x580 && temp <= 0x58f)
				SCD_WRITE(addr, data);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				*(uint32*)&Sound[addr & 0x1fffff] = data;
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				*(uint32*)&VDP1[addr & 0x3ffff] = data;
			else if (temp == 0x5d0)
			{
				*(uint32*)&VDP1[(addr & 0x1f)+0x30000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				*(uint32*)&VDP2[addr & 0xfffff] = data;
			else if (temp == 0x5f0)
				*(uint32*)&VDP2[(addr & 0x1fff)+0x80000] = data;
			else if (temp == 0x5f8)
			{
				*(uint32*)&VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp == 0x5fe)
			{
				*(uint32*)&SCU[addr & 0xff] = swapLong(data);
				checkHW(addr);
			}
			else if (temp >= 0x600 && temp <= 0x60f)
				*(uint32*)&RAM[(addr & 0x3fffff)+0x100000] = data;
			else if (temp == 0x7ff)
			{
				*(uint32*)&Special[(addr & 0xff) + 0x1ff] = data;
				updateMiscRegs(addr);
			}
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			//unknown memory area 
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				casLatency = data;
			else
				*(uint32*)&Special[addr & 0x1ff] = data;
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	}
	dynaFree();
}

uint8 memGetByte(uint32 addr)
{
	uint32 temp;
	uint8 val = 0;
	dynaLock();

	switch ((addr >> 28) & 0xf)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				val = IPL[addr & 0x7ffff];
			else if (temp == 0x10)
				val = SMPC[addr & 0xff];
			else if (temp == 0x18 || temp == 0x19)
				val = Backup[addr & 0x1ffff];
			else if (temp >= 0x20 && temp <= 0x2f)
				val = RAM[addr & 0x1fffff];
			else if (temp >= 0x200 && temp <= 0x57f)
				val = ABus[addr & 0x3fffff];
			else if (temp >= 0x580 && temp <= 0x58f)
				val = (uint8)(SCD_READ(addr) >> 8);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				val = Sound[addr & 0x1fffff];
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				val = VDP1[addr & 0x3ffff];
			else if (temp == 0x5d0)
				val = VDP1[(addr & 0x1f)+0x30000];
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				val = VDP2[addr & 0xfffff];
			else if (temp == 0x5f0)
				val = VDP2[(addr & 0x1fff)+0x80000];
			else if (temp == 0x5f8)
				val = VDP2[(addr & 0x1ff)+0x80000+0x1000];
			else if (temp == 0x5fe)
				val = SCU[addr & 0xff];
			else if (temp >= 0x600 && temp <= 0x60f)
				val = RAM[(addr & 0x3fffff)+0x100000];
			else if (temp == 0x7ff)
				val = Special[(addr & 0xff) + 0x1ff];
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			// unknown memory area
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				val = *(uint8*)&casLatency;

			val = Special[addr & 0x1ff];
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem read: 0x%8x", addr);
	}

	dynaFree();
	return val;
}

uint16 memGetWord(uint32 addr)
{
	uint32 temp;
	uint16 val = 0;
	dynaLock();

	switch ((addr >> 28) & 0xf)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				val = swapWord(*(uint16*)&IPL[addr & 0x7ffff]);
			else if (temp == 0x10)
				val = *(uint16*)&SMPC[addr & 0xff];
			else if (temp == 0x18 || temp == 0x19)
				val = swapWord(*(uint16*)&Backup[addr & 0x1ffff]);
			else if (temp >= 0x20 && temp <= 0x2f)
				val = swapWord(*(uint16*)&RAM[addr & 0x1fffff]);
			else if (temp >= 0x200 && temp <= 0x57f)
				val = *(uint16*)&ABus[addr & 0x3fffff];
			else if (temp >= 0x580 && temp <= 0x58f)
				val = (uint16)SCD_READ(addr);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				val = swapWord(*(uint16*)&Sound[addr & 0x1fffff]);
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				val = *(uint16*)&VDP1[addr & 0x3ffff];
			else if (temp == 0x5d0)
				val = *(uint16*)&VDP1[(addr & 0x1f)+0x30000];
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				val = *(uint16*)&VDP2[addr & 0xfffff];
			else if (temp == 0x5f0)
				val = *(uint16*)&VDP2[(addr & 0x1fff)+0x80000];
			else if (temp == 0x5f8)
				val = *(uint16*)&VDP2[(addr & 0x1ff)+0x80000+0x1000];
			else if (temp == 0x5fe)
				val = *(uint16*)&SCU[addr & 0xff];
			else if (temp >= 0x600 && temp <= 0x60f)
				val = swapWord(*(uint16*)&RAM[(addr & 0x3fffff)+0x100000]);
			else if (temp == 0x7ff)
				val = *(uint16*)&Special[(addr & 0xff) + 0x1ff];
			else
				return 0;
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			// unknown memory area
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				val = *(uint16*)&casLatency;

			val = *(uint16*)&Special[addr & 0x1ff];
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem read: 0x%8x", addr);
	}

	dynaFree();
	return val;
}

uint32 memGetLong(uint32 addr)
{
	uint32 temp, val = 0;
	dynaLock();

	switch ((addr & 0xf0000000)>>28)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				val = swapLong(*(uint32*)&IPL[addr & 0x7ffff]);
			else if (temp == 0x10)
				val = *(uint32*)&SMPC[addr & 0xff];
			else if (temp == 0x18 || temp == 0x19)
				val = swapLong(*(uint32*)&Backup[addr & 0x1ffff]);
			else if (temp >= 0x20 && temp <= 0x2f)
				val = swapLong(*(uint32*)&RAM[addr & 0x1fffff]);
			else if (temp >= 0x200 && temp <= 0x57f)
				val = *(uint32*)&ABus[addr & 0x3fffff];
			else if (temp >= 0x580 && temp <= 0x58f)
				val = SCD_READ(addr);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				val = swapLong(*(uint32*)&Sound[addr & 0x1fffff]);
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				val = *(uint32*)&VDP1[addr & 0x3ffff];
			else if (temp == 0x5d0)
				val = *(uint32*)&VDP1[(addr & 0x1f)+0x30000];
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				val = *(uint32*)&VDP2[addr & 0xfffff];
			else if (temp == 0x5f0)
				val = *(uint32*)&VDP2[(addr & 0x1fff)+0x80000];
			else if (temp == 0x5f8)
				val = *(uint32*)&VDP2[(addr & 0x1ff)+0x80000+0x1000];
			else if (temp == 0x5fe)
				val = *(uint32*)&SCU[addr & 0xff];
			else if (temp >= 0x600 && temp <= 0x60f)
				val = swapLong(*(uint32*)&RAM[(addr & 0x3fffff)+0x100000]);
			else if (temp == 0x7ff)
				val = *(uint32*)&Special[(addr & 0xff) + 0x1ff];
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			// unknown memory area
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				val = casLatency;

			val = *(uint32*)&Special[addr & 0x1ff];
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem read: 0x%8x", addr);
	}

	dynaFree();
	return val;
}

#endif
