// DEBUG.C
// Titan Debugging Module

#include "global.h"
#ifdef TITAN_DEBUG

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <memory.h>
#include "debug.h"
#include "mem.h"
#include "sh2.h"

// Debugging Modes
#define DBG_NONE		0x00
#define DBG_WAIT		0x01
#define DBG_DISASM		0x02
#define DBG_WATCHPC		0x04
#define DBG_WATCHMEM	0x08
#define DBG_WATCHREG	0x16

extern uint32 Count;

#ifdef WIN32
#include <windows.h>

DWORD g_temp, charRead;
HANDLE hConInput, hConOutput;
INPUT_RECORD Buffer;

void DebugMsg(char *message, ...)
{
	va_list data;
	char tempstr[128];
	static DWORD charWritten = 0;

	va_start(data, message);
	vsprintf(tempstr, message, data);
	va_end(data);

	WriteConsole(hConOutput, tempstr, strlen(tempstr), &charWritten, NULL);
}

#define getDebugChar(x)		x = '\0'; \
							do { ReadConsole(hConInput, &x, 1, &charRead, NULL); \
							}while((x == '\n') && (x == '\0')); \
							FlushConsoleInputBuffer(hConInput);

#define keyInput(x)			x = '\0'; GetNumberOfConsoleInputEvents(hConInput, &g_temp); \
							while(g_temp-- > 0) { \
								ReadConsoleInput(hConInput, &Buffer, sizeof(INPUT_RECORD), &charRead); \
								if (Buffer.EventType == KEY_EVENT && Buffer.Event.KeyEvent.bKeyDown) { \
									x = Buffer.Event.KeyEvent.uChar.AsciiChar; \
									FlushConsoleInputBuffer(hConInput);	break; } }\

#else
#define DebugMsg			printf
#define getDebugChar(x)		x = getchar();
#define keyInput(x)			
#endif

void SH2_Disasm(tagSH2 *sh2)
{
	int i;
    unsigned short op;

	op = memGetWord(PC);
	i = opcodeIndex(op);
    DebugMsg("%08X (%04X):  ", PC, op);
    
	if (opcode_list[i].format == ZERO_F)
		DebugMsg("%s", opcode_list[i].mnem);
	else if (opcode_list[i].format == N_F)
		DebugMsg(opcode_list[i].mnem, (op >> 8) & 0xf);
	else if (opcode_list[i].format == M_F)
		DebugMsg(opcode_list[i].mnem, (op >> 8) & 0xf);
	else if (opcode_list[i].format == NM_F)
		DebugMsg(opcode_list[i].mnem, (op >> 4) & 0xf, (op >> 8) & 0xf);
	else if (opcode_list[i].format == MD_F)
	{
		if (op & 0x100)
			DebugMsg(opcode_list[i].mnem, (op & 0xf) * 2, (op >> 4) & 0xf);
		else
			DebugMsg(opcode_list[i].mnem, op & 0xf, (op >> 4) & 0xf);
	}
	else if (opcode_list[i].format == ND4_F)
	{
		if (op & 0x100)
			DebugMsg(opcode_list[i].mnem, (op & 0xf) * 2, (op >> 4) & 0xf);
		else
			DebugMsg(opcode_list[i].mnem, (op & 0xf), (op >> 4) & 0xf);
	}
	else if (opcode_list[i].format == NMD_F)
	{
		if ((op & 0xf000) == 0x1000)
			DebugMsg(opcode_list[i].mnem, (op >> 4) & 0xf, (op & 0xf) * 4, (op >> 8) & 0xf);
		else
			DebugMsg(opcode_list[i].mnem, (op & 0xf) * 4, (op >> 4) & 0xf, (op >> 8) & 0xf);
	}
	else if (opcode_list[i].format == D_F)
	{
		if (opcode_list[i].dat <= 4)
		{
			if ((op & 0xff00) == 0xc700)
			{
				DebugMsg(opcode_list[i].mnem, (op & 0xff) * opcode_list[i].dat + 4);
				DebugMsg("  ; 0x%08X", (op & 0xff) * opcode_list[i].dat + 4 + PC);
			}
			else
				DebugMsg(opcode_list[i].mnem, (op & 0xff) * opcode_list[i].dat);
		}
		else
		{
			if (op & 0x80)  /* sign extend */
				DebugMsg(opcode_list[i].mnem, (((op & 0xff) + 0xffffff00) * 2) + PC + 4);
			else
				DebugMsg(opcode_list[i].mnem, ((op & 0xff) * 2) + PC + 4);
		}        
	}
	else if (opcode_list[i].format == D12_F)
	{
		if (op & 0x800)         /* sign extend */
			DebugMsg(opcode_list[i].mnem, ((op & 0xfff) + 0xfffff000) * 2 + PC + 4);
		else
			DebugMsg(opcode_list[i].mnem, (op & 0xfff) * 2 + PC + 4);
	}
	else if (opcode_list[i].format == ND8_F)
	{
		if ((op & 0xf000) == 0x9000)    /* .W */
		{
			DebugMsg(opcode_list[i].mnem, (op & 0xff) * opcode_list[i].dat + 4, (op >> 8) & 0xf);
			DebugMsg("  ; 0x%08X", (op & 0xff) * opcode_list[i].dat + 4 + PC);
		}
		else  /* .L */
		{
			DebugMsg(opcode_list[i].mnem, (op & 0xff) * opcode_list[i].dat + 4, (op >> 8) & 0xf);
			DebugMsg("  ; 0x%08X", (op & 0xff) * opcode_list[i].dat + 4 + (PC & 0xfffffffc));
		}
	}
	else if (opcode_list[i].format == I_F)
		DebugMsg(opcode_list[i].mnem, op & 0xff);
	else if (opcode_list[i].format == NI_F)
		DebugMsg(opcode_list[i].mnem, op & 0xff, (op >> 8) & 0xf);
	else
	{
		DebugMsg("unrecognized\n");
		return;
	}
	DebugMsg("\n");
}

void debugHook()
{
	char tmp = '\0', temp = '\0';
	static uint16 debug_status = DBG_WAIT;
	static uint8 watchcount = 0;

	if (debug_status & DBG_WAIT) {
		DebugMsg("-");
		do {
			getDebugChar(temp);
			temp = toupper(temp);
		}while((temp < 'A') || (temp > 'Z'));
	}

	if (temp != '\0')
	{
		if (temp == 'G')
			debug_status &= ~DBG_WAIT;
		else if (temp == 'D')
		{
			debug_status |= DBG_DISASM;
			debug_status &= ~DBG_WAIT;
		}
		else if (temp == 'C')
			DebugMsg("COUNT: %u instructions executed\n", Count);
		else
			DebugMsg("Unrecognized Command\n");

		temp = '\0';
	}

	if (debug_status & DBG_DISASM)
		SH2_Disasm(&master);

	// handle key presses to stop watches, disassembly, etc.
	if (!(debug_status & DBG_WAIT))	{
		keyInput(tmp);

		if (tmp != '\0') {
			//DebugMsg("PAUSE: %c pressed\n", tmp);
			debug_status &= ~DBG_DISASM;
			debug_status |= DBG_WAIT;
		}		
	}
}

#endif