// Emu.cpp
// Main Emulation Routines

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include "dynamap.h"
#include "global.h"
#include "m68k.h"
#include "mem.h"
#include "sh2.h"
#include "scu.h"
#include "plugin.h"

#define CLKSPD	28600000
//#define REFSPD	CLKSPD / 60

#ifdef WIN32
#include <windows.h>
#endif

#ifdef CPU_PROFILING
#include <time.h>
#endif

#ifdef TITAN_DEBUG
#include "debug.h"
#endif

tagInternal Flags;
FILE *console;
uint32 Count = 0;

void emulate(void)
{
	int dummy;
	
#ifdef CPU_PROFILING
	static time_t start, finish;
	time(&start);
#endif

	while (Flags.EmuStatus != EMU_STOP)
	{
		while(Flags.EmuStatus == EMU_PAUSE);

		//manual breakpoint
		if (master.SysReg[3] == 0x2000)
		{
			dummy = 0;
		}
#ifdef TITAN_DEBUG
		debugHook();
#endif
		doInst(&master);
		Count++;

#ifdef CPU_PROFILING
		if (Count = 0x1C00000)
		{
			time(&finish);
			start = finish;
			ConsoleMsg(MSG_NORM, "Clock Ratio: %i/1", difftime(start, finish));
			Count = 0;
		}
#endif

		// MC68EC000 CPU (Currently at 10:4 Ratio)
		if (Flags.SoundEnabled)
		{
			if (Count % 10 == 0)
			{
				step68k(4);
			}
		}

		// Slave CPU
		if (Flags.SlaveEnabled)
		{
#ifdef TITAN_DEBUG
			//debugHook();
#endif
			doInst(&slave);
		}

		updateHW();
	}

	// shutdown miscellaneous emu stuff
	if (Flags.SoundEnabled)
	{
		stop68k();
		Flags.SoundEnabled = 0;
	}
}

void dynaEmulate(void)
{
	Block *dCode = NULL;
	dCode = dynaInit(dCode);

	while (Flags.EmuStatus != EMU_STOP)
	{
		while(Flags.EmuStatus == EMU_PAUSE);
		dynaExecute(&master, dCode);
	}

	FREEMEM(dCode);
}

int emuStartUp(void)
{
	int res;

	if (!(res = memInit()))	{
		initHW();
		init68k();
	}

	return res;
}

void emuCleanUp(void)
{
	if (Settings.ConsoleLog)
	{
		ConsoleMsg(MSG_NORM, "Shutting down console...");
		fclose(console);
	}
}

int initConsole(void)
{
	if ((console = fopen("console.log", "w")) == NULL)
		return 1;

	ConsoleMsg(MSG_NORM, "Initializing console...");
	return 0;
}

void ConsoleMsg(int flag, char *message, ...)
{
	va_list data;
	char tempstr[128];

	va_start(data, message);
	vsprintf(tempstr, message, data);
	va_end(data);

	strcat(tempstr, "\n");
	if (flag == MSG_NORM) {
		if (Settings.ConsoleLog)
			fwrite((void*)tempstr, sizeof(char), strlen(tempstr), console);
	}
	else if (flag == MSG_WARN) {
#ifdef WIN32
		MessageBox(g_hWnd, tempstr, "Titan Warning", MB_OK|MB_ICONWARNING);
#else
		printf("MSG: %s\n", tempstr);
#endif
		if (Settings.ConsoleLog)
			fwrite((void*)tempstr, sizeof(char), strlen(tempstr), console);
	}
	else if (flag == MSG_FATAL)
	{
		Flags.EmuStatus = EMU_STOP;
#ifdef WIN32
		MessageBox(g_hWnd, tempstr, "Fatal Error", MB_OK|MB_ICONSTOP);
		DestroyWindow(g_hWnd);
#else
		printf("ERR: %s\n", tempstr);
		if (Settings.ConsoleLog)
			fwrite((void*)tempstr, sizeof(char), strlen(tempstr), console);
		emuCleanUp();
		exit(1);
#endif
	}
}