// MEM.C
// Titan Memory Routines

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include "global.h"
#include "plugin.h"
#include "smpc.h"
#include "scu.h"
#include "mem.h"
#include "sh2.h"

#define MEMMASK		0x7FFFFFF
#define addr2		(addr & 0x7ffffff)

unsigned char *IPL, *SMPC, *Backup, *RAM, *ABus, *Sound, *VDP1, *VDP2, *SCU, *Special;
uint32 casLatency = 0;

#ifdef GENERIC_MEM_FUNC

uint16 swapWord(uint16 temp)
{
	uint16 scrap;

	((uint8*)&scrap)[0] = ((uint8*)&temp)[1];
	((uint8*)&scrap)[1] = ((uint8*)&temp)[0];
	return scrap;
}

uint32 swapLong(uint32 temp)
{
	uint32 scrap;

	((uint8*)&scrap)[0] = ((uint8*)&temp)[3];
	((uint8*)&scrap)[1] = ((uint8*)&temp)[2];
	((uint8*)&scrap)[2] = ((uint8*)&temp)[1];
	((uint8*)&scrap)[3] = ((uint8*)&temp)[0];
	return scrap;
}

#endif

int memInit(void)
{
	FILE *bkfile;

	IPL = (unsigned char*)ALLOCATE(0x80000);
	SMPC = (unsigned char*)ALLOCATE(0x80);
	Backup = (unsigned char*)ALLOCATE(0x10000);
	RAM = (unsigned char*)ALLOCATE(0x200000);
	ABus = (unsigned char*)ALLOCATE(0x3800000);
	Sound = (unsigned char*)ALLOCATE(0x100ee4);
	VDP1 = (unsigned char*)ALLOCATE(0x30000+0x30);
	VDP2 = (unsigned char*)ALLOCATE(0x80000+0x1000+0x120);
	SCU = (unsigned char*)ALLOCATE(0xd0);
	// This could be more efficiant, but it's <1kB, so I'll let it slide
	Special = (unsigned char*)ALLOCATE(0x1ff + 0xff);

	if (IPL == NULL || SMPC == NULL || Backup == NULL || RAM == NULL || ABus == NULL ||
		Sound == NULL || VDP1 == NULL || VDP2 == NULL || SCU == NULL || Special == NULL)
	{
		ConsoleMsg(MSG_FATAL, "Cannot allocate enough memory");
		return 1;
	}

	if ((bkfile = fopen("backup.ram", "r")) == NULL)
		memset((void*)Backup, 0, 0x10000);
	else
	{
		fread((void*)Backup, sizeof(char), 0x10000, bkfile);
		fclose(bkfile);
	}

	memset((void*)Special, 0, 0x1ff);
	memset((void*)&Flags, 0, sizeof(tagInternal));
	ResetCPU(&master);

	return 0;
}

int loadROM(void)
{
	FILE *bios;

	if ((bios = fopen(Settings.BIOSFile, "r")) == NULL)
	{
		ConsoleMsg(MSG_WARN, "Cannot find BIOS File");
		return 1;
	}
   
	fread((void*)IPL, sizeof(char), 0x80000, bios);
	fclose(bios);

	ResetCPU(&master);
	master.SysReg[3] = 0x20000200;
	return 0;
}

int loadCart(char *szFileName)
{
	long fsize = 0;
	FILE *ROMHandle;

	if ((ROMHandle = fopen(szFileName, "r")) == NULL)
	{
		ConsoleMsg(MSG_WARN, "Failed to load EXE");
		return 1;
    }
	else 
	{
		ResetCPU(&master);
        fseek(ROMHandle, 0, SEEK_END);
        fsize = ftell(ROMHandle);
        rewind(ROMHandle);
		fread((void*)(RAM+0x104000), sizeof(uint8), fsize, ROMHandle);
		fclose(ROMHandle);
		master.SysReg[3] = 0x06004000;
	}
	return 0;
}

void memFree(void)
{
	FILE *bkfile;

	// Update Backup RAM File
	if((bkfile = fopen("backup.ram", "wb")) != NULL)
	{
		fwrite((void*)Backup, sizeof(char), 0x10000, bkfile);
		fclose(bkfile);
	}

	FREEMEM(IPL);
	FREEMEM(SMPC);
	FREEMEM(Backup);
	FREEMEM(RAM);
	FREEMEM(ABus);
	FREEMEM(Sound);
	FREEMEM(VDP1);
	FREEMEM(VDP2);
	FREEMEM(SCU);
	FREEMEM(Special);
}

#ifdef GENERIC_MEM_FUNC

void memSetByte(uint32 addr, unsigned char data)
{
	uint32 temp;

	switch ((addr >> 28) & 0xf)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp>=0x0 && temp<=0xf)
				IPL[addr & 0x7ffff] = data;
			else if (temp == 0x10)
			{
				SMPC[addr & 0xff] = data;
				if ((addr & 0xff) == 0x1f)
					updateSMPC();
			}
			else if (temp == 0x18 || temp == 0x19)
				Backup[addr & 0x1ffff] = data;
			else if (temp >= 0x20 && temp <= 0x2f)
				RAM[addr & 0x1fffff] = data;
			else if (temp >= 0x200 && temp <= 0x57f)
				ABus[addr & 0x3fffff] = data;
			else if (temp >= 0x580 && temp <= 0x58f)
				SCD_WRITE(addr, data);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				Sound[addr & 0x1fffff] = data;
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				VDP1[addr & 0x3ffff] = data;
			else if (temp == 0x5d0)
			{
				VDP1[(addr & 0x1f)+0x30000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				VDP2[addr & 0xfffff] = data;
			else if (temp == 0x5f0)
				VDP2[(addr & 0x1fff)+0x80000] = data;
			else if (temp == 0x5f8)
			{
				VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp == 0x5fe)
			{
				SCU[addr & 0xff] = data;
				checkHW(addr);
			}
			else if (temp >= 0x600 && temp <= 0x60f)
				RAM[(addr & 0x3fffff)+0x100000] = data;
			else if (temp == 0x7ff)
			{
				Special[(addr & 0xff) + 0x1ff] = data;
				updateMiscRegs(addr);
			}
			break;
		case (0x4):
		case (0x6): // (0x60000000)
		case (0xc):
			//unknown memory area 
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				casLatency = (uint32)data;
			else
				Special[addr & 0x1ff] = data;
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	}
}

void memSetWord(uint32 addr, uint16 data)
{
	uint32 temp;

	data = swapWord(data);
	switch ((addr & 0xf0000000)>>28)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				*(uint16*)&IPL[addr & 0x7ffff] = data;
			else if (temp == 0x10)
			{
				*(uint16*)&SMPC[addr & 0xff] = data;
				if ((addr & 0xff) == 0x1f)
					updateSMPC();
			}
			else if (temp == 0x18 || temp == 0x19)
				*(uint16*)&Backup[addr & 0x1ffff] = data;
			else if (temp >= 0x20 && temp <= 0x2f)
				*(uint16*)&RAM[addr & 0x1fffff] = data;
			else if (temp >= 0x200 && temp <= 0x57f)
				*(uint16*)&ABus[addr & 0x3fffff] = data;
			else if (temp >= 0x580 && temp <= 0x58f)
				SCD_WRITE(addr, data);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				*(uint16*)&Sound[addr & 0x1fffff] = data;
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				*(uint16*)&VDP1[addr & 0x3ffff] = data;
			else if (temp == 0x5d0)
			{
				*(uint16*)&VDP1[(addr & 0x1f)+0x30000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				*(uint16*)&VDP2[addr & 0xfffff] = data;
			else if (temp == 0x5f0)
				*(uint16*)&VDP2[(addr & 0x1fff)+0x80000] = data;
			else if (temp == 0x5f8)
			{
				*(uint16*)&VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp == 0x5fe)
			{
				*(uint16*)&SCU[addr & 0xff] = swapWord(data);
				checkHW(addr);
			}
			else if (temp >= 0x600 && temp <= 0x60f)
				*(uint16*)&RAM[(addr & 0x3fffff)+0x100000] = data;
			else if (temp == 0x7ff)
			{
				*(uint16*)&Special[(addr & 0xff) + 0x1ff] = data;
				updateMiscRegs(addr);
			}
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			//unknown memory area 
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				casLatency = (uint32)data;
			else
				*(uint16*)&Special[addr & 0x1ff] = data;
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	}
}

void memSetLong(uint32 addr, uint32 data)
{
	uint32 temp;

	data = swapLong(data);
	switch ((addr & 0xf0000000)>>28)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				*(uint32*)&IPL[addr & 0x7ffff] = data;
			else if (temp == 0x10)
			{
				*(uint32*)&SMPC[addr & 0xff] = data;
				// just incase COMREG is referenced like this
				if ((addr & 0xff) == 0x1f)
					updateSMPC();
			}
			else if (temp == 0x18 || temp == 0x19)
				*(uint32*)&Backup[addr & 0x1ffff] = data;
			else if (temp >= 0x20 && temp <= 0x2f)
				*(uint32*)&RAM[addr & 0x1fffff] = data;
			else if (temp >= 0x200 && temp <= 0x57f)
				*(uint32*)&ABus[addr & 0x3fffff] = data;
			else if (temp >= 0x580 && temp <= 0x58f)
				SCD_WRITE(addr, data);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				*(uint32*)&Sound[addr & 0x1fffff] = data;
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				*(uint32*)&VDP1[addr & 0x3ffff] = data;
			else if (temp == 0x5d0)
			{
				*(uint32*)&VDP1[(addr & 0x1f)+0x30000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				*(uint32*)&VDP2[addr & 0xfffff] = data;
			else if (temp == 0x5f0)
				*(uint32*)&VDP2[(addr & 0x1fff)+0x80000] = data;
			else if (temp == 0x5f8)
			{
				*(uint32*)&VDP2[(addr & 0x1ff)+0x80000+0x1000] = data;
				VDP_UPD_REGS(addr);
			}
			else if (temp == 0x5fe)
			{
				*(uint32*)&SCU[addr & 0xff] = swapLong(data);
				checkHW(addr);
			}
			else if (temp >= 0x600 && temp <= 0x60f)
				*(uint32*)&RAM[(addr & 0x3fffff)+0x100000] = data;
			else if (temp == 0x7ff)
			{
				*(uint32*)&Special[(addr & 0xff) + 0x1ff] = data;
				updateMiscRegs(addr);
			}
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			//unknown memory area 
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				casLatency = data;
			else
				*(uint32*)&Special[addr & 0x1ff] = data;
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem write: 0x%8x", addr);
	}
}

unsigned char memGetByte(uint32 addr)
{
	uint32 temp;
	unsigned char val = 0;

	switch ((addr >> 28) & 0xf)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				val = IPL[addr & 0x7ffff];
			else if (temp == 0x10)
				val = SMPC[addr & 0xff];
			else if (temp == 0x18 || temp == 0x19)
				val = Backup[addr & 0x1ffff];
			else if (temp >= 0x20 && temp <= 0x2f)
				val = RAM[addr & 0x1fffff];
			else if (temp >= 0x200 && temp <= 0x57f)
				val = ABus[addr & 0x3fffff];
			else if (temp >= 0x580 && temp <= 0x58f)
				val = (uint8)(SCD_READ(addr) >> 8);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				val = Sound[addr & 0x1fffff];
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				val = VDP1[addr & 0x3ffff];
			else if (temp == 0x5d0)
				val = VDP1[(addr & 0x1f)+0x30000];
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				val = VDP2[addr & 0xfffff];
			else if (temp == 0x5f0)
				val = VDP2[(addr & 0x1fff)+0x80000];
			else if (temp == 0x5f8)
				val = VDP2[(addr & 0x1ff)+0x80000+0x1000];
			else if (temp == 0x5fe)
				val = SCU[addr & 0xff];
			else if (temp >= 0x600 && temp <= 0x60f)
				val = RAM[(addr & 0x3fffff)+0x100000];
			else if (temp == 0x7ff)
				val = Special[(addr & 0xff) + 0x1ff];
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			// unknown memory area
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				val = *(uint8*)&casLatency;

			val = Special[addr & 0x1ff];
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem read: 0x%8x", addr);
	}

	return val;
}

uint16 memGetWord(uint32 addr)
{
	uint32 temp;
	uint16 val = 0;

	switch ((addr >> 28) & 0xf)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				val = swapWord(*(uint16*)&IPL[addr & 0x7ffff]);
			else if (temp == 0x10)
				val = *(uint16*)&SMPC[addr & 0xff];
			else if (temp == 0x18 || temp == 0x19)
				val = swapWord(*(uint16*)&Backup[addr & 0x1ffff]);
			else if (temp >= 0x20 && temp <= 0x2f)
				val = swapWord(*(uint16*)&RAM[addr & 0x1fffff]);
			else if (temp >= 0x200 && temp <= 0x57f)
				val = *(uint16*)&ABus[addr & 0x3fffff];
			else if (temp >= 0x580 && temp <= 0x58f)
				val = (uint16)SCD_READ(addr);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				val = swapWord(*(uint16*)&Sound[addr & 0x1fffff]);
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				val = *(uint16*)&VDP1[addr & 0x3ffff];
			else if (temp == 0x5d0)
				val = *(uint16*)&VDP1[(addr & 0x1f)+0x30000];
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				val = *(uint16*)&VDP2[addr & 0xfffff];
			else if (temp == 0x5f0)
				val = *(uint16*)&VDP2[(addr & 0x1fff)+0x80000];
			else if (temp == 0x5f8)
				val = *(uint16*)&VDP2[(addr & 0x1ff)+0x80000+0x1000];
			else if (temp == 0x5fe)
				val = *(uint16*)&SCU[addr & 0xff];
			else if (temp >= 0x600 && temp <= 0x60f)
				val = swapWord(*(uint16*)&RAM[(addr & 0x3fffff)+0x100000]);
			else if (temp == 0x7ff)
				val = *(uint16*)&Special[(addr & 0xff) + 0x1ff];
			else
				return 0;
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			// unknown memory area
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				val = *(uint16*)&casLatency;

			val = *(uint16*)&Special[addr & 0x1ff];
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem read: 0x%8x", addr);
	}

	return val;
}

uint32 memGetLong(uint32 addr)
{
	uint32 temp, val = 0;

	switch ((addr & 0xf0000000)>>28)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp<=0xf)
				val = swapLong(*(uint32*)&IPL[addr & 0x7ffff]);
			else if (temp == 0x10)
				val = *(uint32*)&SMPC[addr & 0xff];
			else if (temp == 0x18 || temp == 0x19)
				val = swapLong(*(uint32*)&Backup[addr & 0x1ffff]);
			else if (temp >= 0x20 && temp <= 0x2f)
				val = swapLong(*(uint32*)&RAM[addr & 0x1fffff]);
			else if (temp >= 0x200 && temp <= 0x57f)
				val = *(uint32*)&ABus[addr & 0x3fffff];
			else if (temp >= 0x580 && temp <= 0x58f)
				val = SCD_READ(addr);
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				val = swapLong(*(uint32*)&Sound[addr & 0x1fffff]);
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				val = *(uint32*)&VDP1[addr & 0x3ffff];
			else if (temp == 0x5d0)
				val = *(uint32*)&VDP1[(addr & 0x1f)+0x30000];
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				val = *(uint32*)&VDP2[addr & 0xfffff];
			else if (temp == 0x5f0)
				val = *(uint32*)&VDP2[(addr & 0x1fff)+0x80000];
			else if (temp == 0x5f8)
				val = *(uint32*)&VDP2[(addr & 0x1ff)+0x80000+0x1000];
			else if (temp == 0x5fe)
				val = *(uint32*)&SCU[addr & 0xff];
			else if (temp >= 0x600 && temp <= 0x60f)
				val = swapLong(*(uint32*)&RAM[(addr & 0x3fffff)+0x100000]);
			else if (temp == 0x7ff)
				val = *(uint32*)&Special[(addr & 0xff) + 0x1ff];
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			// unknown memory area
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				val = casLatency;

			val = *(uint32*)&Special[addr & 0x1ff];
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem read: 0x%8x", addr);
	}

	return val;
}

#endif

/*void *memAddrConv(uint32 addr)
{
	uint32 temp;
	switch ((addr >> 28) & 0xf)
	{
		case (0x0):
		case (0x1):
		case (0x2):
		case (0x3):
			temp = ((addr & 0x7ffffff)>>16);
			if (temp>=0x0 && temp<=0xf)
				return (void*)&IPL[addr & 0x7ffff];
			else if (temp == 0x10)
				return (void*)&SMPC[addr & 0xff];
			else if (temp == 0x18 || temp == 0x19)
				return (void*)&Backup[addr & 0x1ffff];
			else if (temp >= 0x20 && temp <= 0x2f)
				return (void*)&RAM[addr & 0x1fffff];
			else if (temp >= 0x200 && temp <= 0x58f)
				return (void*)&ABus[addr & 0x3fffff];
			else if (temp >= 0x5a0 && temp <= 0x5b0)
				return (void*)&Sound[addr & 0x1fffff];
			else if (temp >= 0x5c0 && temp <= 0x5cc)
				return (void*)&VDP1[addr & 0x3ffff];
			else if (temp == 0x5d0)
				return (void*)&VDP1[(addr & 0x1f)+0x30000];
			else if (temp >= 0x5e0 && temp <= 0x5e7)
				return (void*)&VDP2[addr & 0xfffff];
			else if (temp == 0x5f0)
				return (void*)&VDP2[(addr & 0x1fff)+0x80000];
			else if (temp == 0x5f8)
				return (void*)&VDP2[(addr & 0x1ff)+0x80000+0x1000];
			else if (temp == 0x5fe)
				return (void*)&SCU[addr & 0xff];
			else if (temp >= 0x600 && temp <= 0x60f)
				return (void*)&RAM[(addr & 0x3fffff)+0x100000];
			else if (temp == 0x7ff)
				return (void*)&Special[(addr & 0xff) + 0x1ff];
			else
				return 0;
			break;
		case (0x4):
		case (0x6):
		case (0xc):
			// unknown memory area
			return 0;
			break;
		case (0xf): //special registers
			if (addr == 0xffff8888)
				return (void*)&casLatency;
			return (void*)&Special[addr & 0x1ff];
			break;
		default:
			ConsoleMsg(MSG_WARN, "MEM: invalid mem read: 0x%8x", addr);
			return 0;
	}
}*/
