// SMPC.C
// System Manager and Peripheral Control

#include <time.h>
#include "global.h"
#include "m68k.h"
#include "smpc.h"
#include "sh2.h"
#include "mem.h"
#include "scu.h"
#include "pad.h"

int SMPC_Flag = SMPC_NOP;
unsigned char smem[4];

void GetStatus(void);
void GetPadStatus(void);
void ResetEnable(int);
int DectoBCD(int);

void updateSMPC(void)
{
	// Check for SH2 Peripheral Access (unhandled)
	if (!IOSEL)
		IOSEL = 0x0;

	// InterruptBack
	if (COMREG == 0x10)
	{
		//Break InterruptBack
		if((IREG[0] & 0x40))
		{
			SMPC_Flag = SMPC_NOP;
			SMPC_SR = 0;
			SMPC_SF = 0;
			COMREG = 0;
			return;
		}

		// Continue InterruptBack
		if((IREG[0] & 0x80))
		{
			InterruptBack();
			IREG[0] = ~0x80;
			//SMPC_SR = 0x40;
			return;
		}
	}

	switch(COMREG)
	{
		case (0x0): //Master SH2 ON
			OREG[31] = 0x0;
		case (0x2): //Slave SH2 ON
            ConsoleMsg(MSG_NORM, "SMPC: Recieved Slave SH2 On");
			Flags.SlaveEnabled = 1;
			OREG[31] = 0x02;
            SMPC_SF = 0x00;
            break;
		case (0x3): //Slave SH2 OFF
            ConsoleMsg(MSG_NORM, "SMPC: Recieved Slave SH2 Off");
            Flags.SlaveEnabled = 0;
			OREG[31] = 0x03;
            SMPC_SF = 0x00;
            break;
		case (0x6): //Sound ON
            ConsoleMsg(MSG_NORM, "SMPC: Recieved Sound On");
			SCU_INT_STATUS |= INT_SOUND;
			Flags.SoundEnabled = 1;
			reset68k();
			OREG[31] = 0x06;
            SMPC_SF = 0x00;
            break;
		case (0x7): //Sound OFF
            ConsoleMsg(MSG_NORM, "SMPC: Recieved Sound Off");
			Flags.SoundEnabled = 0;
            OREG[31] = 0x07;
            SMPC_SF = 0x00;
            break;
        case (0x8): //CDON
            ConsoleMsg(MSG_NORM, "SMPC: Recieved CD On");
            //OREG[31] = 0x08;
            SMPC_SF = 0x00;
            break;
		case (0x9): //CDOFF
            ConsoleMsg(MSG_NORM, "SMPC: Recieved CD Off");
			//OREG[31] = 0x09;
            SMPC_SF = 0x00;
            break;
		case (0xd): //SYSRES
            ConsoleMsg(MSG_NORM, "SMPC: Recieved System Reset");
            SMPC_SF = 0x00;
            break;
		case (0xe): //CKCHG352
            ConsoleMsg(MSG_NORM, "SMPC: Recieved Mode change to 352");

            Flags.SlaveEnabled = 0;
			sh2_ReturnFromFunction(&master);
			OREG[31] = 0x0e;

            SMPC_SF = 0x00;
            break;
		case (0xf): //CKCHG320
            ConsoleMsg(MSG_NORM, "SMPC: Recieved Mode change to 320");

            Flags.SlaveEnabled = 0;
			sh2_ReturnFromFunction(&master);
			OREG[31] = 0x0f;

            SMPC_SF = 0x00;
            break;
		case (0x10): //INTBACK:
            if (SMPC_SF == 0)
				return;
            InterruptBack();
            break;
		case (0x16): //SETTIME
			// will ignore this command
			OREG[31] = 0x16;
			SMPC_SF = 0x00;
			break;
		case (0x17): //SETSMEM:
			smem[0] = (uint8)IREG[0];
			smem[1] = (uint8)IREG[1];
			smem[2] = (uint8)IREG[2];
			smem[3] = (uint8)IREG[3];
			OREG[31] = 0x17;
            SMPC_SF = 0x00;
            break;
		case (0x18): //NMIREQ
            ConsoleMsg(MSG_NORM, "SMPC: Recieved NMIREQ");
            SMPC_SF = 0x00;
            break;
		case (0x19): //RESENAB
			Flags.ResetEnabled = TRUE;
			OREG[31] = 0x19;
            SMPC_SF = 0x00;
            break;
		case (0x1a): //RESDISA
            Flags.ResetEnabled = FALSE;
			OREG[31] = 0x1a;
            SMPC_SF = 0x00;
            break;
		default:
			ConsoleMsg(MSG_NORM, "SMPC: Unhandled Message: 0x%2x", SMPC[0x1f]);
			SMPC_SF = 0;
			break;
	}
}

void InterruptBack(void)
{
    switch(SMPC_Flag)
    {
	    case SMPC_NOP:
			SMPC_SR = 0;

			if(IREG[0] & 0x1) {
				GetStatus();
				SMPC_Flag = SMPC_NOP;

				if(IREG[1] & 0x8) {
					SMPC_Flag = SMPC_INPUT;
					SMPC_SR |= 0x20;
				}
			}
			else {
				SMPC_Flag = SMPC_NOP;
				SMPC_SR = 0;
				SMPC_SF = 0;
				COMREG = 0;
			}
			break;

		case SMPC_INPUT:
	        GetPadStatus();
			SMPC_Flag = SMPC_NOP;
			break;

		case SMPC_STATUS:
	        GetStatus();
			if(IREG[1] & 0x8) {
				SMPC_Flag = SMPC_INPUT;
				SMPC_SR |= 0x20;
		    }
	        else {
				SMPC_Flag = SMPC_NOP;
				SMPC_SF = 0;
				COMREG = 0;
			}
			break;
	}
    SMPC_SF = 0;
    COMREG  = 0;    
	return;
}

void GetStatus()
{
	time_t current;
	struct tm *tmp;

	if (Flags.ResetEnabled)
		OREG[0] &= ~0x40;
	else
		OREG[0] |= 0x40;

    OREG[0] = 0x80;

	time(&current);
	tmp = localtime(&current);

    OREG[1] = DectoBCD((tmp->tm_year + 1900)/100);
    OREG[2] = DectoBCD((tmp->tm_year + 1900)%100);
    OREG[3] = (tmp->tm_wday << 8) | DectoBCD(tmp->tm_mon);
    OREG[4] = DectoBCD(tmp->tm_mday);
    OREG[5] = DectoBCD(tmp->tm_hour);
    OREG[6] = DectoBCD(tmp->tm_min);
    OREG[7] = DectoBCD(tmp->tm_sec);
    OREG[8] = 0x00; //CTG1=0 CTG0=0 (correct??)
    OREG[9] = Settings.AreaCode;

    OREG[10] = 0;
    OREG[11] = 0;

    OREG[12] = smem[0];
    OREG[13] = smem[1];
    OREG[14] = smem[2];
    OREG[15] = smem[3];

    OREG[31] = 0x10;
    SMPC_SR = 0x40;
}

void GetPadStatus()
{
	unsigned int tmp = 0, size;
	void *pad;
	uint8 *ptr = &SMPC[0x21]; //OREG[0]

	// retrieve peripheral data in a platform independant 
	// manner (now I can worry about getPeriData() later :)
	while ( (pad = getPeriData((uint8)(++tmp), &size)) != NULL) {
		memcpy((void*)ptr, (void*)pad, size);
		ptr += size;
	}

    /*OREG[0] = 0xF1;
    OREG[1] = 0x02;
    OREG[2] = 0xEF;
    OREG[3] = 0xFF;*

    OREG[4] = 0x02;
    OREG[5] = 0xFF;
    OREG[6] = 0xFF;*/

    SMPC_SR = 0xC0;
	SMPC_SF = 0;
}

int DectoBCD(int num)
{
	int i, cnt = 0, tmp, res = 0;
	
	while (num > 0) {
		tmp = num;
		while (tmp >= 10) tmp %= 10;
		for (i=0; i<cnt; i++)
			tmp *= 16;
		res += tmp;
		cnt++;
		num /= 10;
	}

	return res;
}