#include "gpulocal.h"
#include "keyboard.h"
#include <malloc.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

static int cur_width,cur_height;
static int dsp_mw,dsp_mh;
static UINT16 dsp_keymap[km_s];

Display *Dsp;
XEvent	E;
Screen	*Scr;
Window	Wnd;
GC	DefaultGC;
UINT8	*XBuf;
UINT16	*XBuf16;
UINT32	*XBuf32;
XImage	*ximage;
int colordepth=0;

static int dsp_init();
static void dsp_exit();
static void dsp_open_window(unsigned int w,unsigned int h);
static void dsp_resize(unsigned int w,unsigned int h);
static void dsp_change_title(char *title);
static void dsp_render_frame(UINT32 *image, int start, int endw);
static int dsp_key_pressed();
static UINT16 dsp_get_key();

static void dsp_fillkeymap();

DisplayStruct Dsp_X11 = {
	dsp_init,
	dsp_exit,
	dsp_open_window,
	dsp_resize,
	dsp_change_title,
	dsp_render_frame,
	dsp_key_pressed,
	dsp_get_key,
};


static int dsp_init() {
	Dsp = XOpenDisplay(NULL);
	if (Dsp==NULL) return -1;
	XCloseDisplay(Dsp);
	dsp_fillkeymap();
	return 0;
}

static void dsp_open_window(unsigned int w,unsigned int h) {
	XSizeHints Hints;
	XWMHints WMHints;
	long unsigned White,Black;
	int i,attr_mask;
	XSetWindowAttributes attr;
	Colormap cmap = 0;

	cur_width=w;
	cur_height=h;
	Dsp=XOpenDisplay(NULL);
	Scr=DefaultScreenOfDisplay(Dsp);
	White=WhitePixelOfScreen(Scr);
	colordepth=DefaultDepthOfScreen(Scr);
	Black=BlackPixelOfScreen(Scr);
	DefaultGC=DefaultGCOfScreen(Scr);
	attr.event_mask = KeyPressMask | KeyReleaseMask;
	attr.background_pixel = Black;
	attr.border_pixel = White;
	attr.backing_store = WhenMapped;
	attr_mask = CWBackPixel | CWEventMask | CWBorderPixel | CWBackingStore;

	Wnd=XCreateWindow(Dsp,RootWindowOfScreen(Scr),0,0,w,h,0,colordepth,
			  InputOutput,DefaultVisualOfScreen(Scr),attr_mask,&attr);

	Hints.flags=PSize|PMinSize|PMaxSize;
	Hints.min_width=Hints.max_width=Hints.base_width=w;
	Hints.min_height=Hints.max_height=Hints.base_height=h;
	WMHints.input=True;WMHints.flags=InputHint;
	XSetWMHints(Dsp,Wnd,&WMHints);
	XSetWMNormalHints(Dsp,Wnd,&Hints);
	XSelectInput(Dsp,Wnd,FocusChangeMask|ExposureMask|KeyPressMask|KeyReleaseMask);
	XMapRaised(Dsp,Wnd);
	XClearWindow(Dsp,Wnd);
	XWindowEvent(Dsp,Wnd,ExposureMask,&E);

	XBuf=(UINT8 *)malloc((colordepth/8)*w*h);
	XBuf16=(UINT16 *)XBuf;
	XBuf32=(UINT32 *)XBuf;
	if (colordepth==8) {
		XColor color;
		cmap = XCreateColormap (Dsp,Wnd,DefaultVisualOfScreen(Scr),AllocNone);
		XSetWindowColormap (Dsp,Wnd, cmap);

		for (i=0;i<64;i++) {
		    color.blue  = (i&0x03) << 14;
		    color.green = (i&0x0c) << 12;
		    color.red   = (i&0x30) << 10;
		    XAllocColor(Dsp,cmap,&color);
		}
	}

	if (colordepth==32)
		ximage=XCreateImage(Dsp,DefaultVisualOfScreen(Scr),24,ZPixmap,0,XBuf,w,h,32,0);
	else
		ximage=XCreateImage(Dsp,DefaultVisualOfScreen(Scr),colordepth,ZPixmap,0,XBuf,w,h,32,0);
	dsp_mw = WidthOfScreen(Scr);
	dsp_mh = HeightOfScreen(Scr);
}

static void dsp_resize(unsigned int w,unsigned int h) {
	XSizeHints Hints;

	Hints.flags = PSize|PMinSize|PMaxSize;
	Hints.min_width = Hints.max_width = Hints.base_width = w;
	Hints.min_height = Hints.max_height = Hints.base_height = h;
	XSetWMNormalHints(Dsp,Wnd,&Hints);

	cur_width = w;
	cur_height = h;
	XResizeWindow(Dsp,Wnd,w,h);
	XMoveWindow(Dsp,Wnd,(dsp_mw-w)/2,(dsp_mh-h)/2);
	ximage->width = w;
	ximage->height = h;
	ximage->bytes_per_line = ((colordepth*w+31)/32)*(32>>3);
}

static void dsp_exit() {
	XDestroyWindow(Dsp,Wnd);
	XCloseDisplay(Dsp);
	free(XBuf);
}
static void dsp_change_title(char *title) {
	XStoreName(Dsp,Wnd,title);
}

static void dsp_render_frame(UINT32 *image, int start, int endw) {
	int unsigned i,j,k,t;
	UINT16 *ll;

	ll=(UINT16*)(image+start);
	k=0;
	switch (colordepth) {
	case 32:
		for (j=0;j<cur_height;j++) {
		    for (i=0;i<cur_width;i++) {
			t=*(ll++);
			XBuf32[k++] = ((t&BMASK)<<9) | ((t&GMASK)<<6) | ((t&BMASK)<<3);
		    }
		    ll+=(endw-cur_width);
		}
		break;
	case 24:
		for (j=0;j<cur_height;j++) {
		    for (i=0;i<cur_width;i++) {
			t=*(ll++);
			XBuf[k++]=(t&RMASK)<<3;
			XBuf[k++]=(t&GMASK)>>2;
			XBuf[k++]=(t&BMASK)>>7;
		    }
		    ll+=(endw-cur_width);
		}
		break;
	case 16:
		for (j=0;j<cur_height;j++) {
		    for (i=0;i<cur_width;i++) {
			t=*(ll++);
			t = (t&RMASK)|((t&GMASK)<<1)|((t&BMASK)<<1);
			XBuf16[k++]=t;
			}
		    ll+=(endw-cur_width);
		}
		break;
	case 8:
		for (j=0;j<cur_height;j++) {
		    for (i=0;i<cur_width;i++) {
			t=*(ll++);
			t = ((t&0x6000)>>9)|((t&0x300)>>6)|((t&0x18)>>3);
			XBuf[k++]=t;
		    }
		    ll+=(endw-cur_width);
		}
		break;
	}
	XPutImage(Dsp,Wnd,DefaultGC,ximage,0,0,0,0,cur_width,cur_height);
	XFlush(Dsp);
}

static int dsp_key_pressed(){
	if (XCheckWindowEvent(Dsp,Wnd,KeyPressMask|KeyReleaseMask,&E)) {
	    if (E.type==KeyPress) return PRESSED_KEY;
	    if (E.type==KeyRelease) return RELEASED_KEY;
	}
	return NO_KEY;
}

static UINT16 dsp_get_key(){
	KeySym key;

	key=XLookupKeysym((XKeyEvent *)&E,0);
	switch(key){
		case XK_Return: return KEY_RETURN;
		case XK_Escape: return KEY_ESCAPE;
		case XK_Up:	return KEY_UP;
		case XK_Down:	return KEY_DOWN;
		case XK_Left:	return KEY_LEFT;
		case XK_Right:	return KEY_RIGHT;
		case XK_F1:	return KEY_F1;
		case XK_F2:	return KEY_F2;
		case XK_F3:	return KEY_F3;
		case XK_F4:	return KEY_F4;
		case XK_F5:	return KEY_F5;
		case XK_F6:	return KEY_F6;
		case XK_F7:	return KEY_F7;
		case XK_F8:	return KEY_F8;
		case XK_F9:	return KEY_F9;
		case XK_F10:	return KEY_F10;
		case XK_F11:	return KEY_F11;
		case XK_F12:	return KEY_F12;
		default:	return key;
	}
}


static void dsp_fillkeymap() {
	int i;

	for (i=0; i<km_s; i++) {
	    dsp_keymap[i] = i;
	}
}