
#include <cstring>
#include <cassert>
#include "t6a04.h"



const int t6a04::SCREEN_WIDTH = 96;
const int t6a04::SCREEN_HEIGHT = 64;



void t6a04::reset() {
  _cursor_mode_changed = false;
  _cursor = CURSOR_INC_LEFT; 
  _x = _y = 0;
  _column_width8 = 1;
  _display_on = 1;
  memset(_screen, 0, sizeof(_screen));
}



unsigned char t6a04::read_control() {
  /*
   bit 7 - controller busy (0=false, 1=true)
   bit 6 - column width (0=6, 1=8)
   bit 5 - display status (0=off, 1=on)
   bit 4 - display controller in the reset state (0=false, 1=true)
   bits 3-2 - always 0
   bits 1-0 - direction of cursor increment (0=left, 1=right, 2=up, 3=down)
  */
  unsigned char control = 0;
  
  if(_column_width8) control |= 0x20;
  if(_display_on)    control |= 0x10;
  
  switch(_cursor) {
  case CURSOR_INC_UP:    control |= 0x2; break;
  case CURSOR_INC_DOWN:  control |= 0x3; break;
  case CURSOR_INC_LEFT:  control |= 0x0; break;
  case CURSOR_INC_RIGHT: control |= 0x1; break;
  }

  return control;
}



void t6a04::write_control(unsigned char cmd) {
  /*
   0x00 - sets word length (ie column width) to 6 (16 columns * 6 pixels)
   0x01 - sets word length (ie column width) to 8 (12 columns * 8 pixels)
   0x02 - turn display off
   0x03 - turn display on
   0x04 - set increment to up
   0x05 - set increment to down 
   0x06 - set increment to left
   0x07 - set increment to right
   0x08-0x0F - op-amp power control 1
   0x10-0x17 - op-amp power control 2
   0x18-0x1F - test mode select
   0x20-0x3F - set X coordinate
   0x40-0x7F - scroll lines
   0x80-0xBF - set Y coordinate
   0xC0-0xFF - set contrast (0x3F=darkest, 0x00=lightest)
 */
  if(cmd == 0x00)
    _column_width8 = 0;
  else if(cmd == 0x01)
    _column_width8 = 1;
  else if(cmd == 0x02)
    _display_on = 0;
  else if(cmd == 0x03)
    _display_on = 1;
  else if(cmd == 0x04) {
    _cursor = CURSOR_INC_UP;
    _cursor_mode_changed = true;
  }
  else if(cmd == 0x05) {
    _cursor = CURSOR_INC_DOWN;
    _cursor_mode_changed = true;
  }
  else if(cmd == 0x06) {
    _cursor = CURSOR_INC_LEFT;
    _cursor_mode_changed = true;
  }
  else if(cmd == 0x07) {
    _cursor = CURSOR_INC_RIGHT;
    _cursor_mode_changed = true;
  }
  else if(cmd >= 0x08 && cmd <= 0x0F) {
    // TODO
  }
  else if(cmd >= 0x10 && cmd <= 0x17) {
    // TODO
  }
  else if(cmd >= 0x18 && cmd <= 0x1F) {
    // TODO
  }
  else if(cmd >= 0x20 && cmd <= 0x3F)
    _x = cmd & 0x1F;
  else if(cmd >= 0x40 && cmd <= 0x7F) {
    // TODO
  }
  else if(cmd >= 0x80 && cmd <= 0xBF)
    _y = cmd & 0x3F;
  else if(cmd >= 0xC0 && cmd <= 0xFF) {
    // TODO
  }
}



void t6a04::update_coordinates() {
  switch(_cursor) {
  case CURSOR_INC_UP:
   --_y; _y &= 0x3F;
   break;
  case CURSOR_INC_DOWN:
   ++_y; _y &= 0x3F;
   break;
  case CURSOR_INC_LEFT:
   --_x; _x &= 0xF;
   break;
  case CURSOR_INC_RIGHT:
   ++_x; _x &= 0xF;
   break;
  }
//  if(_column_width8 && _x > 0xB) _x -= 0xB;
}


  
unsigned char t6a04::read_display() {
  unsigned char data = 0xFF;

  if(_cursor_mode_changed)
    _cursor_mode_changed = false;
  else {
    if(_column_width8) {
      unsigned int offset = _y * 12 + _x;
      assert(offset < 768);
      data = _screen[offset];
    }
    else {
      unsigned int column = _x * 6;
      unsigned int offset = _y * 12 + column / 8;
      assert(offset < 768);
      unsigned char shift = 10 - (column & 7);
      data = ((_screen[offset] << 8) | _screen[offset + 1]) >> shift;
      data &= 0x3F;
    }
    update_coordinates();
  }
  return data;
}



void t6a04::write_display(unsigned char data) {
  if(_column_width8) {
    unsigned int offset = _y * 12 + _x;
    assert(offset < 768);
    _screen[offset] = data;
  } 
  else {
    unsigned int column = _x * 6;
    unsigned int offset = _y * 12 + column / 8;
    assert(offset < 768);
    switch(column & 7) {
    case 0:
      _screen[offset] = (data << 2) | (_screen[offset] & 0x3);
      break;
    case 2: 
      _screen[offset] = (data & 0x3F) | (_screen[offset] & 0xC0);
      break;
    case 4:
      _screen[offset] = ((data >> 2) & 0xF) | (_screen[offset] & 0xF0);
      _screen[offset + 1] = ((data & 0x3) << 6) | (_screen[offset + 1] & 0x3F);
      break;
    case 6:
      _screen[offset] = ((data >> 4) & 0x3) | (_screen[offset] & 0xFC);
      _screen[offset + 1] = ((data & 0xF) << 4) | (_screen[offset + 1] & 0xF);
      break;
    }
  }  	
  update_coordinates(); 	
}
