
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/


_inline void arm7tdmi_set_cpu_mode(u32 mode)
{
	ARM_CPSR |= mode;
	ARM_CPSR &= 0xFFFFFF10 | (~mode);
}

#ifdef	MACRO_EXPAND

#define arm7tdmi_set_sub_flag(a, b, c)	\
{	/*sub,rsb,rsc,cmp*/\
	if(!c)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);\
	if(c & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);\
	if(((a & ~b) | (a & ~c) | (~b & ~c)) & BIT31)ARM_SF(ARM_C); else ARM_RF(ARM_C);\
	if(((a & ~(b | c)) | ((b & c) & ~a)) & BIT31)ARM_SF(ARM_V); else ARM_RF(ARM_V);\
}

#define arm7tdmi_set_add_flag(a, b, c)	\
{	/*add,adc,cmn*/\
	if(!c)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);\
	if(c & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);\
	if(((a & b) | (a & ~c) | (b & ~c)) & BIT31)ARM_SF(ARM_C); else ARM_RF(ARM_C);\
	if(((a & b & ~c) | (~a & ~b & c)) & BIT31)ARM_SF(ARM_V); else ARM_RF(ARM_V);\
}

#define arm7tdmi_set_dp_flags(a)	\
{\
	if(!a)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);\
	if(a & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);\
}

#else	/*MACRO_EXPAND*/

_inline void arm7tdmi_set_sub_flag(u32 a, u32 b, u32 c)
{	/*sub,rsb,rsc,cmp*/
	if(!c)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(c & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
	if(((a & ~b) | (a & ~c) | (~b & ~c)) & BIT31)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(((a & ~(b | c)) | ((b & c) & ~a)) & BIT31)ARM_SF(ARM_V); else ARM_RF(ARM_V);
}

_inline void arm7tdmi_set_add_flag(u32 a, u32 b, u32 c)
{	/*add,adc,cmn*/
	if(!c)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(c & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
	if(((a & b) | (a & ~c) | (b & ~c)) & BIT31)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(((a & b & ~c) | (~a & ~b & c)) & BIT31)ARM_SF(ARM_V); else ARM_RF(ARM_V);
}

_inline void arm7tdmi_set_dp_flags(u32 a)	
{	/*muls,mlas, and,eor,tst,teq,orr,mov,bic,mvn*/
	if(!a)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(a & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
}

#endif	/*MACRO_EXPAND*/


u32 arm7tdmi_imm_shift(u32 opcode)
{	/*Vtgl*/
	u32	bit, temp32;
	u32	shift, rm;
	u8	shift_type;

	rm	= ARM_SOREG;
	shift_type = (u8)((opcode >> 5) & 3);

	if(opcode & 0x08){	/*Rs(8-11)*/
		shift = (opcode >> 8) & 0xF;
	}else{				/*Rs(7-11)*/
		shift = (opcode >> 7) & 0x1F;
	}

	switch(shift_type){
	case 0:	/*00 = logical left(lsl) _IVtg 0*/
		return (arm.reg[rm] << shift);
	case 1:	/*01 = logical right(lsr) _IEVtg 0*/
		return (arm.reg[rm] >> shift);
	case 2:	/*10 = arithmetic right(asr) ZpIEVtg 31rbg̒l*/
		temp32 = (arm.reg[rm] >> shift);
		if(arm.reg[rm] & BIT31){
			for(bit = BIT31; shift; bit >>= 1){
				temp32 |= bit;
				shift--;
			}
		}
		return temp32;
	case 3:	/*11 = rotate right(ror) lւ*/
		return (arm.reg[rm] >> shift) | (arm.reg[rm] << (32 - shift));
	}

	return 0;
}

u32 arm7tdmi_imm_rotate(u32 opcode)
{	/*E[e[gl*/
	u32	shift, imm_val;

	shift	= ((opcode >> 8) & 0xF) << 1;
	imm_val	= opcode & 0xFF;

	return (imm_val >> shift) | (imm_val << (32 - shift));
}

_inline void arm7tdmi_arm_adc()
{	/*Rd := Rn + Op2 + Carry <Data Processing> (L[tZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	operand += ARM_LSB_FC;
	/*if((arm.reg[rn] + operand) > 0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/

	arm.reg[rd] = arm.reg[rn] + operand;

	if(ARM_FLAG_COND)arm7tdmi_set_add_flag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_add()
{	/*Rd := Rn + Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] + operand;

	if(ARM_FLAG_COND)arm7tdmi_set_add_flag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_and()
{	/*Rd := Rn AND Op2 <Data Processing> (_)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;
	
	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] & operand;

	if(ARM_FLAG_COND)arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_b()
{	/*R15 := address ()*/
	u32	opcode,	offset, address;

	opcode	= arm.opcode;
	/*l͂S{*/
	offset	= ((opcode & 0x00FFFFFF) << 2);

	address = ARM_PC + offset + 4;

	/*ItZbg͈̔͂MSB^Ȃ}CiX Zl{̒l*/
	if(offset & BIT25)address -= BIT26;	/*tItZbg*/

	ARM_PC = address;

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_bic()
{	/*Rd := Rn AND NOT Op2 <Data Processing> (rbgNA)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] & ~operand;

	if(ARM_FLAG_COND)arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_bl()
{	/*R14 := R15, R15 := address (ƃN)*/
	u32	opcode,	offset, address;

	opcode	= arm.opcode;
	/*l͂S{*/
	offset	= (opcode & 0x00FFFFFF) << 2;

	ARM_R14 = ARM_PC;// + 4;
	address = ARM_PC + offset + 4;

	/*ItZbg͈̔͂MSB^Ȃ}CiX Zl{̒l*/
	if(offset & BIT25)address -= BIT26;	/*tItZbg*/

	ARM_PC = address;

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_bx()
{	/*R15 := Rn (Ɩ߃Xe[g̕ύX)*/
	u32	opcode;
	u32 rn;
	
	opcode		= arm.opcode;
	rn			= REG_0_4;
	ARM_R15		= arm.reg[rn];

	if(ARM_R15 & BIT1){
		ARM_CPSR |= ARM_T;		/*ŉʃrbgPȂTHUMBXe[gɕύX*/
	}else{
		ARM_CPSR &= ~ARM_T;	/*ŉʃrbgOȂARMXe[gɕύX*/
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_cdp()
{	/*(Coprocessor-specific)*/

	arm.cycle = 2;
}

_inline void arm7tdmi_arm_cmn()
{	/*CPSR flags := Rn + Op2 <Data Processing> (RvZbTf[^)*/
	u32	opcode, operand;
	u32	temp;
	u32	rn;

	opcode	= arm.opcode;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	temp = arm.reg[rn] + operand;

#ifdef	INLINE_FLAG
	if(!temp)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if((arm.reg[rn] - operand) > 0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(temp & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
#else
	arm7tdmi_set_add_flag(arm.reg[rn], operand, temp);
#endif	/*INLINE_FLAG*/

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_cmp()
{	/*CPSR flags := Rn - Op2 <Data Processing> (r)*/
	u32	opcode, operand;
	u32	temp;
	u32	rn;

	opcode	= arm.opcode;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	temp = arm.reg[rn] - operand;

#ifdef	INLINE_FLAG
	if((arm.reg[rn] - operand) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(!temp)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(temp & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
#else
	arm7tdmi_set_sub_flag(arm.reg[rn], operand, temp);
#endif	/*INLINE_FLAG*/

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_eor()
{	/*Rd := (Rn AND NOT Op2) OR (op2 AND NOT Rn) <Data Processing> (rI_a)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;
	
	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	} 

	arm.reg[rd] = arm.reg[rn] ^ operand;

	if(ARM_FLAG_COND)arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_ldc()
{	/*Coprocessor load (RvZbTւ̓])*/
	u32	opcode;
	u32	rd, rn, cp;
	u32	offset;

	opcode	= arm.opcode;
	rd		= (opcode >> 12) & 0x0F;
	rn		= (opcode >> 16) & 0x0F;
	cp		= (opcode) & 0x0F;
	offset	= opcode & 0xF;

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_ldm()
{	/*Stack manipulation (̃WX^̃[h,POP)*/
	u32	opcode,	address, bit;
	u32	rn, lst;
	u32 fprepost;

	arm.cycle = 3;

	opcode	= arm.opcode;
	rn		= (opcode >> 16) & 0x0F;	/*16-19 4bit*/

	address = arm.reg[rn];

	fprepost = (opcode & BIT24)?TRUE:FALSE;

	for(bit = BIT15, lst = 15; bit; bit >>= 1,lst--){
		if(opcode & bit){//TRACE("%2d,%08X\n",lst, bit);
			arm.cycle++;
			if(fprepost){	/*Pre: ]OɃItZbgǉ*/
				if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
			arm.reg[lst] = agb_read_mem32(address);
			if(!fprepost){	/*Post: ]ɃItZbgǉ*/
				if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
		}
	}

	if((opcode & BIT21) || !fprepost){	/*Write-back: SP(rn)̓eXV*/
		arm.reg[rn] = address;
	}
}

_inline void arm7tdmi_arm_ldr()
{	/*Rd := (address) (烌WX^ւ̓ǂݍ)*/
	u32	opcode;
	u32	offset, address;
	u32	rd, rn;
	u32 fpreindex;

	opcode = arm.opcode;
	rd     = ARM_DEREG;
	rn     = (opcode >> 16) & 0xF;	/*x[XWX^*/

	address = arm.reg[rn];

	if(rn == 15)address += 8;	/*ΏۂPC̏ꍇ*/

	fpreindex = opcode & BIT24;

	if(opcode & BIT25){	/*1 = ItZbg̓WX^*/
		offset = arm7tdmi_imm_shift(opcode);
	}else{					/*0 = ItZbg𑦒l*/
		offset = opcode & 0xFFF;
	}

	if(fpreindex){	/*1 = Pre: ItZbgǉOɓ]*/
		if(opcode & BIT23){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	arm.reg[rd] = (opcode & BIT22)?agb_read_mem8(address):agb_read_mem32(address);

	if(!fpreindex){	/*0 = Post: ItZbgǉ]*/
		if(opcode & BIT23){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT21) || !fpreindex){	/*Write-back: ݐ̃AhXcĂ*/
		//arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_ldrs()
{	/*Rd := (address) (WX^ւ̃[h) Register offset*/
	u32	opcode, address;
	u32	rd, rn, rm;
	u32	fprepost;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/
	rm		= opcode & 0xF;

	address = arm.reg[rn];

	fprepost = (opcode & BIT24)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT5){	/*Halfwords*/
		if(opcode & BIT6){	/*Signed*/
			/*11: tn[t[h*/
			arm.reg[rd] = agb_read_mem16(address);
		}else{
			/*01: n[t[h*/
			arm.reg[rd] = agb_read_mem16(address);
		}
	}else{
		if(opcode & BIT6){	/*Signed*/
			/*10: toCg*/
			arm.reg[rd] = agb_read_mem8(address);
		}else{
			/*00: SWP߂Œ`ς*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if((opcode & BIT21) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_ldrs_imm()
{	/*Rd := (address) (WX^ւ̃[h) Immidiate offset*/
	u32	opcode,	offset, address;
	u32	rd, rn;
	u32	fprepost;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/
	offset	= (opcode & 0xF) | ((opcode >> 4) & 0xF0);

	address = arm.reg[rn];

	fprepost = (opcode & BIT24)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT5){	/*Halfwords*/
		if(opcode & BIT6){	/*Signed*/
			/*Signed Halfwords*/
			arm.reg[rd] = agb_read_mem16(address);
		}else{
			/*Unsigned Halfwords*/
			arm.reg[rd] = agb_read_mem16(address);
		}
	}else{					/*Bytes*/
		if(opcode & BIT6){	/*Signed*/
			/*Signed byte*/
			arm.reg[rd] = agb_read_mem8(address);
		}else{
			/*SWP instruction*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT21) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_mcr()
{	/*cRn := rRn {<op>cRm} (CPUWX^RvZbTWX^ւ̓])*/
	u32	opcode;
	u32	rn, rd, rm;

	opcode	= arm.opcode;
	rn		= (opcode >> 16) & 0x0F;
	rd		= (opcode >> 12) & 0x0F;
	rm		= (opcode) & 0x0F;
	
	arm.cycle = 3;
}

_inline void arm7tdmi_arm_mla()
{	/*Rd := (Rm * Rs) + Rn (ϘaZ)*/
	u32	opcode;
	u32	rd, rm, rs, rn;

	opcode	= arm.opcode;
	
	rd		= (opcode >> 16) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;
	rn		= (opcode >> 12) & 0x0F;

	/*if(((arm.reg[rm] * arm.reg[rs]) + arm.reg[rn]) > 0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/

	arm.reg[rd] = (arm.reg[rm] * arm.reg[rs]) + arm.reg[rn];

	arm7tdmi_set_dp_flags(arm.reg[rd]);//

	arm.cycle = 2;
}

_inline void arm7tdmi_arm_mlal()
{	/*RdHigh, RdLow := (Rm * Rs) + Rn (64bitϘaZ)*/
	u32	opcode;
	u32	rdh, rdl, rm, rs;

	opcode	= arm.opcode;
	rdh		= (opcode >> 16) & 0x0F;
	rdl		= (opcode >> 12) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.reg[rdl] = arm.reg[rm] * arm.reg[rs] + arm.reg[rdl];
	arm.reg[rdh] = (u32)((unsigned __int64)arm.reg[rm] * arm.reg[rs] + arm.reg[rdl]) >> 32;

	if(!arm.reg[rdl] && !arm.reg[rdh])ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(arm.reg[rdh] & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
/*
MULL	RL,RH,Rm,Rs

UMULL	R1,R4,R2,R3	; R4,R1:=R2*R3
UMLALS	R1,R5,R2,R3	; R5,R1:=R2*R3+R5,R1 also setting
*/
	arm.cycle = 3;
}

_inline void arm7tdmi_arm_mov()
{	/*Rd : = Op2 <Data Processing> (WX^͒萔̑)*/
	u32	opcode, operand;
	u32	rd;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = operand;

	if(ARM_FLAG_COND)arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_mrc()
{	/*Rn := cRn {<op>cRm} (RvZbTWX^CPUWX^ւ̓])*/
	u32	opcode;
	u32	rn, rd, rm;

	opcode	= arm.opcode;
	rn		= (opcode >> 16) & 0x0F;
	rd		= (opcode >> 12) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_mrs()
{	/*Rn := PSR (Xe[^XWX^烌WX^ւ̓])*/
	u32	opcode;
	u32	rd;

	opcode	= arm.opcode;
	rd		= (opcode >> 12) & 0x0F;

	if(opcode & BIT22){
		switch((u8)(arm.reg[16] & 0xF)){
		case 0x0:	/*USR*/
		case 0xF:	/*SYS*/
			arm.reg[rd] = arm.reg[16];
			break;
		case 0x1:	/*FIQ*/
			arm.reg[rd] = ARM_SPSR_FIQ;
			break;
		case 0x2:	/*IRQ*/
			arm.reg[rd] = ARM_SPSR_IRQ;
			break;
		case 0x3:	/*SVC*/
			arm.reg[rd] = ARM_SPSR_SVC;
			break;
		case 0x7:	/*ABT*/
			arm.reg[rd] = ARM_SPSR_ABT;
			break;
		case 0xB:	/*UND*/
			arm.reg[rd] = ARM_SPSR_UND;
			break;
		}
	}else{
		arm.reg[rd] = arm.reg[16];
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_msr()
{	/*PSR := Rm (WX^Xe[^XWX^ւ̓])*/
	u32	opcode;
	u32	rm;

	opcode	= arm.opcode;
	rm		= opcode & 0x0F;

	if(opcode & BIT22){
		switch((u8)(arm.reg[16] & 0xF)){
		case 0x0:	/*USR*/
		case 0xF:	/*SYS*/
			arm.reg[16] = arm.reg[rm];
			break;
		case 0x1:	/*FIQ*/
			ARM_SPSR_FIQ = arm.reg[rm];
			break;
		case 0x2:	/*IRQ*/
			ARM_SPSR_IRQ = arm.reg[rm];
			break;
		case 0x3:	/*SVC*/
			ARM_SPSR_SVC = arm.reg[rm];
			break;
		case 0x7:	/*ABT*/
			ARM_SPSR_ABT = arm.reg[rm];
			break;
		case 0xB:	/*UND*/
			ARM_SPSR_UND = arm.reg[rm];
			break;
		}
	}else{
		arm.reg[16] = arm.reg[rm];
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_mul()
{	/*Rd := Rm * Rs (ώZ)*/
	u32	opcode;
	u32	rd, rm, rs;

	opcode	= arm.opcode;
	rd		= (opcode >> 16) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;

	/*if((arm.reg[rm] * arm.reg[rs]) > 0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/

	arm.reg[rd] = arm.reg[rm] * arm.reg[rs];

	if(ARM_FLAG_COND)arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_mull()
{	/*RdHigh, RdLow := Rm * Rs (64bitώZ)*/
	u32	opcode;
	u32	rdh, rdl, rm, rs;

	opcode	= arm.opcode;
	rdh		= (opcode >> 16) & 0x0F;
	rdl		= (opcode >> 12) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.reg[rdl] = arm.reg[rm] * arm.reg[rs];
	arm.reg[rdh] = (u32)(((unsigned __int64)arm.reg[rm] * arm.reg[rs]) >> 32);

	if(ARM_FLAG_COND){
		if(!arm.reg[rdl] && !arm.reg[rdh])ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
		if(arm.reg[rdh] & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_mvn()
{	/*Rd := 0xFFFFFFFF EOR Op2 <Data Processing> (1̕␔)*/
	u32	opcode, operand;
	u32	rd;

	opcode	= ARM_OPCODE;
	rd		= ARM_DEREG;
	
	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = 0xFFFFFFFF ^ operand;

	if(ARM_FLAG_COND)arm7tdmi_set_dp_flags(arm.reg[rd]);
	
	arm.cycle = 1;
}

_inline void arm7tdmi_arm_orr()
{	/*Rd := Rn OR Op2 <Data Processing> (_a)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] | operand;

	if(ARM_FLAG_COND)arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_rsb()
{	/*Rd := Op2 - Rn <Data Processing> (tZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = operand - arm.reg[rn];

	if(ARM_FLAG_COND)arm7tdmi_set_sub_flag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_rsc()
{	/*Rd := Op2 - Rn - 1 + Carry <Data Processing> (L[ttZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;
	
	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = operand - arm.reg[rn] - 1 + ARM_LSB_FC;

	if(ARM_FLAG_COND)arm7tdmi_set_sub_flag(arm.reg[rn], (operand - 1 + ARM_LSB_FC), arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_sbc()
{	/*Rd := Rn - Op2 - 1 + Carry <Data Processing> (L[tZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] - operand - 1 + ARM_LSB_FC;

	arm7tdmi_set_sub_flag(arm.reg[rn], (operand - 1 + ARM_LSB_FC), arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_stc()
{	/*address := CRn (RvZbTWX^̓e֊i[)*/
	u32	opcode;
	u32	rd, rn, cp;
	u8	offset;

	opcode	= arm.opcode;
	rd		= (opcode >> 12) & 0x0F;
	rn		= (opcode >> 16) & 0x0F;
	cp		= (opcode) & 0x0F;
	offset	= (u8)(opcode);

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_stm()
{	/*Stack manipulation (̃WX^֊i[,Push)*/
	u32	opcode,	address, bit;
	u32	rn;		/*base register*/
	u32	lst;	/*register list*/
	u32	fprepost;

	arm.cycle = 2;

	opcode	= arm.opcode;
	rn		= (opcode >> 16) & 0x0F;	/*16-19 4bit*/

	/*base address*/
	address = arm.reg[rn];

	fprepost = (opcode & BIT24)?TRUE:FALSE;

	for(bit = 1, lst = 0; bit != BIT16; bit <<= 1 ,lst++){
		if(opcode & bit){
			arm.cycle++;
			if(fprepost){	/*Pre: ItZbg̒ǉOɓ]*/
				if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
			agb_write_mem32(address, arm.reg[lst]);
			if(!fprepost){	/*Post: ItZbgǉɓ]*/
				if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
		}
	}

	if((opcode & BIT21) || !fprepost){	/*Write-back sp(rn)̓eXV*/
		arm.reg[rn] = address;
	}
}

_inline void arm7tdmi_arm_str()
{	/*<address> := Rd, [rn] (WX^֊i[) lItZbg*/
	u32	opcode,offset, address;
	u32	rd, rn;
	u32	fprepost;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/

	address = arm.reg[rn];

	if(rn == 15)address += 8;	/*ΏۂPC̏ꍇ*/

	fprepost = (opcode & BIT24)?TRUE:FALSE;

	if(opcode & BIT25){	/*1 = ItZbg̓WX^*/
		offset = arm7tdmi_imm_shift(opcode);
	}else{					/*0 = ItZbg͑l*/
		offset = opcode & 0xFFF;
	}

	if(fprepost){	/*1 = Pre: ItZbg̒ǉOɓ]*/
		if(opcode & BIT23){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT22){	/*Byte ]*/
		agb_write_mem8(address, arm.reg[rd]);
		//agb_write_mem8(address&~1, arm.reg[rd]);
	}else{					/*Double word ]*/
		agb_write_mem32(address, arm.reg[rd]);
	}

	if(!fprepost){	/*0 = Post: ItZbg̒ǉɓ]*/
		if(opcode & BIT23){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT21) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 2;
}

_inline void arm7tdmi_arm_strs()
{	/*<address> := Rd (WX^֊i[) Register offset*/
	u32	opcode, address;
	u32	rd, rn, rm;
	u32	fprepost;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/
	rm		= opcode & 0xF;

	address = arm.reg[rn];

	fprepost = (opcode & BIT24)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT5){	/*Halfword*/
		if(opcode & BIT6){	/*Sign*/
			/*11: Signed Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}else{
			/*01: Unsigned Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}
	}else{
		if(opcode & BIT6){	/*Sign*/
			/*10: Signed byte*/
			agb_write_mem8(address, arm.reg[rd]);
		}else{
			/*00: SWP instruction*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if((opcode & BIT21) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_strs_imm()
{	/*<address> := Rd (WX^֊i[) Immidiate offset*/
	u32	opcode,	offset, address;
	u32	rd, rn;
	u32	fprepost;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/
	offset	= (opcode & 0xF) | ((opcode >> 4) & 0xF0);

	address = arm.reg[rn];

	fprepost = (opcode & BIT24)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT5){	/*Halfwords*/
		if(opcode & BIT6){	/*Sign*/
			/*Signed Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}else{
			/*Unsigned Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}
	}else{
		if(opcode & BIT6){	/*Sign*/
			/*Signed byte*/
			agb_write_mem8(address, arm.reg[rd]);
		}else{
			/*SWP instruction*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT21) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_sub()
{	/*Rd := Rn - Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= arm.opcode;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	/*if((arm.reg[rn] - operand) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/

	arm.reg[rd] = arm.reg[rn] - operand;

	if(ARM_FLAG_COND)arm7tdmi_set_sub_flag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_swi()
{	/*OS call (\tgEFA荞)*/
	u32	opcode, comment;

	opcode	= arm.opcode;
	comment	= opcode & 0x00FFFFFF;

	switch((u8)comment){
		case 0x00:agb_bios_soft_reset();break;
		case 0x01:agb_bios_register_ram_reset();break;
		case 0x02:agb_bios_halt();break;
		case 0x03:agb_bios_stop();break;
		case 0x04:agb_bios_intr_wait();break;
		case 0x05:agb_bios_vblank_intr_wait();break;
		case 0x06:agb_bios_div();break;
		case 0x07:agb_bios_div_arm();break;
		case 0x08:agb_bios_sqrt();break;
		case 0x09:agb_bios_arc_tan();break;
		case 0x0A:agb_bios_arc_tan2();break;
		default:console_print("Unknow bios call.\n");
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_arm_swp()
{	/*Rd := [Rn], [Rn] := Rm (Pf[^̌)*/
	u32	opcode;
	u32	rd, rn, rm;

	opcode	= arm.opcode;
	rd		= (opcode >> 12) & 0x0F;
	rn		= (opcode >> 16) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.reg[rd] = arm.reg[rn];
	arm.reg[rn] = arm.reg[rm];

	arm.cycle = 4;
}

_inline void arm7tdmi_arm_teq()
{	/*CPSR flags := Rn EOR Op2 <Data Processing> (rbgp^[̔r)*/
	u32	opcode,	operand, temp;
	u32	rn;

	opcode	= arm.opcode;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	/*CPSR = arm.reg[rn] ^ operand;*/
	temp = arm.reg[rn] ^ operand;

	arm7tdmi_set_dp_flags(temp);

	arm.cycle = 1;
}

_inline void arm7tdmi_arm_tst()
{	/*CPSR flags := Rn AND Op2 <Data Processing> (w肵rbg̃eXg)*/
	u32	opcode,	operand, temp;
	u32	rn;

	opcode	= arm.opcode;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){
		operand = arm7tdmi_imm_rotate(opcode);
	}else{
		operand = arm7tdmi_imm_shift(opcode);
	}

	/*CPSR = arm.reg[rn] & operand;*/
	temp = arm.reg[rn] & operand;

	arm7tdmi_set_dp_flags(temp);

	arm.cycle = 1;
}

#ifdef	_AGBWIN32_
void CMainFrame::exec_arm_clk(s32 clk)
{
	do{
		if(ARM_CPSR & ARM_T){
			//exec_thumb_state();
		}else{
			exec_arm_state();
		}
		clk -= arm.cycle;
	}while(clk > 0);
}
#endif	/*_AGBWIN32_*/

#ifdef	_AGBWIN32_
int CMainFrame::exec_arm_state()
#else
int exec_arm_state()
#endif	/*_AGBWIN32_*/
{
	u32	opcode, cond_enable;
	u8	opcode_24_4, cond;
#ifdef	_AGBWIN32_
	char str[64];
#endif	/*_AGBWIN32_*/

	opcode		= ARM_OPCODE;
	arm.opcode	= opcode;
	opcode_24_4	= (u8)((opcode >> 24) & 0x0F);	/*24-27 4bit*/
	cond		= ARM_COND;
	cond_enable	= FALSE;

	switch(cond){	/*s - Condition Field*/
	case 0x0:
		if(ARM_FZ)cond_enable = TRUE;
		break;		/*if Z set		- equal(==)*/
	case 0x1:
		if(!ARM_FZ)cond_enable = TRUE;
		break;		/*if Z clear	- not equal(!=)*/
	case 0x2:
		if(ARM_FC)cond_enable = TRUE;
		break;		/*if C set		- unsigned higher or same(>=)*/
	case 0x3:
		if(!ARM_FC)cond_enable = TRUE;
		break;		/*if C clear	- unsigne lower(<)*/
	case 0x4:
		if(ARM_FN)cond_enable = TRUE;
		break;		/*if N set		- negative(<0)*/
	case 0x5:
		if(!ARM_FN)cond_enable = TRUE;
		break;		/*if N clear	- positive or zero(>=0)*/
	case 0x6:
		if(ARM_FV)cond_enable = TRUE;
		break;		/*if V set		- overflow(>0xFFFFFFFF)*/
	case 0x7:
		if(!ARM_FV)cond_enable = TRUE;
		break;		/*if V clear	- no overflow<=0xFFFFFFFF*/
	case 0x8:
		if(ARM_FC && !ARM_FZ)cond_enable = TRUE;
		break;		/*if C set and Z clear	- unsigned higher(>)*/
	case 0x9:
		if(!ARM_FC || ARM_FZ)cond_enable = TRUE;
		break;		/*if C clear and Z set	- unsigned lower or same(<=)*/
	case 0xA:
		if((ARM_FN && ARM_FV) || (!ARM_FN && !ARM_FV))cond_enable = TRUE;
		break;		/*if ((N set and V set) or (N clear and V clear))	- greater or equal*/
	case 0xB:
		if((ARM_FN && !ARM_FV) || (!ARM_FN && ARM_FV))cond_enable = TRUE;
		break;		/*if ((N set and V clear) or (N clear and V set))	- less than*/
	case 0xC:
		if((!ARM_FZ && (ARM_FN || ARM_FV)) || (!ARM_FZ && (!ARM_FN || !ARM_FV)))cond_enable = TRUE;
		break;		/*if (Z clear and(N or V set) or (N or V clear))	- greater than*/
	case 0xD:
		if((ARM_FZ || (ARM_FN && !ARM_FV)) || (ARM_FZ || (!ARM_FN && ARM_FV)))cond_enable = TRUE;
		break;		/*if (Z set or(N set and V clear) or (N clear and V set))- less than or equal*/
	default:
		cond_enable = TRUE;
		break;		/*(ignored)	- always */
	}

	if(cond_enable){
		switch(opcode_24_4){
		case 0x0:	/*0000*/
			if(((opcode >> 4) & 0x0F) == 0x09){
				if(opcode & BIT23){	/*Multiply Long*/
					if(opcode & BIT21){	/*Accumelate*/
						arm7tdmi_arm_mlal();
						break;
					}else{
						arm7tdmi_arm_mull();
						break;
					}
				}else{					/*Multiply*/
					if(opcode & BIT21){	/*Accumelate*/
						arm7tdmi_arm_mla();
						break;
					}else{
						arm7tdmi_arm_mul();
						break;
					}
				}
			}
		case 0x1:	/*0001*/
			if(((opcode >> 4) & 0xFF) == 0x09){	/*00001001*/
				arm7tdmi_arm_swp();	/*Single Data Swap*/
				break;
			}
			if((opcode & 0x0FFFFFF0) == 0x012FFF10){
				arm7tdmi_arm_bx();	/*Branch and Exchange*/
				break;
			}
			if(opcode & BIT4){	/*Halfword data Transfer:*/
				if(opcode & BIT22){	/*immdiate offset*/
					if(opcode & BIT20){	/*Load from memory*/
						arm7tdmi_arm_ldrs_imm();
					}else{					/*Store to memory*/
						arm7tdmi_arm_strs_imm();
					}
					break;
				}else{					/*register offset*/
					if(opcode & BIT20){	/*Load from memory*/
						arm7tdmi_arm_ldrs();
					}else{					/*Store to memory*/
						arm7tdmi_arm_strs();
					}
					break;
				}
			}
		case 0x2:	/*0010*/
		case 0x3:	/*0011*/
		/*Data Processing*/
			switch((u8)((opcode >> 21) & 0x0F)){
			case 0x0: arm7tdmi_arm_and();break;
			case 0x1: arm7tdmi_arm_eor();break;
			case 0x2: arm7tdmi_arm_sub();break;
			case 0x3: arm7tdmi_arm_rsb();break;
			case 0x4: arm7tdmi_arm_add();break;
			case 0x5: arm7tdmi_arm_adc();break;
			case 0x6: arm7tdmi_arm_sbc();break;
			case 0x7: arm7tdmi_arm_rsc();break;
			case 0x8: arm7tdmi_arm_tst();break;
			case 0x9: arm7tdmi_arm_teq();break;
			case 0xA: arm7tdmi_arm_cmp();break;
			case 0xB: arm7tdmi_arm_cmn();break;
			case 0xC: arm7tdmi_arm_orr();break;
			case 0xD: arm7tdmi_arm_mov();break;
			case 0xE: arm7tdmi_arm_bic();break;
			case 0xF: arm7tdmi_arm_mvn();break;
			}
			break;
		/*Single Data Transfer*/
		case 0x4:	/*0100*/
		case 0x5:	/*0101*/
		case 0x6:	/*0110*/
		case 0x7:	/*0111*/
			if(opcode & BIT20){	/*Load from memory*/
				arm7tdmi_arm_ldr();
				break;
			}else{					/*Store to memory*/
				arm7tdmi_arm_str();
				break;
			}
			if(opcode & BIT4){
		/*`R[h*/
#ifdef	_AGBWIN32_
				LoadString(0, IDS_UNDEFINED_ARM_OPCODE, str, 64);
				sprintf(str, str, opcode, ARM_PC);
				//sprintf(str, "Undefine ARM opcode \"0x%08X\" at 0x%08X", opcode, PC);
				/*MessageBox(message, NULL, MB_ICONSTOP);
				CPUIsRunning = FALSE;*/
				console_print(str);
#endif	/*_AGBWIN32_*/
				arm7tdmi_set_cpu_mode(ARM_MODE_UND);/*UND[h*/
				break;
			}
		/*Block Data Transfer*/
		case 0x8:	/*1000*/
		case 0x9:	/*1001*/
			if(opcode & BIT20){	/*Load from */
				arm7tdmi_arm_ldm();
			}else{
				arm7tdmi_arm_stm();
			}
			break;
		/*Branch*/
		case 0xA:	/*1010*/
			arm7tdmi_arm_b();
			break;
		case 0xB:	/*1011*/
			arm7tdmi_arm_bl();	/*branch with Link*/
			break;
		/*Coprocesser Data Transfer*/
		case 0xC:	/*1100*/
		case 0xD:	/*1101*/
			if(opcode & BIT20){	/*Load from */
				arm7tdmi_arm_ldc();
			}else{
				arm7tdmi_arm_stc();
			}
			break;
		case 0xE:	/*1110*/
			if(opcode & BIT4){
		/*Coprocesser Resgister Transfer*/
				if(opcode & BIT20){	/*Load from Co-Processer*/
					arm7tdmi_arm_mrc();
				}else{					/*Store to Co-Processer*/
					arm7tdmi_arm_mcr();
				}
			}else{
		/*Coprocesser Data Operation*/
				arm7tdmi_arm_cdp();
			}
			break;
		/*Software interrupt*/
		case 0xF:	/*1111*/
			arm7tdmi_arm_swi();
			break;
		default:
			//sprintf(str, "Unknow ARM opcode \"0x%08X\" at 0x%08X", opcode, PC);
#ifdef	_AGBWIN32_
			LoadString(0, IDS_UNKNOW_ARM_OPCODE, str, 64);
			sprintf(str, str, opcode, ARM_PC);
			MessageBox(str, NULL, MB_ICONSTOP);
			console_print(str);
#endif	/*_AGBWIN32_*/
			arm.running = FALSE;
			break;
		}
	}

	ARM_PC += 4;

	return 0;
}

