
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/


extern u32 arm7tdmi_imm_shift(u32 opcode);
extern u32 arm7tdmi_imm_rotate(u32 opcode);

char *cond_table[16] = {	/*RfBVe[u()*/
	"eq",	/*if Z set*/
	"ne",	/*if Z clear*/
	"cs",	/*if C set*/
	"cc",	/*if C clear*/
	"mi",	/*if N set*/
	"pl",	/*if N clear*/
	"vs",	/*if V set*/
	"vc",	/*if V clear*/
	"hi",	/*if C set and Z clear*/
	"ls",	/*if C clear and Z set*/
	"ge",	/*if ((N set and V set) or (N clear and V clear))*/
	"lt",	/*if ((N set and V clear) or (N clear and V set))*/
	"gt",	/*if (Z clear and(N or V set) or (N or V clear))*/
	"le",	/*if (Z set or(N set and V clear) or (N clear and V set))*/
	"",		/*always */
	"",		/*`*/
};

void arm7tdmi_disasm_imm_shift(u32 opcode, char *str)
{	/*Vtgl*/
	u32 offset, bit;
	u32	shift, rm;
	u8	shift_type;

	rm	= ARM_SOREG;
	shift_type = (u8)((opcode >> 5) & 3);

	if(opcode & 0x08){	/*Rs(8-11)*/
		shift = (opcode >> 8) & 0xF;
	}else{				/*Rs(7-11)*/
		shift = (opcode >> 7) & 0x1F;
	}

	if(rm == 15){
		switch(shift_type){
		case 0:	/*00 = logical left(lsl) _IVtg 0*/
			offset = debug_pc << shift;
			break;
		case 1:	/*01 = logical right(lsr) _IEVtg 0*/
			offset = debug_pc >> shift;
			break;
		case 2:	/*10 = arithmetic right(asr) ZpIEVtg 31rbg̒l*/
			offset = debug_pc >> shift;
			if(debug_pc & BIT31){
				for(bit = BIT31; shift; bit >>= 1){
					offset |= bit;
					shift--;
				}
			}
			break;
		case 3:	/*11 = rotate right(ror) lւ*/
			offset =  (debug_pc >> shift) | (debug_pc << (32 - shift));
			break;
		}
		sprintf(str, "#0x%08X", offset + debug_pc);
	}else{
		if(shift){
			switch(shift_type){
			case 0:	/*00 = logical left(lsl) _IVtg 0*/
				sprintf(str, "r%d, lsl #%d", rm, shift);
				break;
			case 1:	/*01 = logical right(lsr) _IEVtg 0*/
				sprintf(str, "r%d, lsr #%d", rm, shift);
				break;
			case 2:	/*10 = arithmetic right(asr) ZpIEVtg 31rbg̒l*/
				sprintf(str, "r%d, asr #%d", rm, shift);
				break;
			case 3:	/*11 = rotate right(ror) lւ*/
				sprintf(str, "r%d, ror #%d", rm, shift);
				break;
			}
		}else{
			sprintf(str, "r%d", rm);
		}
	}
}

void arm7tdmi_disasm_imm_rotate(u32 opcode, char *str)
{	/*E[e[gl*/
	u32	shift, imm_val;
	u32	offset;

	shift	= ((opcode >> 8) & 0xF) << 1;
	imm_val	= opcode & 0xFF;

	offset = (imm_val >> shift) | (imm_val << (32 - shift));

	sprintf(str, "#0x%X", offset);
}


void arm7tdmi_disasm_reg_list(u32 opcode, char *str_lst)
{
	u32 lst, flag;
	char str[8];

	*str_lst = 0;
	flag = FALSE;
	for(lst = 0; lst < 16; lst++){
		if(opcode & (1 << lst)){
			if(opcode & (1 << (lst + 1))){
				if(!flag){
					flag = TRUE;
					sprintf(str, "r%d-", lst);
					strcat(str_lst, str);
				}
			}else{
				sprintf(str, "r%d,", lst);
				strcat(str_lst, str);
			}
		}
	}

	if(str_lst[strlen(str_lst) - 1] == ','){
		str_lst[strlen(str_lst) - 1] = '\0';
	}
}

void arm7tdmi_disasm_arm_adc()
{	/*Rd := Rn + Op2 + Carry <Data Processing> (Z{L[)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X adc%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_add()
{	/*Rd := Rn + Op2 <Data Processing> (Z)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X add%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_and()
{	/*Rd := Rn AND Op2 <Data Processing>*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X and%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_b()
{	/*R15 := address ()*/
	u32	opcode, address;
	s32	offset;/*signed*/
	u32	cond;
	static u32 label_num = 0;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;

	offset = ((opcode & 0x00FFFFFF) << 2);

	address = offset + debug_pc + 8;

	/*ItZbg͈̔͂MSB^Ȃ}CiX*/
	if(offset & BIT25)address -= BIT26;

	sprintf(mnemonic, "%08X %08X b%s #0x%08X", 
		debug_pc, opcode, cond_table[cond], address);
	disasm_print(mnemonic);
	if(m_out_file)fprintf(dfp, "\n.L%d:", label_num++);
}

void arm7tdmi_disasm_arm_bic()
{	/*Rd := Rn AND NOT Op2 <Data Processing> (rbgNA)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X bic%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_bl()
{	/*R14 := R15, R15 := address*/
	u32	opcode, address;
	s32	offset;
	u32	cond;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	offset	= (u32)((opcode & 0x00FFFFFF) << 2);

	/*ItZbg*/
	address = offset + debug_pc + 8;

	/*ItZbg͈̔͂MSB^Ȃ}CiX*/
	if(offset & BIT25)address -= BIT26;

	sprintf(mnemonic, "%08X %08X bl%s #0x%08X", 
		debug_pc, opcode, cond_table[cond], address);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_bx()
{	/*R15 := Rn (Ɩ߃Xe[g̕ύX)*/
	u32	opcode;
	u32	cond, rn;
	
	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= REG_0_4;

	sprintf(mnemonic, "%08X %08X bx%s r%d", 
		debug_pc, opcode, cond_table[cond], rn);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_cdp()
{	/*(Coprocessor-specific) (RvZbT[f[^)*/
	u32	opcode;
	u32	cond;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;

	sprintf(mnemonic, "%08X %08X cdp%s r%d", 
		debug_pc, opcode, cond_table[cond]);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_cmn()
{	/*CPSR flags := Rn + Op2 <Data Processing> (tr)*/
	u32	opcode;
	u32	cond, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X cmn%s r%d, %s", 
		debug_pc, opcode, cond_table[cond], rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_cmp()
{	/*CPSR flags := Rn - Op2 <Data Processing> (r)*/
	u32	opcode;
	u32	cond, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X cmp%s r%d, %s", 
		debug_pc, opcode, cond_table[cond], rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_eor()
{	/*Rd := (Rn AND NOT Op2) OR (op2 AND NOT Rn) <Data Processing> (rI_a)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X eor%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_ldc()
{	/*Coprocessor load (RvZbTւ̓])*/
	u32	opcode;
	u32	cond, rn, cd, cp;
	u32 offset;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= (opcode >> 16) & 0xF;
	cd		= (opcode >> 12) & 0xF;
	cp		= (opcode >> 8) & 0xF;
	offset	= opcode & 0xFF;

	if(offset){
		sprintf(str_shift, "r%d, #0x%X", rn, offset);
	}else{
		sprintf(str_shift, "r%d", rn);
	}

	sprintf(mnemonic, "%08X %08X ldc%s %d, %d, [%s]", 
		debug_pc, opcode, cond_table[cond], 
		cp, cd, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_ldm()
{	/*Stack manipulation (̃WX^̃[h,POP)*/
	u32	opcode,	reg_list;
	u32	cond, rn, mode;
	char str_sp[16], str_mode[16], str_list[128 * 2];

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= (opcode >> 16) & 0x0F;	/*16-19 4bit*/
	mode	= (opcode >> 23) & 0x03;	/*23-24 2bit*/

	reg_list = opcode & 0xFFFF;

	if(rn == 13){	/*sp==r13*/
		strcpy(str_sp, "sp");
		switch(mode){	/*PU*/
		case 0:	/*__*/
			strcpy(str_mode, "ea");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ed");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "fa");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "fd");
			break;
		}
	}else{	/*PU*/
		sprintf(str_sp, "r%d", rn);
		switch(mode){
		case 0:	/*__*/
			strcpy(str_mode, "da");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ia");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "db");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "ib");
			break;
		}
	}

	disasm_reg_list(opcode, str_list);

	sprintf(mnemonic, "%08X %08X ldm%s%s %s%s, {%s}", 
		debug_pc, opcode, str_mode, cond_table[cond], 
		str_sp, ((opcode & BIT21)?"!":""), str_list);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_ldr()
{	/*Rd := (address) (WX^ւ̃[h)*/
	u32	opcode,	offset, address;
	u32	cond, rd, rn;
	char str_shift_temp[STR_OFFSET_SIZE];

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= (opcode >> 16) & 0xF;	/*base register*/

	if(opcode & BIT25){	/*1 = ItZbg̓WX^*/
		offset = arm7tdmi_imm_shift(opcode);
	}else{				/*0 = ItZbg̓WX^*/
		offset = opcode & 0xFFF;	/*0-11bit*/
	}

	if(rn == 15){
		address = debug_pc;
		address += 8;
		if(opcode & BIT24){
			if(opcode & BIT23){
				address += offset;
			}else{
				address -= offset;
			}
		}
		sprintf(str_shift, "[#0x%08X]", address);
	}else{
		if(opcode & BIT25){	/*1 = ItZbg̓WX^*/
			arm7tdmi_disasm_imm_shift(opcode, str_shift_temp);
			sprintf(str_shift, "[%s]", str_shift_temp);
		}else{				/*0 = ItZbg̓WX^*/
			offset = opcode & 0xFFF;	/*0-11bit*/
			if(offset){
				if(opcode & BIT23){
					sprintf(str_shift, "[r%d], #0x%X", rn, offset);
				}else{
					sprintf(str_shift, "[r%d], -#0x%X", rn, offset);
				}
			}else{
				sprintf(str_shift, "[r%d]", rn);
			}
		}
	}


	sprintf(mnemonic, "%08X %08X ldr%s%s r%d, %s%s", 
		debug_pc, opcode, cond_table[cond], 
		((opcode & BIT22)?"b":""), rd, str_shift, ((opcode & BIT21)?"!":""));
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_ldrs()
{	/*Rd := (address) (WX^ւ̃[h) Register offset*/
	u32	opcode;
	u32	cond, rd, rn, rm;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/
	rm		= opcode & 0xF;

	if(opcode & BIT5){	/*Halfwords*/
		if(opcode & BIT6){	/*Signed*/
			/*11SH: Signed Halfwords*/
			strcpy(str_shift, "sh");
		}else{
			/*01: Unsigned Halfwords*/
			strcpy(str_shift, "h");
		}
	}else{
		if(opcode & BIT6){	/*Signed*/
			/*10: Signed byte*/
			strcpy(str_shift, "sb");
		}else{
			/*00: SWP instruction ɖ肠I*/
			strcpy(str_shift, "<SWP>");
		}
	}

	sprintf(mnemonic, "%08X %08X ldr%s%s r%d, [r%d, r%d]%s", 
		debug_pc, opcode, cond_table[cond], 
		str_shift, rd, 
		rn, rm, ((opcode & BIT21)?"!":""));
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_ldrs_imm()
{	/*Rd := (address) (WX^ւ̃[h) Immidiate offset*/
	u32	opcode,	offset;
	u32	cond, rd, rn;
	char str_trans[STR_OFFSET_SIZE];

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;		/*AhX̃x[XWX^*/
	offset	= (opcode & 0xF) | ((opcode >> 4) & 0xF0);

	if(opcode & BIT5){	/*Halfwords*/
		if(opcode & BIT6){	/*Signed*/
			/*Signed Halfwords*/
			strcpy(str_trans, "sh");
		}else{
			/*Unsigned Halfwords*/
			strcpy(str_trans, "h");
		}
	}else{
		if(opcode & BIT6){	/*Signed*/
			/*Signed byte*/
			strcpy(str_trans, "sb");
		}else{
			/*SWP instruction*/
			strcpy(str_trans, "<SWP>");
		}
	}

	if(offset){
		sprintf(str_shift, "[r%d], #0x%02X", rn, offset);
	}else{
		sprintf(str_shift, "[r%d]", rn);
	}

	sprintf(mnemonic, "%08X %08X ldr%s%s r%d, %s%s", 
		debug_pc, opcode, cond_table[cond], 
		str_trans, rd, str_shift, ((opcode & BIT21)?"!":""));
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mcr()
{	/*cRn := rRn {<op>cRm}*/
	u32	opcode;
	u32	cond;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;

	sprintf(mnemonic, "%08X %08X mcr%s", 
		debug_pc, opcode, cond_table[cond]);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mla()
{	/*Rd := (Rm * Rs) + Rn (ϘaZ)*/
	u32	opcode;
	u32	cond, rd, rm, rs, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= (opcode >> 16) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;
	rn		= (opcode >> 12) & 0x0F;

	sprintf(mnemonic, "%08X %08X mla%s r%d, r%d, r%d, r%d", 
		debug_pc, opcode, cond_table[cond], 
		rd, rs, rm, rn);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mlal()
{	/*Rd := (Rm * Rs) + Rn (64bitϘaZ)*/
	u32	opcode;
	u32	cond, rdh, rdl, rm, rs;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rdh		= (opcode >> 16) & 0x0F;
	rdl		= (opcode >> 12) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;

	sprintf(mnemonic, "%08X %08X mlal%s r%d, r%d, r%d, r%d", 
		debug_pc, opcode, cond_table[cond], 
		rdl, rdh, rm, rs);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mov()
{	/*Rd : = Op2 <Data Processing> (WX^͒萔̑)*/
	u32	opcode;
	u32	cond, rd;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X mov%s%s r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, str_shift);

	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mrc()
{	/*Rn := cRn {<op>cRm} (RvZbTWX^CPUWX^ւ̓])*/
	u32	opcode;
	u32	cond;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;

	sprintf(mnemonic, "%08X %08X mrc%s", 
		debug_pc, opcode, cond_table[cond]);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mrs()
{	/*Rn := PSR (Xe[^XWX^烌WX^ւ̓])*/
	u32	opcode;
	u32	cond, rm;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rm		= opcode & 0x0F;

	sprintf(mnemonic, "%08X %08X mrs%s r%d, %s", 
		debug_pc, opcode, cond_table[cond],
			rm, (opcode & BIT22)?"spsr_":"cpsr");
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_msr()
{	/*PSR := Rm (WX^Xe[^XWX^ւ̓])*/
	u32	opcode;
	u32	cond, rd;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= (opcode >> 12) & 0x0F;

	sprintf(mnemonic, "%08X %08X msr%s %s, r%d", 
		debug_pc, opcode, cond_table[cond],
			(opcode & BIT22)?"spsr_":"cpsr", rd);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mul()
{	/*Rd := Rm * Rs (ώZ)*/
	u32	opcode;
	u32	cond, rd, rm, rs;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= (opcode >> 16) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;

	sprintf(mnemonic, "%08X %08X mul%s%s r%d, r%d, r%d", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rm, rs);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mull()
{	/*Rd := Rm * Rs (64bitώZ)*/
	u32	opcode;
	u32	cond, rdh, rdl, rm, rs;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rdh		= (opcode >> 16) & 0x0F;
	rdl		= (opcode >> 12) & 0x0F;
	rs		= (opcode >> 8) & 0x0F;
	rm		= (opcode) & 0x0F;

	sprintf(mnemonic, "%08X %08X mull%s%s r%d, r%d, r%d, r%d", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rdl, rdh, rm, rs);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_mvn()
{	/*Rd := 0xFFFFFFFF EOR Op2 <Data Processing> (1̕␔)*/
	u32	opcode;
	u32	cond, rd;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X mvn%s%s r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_orr()
{	/*Rd := Rn OR Op2 <Data Processing> (_a)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X orr%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_rsb()
{	/*Rd := Op2 - Rn <Data Processing> (tZ)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X rsb%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_rsc()
{	/*Rd := Op2 - Rn <Data Processing> (L[ttZ)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X rsc%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_sbc()
{	/*Rd := Rn - Op2 - 1 + Carry <Data Processing> (L[tZ)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X sbc%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_stc()
{	/*address := CRn (RvZbT̓e֊i[)*/
	u32	opcode;
	u32	cond, rn, cd, cp;
	u32 offset;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= (opcode >> 16) & 0xF;
	cd		= (opcode >> 12) & 0xF;
	cp		= (opcode >> 8) & 0xF;
	offset	= opcode & 0xFF;

	if(offset){
		sprintf(str_shift, "r%d, #0x%X", rn, offset);
	}else{
		sprintf(str_shift, "r%d", rn);
	}

	sprintf(mnemonic, "%08X %08X stc%s %d, %d, [%s]", 
		debug_pc, opcode, cond_table[cond], 
		cp, cd, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_stm()
{	/*Stack manipulation (̃WX^֊i[,PUSH)*/
	u32	opcode;
	u32	cond, rn, mode;
	char str_sp[16], str_mode[16], str_list[128 * 2];

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= (opcode >> 16) & 0x0F;	/*16-19 4bit*/
	mode	= (opcode >> 23) & 0x03;	/*23-24 2bit*/

	if(rn == 13){	/*sp==r13*/
		strcpy(str_sp, "sp");
		switch(mode){	/*PU*/
		case 0:	/*__*/
			strcpy(str_mode, "ed");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ea");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "fd");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "fa");
			break;
		}
	}else{	/*PU*/
		sprintf(str_sp, "r%d", rn);
		switch(mode){
		case 0:	/*__*/
			strcpy(str_mode, "da");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ia");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "db");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "ib");
			break;
		}
	}
/*
	strcpy(str_list, "");
	flag = 1;
	for(bit = 1, lst = 0; bit != BIT16; bit <<= 1, lst++){
		if(opcode & bit){
			//if(flag){
				flag=0;
				sprintf(str, "r%d,", lst);
				strcat(str_list, str);
			//}
		}else{
			flag=1;
		}
	}
*/
	disasm_reg_list(opcode, str_list);

	sprintf(mnemonic, "%08X %08X stm%s%s %s%s, {%s}", 
		debug_pc, opcode, str_mode, cond_table[cond], 
		str_sp, ((opcode & BIT21)?"!":""), str_list);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_str()
{	/*<address> := Rd (WX^̓e֊i[)*/
	u32	opcode,	offset, address;
	u32	cond, rd, rn;
	char str_shift_temp[STR_OFFSET_SIZE];

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/

	if(opcode & BIT25){	/*1 = ItZbg̓WX^*/
		offset = arm7tdmi_imm_shift(opcode);
	}else{				/*0 = ItZbg̓WX^*/
		offset = opcode & 0xFFF;	/*0-11bit*/
	}

	if(rn == 15){
		if(opcode & BIT23){	/*Up: ItZbgx[XɉZ*/
			address = debug_pc + offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address = debug_pc - offset;
		}
		sprintf(str_shift, "[#0x%08X]", address);
	}else{
		if(opcode & BIT25){
			arm7tdmi_disasm_imm_shift(opcode, str_shift_temp);
			sprintf(str_shift, "[%s]", str_shift_temp);
		}else{
			if(offset){
				if(opcode & BIT23){
					sprintf(str_shift, "[r%d], #0x%X", rn, offset);
				}else{
					sprintf(str_shift, "[r%d], -#0x%X", rn, offset);
				}
			}else{
				sprintf(str_shift, "[r%d]", rn);
			}
		}
	}

	sprintf(mnemonic, "%08X %08X str%s%s r%d, %s%s", 
		debug_pc, opcode, cond_table[cond], 
		((opcode & BIT22)?"b":""), 
		rd, str_shift , ((opcode & BIT21)?"!":""));
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_strs()
{	/*<address> := Rd (WX^̓e֊i[) Register offset*/
	u32	opcode;
	u32	cond, rd, rn, rm;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/
	rm		= opcode & 0xF;

	if(opcode & BIT5){	/*Halfwords*/
		if(opcode & BIT6){	/*Signed*/
			/*11SH: Signed Halfwords STR͖`H*/
			strcpy(str_shift, "<sh>");
		}else{
			/*01: Unsigned Halfwords*/
			strcpy(str_shift, "h");
		}
	}else{
		if(opcode & BIT6){	/*Signed*/
			/*10: Signed byte STR͖`H*/
			strcpy(str_shift, "<sb>");
		}else{
			/*00: SWP instruction ɖ肠I*/
			strcpy(str_shift, "<SWP>");
		}
	}

	sprintf(mnemonic, "%08X %08X str%s%s r%d, [r%d, r%d]%s", 
		debug_pc, opcode, cond_table[cond], 
		str_shift, rd, rn, rm, ((opcode & BIT21)?"!":""));
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_strs_imm()
{	/*<address> := Rd (WX^̓e֊i[) Immidiate offset*/
	u32	opcode,	offset;
	u32	cond, rn, rd;
	char str_trans[STR_OFFSET_SIZE];

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;					/*ΏۃWX^*/
	rn		= (opcode >> 16) & 0xF;	/*AhX̃x[XWX^*/
	offset	= (opcode & 0xF) | ((opcode >> 4) & 0xF0);

	if(opcode & BIT5){	/*Halfwords*/
		if(opcode & BIT6){	/*Signed*/
			/*Signed Halfwords*/
			strcpy(str_trans, "sh");
		}else{
			/*Unsigned Halfwords*/
			strcpy(str_trans, "h");
		}
	}else{
		if(opcode & BIT6){	/*Signed*/
			/*Signed byte*/
			strcpy(str_trans, "sb");
		}else{
			/*SWP instruction*/
			strcpy(str_trans, "<SWP>");
		}
	}

	if(offset){
		sprintf(str_shift, "[r%d], #0x%02X", rn, offset);
	}else{
		sprintf(str_shift, "[r%d]", rn);
	}

	sprintf(mnemonic, "%08X %08X str%s%s r%d, %s%s", 
		debug_pc, opcode, cond_table[cond], 
		str_trans, rd, str_shift, ((opcode & BIT21)?"!":""));
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_sub()
{	/*Rd := Rn - Op2 <Data Processing> (Z)*/
	u32	opcode;
	u32	cond, rd, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= ARM_DEREG;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X sub%s%s r%d, r%d, %s", 
		debug_pc, opcode, cond_table[cond], 
		(ARM_FLAG_COND?"s":""), rd, rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_swi()
{	/*OS call (\tgEFA荞)*/
	u32	opcode, comment;
	u32	cond;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	comment	= opcode & 0x00FFFFFF;

	sprintf(mnemonic, "%08X %08X swi%s #0x%X", 
		debug_pc, opcode, cond_table[cond], comment);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_swp()
{	/*Rd := [Rn], [Rn] := Rm (Pf[^̌)*/
	u32	opcode;
	u32	cond, rd, rn, rm;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rd		= (opcode >> 12) & 0x0F;
	rn		= (opcode >> 16) & 0x0F;
	rm		= (opcode) & 0x0F;

	sprintf(mnemonic, "%08X %08X swp%s r%d, r%d, r%d", 
		debug_pc, opcode, cond_table[cond], rd, rn, rm);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_teq()
{	/*CPSR flags := Rn EOR Op2 <Data Processing> (rbgp^[̔r)*/
	u32	opcode;
	u32	cond, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X teq%s r%d, %s", 
		debug_pc, opcode, cond_table[cond], rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_tst()
{	/*CPSR flags := Rn AND Op2 <Data Processing> (w肵rbg̃eXg)*/
	u32	opcode;
	u32	cond, rn;

	opcode	= ARM_OPCODE_D;
	cond	= ARM_COND;
	rn		= ARM_FOREG;

	if(ARM_FLAG_IMM){	/*Immediate Operand*/
		arm7tdmi_disasm_imm_rotate(opcode, str_shift);
	}else{
		arm7tdmi_disasm_imm_shift(opcode, str_shift);
	}

	sprintf(mnemonic, "%08X %08X tst%s r%d, %s", 
		debug_pc, opcode, cond_table[cond], rn, str_shift);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_undef()
{	/*Undefine (`)*/
	u32	opcode;

	opcode = ARM_OPCODE_D;

	sprintf(mnemonic, "%08X %08X <undefine>", 
		debug_pc, opcode);
	disasm_print(mnemonic);
}

void arm7tdmi_disasm_arm_unknow()
{	/*Undefine (`)*/
	u32	opcode;

	opcode = ARM_OPCODE_D;

	sprintf(mnemonic, "%08X %08X <unknow>", 
		debug_pc, opcode);
	disasm_print(mnemonic);
}
