/*-----------------------------------------------------------------------------
	[ADPCM.h]
		`cobl܂B

		Implement ADPCM functions.

	Copyright (C) 2004 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
-----------------------------------------------------------------------------*/
#include <stdio.h>		// debug
#include "ADPCM.h"


//#define	printf
//#define	puts


#define ADPCM_MAXVOLUME			1024


static Uint8	_Ram[0x10000];
static Uint16	_Addr;
static Uint16	_ReadAddr;
static Uint16	_WriteAddr;
static Sint32	_LengthCount;
static Sint32	_PlayLength;
static Sint32	_HalfLength;
static BOOL		_bPlay;
static BOOL		_bRepeat;
static Uint32	_SampleFreq;

/* volume, fadein/fadeout */
static Sint32	_CurrentVolume = ADPCM_MAXVOLUME;
static Sint32	_InitialVolume = ADPCM_MAXVOLUME;
static Sint32	_VolumeStep;
static BOOL		_bFadeIn  = FALSE;
static BOOL		_bFadeOut = FALSE;
static Sint32	_ClockCount;
static Sint32	_FadeCycle;


static Uint32	_Phase;
static Sint32	_DecodeBuffer[0x20000];
static BOOL		_bLowNibble;

static BOOL		_bInterrupt1;


static Sint32	_WrittenSampleCount;
static Sint32	_PlayedSampleCount;
static Sint32	_PlayedSampleCountDecimal;

static Uint8	_ReadBuffer;


static void		(*_pfnNotification)(Uint32 adpcmState);



// for adpcm play
static Sint32	_SrcIndex;
static Sint32	_DstIndex;
static Sint32	_Sample;
static Sint32	_Error;
static Sint32	_nDstSample;
static Sint32	_nSrcSample;


/*
	The original decoder is provided by Dave Shadoff,
	and further optimized by Ki.
*/
static Sint32		ad_sample;
static Sint32		ad_ref_index;

static const Uint32	_LUT[49*16] =
{
	0x0002ffff, 0x0006ffff, 0x000affff, 0x000effff, 0x00120002, 0x00160004, 0x001a0006, 0x001e0008,
	0xfffeffff, 0xfffaffff, 0xfff6ffff, 0xfff2ffff, 0xffee0002, 0xffea0004, 0xffe60006, 0xffe20008,
	0x0002ffff, 0x0006ffff, 0x000affff, 0x000effff, 0x00130002, 0x00170004, 0x001b0006, 0x001f0008,
	0xfffeffff, 0xfffaffff, 0xfff6ffff, 0xfff2ffff, 0xffed0002, 0xffe90004, 0xffe50006, 0xffe10008,
	0x0002ffff, 0x0006ffff, 0x000bffff, 0x000fffff, 0x00150002, 0x00190004, 0x001e0006, 0x00220008,
	0xfffeffff, 0xfffaffff, 0xfff5ffff, 0xfff1ffff, 0xffeb0002, 0xffe70004, 0xffe20006, 0xffde0008,
	0x0002ffff, 0x0007ffff, 0x000cffff, 0x0011ffff, 0x00170002, 0x001c0004, 0x00210006, 0x00260008,
	0xfffeffff, 0xfff9ffff, 0xfff4ffff, 0xffefffff, 0xffe90002, 0xffe40004, 0xffdf0006, 0xffda0008,
	0x0002ffff, 0x0007ffff, 0x000dffff, 0x0012ffff, 0x00190002, 0x001e0004, 0x00240006, 0x00290008,
	0xfffeffff, 0xfff9ffff, 0xfff3ffff, 0xffeeffff, 0xffe70002, 0xffe20004, 0xffdc0006, 0xffd70008,
	0x0003ffff, 0x0009ffff, 0x000fffff, 0x0015ffff, 0x001c0002, 0x00220004, 0x00280006, 0x002e0008,
	0xfffdffff, 0xfff7ffff, 0xfff1ffff, 0xffebffff, 0xffe40002, 0xffde0004, 0xffd80006, 0xffd20008,
	0x0003ffff, 0x000affff, 0x0011ffff, 0x0018ffff, 0x001f0002, 0x00260004, 0x002d0006, 0x00340008,
	0xfffdffff, 0xfff6ffff, 0xffefffff, 0xffe8ffff, 0xffe10002, 0xffda0004, 0xffd30006, 0xffcc0008,
	0x0003ffff, 0x000affff, 0x0012ffff, 0x0019ffff, 0x00220002, 0x00290004, 0x00310006, 0x00380008,
	0xfffdffff, 0xfff6ffff, 0xffeeffff, 0xffe7ffff, 0xffde0002, 0xffd70004, 0xffcf0006, 0xffc80008,
	0x0004ffff, 0x000cffff, 0x0015ffff, 0x001dffff, 0x00260002, 0x002e0004, 0x00370006, 0x003f0008,
	0xfffcffff, 0xfff4ffff, 0xffebffff, 0xffe3ffff, 0xffda0002, 0xffd20004, 0xffc90006, 0xffc10008,
	0x0004ffff, 0x000dffff, 0x0016ffff, 0x001fffff, 0x00290002, 0x00320004, 0x003b0006, 0x00440008,
	0xfffcffff, 0xfff3ffff, 0xffeaffff, 0xffe1ffff, 0xffd70002, 0xffce0004, 0xffc50006, 0xffbc0008,
	0x0005ffff, 0x000fffff, 0x0019ffff, 0x0023ffff, 0x002e0002, 0x00380004, 0x00420006, 0x004c0008,
	0xfffbffff, 0xfff1ffff, 0xffe7ffff, 0xffddffff, 0xffd20002, 0xffc80004, 0xffbe0006, 0xffb40008,
	0x0005ffff, 0x0010ffff, 0x001bffff, 0x0026ffff, 0x00320002, 0x003d0004, 0x00480006, 0x00530008,
	0xfffbffff, 0xfff0ffff, 0xffe5ffff, 0xffdaffff, 0xffce0002, 0xffc30004, 0xffb80006, 0xffad0008,
	0x0006ffff, 0x0012ffff, 0x001fffff, 0x002bffff, 0x00380002, 0x00440004, 0x00510006, 0x005d0008,
	0xfffaffff, 0xffeeffff, 0xffe1ffff, 0xffd5ffff, 0xffc80002, 0xffbc0004, 0xffaf0006, 0xffa30008,
	0x0006ffff, 0x0013ffff, 0x0021ffff, 0x002effff, 0x003d0002, 0x004a0004, 0x00580006, 0x00650008,
	0xfffaffff, 0xffedffff, 0xffdfffff, 0xffd2ffff, 0xffc30002, 0xffb60004, 0xffa80006, 0xff9b0008,
	0x0007ffff, 0x0016ffff, 0x0025ffff, 0x0034ffff, 0x00430002, 0x00520004, 0x00610006, 0x00700008,
	0xfff9ffff, 0xffeaffff, 0xffdbffff, 0xffccffff, 0xffbd0002, 0xffae0004, 0xff9f0006, 0xff900008,
	0x0008ffff, 0x0018ffff, 0x0029ffff, 0x0039ffff, 0x004a0002, 0x005a0004, 0x006b0006, 0x007b0008,
	0xfff8ffff, 0xffe8ffff, 0xffd7ffff, 0xffc7ffff, 0xffb60002, 0xffa60004, 0xff950006, 0xff850008,
	0x0009ffff, 0x001bffff, 0x002dffff, 0x003fffff, 0x00520002, 0x00640004, 0x00760006, 0x00880008,
	0xfff7ffff, 0xffe5ffff, 0xffd3ffff, 0xffc1ffff, 0xffae0002, 0xff9c0004, 0xff8a0006, 0xff780008,
	0x000affff, 0x001effff, 0x0032ffff, 0x0046ffff, 0x005a0002, 0x006e0004, 0x00820006, 0x00960008,
	0xfff6ffff, 0xffe2ffff, 0xffceffff, 0xffbaffff, 0xffa60002, 0xff920004, 0xff7e0006, 0xff6a0008,
	0x000bffff, 0x0021ffff, 0x0037ffff, 0x004dffff, 0x00630002, 0x00790004, 0x008f0006, 0x00a50008,
	0xfff5ffff, 0xffdfffff, 0xffc9ffff, 0xffb3ffff, 0xff9d0002, 0xff870004, 0xff710006, 0xff5b0008,
	0x000cffff, 0x0024ffff, 0x003cffff, 0x0054ffff, 0x006d0002, 0x00850004, 0x009d0006, 0x00b50008,
	0xfff4ffff, 0xffdcffff, 0xffc4ffff, 0xffacffff, 0xff930002, 0xff7b0004, 0xff630006, 0xff4b0008,
	0x000dffff, 0x0027ffff, 0x0042ffff, 0x005cffff, 0x00780002, 0x00920004, 0x00ad0006, 0x00c70008,
	0xfff3ffff, 0xffd9ffff, 0xffbeffff, 0xffa4ffff, 0xff880002, 0xff6e0004, 0xff530006, 0xff390008,
	0x000effff, 0x002bffff, 0x0049ffff, 0x0066ffff, 0x00840002, 0x00a10004, 0x00bf0006, 0x00dc0008,
	0xfff2ffff, 0xffd5ffff, 0xffb7ffff, 0xff9affff, 0xff7c0002, 0xff5f0004, 0xff410006, 0xff240008,
	0x0010ffff, 0x0030ffff, 0x0051ffff, 0x0071ffff, 0x00920002, 0x00b20004, 0x00d30006, 0x00f30008,
	0xfff0ffff, 0xffd0ffff, 0xffafffff, 0xff8fffff, 0xff6e0002, 0xff4e0004, 0xff2d0006, 0xff0d0008,
	0x0011ffff, 0x0034ffff, 0x0058ffff, 0x007bffff, 0x00a00002, 0x00c30004, 0x00e70006, 0x010a0008,
	0xffefffff, 0xffccffff, 0xffa8ffff, 0xff85ffff, 0xff600002, 0xff3d0004, 0xff190006, 0xfef60008,
	0x0013ffff, 0x003affff, 0x0061ffff, 0x0088ffff, 0x00b00002, 0x00d70004, 0x00fe0006, 0x01250008,
	0xffedffff, 0xffc6ffff, 0xff9fffff, 0xff78ffff, 0xff500002, 0xff290004, 0xff020006, 0xfedb0008,
	0x0015ffff, 0x0040ffff, 0x006bffff, 0x0096ffff, 0x00c20002, 0x00ed0004, 0x01180006, 0x01430008,
	0xffebffff, 0xffc0ffff, 0xff95ffff, 0xff6affff, 0xff3e0002, 0xff130004, 0xfee80006, 0xfebd0008,
	0x0017ffff, 0x0046ffff, 0x0076ffff, 0x00a5ffff, 0x00d50002, 0x01040004, 0x01340006, 0x01630008,
	0xffe9ffff, 0xffbaffff, 0xff8affff, 0xff5bffff, 0xff2b0002, 0xfefc0004, 0xfecc0006, 0xfe9d0008,
	0x001affff, 0x004effff, 0x0082ffff, 0x00b6ffff, 0x00eb0002, 0x011f0004, 0x01530006, 0x01870008,
	0xffe6ffff, 0xffb2ffff, 0xff7effff, 0xff4affff, 0xff150002, 0xfee10004, 0xfead0006, 0xfe790008,
	0x001cffff, 0x0055ffff, 0x008fffff, 0x00c8ffff, 0x01020002, 0x013b0004, 0x01750006, 0x01ae0008,
	0xffe4ffff, 0xffabffff, 0xff71ffff, 0xff38ffff, 0xfefe0002, 0xfec50004, 0xfe8b0006, 0xfe520008,
	0x001fffff, 0x005effff, 0x009dffff, 0x00dcffff, 0x011c0002, 0x015b0004, 0x019a0006, 0x01d90008,
	0xffe1ffff, 0xffa2ffff, 0xff63ffff, 0xff24ffff, 0xfee40002, 0xfea50004, 0xfe660006, 0xfe270008,
	0x0022ffff, 0x0067ffff, 0x00adffff, 0x00f2ffff, 0x01390002, 0x017e0004, 0x01c40006, 0x02090008,
	0xffdeffff, 0xff99ffff, 0xff53ffff, 0xff0effff, 0xfec70002, 0xfe820004, 0xfe3c0006, 0xfdf70008,
	0x0026ffff, 0x0072ffff, 0x00bfffff, 0x010bffff, 0x01590002, 0x01a50004, 0x01f20006, 0x023e0008,
	0xffdaffff, 0xff8effff, 0xff41ffff, 0xfef5ffff, 0xfea70002, 0xfe5b0004, 0xfe0e0006, 0xfdc20008,
	0x002affff, 0x007effff, 0x00d2ffff, 0x0126ffff, 0x017b0002, 0x01cf0004, 0x02230006, 0x02770008,
	0xffd6ffff, 0xff82ffff, 0xff2effff, 0xfedaffff, 0xfe850002, 0xfe310004, 0xfddd0006, 0xfd890008,
	0x002effff, 0x008affff, 0x00e7ffff, 0x0143ffff, 0x01a10002, 0x01fd0004, 0x025a0006, 0x02b60008,
	0xffd2ffff, 0xff76ffff, 0xff19ffff, 0xfebdffff, 0xfe5f0002, 0xfe030004, 0xfda60006, 0xfd4a0008,
	0x0033ffff, 0x0099ffff, 0x00ffffff, 0x0165ffff, 0x01cb0002, 0x02310004, 0x02970006, 0x02fd0008,
	0xffcdffff, 0xff67ffff, 0xff01ffff, 0xfe9bffff, 0xfe350002, 0xfdcf0004, 0xfd690006, 0xfd030008,
	0x0038ffff, 0x00a8ffff, 0x0118ffff, 0x0188ffff, 0x01f90002, 0x02690004, 0x02d90006, 0x03490008,
	0xffc8ffff, 0xff58ffff, 0xfee8ffff, 0xfe78ffff, 0xfe070002, 0xfd970004, 0xfd270006, 0xfcb70008,
	0x003dffff, 0x00b8ffff, 0x0134ffff, 0x01afffff, 0x022b0002, 0x02a60004, 0x03220006, 0x039d0008,
	0xffc3ffff, 0xff48ffff, 0xfeccffff, 0xfe51ffff, 0xfdd50002, 0xfd5a0004, 0xfcde0006, 0xfc630008,
	0x0044ffff, 0x00ccffff, 0x0154ffff, 0x01dcffff, 0x02640002, 0x02ec0004, 0x03740006, 0x03fc0008,
	0xffbcffff, 0xff34ffff, 0xfeacffff, 0xfe24ffff, 0xfd9c0002, 0xfd140004, 0xfc8c0006, 0xfc040008,
	0x004affff, 0x00dfffff, 0x0175ffff, 0x020affff, 0x02a00002, 0x03350004, 0x03cb0006, 0x04600008,
	0xffb6ffff, 0xff21ffff, 0xfe8bffff, 0xfdf6ffff, 0xfd600002, 0xfccb0004, 0xfc350006, 0xfba00008,
	0x0052ffff, 0x00f6ffff, 0x019bffff, 0x023fffff, 0x02e40002, 0x03880004, 0x042d0006, 0x04d10008,
	0xffaeffff, 0xff0affff, 0xfe65ffff, 0xfdc1ffff, 0xfd1c0002, 0xfc780004, 0xfbd30006, 0xfb2f0008,
	0x005affff, 0x010fffff, 0x01c4ffff, 0x0279ffff, 0x032e0002, 0x03e30004, 0x04980006, 0x054d0008,
	0xffa6ffff, 0xfef1ffff, 0xfe3cffff, 0xfd87ffff, 0xfcd20002, 0xfc1d0004, 0xfb680006, 0xfab30008,
	0x0063ffff, 0x012affff, 0x01f1ffff, 0x02b8ffff, 0x037f0002, 0x04460004, 0x050d0006, 0x05d40008,
	0xff9dffff, 0xfed6ffff, 0xfe0fffff, 0xfd48ffff, 0xfc810002, 0xfbba0004, 0xfaf30006, 0xfa2c0008,
	0x006dffff, 0x0148ffff, 0x0223ffff, 0x02feffff, 0x03d90002, 0x04b40004, 0x058f0006, 0x066a0008,
	0xff93ffff, 0xfeb8ffff, 0xfdddffff, 0xfd02ffff, 0xfc270002, 0xfb4c0004, 0xfa710006, 0xf9960008,
	0x0078ffff, 0x0168ffff, 0x0259ffff, 0x0349ffff, 0x043b0002, 0x052b0004, 0x061c0006, 0x070c0008,
	0xff88ffff, 0xfe98ffff, 0xfda7ffff, 0xfcb7ffff, 0xfbc50002, 0xfad50004, 0xf9e40006, 0xf8f40008,
	0x0084ffff, 0x018dffff, 0x0296ffff, 0x039fffff, 0x04a80002, 0x05b10004, 0x06ba0006, 0x07c30008,
	0xff7cffff, 0xfe73ffff, 0xfd6affff, 0xfc61ffff, 0xfb580002, 0xfa4f0004, 0xf9460006, 0xf83d0008,
	0x0091ffff, 0x01b4ffff, 0x02d8ffff, 0x03fbffff, 0x051f0002, 0x06420004, 0x07660006, 0x08890008,
	0xff6fffff, 0xfe4cffff, 0xfd28ffff, 0xfc05ffff, 0xfae10002, 0xf9be0004, 0xf89a0006, 0xf7770008,
	0x00a0ffff, 0x01e0ffff, 0x0321ffff, 0x0461ffff, 0x05a20002, 0x06e20004, 0x08230006, 0x09630008,
	0xff60ffff, 0xfe20ffff, 0xfcdfffff, 0xfb9fffff, 0xfa5e0002, 0xf91e0004, 0xf7dd0006, 0xf69d0008,
	0x00b0ffff, 0x0210ffff, 0x0371ffff, 0x04d1ffff, 0x06330002, 0x07930004, 0x08f40006, 0x0a540008,
	0xff50ffff, 0xfdf0ffff, 0xfc8fffff, 0xfb2fffff, 0xf9cd0002, 0xf86d0004, 0xf70c0006, 0xf5ac0008,
	0x00c2ffff, 0x0246ffff, 0x03caffff, 0x054effff, 0x06d20002, 0x08560004, 0x09da0006, 0x0b5e0008,
	0xff3effff, 0xfdbaffff, 0xfc36ffff, 0xfab2ffff, 0xf92e0002, 0xf7aa0004, 0xf6260006, 0xf4a20008
};


#if defined(__GNUC__) && defined(USE_INLINE_ASM)
static
inline
Sint32
saturated_addition(Sint32 val, Sint32 add, Sint32 min, Sint32 max)
{
	__asm__(
			"addl		%1, %0			\n\t"
			"cmpl		%2, %0			\n\t"
			"cmovll		%2, %0			\n\t"
			"cmpl		%3, %0			\n\t"
			"cmovgl		%3, %0			\n\t"
			: "+r" (val)
			: "r" (add), "r" (min), "r" (max)
	);
	return val;
}
#else
static
inline
Sint32
saturated_addition(Sint32 val, Sint32 add, Sint32 min, Sint32 max)
{
	val += add;

	if (val < min)	val = min;
	if (val > max)	val = max;

	return val;
}
#endif


/*-----------------------------------------------------------------------------
	[decode]
		ADPCMGR[hꂽf[^𕄍t16rbgI[fBIf[^
	fR[h܂B
-----------------------------------------------------------------------------*/
static
inline
Sint32
decode(
	Uint8	code)
{
	Uint32 data = _LUT[ad_ref_index * 16 + code];
	ad_sample = saturated_addition(ad_sample, (Sint32)data >> 16, -2048, 2047);
	ad_ref_index = saturated_addition(ad_ref_index, (Sint16)(data & 0xffff), 0, 48);
	return ad_sample;
}


/*-----------------------------------------------------------------------------
	[Init]
		
-----------------------------------------------------------------------------*/
BOOL
ADPCM_Init()
{
	_Phase = 0;
	_bPlay = FALSE;

	_CurrentVolume = _InitialVolume;
	_VolumeStep = _InitialVolume / 10;
	_bFadeIn  = FALSE;
	_bFadeOut = FALSE;
	_ClockCount = 0;
	_FadeCycle = 0;

	ad_sample = 0;
	ad_ref_index = 0;

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[Deinit]
		
-----------------------------------------------------------------------------*/
void
ADPCM_Deinit()
{
	_bPlay = FALSE;

	memset(_Ram, 0, sizeof(_Ram));

	_Addr = 0;
	_ReadAddr = 0;
	_WriteAddr = 0;
	_LengthCount = 0;
	_PlayLength = 0;
	_HalfLength = 0;
	_bRepeat = FALSE;
	_SampleFreq = 0;

	_Phase = 0;

	memset(_DecodeBuffer, 0, sizeof(_DecodeBuffer));
	_bLowNibble = TRUE;
	_bInterrupt1 = FALSE;

	// for adpcm play
	_SrcIndex = 0;
	_DstIndex = 0;
	_Sample = 0;
	_Error = 0;
	_nDstSample = 0;
	_nSrcSample = 0;

	_CurrentVolume = _InitialVolume;
	_VolumeStep = 0;
	_bFadeIn  = FALSE;
	_bFadeOut = FALSE;
	_ClockCount = 0;
	_FadeCycle = 0;

	ad_sample = 0;
	ad_ref_index = 0;
}


/*-----------------------------------------------------------------------------
	[Reset]
		
-----------------------------------------------------------------------------*/
void
ADPCM_Reset()
{
	_Addr			= 0;
	_ReadAddr		= 0;
	_WriteAddr		= 0;
	_LengthCount	= 0;
	_bPlay			= FALSE;
	_SampleFreq	= 0;
	_Phase			= 0;
	_bLowNibble	= FALSE;

	// for adpcm play
	_SrcIndex = 0;
	_DstIndex = 0;
	_Sample = 0;
	_Error = 0;
	_nDstSample = 0;
	_nSrcSample = 0;

	_bInterrupt1 = FALSE;

	_bRepeat = FALSE;

	_WrittenSampleCount = 0;

	_CurrentVolume = _InitialVolume;
	_VolumeStep = _InitialVolume / 10;
	_bFadeIn  = FALSE;
	_bFadeOut = FALSE;
	_ClockCount = 0;
	_FadeCycle = 0;

	ad_sample = 0;
	ad_ref_index = 0;
}


/*-----------------------------------------------------------------------------
	[SetNotificationFunction]
		
-----------------------------------------------------------------------------*/
void
ADPCM_SetNotificationFunction(
	void	(*pfnNotification)(Uint32))
{
	_pfnNotification = pfnNotification;
}


/*-----------------------------------------------------------------------------
	[SetAddrLo]
		
-----------------------------------------------------------------------------*/
void
ADPCM_SetAddrLo(
	Uint8		addrLo)
{
	_Addr &= 0xff00;
	_Addr |= addrLo;
}


/*-----------------------------------------------------------------------------
	[SetAddrHi]
		
-----------------------------------------------------------------------------*/
void
ADPCM_SetAddrHi(
	Uint8		addrHi)
{
	_Addr &= 0xff;
	_Addr |= addrHi << 8;
}


/*-----------------------------------------------------------------------------
	[SetReadAddr]
		
-----------------------------------------------------------------------------*/
void
ADPCM_SetReadAddr()
{
	// -1  $180A ̋ǂ݂QsȂ邱Ƃւ̑΍ 
	_ReadAddr = _Addr - 1;
}


/*-----------------------------------------------------------------------------
	[SetWriteAddr]
		
-----------------------------------------------------------------------------*/
void
ADPCM_SetWriteAddr()
{
	_WriteAddr = _Addr;
}


/*-----------------------------------------------------------------------------
	[SetLength]
		
-----------------------------------------------------------------------------*/
void
ADPCM_SetLength()
{
	_LengthCount = _Addr;
//	printf("ADPCM playback length set to %u\n", _LengthCount);
}


/*-----------------------------------------------------------------------------
	[ReadBuffer]
		
-----------------------------------------------------------------------------*/
Uint8
ADPCM_ReadBuffer()
{
	Uint8			temp;

	temp = _ReadBuffer;
	_ReadBuffer = _Ram[_ReadAddr++];
	return temp;
}


/*-----------------------------------------------------------------------------
	[WriteBuffer]
		
-----------------------------------------------------------------------------*/
void
ADPCM_WriteBuffer(
	Uint8		data)
{
	_Ram[_WriteAddr++] = data;
//	if ((_WriteAddr & 0x1fff) == 0)
//	printf("Write addr = %d\n", _WriteAddr);

	_WrittenSampleCount += 2;
}


static
void
predecode_adpcm(
	Sint32*		pDst,
	Uint8*		pSrc,
	Sint32		nSample)
{
	Uint8		data;

	while (nSample > 0)
	{
		data = *pSrc;

		if (_bLowNibble)
		{
			data &= 0x0f;
			++pSrc;
			--nSample;
		}
		else
			data >>= 4;

		_bLowNibble ^= 1;

		*pDst++ = decode(data);
	}
}


/*-----------------------------------------------------------------------------
	[Play]
		
-----------------------------------------------------------------------------*/
void
ADPCM_Play(
	BOOL		bPlay)
{
	if (bPlay)
	{
		_bLowNibble = FALSE;
		ad_sample = 0;
		ad_ref_index = 0;
		predecode_adpcm(_DecodeBuffer, _Ram+_ReadAddr, _LengthCount+1);
		_Phase = 0;
		_HalfLength = 0x10000 << 13; //(_LengthCount+1) << 13;
		_PlayLength = ((_LengthCount+1) << 13) * 2;
		_bInterrupt1 = FALSE;

		_WrittenSampleCount = (_LengthCount + 1) * 2;
		_PlayedSampleCount = 0;
		_PlayedSampleCountDecimal = 0;
	}
	else
	{
		// XgbvƖɂȂ邾ŁA 
		// cl`͍Ō܂ős\B 
		_WrittenSampleCount = 0;
		_PlayedSampleCount = 0;
		_PlayedSampleCountDecimal = 0;
		_bRepeat = FALSE;
		_Phase = 0;
		_HalfLength = 0;
		_PlayLength = 0;
		_bInterrupt1 = FALSE;

		ad_sample = 0;
		ad_ref_index = 0;

		if (_pfnNotification != NULL)
		{
			_pfnNotification(ADPCM_STATE_NORMAL);
		}
	}

	// ͍ŌɂȂƂ 
	_bPlay = bPlay;

//	printf("_bPlay = %d\n", _bPlay);
//	printf("_bPlay = %d _PlayLength = %d _WrittenSampleCount = %d\n", _bPlay, _PlayLength>>13, _WrittenSampleCount);
}


/*-----------------------------------------------------------------------------
	[Repeat]
		n[hEFAŃ[vĐsȂĂ邩ǂ͕sB
-----------------------------------------------------------------------------*/
void
ADPCM_Repeat(
	BOOL		bRepeat)
{
	if (!_bPlay)
		_bRepeat = bRepeat;
}


/*-----------------------------------------------------------------------------
	[Mix]

-----------------------------------------------------------------------------*/
void
ADPCM_Mix(
	Sint32*			pDst,				// o͐obt@ 
	Uint32			sampleRate,			// n[hEFA̍Đ[g 
	Sint32			nSample)			// oTv 
{
	Sint32			i;
	Uint32			dp;					// delta phase in 16.16 fixed
	Sint32			sample;

	if (!_bPlay)
		return;

	// W{I[o[TvOŃ~bNX̂ 
	// ʏ dp WŊ 
	dp = (Uint32)(8192.0 * _SampleFreq / sampleRate / 8.0);

	for (i = 0; i < nSample; i++)	// mixing loop
	{
		sample  = _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;
		sample += _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;
		sample += _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;
		sample += _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;
		sample += _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;
		sample += _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;
		sample += _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;
		sample += _DecodeBuffer[(_Phase >> 13) & 0x1ffff];	_Phase += dp;

		sample /= 8;
		sample *= _CurrentVolume;
		*pDst++ += sample;
		*pDst++ += sample;

		_PlayLength -= 8 * (Sint32)dp;
		_PlayedSampleCountDecimal += 8 * (Sint32)dp;

		while (_PlayedSampleCountDecimal >= (1 << 13))
		{
			++_PlayedSampleCount;
			_PlayedSampleCountDecimal -= (1 << 13);
		}

		if (_PlayedSampleCount >= _WrittenSampleCount)
		{
			_bPlay = FALSE;
			_bRepeat = FALSE;
			_PlayedSampleCount = 0;
			_PlayedSampleCountDecimal = 0;
			_WrittenSampleCount = 0;

			if (_pfnNotification != NULL)
			{
				_pfnNotification(ADPCM_STATE_STOPPED);
			}
			return;
		}

		if ((_PlayLength >> 13) < 0x10000 && _bInterrupt1 == FALSE)
		{
			/* Đc肪obt@̔ɂȂƂ */
//			puts("half");

//			printf("_PlayLength = %d, _WrittenSampleCount = %d\n", _PlayLength>>13, _WrittenSampleCount);

			if (_WrittenSampleCount >= 0x20000)
			{
//				puts("decode");
				predecode_adpcm(_DecodeBuffer, _Ram, 0x10000);
			}

			if (_pfnNotification != NULL)
			{
				_pfnNotification(ADPCM_STATE_HALF_PLAYED);
			}

			_bInterrupt1 = TRUE;
		}
		else if (_PlayLength <= 0)
		{
//			puts("_PlayLength <= 0");
			if (_bRepeat)
			{
				if (_WrittenSampleCount >= 0x20000)
				{
//					puts("decode");
					predecode_adpcm(_DecodeBuffer, _Ram, 0x10000);
					_PlayLength = 0x10000 << 13;
					_bInterrupt1 = FALSE;
				}
				else
				{
					if (_pfnNotification != NULL)
					{
						_pfnNotification(ADPCM_STATE_FULL_PLAYED);
					}
//					ADPCM_Play(FALSE);
//					puts("stop.");
					return;
				}
			}
//			else
//			{
//				if (_pfnNotification != NULL)
//				{
//					_pfnNotification(ADPCM_STATE_FULL_PLAYED);
//				}

//				ADPCM_Play(FALSE);
//				puts("end.");
//				return;
//			}
		}

//		if ((_PlayLength >> 13) % 1000 == 0)
//			printf("PlayLen=%d HalfLen=%d PlayedSample=%d WrittenSample=%d\n", _PlayLength>>13, _HalfLength>>13, _PlayedSampleCount, _WrittenSampleCount);
	}
}


/*-----------------------------------------------------------------------------
	[SetFreq]
		
-----------------------------------------------------------------------------*/
void
ADPCM_SetFreq(
	Uint32		freq)
{
	_SampleFreq = freq;
}


/*-----------------------------------------------------------------------------
	[IsPlaying]
		
-----------------------------------------------------------------------------*/
BOOL
ADPCM_IsPlaying()
{
	if (_bRepeat)
	{
		return (((_PlayedSampleCount>>13) < _WrittenSampleCount) && _bPlay);
	}

	return _bPlay;
}


void
ADPCM_SetVolume(
	Uint32			volume)		// 0 - 65535
{
	if (volume < 0)	volume = 0;
	if (volume > 65535)	volume = 65535;

	double tempVol = (double)ADPCM_MAXVOLUME;
	tempVol = tempVol * volume / 65535.0;
	_InitialVolume = (Sint32)tempVol;
	_CurrentVolume = (Sint32)tempVol;
}


void
ADPCM_FadeOut(
	Sint32		ms)
{
	if (ms == 0)
	{
		_CurrentVolume = 0;
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
	else if (_CurrentVolume > 0)
	{
		_FadeCycle = ((7159090.0 / ((double)_CurrentVolume / (double)_VolumeStep)) * (double)ms) / 1000.0;
		_bFadeOut	= TRUE;
		_bFadeIn	= FALSE;
	}
	else
	{
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
//	printf("fade-out ms = %d\n", ms);
//	printf("fade-out cycle = %d\n", _FadeCycle);
}


void
ADPCM_FadeIn(
	Sint32		ms)
{
	if (ms == 0)
	{
		_CurrentVolume = _InitialVolume;
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
	else if (_InitialVolume - _CurrentVolume > 0)
	{
		_FadeCycle = ((7159090.0 / (((double)_InitialVolume - (double)_CurrentVolume) / (double)_VolumeStep)) * (double)ms) / 1000.0;
		_bFadeOut = FALSE;
		_bFadeIn  = TRUE;
	}
	else
	{
		_bFadeOut = FALSE;
		_bFadeIn  = FALSE;
		_FadeCycle = 0;
	}
//	printf("fade-in ms = %d\n", ms);
//	printf("fade-in cycle = %d\n", _FadeCycle);
}



/*-----------------------------------------------------------------------------
	[AdvanceClock]
		
-----------------------------------------------------------------------------*/
Uint32
ADPCM_AdvanceClock(
	Sint32			cycles)	// 21477270 / 3 = 7159090 [Hz] ̎gŗB 
{
	if (_bFadeOut || _bFadeIn)
	{
		_ClockCount += cycles;

		while (_ClockCount >= _FadeCycle)
		{
			_ClockCount -= _FadeCycle;

			if (_bFadeOut)
			{
				if (_CurrentVolume > 0)
				{
					_CurrentVolume -= _VolumeStep;
					if (_CurrentVolume < 0)
					{
						_CurrentVolume = 0;
						_bFadeOut = FALSE;
						break;
					}
				}
			}
			else if (_bFadeIn)
			{
				if (_CurrentVolume < _InitialVolume)
				{
					_CurrentVolume += _VolumeStep;
					if (_CurrentVolume > _InitialVolume)
					{
						_CurrentVolume = _InitialVolume;
						_bFadeIn = FALSE;
						break;
					}
				}
			}
		}
	}
	return 0;
}


// save variable
#define SAVE_V(V)	if (fwrite(&V, sizeof(V), 1, p) != 1)	return FALSE
#define LOAD_V(V)	if (fread(&V, sizeof(V), 1, p) != 1)	return FALSE
// save array
#define SAVE_A(A)	if (fwrite(A, sizeof(A), 1, p) != 1)	return FALSE
#define LOAD_A(A)	if (fread(A, sizeof(A), 1, p) != 1)		return FALSE
/*-----------------------------------------------------------------------------
	[SaveState]
		Ԃt@Cɕۑ܂B 
-----------------------------------------------------------------------------*/
BOOL
ADPCM_SaveState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	SAVE_A(_Ram);
	SAVE_A(_DecodeBuffer);

	SAVE_V(_Addr);
	SAVE_V(_ReadAddr);
	SAVE_V(_WriteAddr);
	SAVE_V(_LengthCount);
	SAVE_V(_PlayLength);
	SAVE_V(_HalfLength);
	SAVE_V(_bPlay);
	SAVE_V(_bRepeat);
	SAVE_V(_SampleFreq);
	SAVE_V(_Phase);
	SAVE_V(_bLowNibble);
	SAVE_V(_bInterrupt1);

	SAVE_V(_WrittenSampleCount);
	SAVE_V(_PlayedSampleCount);
	SAVE_V(_PlayedSampleCountDecimal);

	SAVE_V(_ReadBuffer);

	SAVE_V(_SrcIndex);
	SAVE_V(_DstIndex);
	SAVE_V(_Sample);
	SAVE_V(_Error);
	SAVE_V(_nDstSample);
	SAVE_V(_nSrcSample);

	SAVE_V(_CurrentVolume);
	SAVE_V(_InitialVolume);
	SAVE_V(_VolumeStep);
	SAVE_V(_bFadeIn);
	SAVE_V(_bFadeOut);
	SAVE_V(_ClockCount);
	SAVE_V(_FadeCycle);

	SAVE_V(ad_sample);
	SAVE_V(ad_ref_index);

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[LoadState]
		Ԃt@Cǂݍ݂܂B 
-----------------------------------------------------------------------------*/
BOOL
ADPCM_LoadState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	LOAD_A(_Ram);
	LOAD_A(_DecodeBuffer);

	LOAD_V(_Addr);
	LOAD_V(_ReadAddr);
	LOAD_V(_WriteAddr);
	LOAD_V(_LengthCount);
	LOAD_V(_PlayLength);
	LOAD_V(_HalfLength);
	LOAD_V(_bPlay);
	LOAD_V(_bRepeat);
	LOAD_V(_SampleFreq);
	LOAD_V(_Phase);
	LOAD_V(_bLowNibble);
	LOAD_V(_bInterrupt1);

	LOAD_V(_WrittenSampleCount);
	LOAD_V(_PlayedSampleCount);
	LOAD_V(_PlayedSampleCountDecimal);

	LOAD_V(_ReadBuffer);

	LOAD_V(_SrcIndex);
	LOAD_V(_DstIndex);
	LOAD_V(_Sample);
	LOAD_V(_Error);
	LOAD_V(_nDstSample);
	LOAD_V(_nSrcSample);

	LOAD_V(_CurrentVolume);
	LOAD_V(_InitialVolume);
	LOAD_V(_VolumeStep);
	LOAD_V(_bFadeIn);
	LOAD_V(_bFadeOut);
	LOAD_V(_ClockCount);
	LOAD_V(_FadeCycle);

	LOAD_V(ad_sample);
	LOAD_V(ad_ref_index);
	
	return TRUE;
}

#undef SAVE_V
#undef SAVE_A
#undef LOAD_V
#undef LOAD_A
