/*-----------------------------------------------------------------------------
	[AudioConfig.c]

		Implements an audio configuration window.

	Copyright (C) 2005 Ki

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
-----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include "WinMain.h"
#include "AudioConfig.h"
#include "APU.h"
#include "ADPCM.h"
#include "CdFader.h"


#define CAPTION 	"AUDIO VOLUME CONFIGURATION"

#define N_LINES 	11
#define LINE_LEN	47


static Uint32		_FontWidth;
static Uint32		_FontHeight;
static const char*	_pCaption = CAPTION;
static HINSTANCE	_hInstance;
static HWND 		_hWnd;


static HWND 		_hApuSB;
static HWND 		_hAdpcmSB;
static HWND 		_hMixMasterSB;
static HWND 		_hMixCdSB;
static HWND 		_hMixWaveSB;

static Uint32*		_pApu = 0;
static Uint32*		_pAdpcm = 0;
static Uint32*		_pMixMaster = 0;
static Uint32*		_pMixCd = 0;
static Uint32*		_pMixWave = 0;


/* audio config thread */
static HANDLE	_hThread = INVALID_HANDLE_VALUE;
static DWORD	_dwThreadID;


/* tHg̍擾 */
static
Uint32
get_font_height(
	HWND			hWnd)
{
	HDC 			hDC;
	HFONT			hFont;
	HFONT			hFontOld;
	TEXTMETRIC		tm;

	hDC 	 = GetDC(hWnd);
	hFont	 = GetStockObject(OEM_FIXED_FONT);	 /* Œsb`tHg */
	hFontOld = SelectObject(hDC, hFont);

	GetTextMetrics(hDC, &tm);

	SelectObject(hDC, hFontOld);
	DeleteObject(hFont);
	ReleaseDC(hWnd, hDC);

	return (Uint32)(tm.tmHeight);
}

/* tHg̉擾 */
static
Uint32
get_font_width(
	HWND			hWnd)
{
	HDC 			hDC;
	HFONT			hFont;
	HFONT			hFontOld;
	TEXTMETRIC		tm;

	hDC 	 = GetDC(hWnd);
	hFont	 = GetStockObject(OEM_FIXED_FONT);	 /* Œsb`tHg */
	hFontOld = SelectObject(hDC, hFont);

	GetTextMetrics(hDC, &tm);

	SelectObject(hDC, hFontOld);
	DeleteObject(hFont);
	ReleaseDC(hWnd, hDC);

	return (Uint32)tm.tmAveCharWidth;
}


static
void
set_window_size(
	HWND			hWnd)
{
	RECT			rc;
	Uint32			wndW = _FontWidth  * LINE_LEN;
	Uint32			wndH = _FontHeight * N_LINES;

	SetRect(&rc, 0, 0, wndW, wndH);
	AdjustWindowRectEx(&rc, GetWindowLong(hWnd, GWL_STYLE),
						GetMenu(hWnd) != NULL, GetWindowLong(hWnd, GWL_EXSTYLE));
	wndW = rc.right - rc.left;
	wndH = rc.bottom - rc.top;
	MoveWindow(hWnd, 50, 50, wndW, wndH, TRUE);
}


static
void
update_scrollbars()
{
	SCROLLINFO		si;

	si.cbSize = sizeof(si);
	si.fMask  = SIF_RANGE | SIF_POS | SIF_PAGE;
	si.nMin   = 0;
	si.nMax   = 65535;
	si.nPage  = 655;

	si.nPos   = *_pApu; 		SetScrollInfo(_hApuSB,		 SB_CTL, &si, TRUE);
	si.nPos   = *_pAdpcm;		SetScrollInfo(_hAdpcmSB,	 SB_CTL, &si, TRUE);
	si.nPos   = *_pMixMaster;	SetScrollInfo(_hMixMasterSB, SB_CTL, &si, TRUE);
	si.nPos   = *_pMixCd;		SetScrollInfo(_hMixCdSB,	 SB_CTL, &si, TRUE);
	si.nPos   = *_pMixWave; 	SetScrollInfo(_hMixWaveSB,	 SB_CTL, &si, TRUE);
}


static
void
update_window(
	HWND			hWnd)
{
	HDC 			hDC;
	HFONT			hFont;
	HFONT			hFontOld;
	PAINTSTRUCT 	ps;
	Uint32			x;
	Uint32			y;
	char			buf[256];
	int 			percent;

	/* `揀 */
	hDC 	 = BeginPaint(hWnd, &ps);
	hFont	 = GetStockObject(OEM_FIXED_FONT);
	hFontOld = SelectObject(hDC, hFont);
	SetBkColor(hDC, RGB(0,0,0));
	SetTextColor(hDC, RGB(224, 224, 224));

	/* ̔wihԂ */
	SetBkMode(hDC, OPAQUE);

	x = _FontWidth;
	y = _FontHeight + 1;
	TextOut(hDC, x, y, "APU MASTER",   strlen("APU MASTER"));	y += 2*_FontHeight;
	TextOut(hDC, x, y, "ADPCM", 	   strlen("ADPCM"));		y += 2*_FontHeight;
	TextOut(hDC, x, y, "MIXER MASTER", strlen("MIXER MASTER")); y += 2*_FontHeight;
	TextOut(hDC, x, y, "MIXER CD",	   strlen("MIXER CD")); 	y += 2*_FontHeight;
	TextOut(hDC, x, y, "MIXER WAVE",   strlen("MIXER WAVE"));

	x = _FontWidth + _FontWidth*14 + _FontWidth*25;
	y = _FontHeight + 1;

	percent = (double)*_pApu * 100.0 / 65535.0 + 1.0;
	if (percent > 100)	percent = 100;
	sprintf(buf, "%3d[%%]", percent);	TextOut(hDC, x, y, buf, strlen(buf));	y += 2*_FontHeight;

	percent = (double)*_pAdpcm * 100.0 / 65535.0 + 1.0;
	if (percent > 100)	percent = 100;
	sprintf(buf, "%3d[%%]", percent);	TextOut(hDC, x, y, buf, strlen(buf));	y += 2*_FontHeight;

	percent = (double)*_pMixMaster * 100.0 / 65535.0 + 1.0;
	if (percent > 100)	percent = 100;
	sprintf(buf, "%3d[%%]", percent);	TextOut(hDC, x, y, buf, strlen(buf));	y += 2*_FontHeight;

	percent = (double)*_pMixCd * 100.0 / 65535.0 + 1.0;
	if (percent > 100)	percent = 100;
	sprintf(buf, "%3d[%%]", percent);	TextOut(hDC, x, y, buf, strlen(buf));	y += 2*_FontHeight;

	percent = (double)*_pMixWave * 100.0 / 65535.0 + 1.0;
	if (percent > 100)	percent = 100;
	sprintf(buf, "%3d[%%]", percent);	TextOut(hDC, x, y, buf, strlen(buf));

	/* I */
	EndPaint(hWnd, &ps);
	SelectObject(hDC, hFontOld);
	DeleteObject(hFont);
	ReleaseDC(hWnd, hDC);
}


static
Uint32
on_scroll(
	HWND		hWnd,
	Uint32		code)
{
	SCROLLINFO		si;

	si.cbSize = sizeof(si);

	switch (code)
	{
	case SB_LINEDOWN:
		si.fMask  = SIF_RANGE | SIF_POS | SIF_PAGE;
		GetScrollInfo(hWnd, SB_CTL, &si);
		si.nPos += si.nPage;
		if (si.nPos > si.nMax)
			si.nPos = si.nMax;
		break;

	case SB_LINEUP:
		si.fMask  = SIF_POS | SIF_PAGE;
		GetScrollInfo(hWnd, SB_CTL, &si);
		if (si.nPos >= si.nPage)
			si.nPos -= si.nPage;
		else
			si.nPos = 0;
		break;

	case SB_PAGEDOWN:
		si.fMask  = SIF_RANGE | SIF_POS | SIF_PAGE;
		GetScrollInfo(hWnd, SB_CTL, &si);
		si.nPos += si.nPage*10;
		if (si.nPos > si.nMax)
			si.nPos = si.nMax;
		break;

	case SB_PAGEUP:
		si.fMask  = SIF_POS | SIF_PAGE;
		GetScrollInfo(hWnd, SB_CTL, &si);
		if (si.nPos >= si.nPage*10)
			si.nPos -= si.nPage*10;
		else
			si.nPos = 0;
		break;

	case SB_THUMBTRACK:
		si.fMask = SIF_POS | SIF_TRACKPOS | SIF_PAGE;
		GetScrollInfo(hWnd, SB_CTL, &si);
		si.nPos = (si.nTrackPos / si.nPage) * si.nPage;
		break;

	case SB_THUMBPOSITION:
	case SB_ENDSCROLL:
		si.fMask = SIF_POS;
		GetScrollInfo(hWnd, SB_CTL, &si);
		break;
	}

	si.cbSize = sizeof(si);
	si.fMask = SIF_POS;
	SetScrollInfo(hWnd, SB_CTL, &si, TRUE);

	return si.nPos;
}


static
LRESULT
CALLBACK
audioconfig_wnd_proc(
	HWND		hWnd,
	UINT		uMsg,
	WPARAM		wParam,
	LPARAM		lParam)
{
	Uint32		val;

	switch(uMsg)
	{
	case WM_CREATE:
		_FontWidth	= get_font_width(hWnd);
		_FontHeight = get_font_height(hWnd);
		set_window_size(hWnd);
		break;

	case WM_PAINT:
		update_scrollbars();
		update_window(hWnd);
		break;

	case WM_KEYDOWN:
		if (wParam == VK_ESCAPE)
			PostMessage(hWnd, WM_CLOSE, 0, 0);
		break;

	case WM_HSCROLL:
		val = on_scroll((HWND)lParam, LOWORD(wParam));

		if ((HWND)lParam == _hApuSB)
		{
			*_pApu		 = val;
			APU_SetVolume(*_pApu);
		}
		else if ((HWND)lParam == _hAdpcmSB)
		{
			*_pAdpcm	 = val;
			ADPCM_SetVolume(*_pAdpcm);
		}
		else if ((HWND)lParam == _hMixMasterSB)
		{
			if (*_pMixMaster != val)
			{
				*_pMixMaster = val;
				CDFADER_SetMasterVolume(*_pMixMaster);
			}
		}
		else if ((HWND)lParam == _hMixCdSB)
		{
			*_pMixCd	 = val;
			CDFADER_SetCdVolume(*_pMixCd);
		}
		else if ((HWND)lParam == _hMixWaveSB)
		{
			*_pMixWave	 = val;
			CDFADER_SetWaveVolume(*_pMixWave);
		}
		InvalidateRect(hWnd, NULL, FALSE);
		break;
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}


static
HWND
create_scrollbar(
	const char* 	pText,
	HWND			hWnd,
	HINSTANCE		hInstance,
	Uint32			x,
	Uint32			y,
	Uint32			w,
	Uint32			h)
{
	return CreateWindow(
							TEXT("SCROLLBAR"),
							TEXT(pText),
							WS_CHILD | WS_VISIBLE,
							x,
							y,
							w,
							h,
							hWnd,
							NULL,
							hInstance,
							NULL);
}



static
DWORD
WINAPI
audio_config_thread(
	LPVOID	param)
{
	WNDCLASS		wc;
	HWND			hWnd;
	MSG 			msg;
	int 			ret;
	Uint32			y;

	ZeroMemory(&wc, sizeof(wc));
	wc.style		 = 0;
	wc.lpfnWndProc	 = audioconfig_wnd_proc;
	wc.cbClsExtra	 = 0;
	wc.cbWndExtra	 = 0;
	wc.hInstance	 = _hInstance;
	wc.hIcon		 = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor		 = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wc.lpszMenuName  = "";
	wc.lpszClassName = _pCaption;

	if (RegisterClass(&wc) == 0)
		return 0;

	hWnd = CreateWindow(
		_pCaption,
		_pCaption,
		WS_MINIMIZEBOX | WS_SYSMENU | WS_CAPTION,
		CW_USEDEFAULT,
		CW_USEDEFAULT,
		0,
		0,
		NULL,
		NULL,
		_hInstance,
		NULL
	);

	if (hWnd == NULL)
		return FALSE;

	_hWnd	   = hWnd;

	y = _FontHeight;
	_hApuSB = create_scrollbar(
					"APU",
					hWnd,
					_hInstance,
					_FontWidth*14,
					y,
					_FontWidth*25,
					_FontHeight+2);

	y += 2*_FontHeight;
	_hAdpcmSB = create_scrollbar(
					"ADPCM",
					hWnd,
					_hInstance,
					_FontWidth*14,
					y,
					_FontWidth*25,
					_FontHeight+2);

	y += 2*_FontHeight;
	_hMixMasterSB = create_scrollbar(
					"MixMaster",
					hWnd,
					_hInstance,
					_FontWidth*14,
					y,
					_FontWidth*25,
					_FontHeight+2);

	y += 2*_FontHeight;
	_hMixCdSB = create_scrollbar(
					"MixCd",
					hWnd,
					_hInstance,
					_FontWidth*14,
					y,
					_FontWidth*25,
					_FontHeight+2);

	y += 2*_FontHeight;
	_hMixWaveSB = create_scrollbar(
					"MixWave",
					hWnd,
					_hInstance,
					_FontWidth*14,
					y,
					_FontWidth*25,
					_FontHeight+2);

	update_scrollbars();

	ShowWindow(hWnd, SW_SHOWNORMAL);
	UpdateWindow(hWnd);

	// bZ[W[v 
	while ((ret = (int)GetMessage(&msg, NULL, 0, 0)))
	{
		if (ret == -1)
		{
			puts("GetMessage Error!");
			break;
		}
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	return 0;
}


BOOL
AUDIOCONFIG_Init(
	HINSTANCE		hInstance,
	Uint32* 		apuVal,
	Uint32* 		adpcmVal,
	Uint32* 		mixMasterVal,
	Uint32* 		mixCdVal,
	Uint32* 		mixWaveVal)
{
	if (_hInstance != NULL)
		AUDIOCONFIG_Deinit();

	_hInstance = hInstance;

	_pApu		= apuVal;
	_pAdpcm 	= adpcmVal;
	_pMixMaster = mixMasterVal;
	_pMixCd 	= mixCdVal;
	_pMixWave	= mixWaveVal;

	// Xbh쐬Ďs	
	_hThread = CreateThread(NULL, 0, audio_config_thread, NULL, 0, &_dwThreadID);
	if (_hThread == NULL)
	{
		return FALSE;
	}

	return TRUE;
}


void
AUDIOCONFIG_Deinit()
{
	if (_hInstance != NULL)
	{
		PostThreadMessage(_dwThreadID, WM_QUIT, 0, 0);

		// Xbh̏I҂ 
		if (_hThread != INVALID_HANDLE_VALUE)
		{
			WaitForSingleObject(_hThread, INFINITE);
			CloseHandle(_hThread);
			_hThread = INVALID_HANDLE_VALUE;
		}
		UnregisterClass(_pCaption, _hInstance);
		_dwThreadID = 0;
		_hInstance = NULL;
		_hWnd = NULL;
	}
}


void
AUDIOCONFIG_Update()
{
	if (_hWnd != NULL && _hInstance != NULL)
	{
		BOOL		bUpdate = FALSE;
		Uint32		val;

		val = CDFADER_GetMasterVolume();
		if (val != *_pMixMaster)
		{
			*_pMixMaster = val;
			bUpdate = TRUE;
		}

		val = CDFADER_GetCdVolume();
		if (val != *_pMixCd)
		{
			*_pMixCd = val;
			bUpdate = TRUE;
		}

		val = CDFADER_GetWaveVolume();
		if (val != *_pMixWave)
		{
			*_pMixWave = val;
			bUpdate = TRUE;
		}

		if (bUpdate)
		{
			update_scrollbars();
			update_window(_hWnd);
			InvalidateRect(_hWnd, NULL, FALSE);
		}
	}
}


